// envelope.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "controller.h"
#include "envelope.h"
#include "envelopeheader.h"
#include "request.h"

Envelope::Envelope() : FrameData(FloatData, DefaultEnvelopeSize, 1, 44100, 1.0) {
	init();
	setMappingLength(DefaultEnvelopeSize);
}

Envelope::Envelope(int len) : FrameData(FloatData, len, 1, 44100, 1.0) {
	init();
	setMappingLength(len);
}

Envelope::~Envelope() {
	delete [] frameLabel;
	delete [] channelLabel;
}

Data *
Envelope::newData() { return new Envelope(this); }

Data *
Envelope::newData(int len) { return new Envelope(this, len); }

Data *
Envelope::clone(const Range &r) { return new Envelope(this, r); }

Data *
Envelope::clone(const Range &r, const Range &c) {
	return new Envelope(this, r, c);
}	

void
Envelope::information(Controller *controller) {
	AlertRequest request("Envelope Information:");
	request.setBell(false);
	char str[128];
	request.appendLabel("------------");
	request.appendLabel("Filename: ", controller->fileName());
	request.appendLabel("Number of Frames: ", toString(str, length()));
	request.appendLabel("File Size (bytes): ", toString(str, sizeInBytes()));
	controller->handleRequest(request);
}

void
Envelope::init() {
	frameLabel = nil;
	channelLabel = nil;
	rangeFactor = 1;
	rangeOffset = 0;
	reset();
}

void
Envelope::reset() {
	currentVal = get(0);
	nextVal = currentVal;
	currentIndex = fraction = oldindex = 0;
}

void
Envelope::checkValues() {
	Super::checkValues();
	reset();
}

// this returns a linearly interpolated value from the array

double
Envelope::next() {
	int index = int(currentIndex);
	if(index > oldindex) {	// only get new values if index moves up
		currentVal = nextVal; 
		oldindex = index;
		// if next value is past end, return currentVal
		if (index+1 < length())
		    nextVal = get(index+1);
		else
		    return currentVal;
	}
	fraction = currentIndex - index;
	currentIndex += increment;
	return (currentVal + fraction * (nextVal - currentVal));
}

// Envelope objects may want to have a custom label for the horizontal scale

void
Envelope::setFrameRangeLabel(const char* label) {
	delete [] frameLabel;
	frameLabel = newstr(label);
}

Range
Envelope::frameRange(RangeUnit) const {
	Range fr = Super::frameRange(FrameUnit);	// always use frame units
	fr *= rangeFactor;
	fr += int(rangeOffset);
	return fr;
}

void
Envelope::setChannelName(const char* name) {
	delete [] channelLabel;
	channelLabel = newstr(name);
}

const char *
Envelope::frameRangeLabel(RangeUnit units) const {
	return frameLabel ? frameLabel : Data::frameRangeLabel(units);
}

const char *
Envelope::channelName(int chan) const {
	static char string[16];
	if(channelLabel)
		return channelLabel;
	else {
		sprintf(string, "Amp, Ch. %d", chan);
		return string;
	}
}

// protected

Header *
Envelope::createHeader(DataFile *, boolean reading) {
	Header *hdr = new EnvelopeHeader(nFrames(), sRate());
	if(!reading) configureHeader(hdr);
	return hdr;
}

void
Envelope::readFromHeader(Header *h) {
	Data::readFromHeader(h);
	EnvelopeHeader *header = (EnvelopeHeader *) h;
	sr = header->sampleRate();
}
