/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure_parser.cpp
 * @brief Parser for figure elements, which are converted from XML to our internal model.
 * 
 */

#include "myx_gc.h"

#include <libxml/xmlmemory.h>
#include <libxml/tree.h>

#include "myx_gc_figure_parser.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_model.h"
#include "myx_gc_const.h"
#include "myx_gc_font_manager.h"

static hash_map<string, TFigureElementLayout> LayoutLookup;
static hash_map<string, TAlignment> AlignmentLookup;

//----------------- CFigureParser --------------------------------------------------------------------------------------

CFigureParser::CFigureParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

CFigureParser::~CFigureParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if the static lookup tables are set up already. If not then it is done.
 */
void CFigureParser::CheckLookupTables(void)
{

  if (LayoutLookup.size() == 0)
  {
    LayoutLookup["row"] = GC_LAYOUT_ROW;
    LayoutLookup["column"] = GC_LAYOUT_COLUMN;
  };
  if (AlignmentLookup.size() == 0)
  {
    AlignmentLookup["left"] = GC_ALIGN_LEFT_TOP;
    AlignmentLookup["center"] = GC_ALIGN_CENTER;
    AlignmentLookup["centered"] = GC_ALIGN_CENTER;
    AlignmentLookup["middle"] = GC_ALIGN_CENTER;
    AlignmentLookup["right"] = GC_ALIGN_RIGHT_BOTTOM;
    AlignmentLookup["top"] = GC_ALIGN_LEFT_TOP; 
    AlignmentLookup["bottom"] = GC_ALIGN_RIGHT_BOTTOM; 
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given XML node and interprets it as a caption definition.
 *
 * @param Node The XML node to parse.
 * @return A new caption element.
 */
CCaptionElementTemplate* CFigureParser::ParseCaption(xmlNodePtr Node)
{
  wstring ID = Utf8ToUtf16(GetStringAttributeDef(Node, "id", "unnamed"));
  wstring Text = Utf8ToUtf16(GetStringAttributeDef(Node, "default", ""));
  float X = GetFloatAttributeDef(Node, "x", 0);
  float Y = GetFloatAttributeDef(Node, "y", 0);
  string FontFamily = GetStringAttributeDef(Node, "font-family", DefaultFontFamily);
  int FontSize = GetIntAttributeDef(Node, "font-size", DefaultFontSize);
  string FontStyle = GetStringAttributeDef(Node, "font-style", DefaultFontStyle);
  string FontWeightString = GetStringAttributeDef(Node, "font-weight", DefaultFontWeight);
  int Weight = ConvertFontWeight(FontWeightString);
  string Attribute = GetStringAttributeDef(Node, "horizontal-alignment", "left");
  TAlignment HorizontalAlignment = AlignmentLookup[Attribute];
  Attribute = GetStringAttributeDef(Node, "vertical-alignment", "top");
  TAlignment VerticalAlignment = AlignmentLookup[Attribute];
  GLubyte Color[4];
  GLubyte* ColorPtr = Color;
  if (ConvertColor(Node, "fill", Color) != 0)
    ColorPtr = NULL;

  TConstraints Constraints;
  Constraints.MaxHeight = GetFloatAttributeDef(Node, "max-height", -1);
  Constraints.MaxWidth = GetFloatAttributeDef(Node, "max-width", -1);
  Constraints.MinHeight = GetFloatAttributeDef(Node, "min-height", -1);
  Constraints.MinWidth = GetFloatAttributeDef(Node, "min-width", -1);

  CCaptionElementTemplate* Result = new CCaptionElementTemplate(ID);
  Result->Initialize(Text, X, Y, FontFamily, FontSize, Weight, FontStyle, HorizontalAlignment, VerticalAlignment, 
    ColorPtr, Constraints);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single element and returns a new figure element instance. Can be called recursively.
 *
 * @param Node The XML node to parse.
 * @return The new figure element instance created out of the element description.
 */
CFigureElementTemplate* CFigureParser::ParseElement(xmlNodePtr Node, CGCModel* Model)
{
  wstring ID = Utf8ToUtf16(GetStringAttributeDef(Node, "id", "unnamed"));
  CFigureElementTemplate* Result = new CFigureElementTemplate(ID);

  string Attribute = GetStringAttributeDef(Node, "layout", DefaultLayout);
  TFigureElementLayout Layout = LayoutLookup[Attribute];

  string CanResize = GetStringAttributeDef(Node, "resize", DefaultResize);
  wstring StyleName;
  if (GetStringAttribute(Node, "template", Attribute))
    StyleName = Utf8ToUtf16(Attribute);

  TConstraints Constraints;
  Constraints.MaxHeight = GetFloatAttributeDef(Node, "max-height", -1);
  Constraints.MaxWidth = GetFloatAttributeDef(Node, "max-width", -1);
  Constraints.MinHeight = GetFloatAttributeDef(Node, "min-height", -1);
  Constraints.MinWidth = GetFloatAttributeDef(Node, "min-width", -1);
  int MinOccurs = GetIntAttributeDef(Node, "min-occurs", 0);
  if (MinOccurs < 0)
    MinOccurs = 0;
  int MaxOccurs = GetIntAttributeDef(Node, "max-occurs", -1);

  Result->Initialize(Layout, CanResize == "true", StyleName, Constraints, MinOccurs, MaxOccurs);

  // Now go through the child nodes. There can either be child elements or a caption node.
  xmlNodePtr Run = Node->children;
  while (Run != NULL)
  {
    if (XML_IS(Run, "caption"))
    {
      CCaptionElementTemplate* ChildElement = ParseCaption(Run);
      Result->SetCaption(ChildElement);
    }
    else
      if (XML_IS(Run, "element"))
      {
        CFigureElementTemplate* ChildElement = ParseElement(Run, Model);
        Result->AddChild(ChildElement);
      };  

    Run = Run->next;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single layout definition and creates a figure template from it.
 *
 * @param Definition The definition to parse.
 * @param The model class that gets the new template.
 */
void CFigureParser::ParseLayoutDefinition(xmlNodePtr Definition, CGCModel* Model)
{
  CheckLookupTables();

  string ID;
  if (GetStringAttribute(Definition, "id", ID))
  {
    CFigureTemplate* Template = Model->Layout(Utf8ToUtf16(ID));
    
    // Find first <element> node. This is our root.
    xmlNodePtr Run = Definition->children;
    while (Run != NULL)
    {
      if (XML_IS(Run, "element"))
        break;
      Run = Run->next;
    };

    if (Run != NULL)
      Template->FContent = ParseElement(Run, Model);
  };
}

//----------------------------------------------------------------------------------------------------------------------

