/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_gl_helper.cpp
 * @brief Helper functions for creating OpenGL data and structures out of XML data.
 */

#include "myx_gc.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "myx_gc_datatypes.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_utilities.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"
#include "myx_gc_const.h"

static TColorMap PredefinedColors;

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a float attribute.
bool GetFloatAttribute(xmlNodePtr Element, const char* Name, float& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = (float) atof((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to retrieve an integer attribute. If it cannot be found a default value will be used instead.
 */
float GetFloatAttributeDef(xmlNodePtr Element, const char* Name, float Default)
{
  float Result;
  if (!GetFloatAttribute(Element, Name, Result))
    Result = Default;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute.
bool GetIntAttribute(xmlNodePtr Element, const char* Name, int& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = atoi((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute. If it cannot be found a default value will be used instead.
int GetIntAttributeDef(xmlNodePtr Element, const char* Name, int Default)
{
  int Result;
  if (!GetIntAttribute(Element, Name, Result))
    Result = Default;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute.
// If the attribute could be found then true is returned and Value is set to the value of the attribute.
// Otherwise false is returned and Value is not touched.
bool GetStringAttribute(xmlNodePtr Element, const char* Name, string& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  else
  {
    Value.clear();
    Value.append((char*) Attribute);
    return true;
  }
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute. If the attribute is empty or cannot be found then a default value is returned.
string GetStringAttributeDef(xmlNodePtr Element, const char* Name, const string Default)
{
  string Result;
  xmlChar* Value = xmlGetProp(Element, (xmlChar*) Name);
  if ((Value == NULL) || (*Value == '\0'))
    Result += Default;
  else
    Result += (char*) Value;

  if (Value != NULL)
    xmlFree(Value);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Converts the two hex digits given by Upper and Lower to an unsigned byte.
unsigned char HexToByte(char Upper, char Lower)
{
  Upper -= '0';
  if (Upper > 9)
    Upper -= 7;
  Lower -= '0';
  if (Lower > 9)
    Lower -= 7;
  return Upper * 16 + Lower;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Searchs the predifined colors and tries to find one with the given name.
 *
 * @param Name The name of the color to find.
 * @param Color [out] The color data if it could be found.
 * @returns true if the color could be found, otherwise false;
 */
bool ColorByName(string Name, GLubyte* Color)
{

  // Fill the color map if not already done.
  if (PredefinedColors.size() == 0)
  {
    for (unsigned int I = 0; I < COLOR_COUNT; I++)
      PredefinedColors[Colors[I].Name] = Colors[I].Color;

#ifdef _WINDOWS
    for (unsigned int I = 0; I < SYS_COLOR_COUNT; I++)
    {
        COLORREF Reference = GetSysColor(SystemColors[I].Value);
        SystemColors[I].Color[0] = GetRValue(Reference);
        SystemColors[I].Color[1] = GetGValue(Reference);
        SystemColors[I].Color[2] = GetBValue(Reference);
        SystemColors[I].Color[3] = 1;
      PredefinedColors[SystemColors[I].Name] = SystemColors[I].Color;
    };
#endif // #ifdef _WINDOWS
  };

  TColorMapIterator Iterator = PredefinedColors.find(Name);
  bool Result = Iterator != PredefinedColors.end();
  if (Result)
  {
    Color[0] = Iterator->second[0];
    Color[1] = Iterator->second[1];
    Color[2] = Iterator->second[2];
    Color[3] = Iterator->second[3];
  };

  return Result;
}


//----------------------------------------------------------------------------------------------------------------------

/**
 * Registers predifined colors.
 *
 * @param ColorMap Colors to add to the predefined color list.
 */
void RegisterSystemColors(const TColorMap &ColorMap)
{
  for (TColorMapIterator Iterator = ColorMap.begin(); 
       Iterator != ColorMap.end(); ++Iterator)
  {
    PredefinedColors[Iterator->first]= Iterator->second;
  }
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Reads attribute Name from Element and tries to treat the string as a color.
 * The allowed syntax for colors is (as given by the SVG specification) either an HTML like
 * value (e.g. #FFFFFF, #FFF) or a function like form (e.g. rgb(100, 255, 255), rgb(10%, 100%, 0%)).
 *
 * @param Element The XML element to parse.
 * @param Name The name of the color attribute.
 * @param Color [out] The converted color.
 * @return 
 *   0 - If a color could be found and converted.
 *   1 - If a color could be found but a conversion error occured.
 *   2 - No color was given.
 *   3 - The special color "none" was found.
 */

int ConvertColor(xmlNodePtr Element, const char* Name, GLubyte* Color)
{
  int Result = 2;
  
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute != NULL)
  {
    Color[0] = 0;
    Color[1] = 0; 
    Color[2] = 0; 
    xmlChar* Head = Attribute;
    xmlChar* Tail;

    // Start by skipping leading white spaces. We have only simple ASCII compatible strings here,
    // so we don't need to care for conversion from UTF-8.
    while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
      Head++;

    if (*Head != '\0')
    {
      if (*Head == '#')
      {
        // HTML color.
        Head++;
        Tail = Head;
        while ((*Tail >= '0' && *Tail <= '9') || (*Tail >= 'a' && *Tail <= 'f') || (*Tail >= 'A' && *Tail <= 'F'))
          Tail++;
        switch (Tail - Head)
        {
          // Support only the two defined cases. Other values are simply ignored.
          case 3:
            {
              Color[0] = HexToByte(*Head, *Head);
              Head++;
              Color[1] = HexToByte(*Head, *Head);
              Head++;
              Color[2] = HexToByte(*Head, *Head);
              Result = 0;

              break;
            };
          case 6:
            {
              Tail = Head + 1;
              Color[0] = HexToByte(*Head, *Tail);
              Head += 2; Tail += 2;
              Color[1] = HexToByte(*Head, *Tail);
              Head += 2; Tail += 2;
              Color[2] = HexToByte(*Head, *Tail);
              Result = 0;

              break;
            };
        }
      }
      else
        if (xmlStrncasecmp(Head, (xmlChar*) "none", 4) == 0)
        {
          // Do not fill at all.
          Result = 3;
        }
        else
        {
          bool isRGB = xmlStrncasecmp(Head, (xmlChar*) "rgb(", 4) == 0;
          if (isRGB)
          {
            // Found a function call like specification. Split the entries and look if they are absolute
            // or percentage values.
            int Index = 0;
            int Value;
            int ComponentCount = 3;
            Head += ComponentCount + 1;
            while (Index < ComponentCount)
            {
              Value = 0;
              
              // Skip leading white spaces.
              while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
                Head++;
              while ((*Head >= '0') && (*Head <= '9'))
                Value = Value * 10 + *Head++ - '0';

              if (Value < 0)
                Value = 0;
              if (*Head == '%')
              {
                if (Value > 100)
                  Value = 100;
                Value = (Value * 255) / 100;
                Head++;
              }
              else
              {
                if (Value > 255)
                  Value = 255;
              };

              Color[Index++] = Value;

              // Skip trailing white spaces.
              while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
                Head++;

              // If there is no comma or closing parenthesis then there is something wrong.
              if (*Head == ')') 
                break;
              if (*Head != ',')
                return 1;

              Head++;
            };
            if ((*Head == ')') && (Index == ComponentCount))
              Result = 0;
            else
              Result = 1;
          }
          else
          {
            // Last chance are color names. Try to find the text in the color constants.
            if (ColorByName((char*) Head, Color))
              Result = 0;
            else
              Result = 1;
          };
        };
    };

    xmlFree(Attribute);
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Parses the given XML node for texture information and creates a new entry in the texture manager.
void ParseTextureEntry(xmlNodePtr XML)
{
  // Collect font information.
  string ID;
  string Filename;
  if (GetStringAttribute(XML, "id", ID))
  {
    string WrapModeH  = GetStringAttributeDef(XML, "wrap-mode-horizontal", DefaultTextureWrapMode);
    string WrapModeV  = GetStringAttributeDef(XML, "wrap-mode-vertical", DefaultTextureWrapMode);
    string MinFilter = GetStringAttributeDef(XML, "min-filter", DefaultTextureMinFilter);
    string MaxFilter = GetStringAttributeDef(XML, "mag-filter", DefaultTextureMagFilter);
    int Dimensions = GetIntAttributeDef(XML, "dimensions", DefaultTextureDimensions);
    string Mode = GetStringAttributeDef(XML, "mode", DefaultTextureMode);

    // See if there are any LOD sub elements and if there is a LOD 0 entry as parameter.
    TLODList LODs;
    if (GetStringAttribute(XML, "location", Filename))
    {
      // The location paremter in the texture tag always gives the top level of detail.
      LODs.push_back(Filename);
    };
    xmlNodePtr Run = XML->children;
    while (Run != NULL)
    {
      if (xmlStrcmp(Run->name, (const xmlChar *) "texture-lod") == 0)
      {
        TLODList::size_type Level = GetIntAttributeDef(Run, "lod", 0);
        if (LODs.size() <= Level)
          LODs.resize(Level + 1);
        LODs[Level] = GetStringAttributeDef(Run, "location", "");
      };
      Run = Run->next;
    };

    TextureManager()->CreateTextureEntry(LODs, ID, WrapModeH, WrapModeV, MinFilter, MaxFilter, Dimensions, Mode);
  };
}

//----------------------------------------------------------------------------------------------------------------------

