/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <myx_public_interface.h>
#include <myx_grt_public_interface.h>
#include <myx_grt_builtin_module_public_interface.h>

#include "myx_grt_mysql.h"

#define STR_FIELD_TO_DICT_ITEM(i, d, k) if (fi[i] > -1) \
  {\
    char *val= myx_convert_dbstr_utf8(mysql, row[fi[i]]);\
    myx_grt_dict_item_set_value_from_string(d, k, val);\
    g_free(val);\
  }

/**
 ****************************************************************************
 * @brief Tests the given connection
 *
 *   Connects to a MySQL server using the given connection to test the 
 * connection
 *
 * @param param the connection information stored in a GRT value
 * @param data buildin module private pointer to the GRT struct
 * 
 * @return Returns the list of schema names in a GRT module function return value 
 *****************************************************************************/
MYX_GRT_VALUE *test_connection(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *value= NULL;
  MYSQL *mysql= grt_mysql_connect(param, &value);
  if (!mysql)
  {
    // if the connection was not successful, return the error GRT value from connect_mysql()
    return value;
  }

  value= myx_grt_value_from_int(1);

  return make_return_value(value);
}

/**
 ****************************************************************************
 * @brief Returns all schemata names of a given database
 *
 *   Connects to a MySQL server using the given connection parameters defined
 * in a GRT value and retrieves a list of schema names.
 *
 * @param param the connection information stored in a GRT value
 * @param data buildin module private pointer to the GRT struct
 * 
 * @return Returns the list of schema names in a GRT module function return value 
 *****************************************************************************/
MYX_GRT_VALUE *get_schemata(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *value= NULL;
  MYSQL *mysql;
  MYSQL_RES *res;
  MYSQL_ROW row;

  mysql= grt_mysql_connect(param, &value);
  if (!mysql)
  {
    // if the connection was not successful, return the error GRT value from connect_mysql()
    return value;
  }

  // execute query that will return all schemata names
  if (!(res= mysql_list_dbs(mysql, (char*) NULL)))
  {
    // return error on failure
    return make_return_value_mysql_error_and_close(mysql, "The schemata names could not be retrieved.", NULL);
  }

  // loop over resultset
  value= myx_grt_list_new(MYX_STRING_VALUE, NULL);
  while ((row= mysql_fetch_row(res)))
  {
    char *schema_name= myx_convert_dbstr_utf8(mysql, row[0]);

    myx_grt_list_item_add_as_string(value, schema_name);
    g_free(schema_name);
  }
  myx_mysql_close(mysql);

  return make_return_value(value);
}

/**
 ****************************************************************************
 * @brief Reverse engineers all database object of the list of given schemata
 *
 *   Connects to a MySQL server using the given connection parameters defined
 * in a GRT value and reverse engineers all database object of the list of given 
 * schemata
 *
 * @param param the connection information stored in a GRT value
 * @param data buildin module private pointer to the GRT struct
 * 
 * @return Returns the list of schema names in a GRT module function return value 
 *****************************************************************************/
MYX_GRT_VALUE *reverse_engineer(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *catalog;
  MYX_GRT_VALUE *schemata;
  MYX_GRT_VALUE *schema_names;
  MYX_GRT_VALUE *error= NULL;
  MYSQL *mysql;
  unsigned int i;
  
  // check number of parameter
  if ((!param) || (myx_grt_value_get_type(param) != MYX_LIST_VALUE) ||
    ((myx_grt_value_get_type(param) == MYX_LIST_VALUE) && (myx_grt_list_item_count(param) != 2)))
  {
    return make_return_value_error("Bad parameters.", "This function takes two parameters, "
      "the connection information in a dictionary and the list of schema names.");
  }

  schema_names= myx_grt_list_item_get(param, 1);
  if (myx_grt_value_get_type(schema_names) != MYX_LIST_VALUE)
  {
    return make_return_value_error("Bad parameters.", "The second parameter has to be a string list of "
      "schema names.");
  }

  mysql= grt_mysql_connect(myx_grt_list_item_get(param, 0), &error);
  if (!mysql)
  {
    // if the connection was not successful, return the error GRT value from connect_mysql()
    return error;
  }

  // build the catalog dict
  catalog= myx_grt_dict_new("db.mysql.Catalog");
  myx_grt_dict_generate_id(catalog);
  myx_grt_dict_item_set_value_from_string(catalog, "name", "dev");

  // create schemata list
  schemata= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Schema");
  myx_grt_dict_item_set_value(catalog, "schemata", schemata);
  myx_grt_value_release(schemata);
  
  // loop over all schemata
  for (i= 0; i < myx_grt_list_item_count(schema_names); i++)
  {
    const char *schema_name= myx_grt_list_item_get_as_string(schema_names, i);

    error= reverse_engineer_schema(mysql, catalog, schema_name);
    if (error)
      break;
  }

  myx_mysql_close(mysql);
  
  // return error or success
  if (error)
  {
    myx_grt_value_release(catalog);
    return error;
  }
  else
    return make_return_value(catalog);
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_schema

static const char * show_create_database_fields[]=
{
  "Database",             // 0
  "Create Database",      // 1
};
static const char ** show_create_database_fields_end=
             show_create_database_fields + sizeof(show_create_database_fields)/sizeof(char*);

static const char *sql_show_create_db= "SHOW CREATE DATABASE %s;";

static const char *regex_character_set= "CHARACTER SET (\\w*)";
static const char *regex_collate= "COLLATE (\\w*)";

static MYX_GRT_VALUE *reverse_engineer_schema(MYSQL *mysql, MYX_GRT_VALUE *catalog, const char *schema_name)
{
  MYX_GRT_VALUE *schamata= myx_grt_dict_item_get_value(catalog, "schemata");
  MYX_GRT_VALUE *schema= myx_grt_dict_new("db.mysql.Schema");
  MYX_GRT_VALUE *schema_assets;
  MYX_GRT_VALUE *error= NULL;
  MYSQL_RES *res;
  MYSQL_ROW row;
  MYSQL_FIELD *fields;
  int num_fields;

  // create schema
  myx_grt_dict_generate_id(schema);
  myx_grt_dict_item_set_value_from_string(schema, "name", schema_name);
  myx_grt_dict_item_set_value_from_string(schema, "owner", myx_grt_dict_item_get_as_string(catalog, "_id"));

  // add table list
  schema_assets= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Table");
  myx_grt_dict_item_set_value(schema, "tables", schema_assets);
  myx_grt_value_release(schema_assets);

  // add view list
  schema_assets= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.View");
  myx_grt_dict_item_set_value(schema, "views", schema_assets);
  myx_grt_value_release(schema_assets);

  // add procedure list
  schema_assets= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Procedure");
  myx_grt_dict_item_set_value(schema, "procedures", schema_assets);
  myx_grt_value_release(schema_assets);


  // add schema to schemata list
  myx_grt_list_item_add(schamata, schema);
  myx_grt_value_release(schema);

  // for MySQL Servers 4.1 or later, fetch character set information
  if (mysql_version_is_later_or_equal_than(mysql, 4, 1))
  {    
    // execute SQL
    if (myx_mysql_query_esc(mysql, sql_show_create_db, schema_name) ||
        !(res= mysql_store_result(mysql)))
    {
      return make_return_value_mysql_error(mysql, "Cannot fetch the charset information.", NULL);
    }

    if (row= mysql_fetch_row(res))
    {
      int fi[3];

      //Depending on the version of the server there might be different columns
      num_fields= mysql_num_fields(res);
      fields= mysql_fetch_fields(res);

      build_field_subst(show_create_database_fields, show_create_database_fields_end,
        fields, fields + num_fields, fi);

      if (fi[1] != -1)
      {
        // use a regex to get charset and collation
        char *row_create_database= myx_convert_dbstr_utf8(mysql, row[fi[1]]);
        char *character_set= get_value_from_text(row_create_database, (int)strlen(row_create_database), regex_character_set);
        char *collate= get_value_from_text(row_create_database, (int)strlen(row_create_database), regex_collate);

        if (character_set)
          myx_grt_dict_item_set_value_from_string(schema, "defaultCharacterSetName", character_set);

        if (collate)
          myx_grt_dict_item_set_value_from_string(schema, "defaultCollationName", collate);

        g_free(row_create_database);
        g_free(character_set);
        g_free(collate);
      }

      mysql_free_result(res);
    }
  }

  // get the tables and views
  error= reverse_engineer_tables_views(mysql, catalog, schema);

  // get the stored procedures
  if (!error)
    error= reverse_engineer_sp(mysql, catalog, schema);

  return error;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_tables_views

static MYX_GRT_VALUE *reverse_engineer_tables_views(MYSQL *mysql, MYX_GRT_VALUE *catalog, MYX_GRT_VALUE *schema)
{
  MYX_GRT_VALUE *schema_tables= myx_grt_dict_item_get_value(schema, "tables");
  MYX_GRT_VALUE *schema_views= myx_grt_dict_item_get_value(schema, "views");
  MYX_GRT_VALUE *error= NULL;
  MYSQL_RES *res;
  MYSQL_ROW row;
  char *old_db;

  // change to the correct schema
  if (!use_schema_store_old_one(mysql, myx_grt_dict_item_get_as_string(schema, "name"), &old_db))
  {
    return make_return_value_mysql_error(mysql, "Could not change to given schema.", myx_grt_dict_item_get_as_string(schema, "name"));
  }

  // get SHOW TABLES; result set
  if (!(res= mysql_list_tables(mysql, (char*) NULL)))
  {
    restore_old_schema(mysql, old_db);
    return make_return_value_mysql_error(mysql, "Could not list schema tables.", NULL);
  }

  while ((row= mysql_fetch_row(res)))
  {
    // check if the object is a table, < 5.0 return only 1 row and >= 5.0 return 2 rows, 
    // the second being the "BASE TABLE" || "VIEW"
    if ((mysql_num_fields(res) == 1) || (strcmp2(row[1], "BASE TABLE") == 0))
    {
      // create table
      MYX_GRT_VALUE *table= myx_grt_dict_new("db.mysql.Table");
      char *table_name= myx_convert_dbstr_utf8(mysql, row[0]);

      myx_grt_dict_generate_id(table);
      myx_grt_dict_item_set_value_from_string(table, "name", table_name);
      myx_grt_dict_item_set_value_from_string(table, "owner", myx_grt_dict_item_get_as_string(schema, "_id"));

      g_free(table_name);

      // add table to schema
      myx_grt_list_item_add(schema_tables, table);
      myx_grt_value_release(table);


      // retrieve table data
      if (error= reverse_engineer_table_data(mysql, catalog, schema, table))
        break;

      // retrieve table columns
      if (error= reverse_engineer_table_columns(mysql, catalog, schema, table))
        break;

      // retrieve table indices
      if (error= reverse_engineer_table_indices(mysql, catalog, schema, table))
        break;

      // retrieve foreign keys
      if (error= reverse_engineer_table_foreign_keys(mysql, catalog, schema, table))
        break;
    }
    else if (strcmp2(row[1], "VIEW") == 0)
    {
      // create view
      MYX_GRT_VALUE *view= myx_grt_dict_new("db.mysql.View");
      char *view_name= myx_convert_dbstr_utf8(mysql, row[0]);

      myx_grt_dict_generate_id(view);
      myx_grt_dict_item_set_value_from_string(view, "name", view_name);
      myx_grt_dict_item_set_value_from_string(view, "owner", myx_grt_dict_item_get_as_string(schema, "_id"));

      g_free(view_name);

      // add view to schema
      myx_grt_list_item_add(schema_views, view);
      myx_grt_value_release(view);

      // retrieve view data
      if (error= reverse_engineer_view_data(mysql, catalog, schema, view))
        break;
    }
  }

  mysql_free_result(res);

  restore_old_schema(mysql, old_db);

  return error;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_table_data

static const char * show_table_status_fields[]=
{
  "Name",              // 0
  "Engine",            // 1
  "Type",              // 2
  "Version",           // 3
  "Row_format",        // 4
  "Rows",              // 5
  "Avg_row_length",    // 6
  "Data_length",       // 7
  "Max_data_length",   // 8
  "Index_length",      // 9
  "Data_free",         // 10
  "Auto_increment",    // 11
  "Create_time",       // 12
  "Update_time",       // 13
  "Check_time",        // 14
  "Collation",         // 15
  "Checksum",          // 16
  "Create_options",    // 17
  "Comment"            // 18
};
static const char ** show_table_status_fields_end=
             show_table_status_fields + sizeof(show_table_status_fields)/sizeof(char*);

static const char *sql_show_table_create= "SHOW CREATE TABLE %s;";
static const char *sql_show_table_status= "SHOW TABLE STATUS LIKE '%s';";

static MYX_GRT_VALUE *reverse_engineer_table_data(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                  MYX_GRT_VALUE *schema, MYX_GRT_VALUE *table)
{

  char *sql= myx_mysql_esc(mysql, sql_show_table_create, myx_grt_dict_item_get_as_string(table, "name"));
  MYX_GRT_VALUE *error;
  MYSQL_RES *res;
  MYSQL_ROW row;
  char *create_table_statement= NULL;
  char *tbl_options= NULL;

  // get create table statement
  if (error= grt_mysql_execute_and_free(mysql, &res, sql, "Could not execute SHOW CREATE TABLE statement."))
    return error;

  if (row= mysql_fetch_row(res))
  {
    create_table_statement= myx_convert_dbstr_utf8(mysql, row[1]);
    myx_grt_dict_item_set_value_from_string(table, "sql", create_table_statement);
  }

  mysql_free_result(res);

  // get table status
  sql= g_strdup_printf(sql_show_table_status, myx_grt_dict_item_get_as_string(table, "name"));

  if (error= grt_mysql_execute_and_free(mysql, &res, sql, "Could not execute SHOW TABLE STATUS statement."))
    return error;

  if (row= mysql_fetch_row(res))
  {
    int fi[19];
    MYSQL_FIELD *fields;
    int num_fields;
    unsigned long *lengths;

    //Depending on the version of the server there might be different columns
    num_fields= mysql_num_fields(res);
    fields= mysql_fetch_fields(res);
    lengths = mysql_fetch_lengths(res);

    build_field_subst(show_table_status_fields, show_table_status_fields_end,
      fields, fields + num_fields, fi);

    // tableEngine
    if ((fi[1] > -1) || (fi[2] > -1))
    {
      char *val= (fi[1] > -1) ? myx_convert_dbstr_utf8(mysql, row[fi[1]]) : myx_convert_dbstr_utf8(mysql, row[fi[2]]);
      myx_grt_dict_item_set_value_from_string(table, "tableEngine", val);
      g_free(val);
    }

    // rowFormat
    STR_FIELD_TO_DICT_ITEM(4, table, "rowFormat");

    // nextAutoInc
    STR_FIELD_TO_DICT_ITEM(11, table, "nextAutoInc");
    
    // defaultCollationName
    STR_FIELD_TO_DICT_ITEM(15, table, "defaultCollationName");

    // comment
    STR_FIELD_TO_DICT_ITEM(18, table, "comment");

    // table_options
    if (fi[17] > -1)
      tbl_options= myx_convert_dbstr_utf8(mysql, row[fi[17]]);
  }

  mysql_free_result(res);

#define STR_REGEX_TO_DICT_TIME(txt, txtlen, regex, d, k) {\
      char *tmp= get_value_from_text(txt, txtlen, regex);\
      if (tmp)\
        myx_grt_dict_item_set_value_from_string(d, k, tmp);\
      g_free(tmp);\
    }

  if (tbl_options && tbl_options[0])
  {
    unsigned int len= (unsigned int)strlen(tbl_options);
    char *tmp;

    // delayKeyWrite
    tmp= get_value_from_text(tbl_options, len, "DELAY_KEY_WRITE\\s*=\\s*(0|1)");
    if (tmp)
      myx_grt_dict_item_set_value_from_int(table, "delayKeyWrite", (strcmp2(tmp, "1") == 0) ? 1 : 0);
    g_free(tmp);

    // packKeys
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "PACK_KEYS\\s*=\\s*(\\w+)", table, "packKeys");

    // avgRowLength
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "AVG_ROW_LENGTH\\s*=\\s*(\\w+)", table, "avgRowLength");

    // minRows
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "MIN_ROWS\\s*=\\s*(\\w+)", table, "minRows");

    // maxRows
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "MAX_ROWS\\s*=\\s*(\\w+)", table, "maxRows");

    // checksum
    tmp= get_value_from_text(tbl_options, len, "CHECKSUM=\\s*=\\s*(1)");
    if (tmp)
      myx_grt_dict_item_set_value_from_int(table, "checksum", (strcmp2(tmp, "1") == 0) ? 1 : 0);
    g_free(tmp);

    g_free(tbl_options);
  }

  // process CREATE TABLE statement
  if(create_table_statement)
  {
    unsigned int len= (unsigned int)strlen(tbl_options);

    // mask out strings
    tbl_options= mask_out_string(g_strdup(create_table_statement), '\'', '\'', 'x');
    

    // mergeUnion
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "UNION\\s*=\\s*\\((.+)\\)", table, "mergeUnion");

    // mergeInsert
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "INSERT_METHOD\\s*=\\s*(\\w+)", table, "mergeInsert");

    // tableDataDir
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "DATA DIRECTORY\\s*=\\s*'(.+)'", table, "tableDataDir");

    // tableIndexDir
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "INDEX DIRECTORY\\s*=\\s*'(.+)'", table, "tableIndexDir");
    
    // raidType
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "RAID_TYPE\\s*=\\s*(\\w+)", table, "raidType");

    // raidChunks
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "RAID_CHUNKS\\s*=\\s*(\\w+)", table, "raidChunks");

    // raidChunkSize
    STR_REGEX_TO_DICT_TIME(tbl_options, len, "RAID_CHUNKSIZE\\s*=\\s*(\\w+)", table, "raidChunkSize");

    g_free(tbl_options);
    g_free(create_table_statement);
  }

#undef STR_REGEX_TO_DICT_TIME

  return NULL;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_table_columns

static const char * show_table_columns_fields[]=
{
  "Field",       // 0
  "Type",        // 1
  "Collation",   // 2
  "Null",        // 3
  "Key",         // 4
  "Default",     // 5
  "Extra",       // 6
  "Privileges",  // 7
  "Comment"      // 8
};
static const char ** show_table_columns_fields_end=
             show_table_columns_fields + sizeof(show_table_columns_fields)/sizeof(char*);

static const char *sql_show_table_columns= "SHOW FULL COLUMNS FROM %s;";

static MYX_GRT_VALUE *reverse_engineer_table_columns(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                     MYX_GRT_VALUE *schema, MYX_GRT_VALUE *table)
{
  MYX_GRT_VALUE *columns= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Column");
  MYX_GRT_VALUE *error;
  char *sql= myx_mysql_esc(mysql, sql_show_table_columns, myx_grt_dict_item_get_as_string(table, "name"));
  MYSQL_RES *res;
  MYSQL_ROW row;

  // add column list to table
  myx_grt_dict_item_set_value(table, "columns", columns);
  myx_grt_value_release(columns);

  // execute SQL
  if (error= grt_mysql_execute_and_free(mysql, &res, sql, "Could not get table columns."))
    return error;

  while (row= mysql_fetch_row(res))
  {
    int fi[9];
    MYX_GRT_VALUE *column= myx_grt_dict_new("db.mysql.Column");
    char *val;
    MYSQL_FIELD *fields;
    int num_fields;
    unsigned long *lengths;

    myx_grt_dict_generate_id(column);

    //Depending on the version of the server there might be different columns
    num_fields= mysql_num_fields(res);
    fields= mysql_fetch_fields(res);
    lengths = mysql_fetch_lengths(res);

    build_field_subst(show_table_columns_fields, show_table_columns_fields_end,
      fields, fields + num_fields, fi);

    // column name
    val= myx_convert_dbstr_utf8(mysql, row[fi[0]]);
    myx_grt_dict_item_set_value_from_string(column, "name", val);
    g_free(val);

    // column datatype definition
    val= myx_convert_dbstr_utf8(mysql, row[fi[1]]);
    myx_grt_dict_item_set_value_from_string(column, "datatypeName", val);
    g_free(val);

    // collation
    if ((fi[2] > -1) && (row[fi[2]]) && (strcmp2(row[fi[2]], "NULL") != 0))
    {
      val= myx_convert_dbstr_utf8(mysql, row[fi[2]]);
      myx_grt_dict_item_set_value_from_string(column, "collationName", val);
      g_free(val);
    }

    // isNullable
    myx_grt_dict_item_set_value_from_int(column, "isNullable", strcmp2(row[fi[3]], "YES") ? 1 : 0);

    // defaultValue
    if (row[fi[5]])
    {
      val= myx_convert_dbstr_utf8(mysql, row[fi[5]]);
      myx_grt_dict_item_set_value_from_string(column, "defaultValue", val);
      g_free(val);
    }

    // autoIncrement, check if auto_increment is in the extra field
    if ((fi[6] > -1) && (row[fi[6]]))
      myx_grt_dict_item_set_value_from_int(column, "autoIncrement", 
        (g_strstr_len(row[fi[6]], lengths[fi[6]], "auto_increment")) ? 1 : 0);

    // comment
    if ((fi[8] > -1) && (row[fi[8]]))
    {
      val= myx_convert_dbstr_utf8(mysql, row[fi[8]]);
      myx_grt_dict_item_set_value_from_string(column, "comment", val);
      g_free(val);
    }

    // add the column to the table
    myx_grt_list_item_add(columns, column);
    myx_grt_value_release(column);

    // primary key
    if ((row[fi[4]]) && (strcmp2(row[fi[4]], "PRI")))
    {
      MYX_GRT_VALUE *pk= myx_grt_dict_item_get_value(table, "primaryKey");
      MYX_GRT_VALUE *columns;

      if (!pk)
      {
        char *pk_name= g_strdup_printf("%sPrimaryKey", myx_grt_dict_item_get_as_string(table, "name"));

        pk= myx_grt_dict_new("db.mysql.PrimaryKey");
        myx_grt_dict_generate_id(pk);
        myx_grt_dict_item_set_value_from_string(pk, "name", pk_name);
        myx_grt_dict_item_set_value_from_string(pk, "owner", myx_grt_dict_item_get_as_string(table, "_id"));

        g_free(pk_name);

        columns= myx_grt_list_new(MYX_STRING_VALUE, "db.Column");
        myx_grt_dict_item_set_value(pk, "columns", columns);
        myx_grt_value_release(columns);

        myx_grt_dict_item_set_value(table, "primaryKey", pk);
        myx_grt_value_release(pk);
      }
      else
        columns= myx_grt_dict_item_get_value(pk, "columns");

      myx_grt_list_item_add_as_string(columns, myx_grt_dict_item_get_as_string(column, "_id"));
    }
  }

  mysql_free_result(res);

  return NULL;
}


// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_table_indices

static const char * show_table_indices_fields[]=
{
  "Table",        // 0
  "Non_unique",   // 1
  "Key_name",     // 2
  "Seq_in_index", // 3
  "Column_name",  // 4
  "Collation",    // 5
  "Cardinality",  // 6
  "Sub_part",     // 7
  "Packed",       // 8
  "Null",         // 9
  "Index_type",   // 10
  "Comment"       // 11
};
static const char ** show_table_indices_fields_end=
             show_table_indices_fields + sizeof(show_table_indices_fields)/sizeof(char*);

static const char *sql_show_table_indices= "SHOW INDEX FROM %s;";

static MYX_GRT_VALUE *reverse_engineer_table_indices(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                     MYX_GRT_VALUE *schema, MYX_GRT_VALUE *table)
{
  MYX_GRT_VALUE *indices= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Index");
  MYX_GRT_VALUE *index= NULL;
  MYX_GRT_VALUE *index_columns;
  MYX_GRT_VALUE *error;
  char *sql= myx_mysql_esc(mysql, sql_show_table_indices, myx_grt_dict_item_get_as_string(table, "name"));
  MYSQL_RES *res;
  MYSQL_ROW row;
  char *current_index_name= NULL;

  // add index list to table
  myx_grt_dict_item_set_value(table, "indices", indices);
  myx_grt_value_release(indices);

  // execute SQL
  if (error= grt_mysql_execute_and_free(mysql, &res, sql, "Could not get table indices."))
    return error;

  while (row= mysql_fetch_row(res))
  {
    MYX_GRT_VALUE *index_column;
    int fi[12];
    MYSQL_FIELD *fields;
    int num_fields;
    unsigned long *lengths;

    //Depending on the version of the server there might be different columns
    num_fields= mysql_num_fields(res);
    fields= mysql_fetch_fields(res);
    lengths = mysql_fetch_lengths(res);

    build_field_subst(show_table_indices_fields, show_table_indices_fields_end,
      fields, fields + num_fields, fi);

    // check if this is a new index (and not just the 2nd or 3rd column of the last index)
    // if it is a new index, create the GRT value for it
    if (strcmp2(current_index_name, row[fi[2]]) != 0)
    {
      char *index_type= row[fi[10]];

      current_index_name= row[fi[2]];

      index= myx_grt_dict_new("db.mysql.Index");
      myx_grt_dict_generate_id(index);

      // name
      STR_FIELD_TO_DICT_ITEM(2, index, "name");

      // owner
      myx_grt_dict_item_set_value_from_string(index, "owner", myx_grt_dict_item_get_as_string(table, "_id"));


      // comment
      STR_FIELD_TO_DICT_ITEM(11, index, "comment");

      // unique
      if (fi[1] > -1)
        myx_grt_dict_item_set_value_from_int(index, "unique", strcmp2(row[fi[2]], "1") ? 1 : 0);

      // indexKind and indexType
      if (strcmp2(row[fi[2]], "PRIMARY") == 0)
        myx_grt_dict_item_set_value_from_string(index, "indexKind", "PRIMARY");
      else
        myx_grt_dict_item_set_value_from_string(index, "indexKind", "");

      if(strcmp2(index_type, "BTREE") == 0)
      {
        myx_grt_dict_item_set_value_from_string(index, "indexType", "BTREE");
      }
      else if(strcmp2(index_type, "RTREE") == 0)
      {
        myx_grt_dict_item_set_value_from_string(index, "indexType", "RTREE");
      }
      else if(strcmp2(index_type, "HASH") == 0)
      {
        myx_grt_dict_item_set_value_from_string(index, "indexType", "HASH");
      }
      else if(strcmp2(index_type, "FULLTEXT") == 0)
      {
        myx_grt_dict_item_set_value_from_string(index, "indexKind", "");
        myx_grt_dict_item_set_value_from_string(index, "indexType", "FULLTEXT");
      }
      else if(strcmp2(index_type, "SPATIAL") == 0)
      {
        myx_grt_dict_item_set_value_from_string(index, "indexKind", "");
        myx_grt_dict_item_set_value_from_string(index, "indexType", "SPATIAL");
      }

      if ( (myx_grt_dict_item_get_as_int(index, "unique") == 1) && 
        (strcmp2(row[fi[2]], "PRIMARY") != 0) )
      {
        myx_grt_dict_item_set_value_from_string(index, "indexKind", "UNIQUE");
      }

      // create index column list
      index_columns= myx_grt_list_new(MYX_DICT_VALUE, "db.IndexColumn");

      // add index column list to index
      myx_grt_dict_item_set_value(index, "columns", index_columns);
      myx_grt_value_release(index_columns);

      // add index to the table index list
      myx_grt_list_item_add(indices, index);
      myx_grt_value_release(index);
    }
    
    // create index column
    index_column= myx_grt_dict_new("db.mysql.IndexColumn");
    myx_grt_dict_generate_id(index_column);

    // name
    STR_FIELD_TO_DICT_ITEM(4, index_column, "name");

    // owner
    myx_grt_dict_item_set_value_from_string(index_column, "owner", myx_grt_dict_item_get_as_string(index, "_id"));

    // columnLength
    if ((fi[7] > -1) && (row[fi[7]]) && (row[fi[7]][0]))
      myx_grt_dict_item_set_value_from_int(index_column, "columnLength", atoi(row[fi[7]]));

    // descend
    if (fi[5] > -1)
      myx_grt_dict_item_set_value_from_int(index_column, "descend", (strcmp2(row[fi[5]], "A") == 0) ? 0 : 1);
    else
      myx_grt_dict_item_set_value_from_int(index_column, "descend", 0);

    // add index column to index
    myx_grt_list_item_add(index_columns, index_column);
    myx_grt_value_release(index_column);
  }

  mysql_free_result(res);

  return NULL;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_table_foreign_keys

#define QUOTED_ID "(\\w+|`.+?`|\".+?\")"
#define QUOTED_ID_ "(?:\\w+|`.+?`|\".+?\")"

static const char *regex_foreign_keys= ".*?CONSTRAINT "QUOTED_ID
    " FOREIGN KEY \\(([^)]+)\\) REFERENCES ((?:"QUOTED_ID_"."QUOTED_ID_")|"QUOTED_ID_")"
    " \\(([^)]+)\\)((?: ON \\w+ (?:NO\\sACTION|SET\\sNULL|\\w+))*)";

static MYX_GRT_VALUE *reverse_engineer_table_foreign_keys(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                          MYX_GRT_VALUE *schema, MYX_GRT_VALUE *table)
{
  MYX_GRT_VALUE *fks= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.ForeignKey");
  MYX_GRT_VALUE *table_columns= myx_grt_dict_item_get_value(table, "columns");
  pcre *pcre_exp;
  const char *error_str;
  int error_offset;
  int o_vector[256];
  int rc;
  int offset;
  static struct {
    char *kw;
    MYX_DBM_FK_ACTION action;
  } fk_actions[]= {
    {"CASCADE", MYX_DBM_FA_CASCADE},
    {"SET NULL", MYX_DBM_FA_SET_NULL},
    {"NO ACTION", MYX_DBM_FA_NO_ACTION},
    {"RESTRICT", MYX_DBM_FA_RESTRICT},
    {NULL, MYX_DBM_FA_NO_ACTION}
  };
  char *table_def= mask_out_string(g_strdup(myx_grt_dict_item_get_as_string(table, "sql")), '\'', '\'', 'x');

  myx_grt_dict_item_set_value(table, "foreignKeys", fks);
  myx_grt_value_release(fks);

  pcre_exp= pcre_compile(regex_foreign_keys, PCRE_CASELESS|PCRE_UTF8|PCRE_DOTALL, &error_str, &error_offset, NULL);
  
  if (pcre_exp)
  {
    offset= 0;
    while ((rc= pcre_exec(pcre_exp, NULL, table_def, (int)strlen(table_def),
                       offset, 0, o_vector, sizeof(o_vector)/sizeof(int))) > 0)
    {
      MYX_GRT_VALUE *fk= myx_grt_dict_new("db.mysql.ForeignKey");
      MYX_GRT_VALUE *fk_columns= myx_grt_list_new(MYX_STRING_VALUE, "db.mysql.Column");
      MYX_GRT_VALUE *fk_refered_column_names= myx_grt_list_new(MYX_STRING_VALUE, NULL);
      MYX_GRT_VALUE *fk_refered_columns= myx_grt_list_new(MYX_STRING_VALUE, "db.mysql.Column");
      const char *value;
      char **src;
      char **dst;
      int i;
      char *tmp;
      char *reference_schema_name;
      char *reference_table_name;

      // _id
      myx_grt_dict_generate_id(fk);

      // name
      pcre_get_substring(table_def, o_vector, rc, 1, &value);
      tmp= unquote_identifier(g_strdup(value));
      myx_grt_dict_item_set_value_from_string(fk, "name", tmp);
      g_free(tmp);
      pcre_free_substring(value);

      // add column list
      myx_grt_dict_item_set_value(fk, "columns", fk_columns);
      myx_grt_value_release(fk_columns);

      // add refered column name list
      myx_grt_dict_item_set_value(fk, "referedColumnNames", fk_refered_column_names);
      myx_grt_value_release(fk_refered_column_names);

      // add refered column list
      myx_grt_dict_item_set_value(fk, "referedColumns", fk_refered_columns);
      myx_grt_value_release(fk_refered_columns);

      // add FK to the table's foreign key list
      myx_grt_list_item_add(fks, fk);
      myx_grt_value_release(fk);


      // source columns
      pcre_get_substring(table_def, o_vector, rc, 2, &value);
      src= g_strsplit(value, ", ", 0);
      pcre_free_substring(value);

      // referedTableSchemaName and referedTableName
      pcre_get_substring(table_def, o_vector, rc, 3, &value);
      split_schema_table(g_strdup(value),
                         &reference_schema_name,
                         &reference_table_name);

      myx_grt_dict_item_set_value_from_string(fk, "referedTableSchemaName", reference_schema_name);
      myx_grt_dict_item_set_value_from_string(fk, "referedTableName", reference_table_name);

      g_free(reference_schema_name);
      g_free(reference_table_name);
      
      pcre_free_substring(value);


      // target columns
      pcre_get_substring(table_def, o_vector, rc, 4, &value);
      dst= g_strsplit(value, ", ", 0);
      pcre_free_substring(value);
      
      // check column count
      for (i= 0; src[i] && dst[i]; i++);
      g_assert(!src[i] && !dst[i]);
      
      for (i= 0; src[i]; i++)
      {
        unsigned int j;

        unquote_identifier(src[i]);

        // look for the _id of the source column
        for (j= 0; j < myx_grt_list_item_count(table_columns); j++)
        {
          MYX_GRT_VALUE *table_column= myx_grt_list_item_get(table_columns, j);

          if (strcmp2(myx_grt_dict_item_get_as_string(table_column, "name"), src[i]) == 0)
          {
            myx_grt_list_item_add_as_string(fk_columns, myx_grt_dict_item_get_as_string(table_column, "_id"));
            break;
          }
        }

        // add target column
        myx_grt_list_item_add_as_string(fk_refered_column_names, unquote_identifier(dst[i]));
      }
      g_strfreev(src);
      g_strfreev(dst);

      myx_grt_dict_item_set_value_from_string(fk, "deleteRule", "RESTRICT");
      myx_grt_dict_item_set_value_from_string(fk, "updateRule", "RESTRICT");

      // actions
      pcre_get_substring(table_def, o_vector, rc, 5, &value);
      {
        char *ptr;
        
        ptr= strstr(value, "ON UPDATE ");
        if (ptr)
        {
          ptr += sizeof("ON UPDATE ")-1;
          for (i= 0; fk_actions[i].kw; i++)
          {
            if (strncmp(ptr, fk_actions[i].kw, strlen(fk_actions[i].kw))==0)
            {
              myx_grt_dict_item_set_value_from_string(fk, "updateRule", fk_actions[i].kw);
              break;
            }
          }
        }
        ptr= strstr(value, "ON DELETE ");
        if (ptr)
        {
          ptr += sizeof("ON DELETE ")-1;
          for (i= 0; fk_actions[i].kw; i++)
          {
            if (strncmp(ptr, fk_actions[i].kw, strlen(fk_actions[i].kw))==0)
            {
              myx_grt_dict_item_set_value_from_string(fk, "deleteRule", fk_actions[i].kw);
              break;
            }
          }
        }
      }
     
      pcre_free_substring(value);

      //Move offset
      offset= o_vector[1];
    }
    pcre_free(pcre_exp);
  }

  return NULL;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_view_data

static const char *sql_show_view_create= "SHOW CREATE VIEW %s;";

static const char *regex_with_check_option= "WITH\\s*(\\w*)\\s*CHECK OPTION";

static MYX_GRT_VALUE *reverse_engineer_view_data(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                 MYX_GRT_VALUE *schema, MYX_GRT_VALUE *view)
{
  char *sql= myx_mysql_esc(mysql, sql_show_view_create, myx_grt_dict_item_get_as_string(view, "name"));
  MYSQL_RES *res;
  MYSQL_ROW row;

  // execute SQL
  if (myx_mysql_query(mysql, sql) ||
    !(res= mysql_store_result(mysql)))
  {
    MYX_GRT_VALUE *error= make_return_value_mysql_error(mysql, "Could not execute SHOW CREATE VIEW statement.", sql);

    g_free(sql);

    return error;
  }
  g_free(sql);

  if (row= mysql_fetch_row(res))
  {
    // set view data
    char *view_sql= myx_convert_dbstr_utf8(mysql, row[1]);
    char *check_option= get_value_from_text(view_sql, (int)strlen(view_sql), regex_with_check_option);

    myx_grt_dict_item_set_value_from_string(view, "queryExpression", view_sql);

    // check if the sql contains a check condition
    myx_grt_dict_item_set_value_from_int(view, "withCheckCondition", (check_option) ? 1 : 0);

    g_free(view_sql);
    g_free(check_option);
  }

  mysql_free_result(res);

  return NULL;
}

// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_sp

static const char * sps_names[]=
{
  "Db",              // 0
  "Name",            // 1
  "Type",            // 2
  "Definer",         // 3
  "Modified",        // 4
  "Created",         // 5
  "Security_type",   // 6
  "Comment",         // 7
};
static const char ** sps_names_end=
                 sps_names + sizeof(sps_names)/sizeof(char*);


static MYX_GRT_VALUE *reverse_engineer_sp(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                                 MYX_GRT_VALUE *schema)
{
  MYX_GRT_VALUE *sps= myx_grt_dict_item_get_value(schema, "procedures");
  MYSQL_RES *res;
  MYSQL_ROW row;
  MYSQL_FIELD *fields;
  unsigned int num_fields;
  int q;
  static const char *queries[]={
    "SHOW PROCEDURE STATUS;",
    "SHOW FUNCTION STATUS;"
  };

  // query procedures and functions
  for (q = 0; q < 2; q++)
  {
    if (myx_mysql_query(mysql, queries[q]) ||
        !(res= mysql_store_result(mysql)))
    {
      return make_return_value_mysql_error(mysql, "Could not get stored procedures.", queries[q]);
    }
    else
    {
      int fi[8];

      //Depending on the version of the server there might be different columns
      num_fields= mysql_num_fields(res);
      fields= mysql_fetch_fields(res);
      
      build_field_subst(sps_names, sps_names_end,
                        fields, fields+num_fields, fi);
      
      while ((row= mysql_fetch_row(res)))
      {
        MYX_GRT_VALUE *sp;

        //Ignore SPs from wrong DB
        if (fi[0] == -1 ? 1 : (strcmp2(row[fi[0]], myx_grt_dict_item_get_as_string(schema, "name")) != 0))
          continue;

        sp= myx_grt_dict_new("db.mysql.Procedure");

        // _id
        myx_grt_dict_generate_id(sp);

        // name
        STR_FIELD_TO_DICT_ITEM(1, sp, "name");

        // owner
        myx_grt_dict_item_set_value_from_string(sp, "owner", myx_grt_dict_item_get_as_string(schema, "_id"));

        // add to procedure list
        myx_grt_list_item_add(sps, sp);
        myx_grt_value_release(sp);

        
        // procedureType (PROCEDURE || FUNCTION)
        STR_FIELD_TO_DICT_ITEM(2, sp, "procedureType");

        // security (DEFINER || INVOKER)
        STR_FIELD_TO_DICT_ITEM(6, sp, "security");

        // comment
        STR_FIELD_TO_DICT_ITEM(7, sp, "comment");

        reverse_engineer_sp_data(mysql, catalog, schema, sp);
      }
      mysql_free_result(res);
    }
  }

  return NULL;
}


// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_sp_data

static const char * sp_show_create[]=
{
  "Procedure",       // 0 (col 1)
  "Function",        // 1 (col 1)
  "sql_mode",        // 2 (col 2)
  "Create Procedure",// 3 (col 3)
  "Create Function", // 4 (col 3)
};
static const char ** sp_show_create_end=
                 sp_show_create + sizeof(sp_show_create)/sizeof(char*);


static MYX_GRT_VALUE *reverse_engineer_sp_data(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                               MYX_GRT_VALUE *schema, MYX_GRT_VALUE *sp)
{
  MYX_GRT_VALUE *error= NULL;
  char *sqlcmd, *sp_code;
  MYSQL_RES *res;
  MYSQL_ROW row;
  MYSQL_FIELD *fields;
  unsigned int num_fields;

  //Get SP parameter
  if (strcmp2(myx_grt_dict_item_get_as_string(sp, "procedureType"), "PROCEDURE") == 0)
    sqlcmd= myx_mysql_esc(mysql, "SHOW CREATE PROCEDURE %s;", myx_grt_dict_item_get_as_string(sp, "name"));
  else
    sqlcmd= myx_mysql_esc(mysql, "SHOW CREATE FUNCTION %s;", myx_grt_dict_item_get_as_string(sp, "name"));

  if (myx_mysql_query(mysql, sqlcmd) || !(res= mysql_store_result(mysql)))
  {
    MYX_GRT_VALUE *error= make_return_value_mysql_error(mysql, "Could not get stored procedure code.", sqlcmd);

    g_free(sqlcmd);

    return error;
  }
  else
  {
    int fi[5];
    //Depending on the version of the server there might 
    //   be different columns
    num_fields= mysql_num_fields(res);
    fields= mysql_fetch_fields(res);

    build_field_subst(sp_show_create,sp_show_create_end,
                      fields,fields+num_fields,fi);

    if ((row= mysql_fetch_row(res)))
    {
      if (fi[3]!=-1)
        sp_code= myx_convert_dbstr_utf8(mysql, row[fi[3]]);
      else
        sp_code= fi[4]==-1 ? 0 : myx_convert_dbstr_utf8(mysql, row[fi[4]]);

      if (sp_code)
      {
        MYX_GRT_VALUE *sp_params= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.ProcedureParam");

        // add param list to sp
        myx_grt_dict_item_set_value(sp, "params", sp_params);
        myx_grt_value_release(sp_params);


        // procedureCode
        myx_grt_dict_item_set_value_from_string(sp, "procedureCode", sp_code);

        // analyze params/return type
        error= reverse_engineer_sp_params(mysql, catalog, schema, sp);
      }

      g_free(sp_code);
    }
    mysql_free_result(res);
  }
  g_free(sqlcmd);

  return error;
}


// -----------------------------------------------------------------------------------------------------------
// reverse_engineer_sp_params

///////////////////////////////////////////////////////////////////////////////
/** @brief extracts only the header of a SP code

    @param sp_code      code of SP
    
    @return new allocated header or NULL
*//////////////////////////////////////////////////////////////////////////////

static char *filter_sp_header(char *sp_code)
{
  unsigned int i= 0, len= (unsigned int)strlen(sp_code);
  unsigned int b_count= 0;
  unsigned int header_start= 0;
  char *sp_header= NULL;

  if (!sp_code)
    return 0;

  while (i<len)
  {
    if (sp_code[i]=='(')
    {
      if(b_count==0)
        header_start= i;
      b_count++;
    }
    else if (sp_code[i]==')')
    {
      b_count--;

      if(b_count==0)
        break;
    }

    i++;
  }

  if (header_start>0) 
    sp_header= g_strndup(sp_code+header_start, i-header_start+1);

  g_free(sp_code);

  return sp_header;
}

//use the following regex to get all params of a SP
//(IN|OUT|INOUT)?\s?([\w\d]+)\s+([\w\d\(\)\,]+)\s*(\)|\,)
// 1: param type (IN, OUT, INOUT), IN is not existing
// 2: param name
// 3: param datatype
// 4: param datatype name
// 5: param datatype options in ()
// 6: separator , or )

#define SP_PARAMS "(IN|OUT|INOUT)?\\s?([\\w\\d]+)\\s+([\\w\\d]+)\\s*((\\([\\s\\w\\d,]+\\))?)\\s*(\\)|\\,)"

//use the following regex to get the returntype of a function
//RETURNS\s+([\w\d\(\)\,\s]+)\s+(LANGUAGE|NOT|DETERMINISTIC|SQL|COMMENT|RETURN|BEGIN)
// 1: datatype
// 2: keyword after datatype (not important)

#define SP_RETURN_TYPE "RETURNS\\s+([\\w\\d\\(\\)\\,\\s]+?)\\s+(LANGUAGE|NOT|DETERMINISTIC|SQL|COMMENT|RETURN|BEGIN)"


static MYX_GRT_VALUE *reverse_engineer_sp_params(MYSQL *mysql, MYX_GRT_VALUE *catalog, 
                                               MYX_GRT_VALUE *schema, MYX_GRT_VALUE *sp)
{
  MYX_GRT_VALUE *sp_params= myx_grt_dict_item_get_value(sp, "params");
  char *sp_code= g_strdup(myx_grt_dict_item_get_as_string(sp, "procedureCode"));
  pcre *pcre_exp, *pcre_exp2;
  int erroffset, offset;
  const char *error_str;
  const char *param_type, *param_name, *param_datatype;
  int matched[60], rc;

  // compile regular expressions
  pcre_exp= pcre_compile(SP_PARAMS, PCRE_CASELESS, &error_str, &erroffset, NULL);
  if (!pcre_exp)
  {
    return make_return_value_error("Could not compile regular expression for stored procedure retrieval.", SP_PARAMS);
  }

  pcre_exp2= pcre_compile(SP_RETURN_TYPE, PCRE_CASELESS, &error_str, &erroffset, NULL);
  if (!pcre_exp)
  {
    pcre_free(pcre_exp);
    return make_return_value_error("Could not compile regular expression for stored procedure retrieval.", SP_RETURN_TYPE);
  }

  //Get function return type
  if (strcmp2(myx_grt_dict_item_get_as_string(sp, "procedureType"), "FUNCTION") == 0)
  {
    const char *return_type;
    int matched[60], rc;

    if((rc= pcre_exec(pcre_exp2, NULL, sp_code, (int)strlen(sp_code), 0, 0, matched, sizeof(matched)/sizeof(*matched)))>=0)
    {
      pcre_get_substring(sp_code, matched, rc, 1, &return_type);
      myx_grt_dict_item_set_value_from_string(sp, "returnDatatype", return_type);
      pcre_free_substring(return_type);
    }
  }

  //Look only at params
  sp_code= filter_sp_header(sp_code);        
  
  offset= 0;

  while((rc= pcre_exec(pcre_exp, NULL, sp_code, (int)strlen(sp_code), offset, 0, matched, sizeof(matched)/sizeof(*matched)))>=0)
  {
    MYX_GRT_VALUE *sp_param= myx_grt_dict_new("db.mysql.ProcedureParam");

    // _id
    myx_grt_dict_generate_id(sp_param);

    pcre_get_substring(sp_code, matched, rc, 1, &param_type);
    pcre_get_substring(sp_code, matched, rc, 2, &param_name);                
    pcre_get_substring(sp_code, matched, rc, 3, &param_datatype);

    // name
    myx_grt_dict_item_set_value_from_string(sp_param, "name", param_name);

    // owner
    myx_grt_dict_item_set_value_from_string(sp_param, "owner", myx_grt_dict_item_get_as_string(sp, "_id"));

    // add param to param list
    myx_grt_list_item_add(sp_params, sp_param);
    myx_grt_value_release(sp_param);


    // datatype
    myx_grt_dict_item_set_value_from_string(sp_param, "datatype", param_datatype);

    // paramType
    if (strcmp2(param_type, "")==0)
      myx_grt_dict_item_set_value_from_string(sp_param, "paramType", "IN");
    else
      if (g_ascii_strncasecmp(param_type, "OUT", 3)==0)
        myx_grt_dict_item_set_value_from_string(sp_param, "paramType", "OUT");
      else
        if (g_ascii_strncasecmp(param_type, "INOUT", 5)==0)
          myx_grt_dict_item_set_value_from_string(sp_param, "paramType", "INOUT");
        else
          myx_grt_dict_item_set_value_from_string(sp_param, "paramType", "IN");

    pcre_free_substring(param_type);
    pcre_free_substring(param_name);
    pcre_free_substring(param_datatype);

    //Move offset
    offset= matched[1];
  }

  g_free(sp_code);
  pcre_free(pcre_exp2);
  pcre_free(pcre_exp);

  return NULL;
}

#undef STR_FIELD_TO_DICT_ITEM
