# amx-lib.tcl --
#
#       Low-level code that talks to the AMX box and provides some simple
#       abstractions for controlling through the serial port.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

source amx-header.tcl

global serialFd_

#
# returns the string name of the device for the specified device number
#
# it is an invariant that $g_amxDevices([amx_deviceLookup x]) == x
#
proc amx_deviceLookup {dev} {
    global g_amxDevices g_amxInvDevices

    if {[info exists g_amxInvDevices($dev)]} {
	return $g_amxInvDevices($dev)
    }
    # if we got here, it's not a recognized device
    return $dev
}   

proc amx_channelLookup {dev channel} {
    global g_amxDevices g_amxInvChannels

    if {[info exists g_amxInvChannels($dev,$channel)]} {
	return $g_amxInvChannels($dev,$channel)
    }
    # if we got here, it's not a recognized device or channel
    return $channel
}   

proc amx_commandLookup {cmd} {
    global g_amxInvCommands
    
    if {[info exists g_amxInvCommands($cmd)]} {
	return $g_amxInvCommands($cmd)
    }
    # if we got here, it's not a recognized command
    return $cmd
}

#
# If no filename is specified, the serial port on htsr is used
#
proc setup-AMX-control { {filename "/dev/cuac02"} } {
    global serialFd_

    if {[catch {open "$filename" {RDWR NONBLOCK NOCTTY}} serialFd_]} {
	puts stderr "Fatal error:  cannot open serial port: $filename"
	return -code error "Unable to open serial port: $filename"
    } else {
	if {[catch {fconfigure $serialFd_ -blocking 0 -mode "9600,n,8,0" -buffering none -translation binary} result]} {
	    puts stderr "Fatal error: cannot configure serial port"
	    puts stderr $result
	    return -code error "Unable to configure serial port: $filename"
	}
    }
    # if we got here, it's set up ok
    amx_clearBuffer
}

proc close-AMX-control {} {
    global serialFd_

    catch {close $serialFd_}
}

# FIXME - not sure if anyone needs this function
#   currently, it doesn't check anything, just like the old server code!
proc is-control-present {} {
    return "Room control is present"
}

# currently supports command codes of
# 1-11, 15
#
proc send-AMX-command {cmd device {chan_str ""}} {
    switch -exact $cmd {
	1 -
	2 -
	3 -
	6 -
	7 -
	9 - 
	10 -
	15 {
	    if {[string compare $chan_str ""] == 0} {
		switch -exact $cmd {
		    9 -
		    15 { set param "address" }
		    default { set param "channel" }
		}
		return -code error "Wrong number of args: wanted <cmd> <device> <$param>"
	    }
	    sendSimpleCmd $cmd $device $chan_str
	}
	4 -
	5 {
	    if {[string compare $chan_str ""] == 0} {
		return -code error "Wrong number of args: wanted <cmd> <device> <string>"
	    }
	    sendStringCmd $cmd $device $chan_str
	}
	8 -
	11 {
	    sendSingleCmd $cmd $device
	}
	default {
	    return -code error "Unsupported command code in send-AMX-command"
	}
    }
    return "sent AMX command"
}

# send a command with device,channel or device,address specified
#
# used for command codes 1,2,3,6,7,9,10,15
#
proc sendSimpleCmd { cmd dev data } {
    global serialFd_

    set str [binary format "accc" "*" $cmd $dev $data]
    set checkSum [computeChecksum $str]
    set str [binary format "a*c" $str $checkSum]
    if {[catch {puts -nonewline $serialFd_ $str} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
    if {[catch {flush $serialFd_} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
	
}

# send a command with device, string specified
#
# used for command codes 4, 5
#
proc sendStringCmd { cmd dev strData } {
    global serialFd_

# there is some bug in TCL which can mess things up when the string is printed,
#   if it contains the character with the ASCII value 5 (not the character '5')
#   currently, it causes "xterm" to be printed to the terminal we are working
#   on, as if the user typed it!
#
# apparently, ascii 5 = ^E

    set str [binary format "accc" "*" $cmd $dev [string length $strData]]
    append str $strData
    set checkSum [computeChecksum $str]
    set str [binary format "a*c" $str $checkSum]

    if {[catch {puts -nonewline $serialFd_ $str} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
    if {[catch {flush $serialFd_} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
}

# send a command with only device specified
#
# used for command codes 8, 11
#
proc sendSingleCmd { cmd dev } {
    global serialFd_

    set str [binary format "acc" "*" $cmd $dev]
    set checkSum [computeChecksum $str]
    set str [binary format "a*c" $str $checkSum]
    if {[catch {puts -nonewline $serialFd_ $str} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
    if {[catch {flush $serialFd_} result]} {
	puts stderr $result
	return -code error "Error: unable to write to AMX"
    }
}

proc amx_clearBuffer {} {
    set retList [receive-AMX-command]
    while {[llength $retList] != 0} {
	set retList [receive-AMX-command]
    }
}

proc amx_setCallback {procName} {
    global serialFd_

    if {[catch {fileevent $serialFd_ readable "$procName"} result]} {
	return -code error "Error: unable to register AMX callback: $result"
    }
}

# FIXME - not sure if this thing works
proc amx_getCallback {} {
    global serialFd_

    if {[catch {fileevent $serialFd_ readable} result]} {
	return -code error "Error: unable to get AMX callback:"
    }
    return $result
}

# currently supports command codes of
#   1, 2, 4, 6, 7, 8, 10
#
proc receive-AMX-command {} {
    global g_amxResponses
    
    set header [getChar]

#    puts stdout "receive-AMX-command: header = $header"

    set retList [list]

    # if the first byte is '&', then it is a response packet
    #
    # if it isn't '&', something is wrong, or the AMX functionality has changed
    #    the right thing will happen, with the fileevent generating callbacks
    #    until the buffer is empty or a '&' has been found
    if {[string compare $header "&"] == 0} {
	set cmd [getByte]
	set dev [getByte]
#	set devName [amx_deviceLookup $dev]
#	puts stdout "receive-AMX-command: cmd = $cmd, dev = $dev"
	switch -exact $cmd {
	    1 -
	    2 -
	    6 -
	    7 -
	    8 -
	    10 {
		set chan [getByte]
#		puts stdout "receive-AMX-command: chan = $chan"
#		set respStr [amx_commandLookup $cmd]
#		set retList [list $respStr $devName [amx_channelLookup $devName $chan]]
		set retList [list $cmd $dev $chan]
	    }
	    4 {
		set numBytes [getByte]
#		puts stdout "receive-AMX-command: numBytes = $numBytes"
		set str [getCharBlocking $numBytes]
#		set respStr [amx_commandLookup $cmd]
#		puts stdout "receive-AMX-command: str = $str"
#		set retList [list $respStr $devName $str]
		set retList [list $cmd $dev $str]
	    }
	}
	set checksum [getByte]
    }
    return $retList
}

# returns an 8-bit integer in the range 0-255, or "" if nothing is available
#   from serialFd_
#
# this routine blocks until a byte is read!
proc getByte {} {
    set retVal [getCharBlocking]
    set ascVal [charToInt $retVal]
    return $ascVal
}

# this takes a string and converts the first character in it to an integer
#   in the range 0-255
#
# if the string is empty, returns an empty string
#
proc charToInt { ch } {
    set ascVal ""
    set numConv [binary scan $ch "c" ascVal]
#    puts stdout "num Conv = $numConv; ascVal = $ascVal"
    if {$numConv == 1} {
	if {$ascVal < 0} {
	    # this is necessary because Tcl can't handle unsigned chars
	    set ascVal [expr $ascVal + 256]
	}
    }
    return "$ascVal"
}

# by default, reads one character and returns it as a string
#
# can specify more characters to read
#
# this is non-blocking, so if the buffer is empty, will return ""
proc getChar { {numToRead 1} } {
    global serialFd_

    if {[catch {read $serialFd_ $numToRead} retVal]} {
	return ""
    } else {
	set len [string length $retVal]
#	puts stdout "read $len, expected $numToRead:     $retVal"
	return "$retVal"
    }
}

proc getCharBlocking { {numToRead 1} } {
    global serialFd_

    set retVal ""

    while {$numToRead} {
	if {[catch {read $serialFd_ $numToRead} readStr]} {
	    return ""
	} else {
	    set len [string length $readStr]
#puts stdout "len=$len, numToRead=$numToRead"
	    set numToRead [expr $numToRead - $len]
	    append retVal $readStr
	    if {$numToRead == 0} {
#		puts stdout "getCharBlocking: returning $retVal"
		return $retVal
	    }		
	}
#	puts stdout "getCharBlocking: waiting for available character"
	after 100
    }
#    puts stdout "getCharBlocking: returning $retVal"
    return $retVal
}

# takes a string and computes the checksum
#
# the checksum is the sum of all the ascii character values, (0-255), mod 255
#
proc computeChecksum {str} {
    set size [string length $str]
    set sum 0
    for {set x 0} {$x < $size} {incr x 1} {
	set tmp [charToInt [string index $str $x]]
	set sum [expr $sum + $tmp]
    }
    set sum [expr $sum % 256]
    return $sum
}

