/*
 * rtp-play.cc --
 *
 *      RTP Play for archive streams
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <tclcl.h>
#include "net.h"
#include "rtp.h"
#include "misc/mtrace.h"
#include "archive/archive-stream.h"
#include "archive/rtp-play.h"
#include "misc/nethost.h"





DEFINE_OTCL_CLASS(RTPPlaybackStream, "ArchiveStream/Play/RTP") {
	INSTPROC(buffer_pool);
	INSTPROC(attach_agent);
	INSTPROC(header_info);
}

DEFINE_OTCL_CLASS(RTPPlayRcvr, "Module/RTPPlay") {
}


RTPPlaybackStream::RTPPlaybackStream()
	: PlaybackStream(), startTS_(0), endTS_(0), dnet_(NULL), cnet_(NULL),
	  free_(NULL), eof_(0), skipctrl_(1), converter_(0.0),
	  firstTime_(TRUE), bufferPool_(NULL), cs_(0), firstseq_(0),
	  p_(0), lastp_(0), agent_(NULL), lastNotify_(0.0)
{
}

RTPPlaybackStream::~RTPPlaybackStream() {
	printf("In rtp-play.cc destroy\n");
	agent_->send_bye();
}


int
RTPPlaybackStream::header_info(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	int returnValue;
	FileHeader hdrNet;
	RTPprivatehdr phdrNet;
	const char *hdrArray;

	double last_recv;
	u_int32_t last_rtp;
	double first_recv, new_scale;

	int modend, endpos;

	DataFile* file = DataFile_();
	IndexFile* ifile = IndexFile_();

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(hdrArray);
	END_PARSE_ARGS;

	if (file->IsOpen()==FALSE) {
		tcl.resultf("file not open");
		goto error;
	}

	returnValue = file->Read(&hdrNet, (u_char *)&phdrNet, sizeof(RTPprivatehdr));

	net2host(hdrNet, hdr_);
	net2host(phdrNet, phdr_);
	MTrace(trcArchive|trcVerbose, ("%s Default Scale = %lu\n",hdr_.name,  phdr_.scale));

	// Calculate scale, using recv timestamps
	IndexRecord iNet, iHost;

	if (ifile->Seek(0, SEEK_END)==TCL_ERROR)
		return TCL_ERROR;
	endpos = ifile->Tell();
	// This is only necessary in pathological cases, where the disk filled up, etc
	modend = ((endpos - sizeof(FileHeader)) % sizeof(IndexRecord));
	if (modend != 0) {
		MTrace(trcArchive, ("Wrong Size, truncing"));
		endpos = endpos - modend;
	}

	if (ifile->Seek(endpos - sizeof(IndexRecord), SEEK_SET)==TCL_ERROR)
		return TCL_ERROR;


	//if (ifile->Seek(-1*((int)sizeof(IndexRecord)), SEEK_END)==TCL_ERROR)  {
	//return TCL_ERROR;
	//}

	if (ifile->Read(&iNet)==TCL_ERROR) {
			return TCL_ERROR;
	}
	net2host(iNet, iHost);

	last_rtp = logical2rtp(iHost.sentTS_sec, iHost.sentTS_usec, phdr_.scale, phdr_.ref_rtp, phdr_.ref_tv_sec, phdr_.ref_tv_usec);
	last_recv = iHost.recvTS_sec + iHost.recvTS_usec/1000000.0;
	first_recv = phdr_.ref_tv_sec + phdr_.ref_tv_usec/1000000.0;
	new_scale = (last_rtp - phdr_.ref_rtp) / (last_recv - first_recv);
/*FIXME*/
if (new_scale > 7800 && new_scale < 8200) {
	//printf("warning mccanne hack\n");
	new_scale = 8000;
}
	MTrace(trcArchive|trcVerbose, ("New Scale = %g\n", new_scale));
	phdr_.scale = (int) new_scale;
	//printf("first_recv=%g last_recv=%g last_rtp=%u first_rtp=%g\n", first_recv, last_recv, last_rtp, phdr_.ref_rtp);
	//

	char c_ssrc[16];
	sprintf(c_ssrc, "%u", host2net(phdr_.ssrc));

	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "cname",
			hdr_.cname, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "name",
			hdr_.name, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "email",
			phdr_.email, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "ssrc",
			c_ssrc, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "phone",
			phdr_.phone, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "loc",
			phdr_.loc, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "tool",
			phdr_.tool, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	tcl.resultf("");
	return TCL_OK;

error:
	tcl.add_errorf("\ninvoked from within RTPPlaybackStream::Header_Info()");
	return TCL_ERROR;
}




int
RTPPlaybackStream::attach_agent(int argc, const char * const *argv)
{
	TclObject* agent;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(agent);
	END_PARSE_ARGS;

	agent_=(RTPPlaySession *)agent;
	return TCL_OK;
}

int
RTPPlaybackStream::buffer_pool(int argc, const char * const *argv)
{
	TclObject* bufferPool;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(bufferPool);
	END_PARSE_ARGS;

	bufferPool_=(BufferPool *)bufferPool;

	/*FIXME*/
        int npack = 64;/*FIXME*/
        for (int i = 0; i < npack; ++i) {
		RTPPacket* p = new RTPPacket;
		p->buf=bufferPool_->alloc();
		free(p);
        }
	return TCL_OK;
}


#if OLD
int
RTPPlaybackStream::Datanet(int argc, const char * const *argv)
{
	TclObject* datanet;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(datanet);
	END_PARSE_ARGS;

	dnet_=(Network *)datanet;
	return TCL_OK;
}

int
RTPPlaybackStream::Ctrlnet(int argc, const char * const *argv)
{
	TclObject* ctrlnet;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(ctrlnet);
	END_PARSE_ARGS;

	cnet_=(Network *)ctrlnet;
	return TCL_OK;
}
#endif


void
RTPPlaybackStream::Clip(timeval start, timeval end)
{
	startTS_= logical2rtp(start, phdr_.scale, phdr_.ref_rtp,
				phdr_.ref_tv_sec, phdr_.ref_tv_usec);
	endTS_ = logical2rtp(end, phdr_.scale, phdr_.ref_rtp,
				phdr_.ref_tv_sec, phdr_.ref_tv_usec);
	//startTS_ = logical2rtp(start.tv_sec, start.tv_usec, converter_);
	//endTS_ = logical2rtp(end.tv_sec, end.tv_usec, converter_);
}



/*
 * add packet p to the queue of waiting packets for this
 * source.  The queue is ordered by seqno.
 */
void
RTPPlaybackStream::addpacket(RTPPacket* p, u_int ts, u_int seqno)
{
	//MTrace(trcArchive, ("addpacket %u %u",ts, seqno));

	/* convert the seqno to 32 bits */
	if (p->type) {  // if it's a control packet
		MTrace(trcArchive|trcVerbose, ("addpacket control pkt"));
		seqno = firstseq_;
		if (lastp_) {
			seqno += lastp_->seq;
			ts = lastp_->its;
		}
	} else {  // Data packet
		register int c = cs_;
		register int d = seqno - c;
		if (d < -1024 || d > 1024) {
			cs_ = seqno;
			if (seqno < 512 && c > 0x10000-512) {
				/*
				 * seq no. wrapped - subtract 64k from firstseq to
				 * account for it.
				 */
				firstseq_ -= 0x10000;
			} else {
				/*
				 * the seq. no made a very large
				 * jump.  assume that the other
				 * side restarted without telling
				 * us about it so just re-sync
				 * (i.e., pretend this was the
				 * first packet).
				 */
				firstseq_ = seqno - 1;
			}
		} else if (d > 0) {
			/*
			 * d <= 0 means duplicate or reordered packet so
			 * don't move cs_
			 */
			cs_ = seqno;
		}
	}
	seqno -= firstseq_;
	p->seq = seqno;
	MTrace(trcArchive, ("addpacket %u", seqno));

	p->its=ts;
	//MTrace(trcArchive, ("addpacket %u",ts));
	/*
	 * find where in the queue to insert this packet.
	 * the usual case of inserting at the tail is special cased
	 * for performance reasons.
	 */
	if (lastp_ == 0) {
		lastp_ = p;
		p_ = p;
		p->next = 0;
	} else if (lastp_->its < p->its ||
		   (p->its == lastp_->its && int(seqno - lastp_->seq) > 0)) {
		lastp_->next = p;
		lastp_ = p;
		p->next = 0;
	} else if (p->its < p_->its ||
		   (p->its == p_->its && int(seqno - p_->seq) <= 0)) {
		p->next = p_;
		p_ = p;
	} else {
		RTPPacket* np;
		for (RTPPacket* lp = p_; (np = lp->next) != 0; lp = np) {
			if (p->its < np->its ||
			    (p->its == np->its && int(seqno - np->seq) <= 0)) {
				lp->next = p;
				p->next = np;
				break;
			}
		}
	}
}



u_int
RTPPlaybackStream::ts() const
{
	if (p_ == 0)
		/*FIXME*/
		return (0);

	return (p_->its);
}

RTPPacket*
RTPPlaybackStream::getpkt(u_int now)
{
	RTPPacket* p = p_;
	//MTrace(trcArchive, ("getpkt %f %f %d", p->ts, ts));
	if (p && ((int)(p->its - now) <= 0 || now == 0)) {
		RTPPacket* np = p->next;
		p_ = np;
		if (np == 0)
			lastp_ = 0;
		return (p);
	}
	return (0);
}

RTPPacket*
RTPPlaybackStream::allocate()
{

	RTPPacket* temp = free_;
	free_ = temp->next;
	temp->buf = bufferPool_->alloc();
	return temp;

}

void
RTPPlaybackStream::EmptyBuffs()
{

	RTPPacket *p; RTPPacket *np;

	p = p_;
	while (p) {
		np=p->next;
		free (p);
		p=np;
	}
	p_=0;
	lastp_=0;

}

void
RTPPlaybackStream::IterateIndex(int real_endpos) {

	IndexFile* ifile;
	DataFile* dfile;
	IndexRecord iNet, iHost;

	ifile = IndexFile_();
	dfile = DataFile_();


	int curpos = ifile->Tell();
	while (curpos < real_endpos) {
		if (ifile->Read(&iNet)==TCL_ERROR)
			return;
		net2host(iNet, iHost);
		printf("recv-sent = %d \n", iHost.recvTS_sec - iHost.sentTS_sec);
		curpos = ifile->Tell();
	}


}


Bool
RTPPlaybackStream::SeekTime(timeval goal) {

	IndexFile* ifile;
	DataFile* dfile;
	IndexRecord iNet, iHost;
	int actual_end, endpos;


	eof_=0;

	u_int32_t goalsec= (u_int32_t) goal.tv_sec;
	u_int32_t goalusec= (u_int32_t)goal.tv_usec;

	ifile = IndexFile_();
	dfile = DataFile_();

	if (ifile->Seek(sizeof(FileHeader), SEEK_SET)==TCL_ERROR)
		return TCL_ERROR;
	if (ifile->Read(&iNet)==TCL_ERROR)
			return TCL_ERROR;
	net2host(iNet, iHost);
	if ((goalsec < iHost.sentTS_sec) || ((goalsec == iHost.sentTS_sec) && (goalusec <= iHost.sentTS_usec))) {
		dfile->Seek(sizeof(FileHeader) + sizeof(RTPprivatehdr), SEEK_SET);
		return TCL_OK;
	}

	int startpos = sizeof(FileHeader);

	if (ifile->Seek(0, SEEK_END)==TCL_ERROR)
		return TCL_ERROR;
	actual_end = ifile->Tell();
	endpos = actual_end;
	// This is only necessary in pathological cases, where the disk filled up, etc
	int modend = ((endpos - sizeof(FileHeader)) % sizeof(IndexRecord));
	if (modend != 0) {
		MTrace(trcArchive, ("Wrong Size, truncing"));
		endpos = endpos - modend;
	}

	if (ifile->Seek(endpos - sizeof(IndexRecord), SEEK_SET)==TCL_ERROR)
		return TCL_ERROR;
	//int real_endpos = endpos;
	if (ifile->Read(&iNet)==TCL_ERROR)
			return TCL_ERROR;
	net2host(iNet, iHost);

	//ref_end_rec_ = iHost.recvTS_sec + iHost.recvTS_usec/1000000.0;

	if (goalsec > iHost.sentTS_sec) {
		if (dfile->Seek(0,SEEK_END)==TCL_ERROR)
			return TCL_ERROR;
		MTrace(trcArchive, ("Sought too end"));
		return TCL_OK;
	}
	if ((goalsec == iHost.sentTS_sec) &&
	    (goalusec >= iHost.sentTS_usec))
	{
		if (dfile->Seek(iHost.filePointer, SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
		MTrace(trcArchive, ("Sought too %u",iHost.seqno));
		return TCL_OK;
	}
	int newpos=((((endpos-startpos)/sizeof(IndexRecord))/2) * sizeof(IndexRecord)) + startpos;


	int curpos;


	for (;;) {
		if (ifile->Seek(newpos, SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
		if (ifile->Read(&iNet)==TCL_ERROR)
			return TCL_ERROR;
		net2host(iNet, iHost);

		if ((goalsec == iHost.sentTS_sec)
		    && (goalusec == iHost.sentTS_usec)) {
			if (dfile->Seek(iHost.filePointer, SEEK_SET)==TCL_ERROR)
				return TCL_ERROR;
			MTrace(trcArchive, ("Sought too %u",iHost.seqno));


			return TCL_OK;
		}
		if ((goalsec > iHost.sentTS_sec) || ((goalsec == iHost.sentTS_sec) && (goalusec >= iHost.sentTS_usec))) {
			startpos = newpos + sizeof(IndexRecord);
			newpos=((((endpos-startpos)/sizeof(IndexRecord))/2) * sizeof(IndexRecord)) + startpos;
		}
		if ((goalsec < iHost.sentTS_sec) || ((goalsec == iHost.sentTS_sec) && (goalusec <= iHost.sentTS_usec))) {
			endpos = newpos;
			newpos=((((endpos-startpos)/sizeof(IndexRecord))/2) * sizeof(IndexRecord)) + startpos;
			if (newpos == endpos) {
				if (dfile->Seek(iHost.filePointer, SEEK_SET)==TCL_ERROR)
					return TCL_ERROR;
				MTrace(trcArchive, ("Sought too %u",iHost.seqno));
				curpos = ifile->Tell();



				return TCL_OK;
			}
		}
	}



}


void
RTPPlaybackStream::LTS_Speed()
{
	PlaybackStream::LTS_Speed();
}


void
RTPPlaybackStream::LTS_Reference()
{
	EmptyBuffs();
	timeval now = LTS_()->NowLogical();
	MTrace (trcArchive, ("LTS_Reference %u %u", now.tv_sec, now.tv_usec));
	if (SeekTime(now)==TCL_ERROR) {
		// FIXME: LTS_Reference should have an error-returning mechanism
		return;
	}

	PlaybackStream::LTS_Reference();
}

void
RTPPlaybackStream::free(RTPPacket* p)
{
	p->next = free_;
	free_ = p;
	p->buf->release();
}



int
RTPPlaybackStream::nextpkt(RTPPacket* p)
{

	DataFile *file;
	struct recordhdr *rh = (recordhdr *) p->buf->data;

	file = DataFile_();
	if (firstTime_==TRUE) {

		if (file->Seek(file->getHeaderSize(), SEEK_SET)==TCL_ERROR)
			return (-1);

	}


	// read header (gives us len and type)

	if (file->Read((unsigned char*)rh, sizeof(recordhdr))==0) {
		eof_ = 1;
		return (-1);
	}

	int len = net2host(rh->len);
	int type = rh->type;

	//MTrace(trcArchive, ("nextpkt type %d", type));

	int amt = len;
	if (len > MTU - 4) {
		/* FIXME packet too big */
		for (len -= amt; len > 0; len -= amt) {
			amt = len;
			if (amt > MTU)
				amt = MTU;
			if (file->Read((u_char*) p->buf->data, amt) == 0) {
				eof_ = 1;
				return (-1);
			}
		}
		return (1);
	}

	// now read the rtp packet itself; overwrites recordhdr
	if (file->Read((u_char*) p->buf->data, amt) == 0) {
		eof_ = 1;
		return (-1);
	}

	// assign relevant fields
	// others in the RTPPacket struct are assigned in
	// addpacket

	p->type = type;
	p->len = amt;




	return (len);
}



int
RTPPlaybackStream::fillone()
{

	int rval;
	u_int id, ts, seq;


	RTPPacket *p = allocate();
	rval = nextpkt(p);  /* Actually read from disk */

	if (rval < 0) {
		eof_ = 1;
		p->next = free_;
		free_ = p;
		return (rval);
	}

	if (rval < 2) {
		return (rval);
	}

	struct rtphdr *rh = (struct rtphdr *)p->buf->data;

	// p.type and p.len will have been filled in by nextpkt above

	switch (p->type) {
	default:
		return (1);
	case 0:
		// DATA packet (RTP)

		skipctrl_ = 0;
		id = net2host(rh->rh_ssrc);
		ts = net2host(rh->rh_ts);
		seq = net2host(rh->rh_seqno);
		MTrace (trcArchive|trcVerbose,
			("fillone seqno=%u ts=%u", seq, ts));

		break;

	case 0x80:
		// CONTROL packet (RTCP)
		// Currently, shouldn't get here
		// FIX THIS
		struct rtcp_compound {
			rtcphdr rt;
			rtcp_sr sr;
		} *th = (rtcp_compound *) p->buf->data;
		if (skipctrl_ || (ntohs(th->rt.rh_flags) & 0xff) \
		    != RTCP_PT_SR) {
			/* not a sender report */
			return (1);
		}
		id = net2host(th->rt.rh_ssrc);
		ts = net2host(th->sr.sr_ts);
		seq = ~0;
		break;
	}

	if (firstTime_ == TRUE) {
		// Move this to nextpkt after we fix ctrl pkts
		cs_=seq;
		firstseq_=seq;
		firstTime_=FALSE;
	}

	addpacket(p, ts, seq);  /* Put the packet in the queue */
	return (0);
}


int
RTPPlaybackStream::IsDone()
{
	if ((p_==0) && (eof_==1)) {
		return 1;
	}
	return 0;


        /*if ((eof_ == 0) && (p_ != 0))
                return (0);

	if (p_!=0)
		return (0);
        return (1); */
}

int
RTPPlaybackStream::NextEvent(timeval &logical)
{

        /* fill all free packet buffers with disk data */
        while (free_ && (fillone() >= 0)) {
        }


	if (IsDone()) {
		MTrace(trcArchive, ("RTP playback source is done"));
		logical.tv_sec = logical.tv_usec = 0;
		EmptyBuffs();
		return TCL_OK;

	}

	u_int dt=ts();
	if (endTS_ > 0 && dt > endTS_) {
		//we are done
		MTrace(trcArchive, ("Exceeded ending ts (%lu): %lu",
				    (unsigned long) endTS_,
				    (unsigned long) dt));
		logical.tv_sec = logical.tv_usec = 0;
		return TCL_OK;
	}


	//set logical to be time when next pkt should play
	timeval ref;
	ref.tv_sec = phdr_.ref_tv_sec;
	ref.tv_usec = phdr_.ref_tv_usec;
	//FIXME logical time starts at 0!
	//ref.tv_sec = ref.tv_usec = 0;
	//ref.tv_usec = 1;//except 0 has special meaning... FIXME

	logical = rtp2logical(dt, phdr_.scale, phdr_.ref_rtp,
				     ref);
	MTrace(trcArchive|trcVerbose,
	       ("RTP playback source returned logical timeof %u %u.%u %u",
		dt, logical.tv_sec, logical.tv_usec, p_->seq));
	return TCL_OK;

}

void
RTPPlaybackStream::DoEvent()
{
	double now;
	timeval ts = LTS_()->NowLogical();
	u_int dts = logical2rtp(ts, phdr_.scale, phdr_.ref_rtp,
				phdr_.ref_tv_sec, phdr_.ref_tv_usec);
	for (;;) {
		RTPPacket *p = getpkt(dts);
		if (p == 0)
			return;

		MTrace(trcArchive, ("Send seq=%d ",p->seq));

		p->buf->len = p->len;
		p->buf->next = 0; //??

		//p->buf->layer = 1;  //Change this when we support layered video
		agent_->send(p->buf);

		now = tvtof(LTS_()->NowSystem());
		if (now - lastNotify_ > 0.25) {
			Invokef("notify_observers bytes_sent 0");
			lastNotify_ = now;
		}

		p->next = free_;
		free_ = p;

		// p->buf is released in transmitter-rtp
	}
}
