/*
 * audio-oss.cc --
 *
 *      Audio driver for OSS (Open Sound) API, which is primarily
 *      used on FreeBSD and Linux.
 */

/*
 * Full Duplex audio module for the new sound driver and full duplex
 * cards. Luigi Rizzo, from original sources supplied by Amancio Hasty.
 *
 * This includes some enhancements:
 * - limit the maximum size of the playout queue to approx 4 frames;
 *   this is necessary if the write channel is slower than expected;
 *   the fix is based on two new ioctls, AIOGCAP and AIONWRITE,
 *   but the code should compile with the old driver as well.
 * - use whatever format is available from the card (included split
 *   format e.g. for the sb16);
 */

#ifdef __FreeBSD__
    /* included for __FreeBSD_version */
#   include <sys/param.h>
#   if (__FreeBSD_version >= 400000)
#       include <sys/soundcard.h>
#   else
#       include <machine/soundcard.h>
#   endif
#else
#   include <sys/soundcard.h>
#endif

//#include <osfcn.h>
#include <fcntl.h>
#include "mulaw.h"
#include "Tcl.h"
#include "audio-oss.h"

#define ULAW_ZERO 0x7f
#define ABUFLOG2 7
#define ABUFLEN ( 1<< ABUFLOG2)
#define NFRAG 5
#define NDRVRBUF 5

// this is the OTcl wrapper to make the C++ object a Tcl class
static class OSSAudioClass : public TclClass {
public:
    OSSAudioClass() : TclClass("Audio/OSS") {}
    TclObject* create(int, const char*const*) {
        return (new OSSAudio);

    }
} oss_audio_class;

OSSAudio::OSSAudio() : mixerfd(-1)
{
  // the "*2" is because we want to accomodate 16 bit samples as well as 8.
    readbuf = new u_char[ABUFLEN*NDRVRBUF*2];
    s16_buf = new u_short[blksize_];
    readCopyInPtr = readbuf;
    readCopyOutPtr = readbuf;

    //typically we will be using 8 bit samples in u-law mode...
    memset(readbuf, ULAW_ZERO, blksize_);

    /*
     * The only way to determine if the device is full duplex
     * or not is by actually opening it.  Unfortunately, we might
     * not be able to open it because some other process is
     * using it.  Assume half-duplex.  Obtain() will override
     * if appropriate.
     */
    duplex_ = 0;

    input_names_ = "mike linein cd";
    output_names_ = "speaker lineout";
    played=0;
    playedDuringLast=0;
    sample_rate = 8000; // default value is 8K
    use16RawPCM = 0;    // default value is to use 8 bit mulaw...
    stereo = 0; // default value is mono
}

void
OSSAudio::Obtain()
{
    char *thedev;
    char buf[64];
    int d = -1;
    int capabilities = 0;
    played=0;

    if (haveaudio())
	abort();

    // get ready to open the audio device
    if(device_ < 0)
    {
	thedev=getenv("AUDIODEV");
	if (thedev==NULL)
	    thedev="/dev/audio";
	else if (thedev[0]>='0') {
	    d = atoi(thedev);
	    sprintf(buf,"/dev/audio%d", d);
	    thedev = buf ;
	}
    }
    else
    {
	d = device_;
	sprintf(buf, "/dev/audio%d", d);
	thedev = buf;
    }
    // open the audio device
#if 0
    fd_ = open(thedev, O_WRONLY);
    ioctl(fd_, SNDCTL_DSP_GETCAPS, &capabilities);
    if (capabilities & DSP_CAP_DUPLEX) {
        close(fd_);
#endif
	fd_ = open(thedev, O_RDWR);
#if 0
    }
#endif

    // get ready to open the mixer
    if(device_ < 0)
    {
	thedev=getenv("MIXERDEV");
	if (thedev == NULL)
	{
	    if (d < 0)
		thedev = "/dev/mixer";
	    else {
		sprintf(buf,"/dev/mixer%d", d);
		thedev = buf ;
	    }
	}
    }
    else
    {
      sprintf(buf,"/dev/mixer%d", d);
      thedev = buf ;
    }
    // open the mixer
    mixerfd = open(thedev, O_RDWR); // O_NDELAY

    // check for errors in opening the device
    if(fd_ <0 ) {
	printf("couldn't open audio device\n"); 
	// retry using read only(?)
	// -- if so, then will need to track the last mode that was used
    }
    if(mixerfd <0 ) 
	printf("couldn't open mixer device\n");

    if (fd_ >= 0) {
	// do the IOCTL to control the audio device and mixer
        int channels = stereo+1;
        int rec_rate=sample_rate;
        int formats=0;
        int playformat=0, recordformat=0;
        int ret1=0;
#ifdef SNDCTL_DSP_CHANNELS
	ret1=ioctl(fd_, SNDCTL_DSP_CHANNELS, &channels);
#else	
	if (channels == 2)
	    ret1=ioctl(fd_, SNDCTL_DSP_STEREO, &channels);
#endif
	if(ret1<0) {
	  perror("failed to set number of channels correctly");
	}
	ret1=ioctl(fd_, SNDCTL_DSP_SPEED, &rec_rate);
	if(ret1<0) {
	  perror("failed to set sampling rate");
	} 
        capabilities=0;
        if (ioctl(fd_, SNDCTL_DSP_GETCAPS, &capabilities) <0) {
	    perror("failed to get capabilities");
	}
        if( ioctl(fd_, SNDCTL_DSP_GETFMTS, &formats) <0) {
	    perror("failed to get formats");
	}

	int sources=0;
	if ( ioctl(mixerfd, SOUND_MIXER_READ_DEVMASK, &sources) == -1 ) {
	    perror("failed to poll inputs");
	}

	if(!(sources & (1 << SOUND_MIXER_RECLEV))) {
	    vol_device=SOUND_MIXER_IGAIN;
	}
	else vol_device=SOUND_MIXER_RECLEV;

	switch (capabilities & (DSP_CAP_DUPLEX) ) {
	case DSP_CAP_DUPLEX :
#ifdef AFMT_WEIRD
	    if(formats & AFMT_WEIRD) { /* this is the sb16... */
		if (formats & AFMT_S16_LE) {
    		    playformat = AFMT_U8 ;
	 	    recordformat = AFMT_S16_LE;
		} else {
		    printf("sorry, no supported formats\n");
		    close(fd_);
		    close(mixerfd);
		    fd_ = -1 ;
		    return;
		}
	    }
	    else { // well-behaved soundcard
#endif
	    /*
	     * this entry for cards with decent full duplex. Use s16
	     * preferably (some are broken in ulaw) or ulaw or u8 otherwise.
	     */
	    if (formats & AFMT_S16_LE)
		playformat = recordformat = AFMT_S16_LE ;
	    else if (formats & AFMT_MU_LAW)
		playformat = recordformat = AFMT_MU_LAW ;
	    else if (formats & AFMT_U8)
		playformat = recordformat = AFMT_U8 ;
	    else {
		printf("sorry, no supported formats\n");
		close(fd_);
		close(mixerfd);
		fd_ = -1 ;
		return;
	    }
#ifdef AFMT_WEIRD
         }
#endif	    
	    if ((playformat != AFMT_S16_LE) && (use16RawPCM == 1))
	      printf("sorry, can't do 16 bit samples with this sound card...\n");
	    break ;

	default :
	    // start of new section
	    //printf("sorry don't know how to deal with this card\n");
	    /*
	     * this entry for cards with half duplex. Use s16
	     * preferably (some are broken in ulaw) or ulaw or u8 otherwise.
	     */
	    if (formats & AFMT_S16_LE)
		playformat = recordformat = AFMT_S16_LE ;
	    else if (formats & AFMT_MU_LAW)
		playformat = recordformat = AFMT_MU_LAW ;
	    else if (formats & AFMT_U8)
		playformat = recordformat = AFMT_U8 ;
	    else {
		printf("sorry, no supported formats\n");
		// end of new section
		close (fd_);
		close(mixerfd);
		fd_ = -1;
		// start of new section
	    }
	    // end of new section
	    if ((playformat != AFMT_S16_LE) && (use16RawPCM == 1))
	      printf("sorry, can't do 16 bit samples with this sound card...\n");
	    break;
	}

#ifdef AIOSFMT
	if(playformat!=recordformat) {
		// printf("specifing two formats to your poor soundcard\n");
		snd_chan_param params;
		params.play_rate=sample_rate;
		params.rec_rate=sample_rate;
		params.play_format=playformat;
		params.rec_format=recordformat;
		int stat=ioctl(fd_, AIOSFMT, &params);
		if(stat<0) {
			printf("dual-format specification failed\n");
		}
	}
	else {
	        ioctl(fd_, SNDCTL_DSP_SETFMT, &playformat);
	}
        play_fmt=playformat;
        rec_fmt=recordformat;
#else
	if(playformat!=recordformat) {
		printf("Different formats for playback and capture: binary not compiled for this\n");
		close(fd_);
		close(mixerfd);
		fd_=-1;
		return;
 	}
        ioctl(fd_, SNDCTL_DSP_SETFMT, &playformat);
        play_fmt=playformat;
        rec_fmt=recordformat;
#endif

        int frag= (NFRAG << 16) | ABUFLOG2;
        int ret=ioctl(fd_, SNDCTL_DSP_SETFRAGMENT, &frag);

        duplex_ = (capabilities & DSP_CAP_DUPLEX) ? 1:0;

	/*
	 * Set the line input level to 0 to shut
	 * off the analog side-tone gain between
	 * the line-in and line-out.  This would otherwise
	 * wreak havoc on an echo canceler, for example,
	 * plugged into the audio adaptor.
	 */
	int v = 0;
	ret=ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_LINE), &v);
	ret=ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_MIC), &v);
        if( sources & (1 << SOUND_MIXER_DIGITAL1) ) {
             ret=ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_DIGITAL1),&v);
        }
        if( sources & (1 << SOUND_MIXER_DIGITAL2) ) {
             ret=ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_DIGITAL2),&v);
        }
        int device = 1 << SOUND_MIXER_MIC;
        if ( ioctl(mixerfd, SOUND_MIXER_WRITE_RECSRC, &device) == -1 ) {
          perror("failed to select input");
        }

        /*
         * Restore the hardware settings in case
         * some other vat changed them.
         */

        InputPort(iport_);
        SetRGain(rgain_);
        SetPGain(pgain_);

        if (duplex_)
	    Audio::Obtain();
        else
	    notify();
    }
}


void OSSAudio::Release()
{
    if (haveaudio()) {
	if (mixerfd > 0) {
	    close(mixerfd);
        }
        mixerfd = -1;
        Audio::Release();
    }
}

// Called when you want to write audio out to the sound
// card. Called from within controller.cc
void OSSAudio::Write(u_char *cp)
{
    // blksize_ is inherited from Audio parent class (audio.h/.cc)
    // it is the number of samples NOT the size of the samples
    unsigned int i = blksize_;
    int l;

    if (play_fmt == AFMT_S16_LE) {	
      if (use16RawPCM == 0) {
	// translate it from 8 bit mulaw to 16 bit linear
	for (i=0; i< blksize_; i++)
	  s16_buf[i] = mulawtolin[cp[i]] ;
	cp = (u_char *)s16_buf;
      }
      // size goes from 8 bit to 16 bit, so you need to double i.
      i = 2 *blksize_ ;
    }
    else if (play_fmt == AFMT_S8) {
	for (i=0; i< blksize_; i++) {
	    int x = mulawtolin[cp[i]] ;
	    x =  (x >> 8 ) & 0xff;
	    cp[i] = (u_char)x ;
	}
	i = blksize_ ;
    } else if (play_fmt == AFMT_U8) {
	for (i=0; i< blksize_; i++) {
	    int x = mulawtolin[cp[i]] ;
	    x =  (x >> 8 ) & 0xff;
	    x = (x ^ 0x80) & 0xff ;
	    cp[i] = (u_char)x ;
	}
	i = blksize_ ;
    }
    played=1;
    audio_buf_info info;
    ioctl(fd_, SNDCTL_DSP_GETOSPACE,&info);
    if(info.bytes>=(info.fragstotal*info.fragsize)-(info.fragsize>>1) ) {
      if(playedDuringLast) { // adjusting for fast sound card
	  int locali=i/2;
	    unsigned char * localcp=cp;
	    for ( ; locali > 0 ; locali -= l) {
		l = write(fd_, localcp, locali);
		localcp += l;
	    } 
	} else { // would adjust, but haven't played during last
	    unsigned char buf[blksize_/2];
	    memset(buf,cp[0],blksize_/2);
	    write(fd_, buf, blksize_/2);
	}
    }
    // write the audio data to the sound card.
    for ( ; i > 0 ; i -= l) {
	l = write(fd_, cp, i);
	cp += l;
    }
}

// The Read call is to read data in from the sound
// card (called in controller.cc). Basically it just
// reads from a ring buffer. The data is put into the
// buffer by FrameReady()
u_char* OSSAudio::Read()
{
  playedDuringLast=played;
  played=0;
    
  u_char* ptr=readCopyOutPtr;
  if (use16RawPCM == 0) {
    // these are 8 bit samples
    readCopyOutPtr+=blksize_;
    // since it is 8 bit, we only use half the buffer
    if(readCopyOutPtr>=readbuf+ABUFLEN*NDRVRBUF) {
      if(readCopyOutPtr>readbuf+ABUFLEN*NDRVRBUF) {
	printf("::read() pointer error: %d\n",
	       readCopyOutPtr-readbuf+ABUFLEN*NDRVRBUF );
      }
      readCopyOutPtr=readbuf;
    }
  } else {
    // these are 16 bit samples, so we have blksize_ * 2 bytes
    readCopyOutPtr+=(blksize_*2);
    // since it is 16 bit samples, we use the whole buffer
    if(readCopyOutPtr>=readbuf+(ABUFLEN*NDRVRBUF*2)) {
      if(readCopyOutPtr>readbuf+(ABUFLEN*NDRVRBUF*2)) {
	printf("::read() pointer error: %d\n",
	       readCopyOutPtr-readbuf+(ABUFLEN*NDRVRBUF*2));
      }
      readCopyOutPtr=readbuf;
    }
  }
  return ptr;
}

/*
 * In most mixer devices, there is only a master volume control on
 * the capture channel, so the following code does not really work
 * as expected. The only (partial) exception is the MIC line, where
 * there is generally a 20dB boost which can be enabled or not
 * depending on the type of device.
 */
void OSSAudio::SetRGain(int level)
{    
    // start of new section
  /*    if (!haveaudio())
	Obtain(); */
    // end of new section

    rgain_ = level;
    float x = level;
    level = (int) (x/2.56);
    int foo = (level<<8) | level;
    switch (iport_) {
    case 2:
	if (ioctl(mixerfd, MIXER_WRITE(vol_device), &foo) == -1)
	   perror("failed set input line volume for 2");
	break;
    case 1:
	if (ioctl(mixerfd, MIXER_WRITE(vol_device), &foo) == -1)
	   perror("failed set input line volume for 1");
	break;
    case 0:
	if (ioctl(mixerfd, MIXER_WRITE(vol_device), &foo) == -1)
	   perror("failed set input line volume for 0");
	break;
    }
}

// set the playback gain (volume)
void OSSAudio::SetPGain(int level)
{
    // start of new section
  /*    if (!haveaudio())
		Obtain(); */
    // end of new section
    pgain_ = level;
    float x = level;
    level = (int) (x/2.56);
    int foo = (level<<8) | level;
    if (mixerfd >= 0) {
            if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_PCM), &foo) == -1) {
                    perror("failed to output level");
            }
    }
}

void OSSAudio::OutputPort(int p)
{
    oport_ = p;
}

void OSSAudio::InputPort(int p)
{
    int   zero = 0;

    switch(p) {
    case 2:
	zero = 1 << SOUND_MIXER_CD;
	break;
    case 1:
        zero = 1 << SOUND_MIXER_LINE;
	break;
    case 0 :
	zero = 1 << SOUND_MIXER_MIC;
	break;
    }
    int i=0;
    if ( ioctl(mixerfd, SOUND_MIXER_READ_DEVMASK, &i) == -1 ) {
        perror("failed to select input");
        p = 0;
    }
    if ( ioctl(mixerfd, SOUND_MIXER_READ_RECMASK, &i) == -1 ) {
        perror("failed to select input");
        p = 0;
    }
    if ( ioctl(mixerfd, SOUND_MIXER_READ_RECSRC, &i) == -1 ) {
        perror("failed to select input");
        p = 0;
    }
    if ( ioctl(mixerfd, SOUND_MIXER_WRITE_RECSRC, &zero) == -1 ) {
	perror("failed to select input");
        p = 0;
    }
    iport_ = p;
    if ( ioctl(mixerfd, SOUND_MIXER_READ_RECSRC, &i) == -1 ) {
        perror("failed to select input");
        p = 0;
    }
}

/*
 * FrameReady must return 0 every so often, or the system will keep
 * processing mike data and not other events. This function is called
 * by Audio::dispatch() and if it returns 1 then dispatch will go on
 * to call Controller::handle_audio(). Basically, this function gets
 * audio data from the sound card and stores it in a ring buffer (which
 * is read when the Read() function is called). It returns when there is
 * a full frame of audio data ready.
 */
int OSSAudio::FrameReady()
{
  if (rec_fmt == AFMT_S16_LE) {
    if (use16RawPCM == 0) {
      // we want to store 8 bit mulaw audio in the ring buffer.
      u_short s16Buffer[ABUFLEN];
      if( read(fd_, (u_char*)s16Buffer, ABUFLEN) != (int)(ABUFLEN)) {
	// there isn't a full buffer of audio yet, keep waiting...
	perror("sound device read");
      } else {
	// translate it from 16 bit linear to 8 bit mulaw
	for (int i=0; i< (int)ABUFLEN>>1; i++) {
	  readCopyInPtr[i] = lintomulaw[ s16Buffer[i] & 0xffff ] ;
	}
	readCopyInPtr+=(ABUFLEN>>1);
	// write it to the ring buffer
	if(readCopyInPtr>=readbuf+ABUFLEN*NDRVRBUF) {
	  if(readCopyInPtr>readbuf+ABUFLEN*NDRVRBUF) {
	    printf("ossaudio::frameready pointer error by %d\n",
		   readCopyInPtr-readbuf+ABUFLEN*NDRVRBUF);
	  }
	  readCopyInPtr=readbuf;
	}
      }
    } else {
      // we want to store 16 bit linear audio in the ring buffer.
      if( read(fd_, (u_char*)readCopyInPtr, ABUFLEN) != (int)(ABUFLEN)) {
	// there isn't a full buffer of audio yet, keep waiting...
	perror("sound device read");
      } else
	readCopyInPtr+=ABUFLEN;
      // store it in the ring buffer
      if(readCopyInPtr>=readbuf+(ABUFLEN*NDRVRBUF*2)) {
	if(readCopyInPtr>readbuf+(ABUFLEN*NDRVRBUF*2)) {
	  printf("ossaudio::frameready pointer error by %d\n",
		 readCopyInPtr-readbuf+(ABUFLEN*NDRVRBUF*2));
	}
	readCopyInPtr=readbuf;
      }
    }
  } else {
    // here we are dealing with legacy 8 bit cards...
    if( read(fd_, readCopyInPtr, ABUFLEN) != (int)(ABUFLEN) ) {
      perror("sound device read");
    }
    else {
      if (rec_fmt == AFMT_S8) {
	int i;
	for (i=0; i< (int)ABUFLEN; i++) {
	  readCopyInPtr[i] = lintomulaw[ readCopyInPtr[i]<<8 ] ;
	}
      }
      else if (rec_fmt == AFMT_U8) {
	for (int i=0; i< (int)ABUFLEN; i++) {
	  readCopyInPtr[i] = lintomulaw[ (readCopyInPtr[i]<<8) ^ 0x8000 ] ;
	}
      }
      readCopyInPtr+=ABUFLEN;
      if(readCopyInPtr>=readbuf+ABUFLEN*NDRVRBUF)
	readCopyInPtr=readbuf;   
    }
  }

  int ret;
    
  // return value is dependant on how much data we have saved up...
  if( (unsigned int)(readCopyInPtr)>=(unsigned int)(readCopyOutPtr) ) {
    // if the data doesn't wrap around the end of the buffer
    if (use16RawPCM == 1)
      // wait for a 16 bit frame
      ret= ( (unsigned int)(readCopyInPtr)-(unsigned int)(readCopyOutPtr)>=(blksize_*2));
    else
      // wait for an 8 bit frame
      ret= ( (unsigned int)(readCopyInPtr)-(unsigned int)(readCopyOutPtr)>=blksize_);
    return ret;
  }
  else {
    // the data wraps around the end of the buffer...
    if (use16RawPCM == 1)
      // wait for a 16 bit frame
      ret=((readbuf+(ABUFLEN*NDRVRBUF*2)-readCopyOutPtr+readCopyInPtr-readbuf)
	       >=(int)(blksize_*2));
    else
      // wait for an 8 bit frame
      ret=((readbuf+ABUFLEN*NDRVRBUF-readCopyOutPtr+readCopyInPtr-readbuf)
	       >=(int)blksize_);
    return ret;
  }
}

// commands that allow control from Tcl code
int OSSAudio::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
	  if (strcmp(argv[1], "useRaw16PCM") == 0) {
	    use16RawPCM = 1;
	    return (TCL_OK);
	  }
	  if (strcmp(argv[1], "useStereo") == 0) {
	    stereo = 1;
	    return (TCL_OK);
	  }
	  if (strcmp(argv[1], "in16bitMode") == 0) {
	    sprintf(tcl.result(), "%d", use16RawPCM);
	    return (TCL_OK);
	  }
	  if (strcmp(argv[1], "inStereoMode") == 0) {
	    sprintf(tcl.result(), "%d", stereo);
	    return (TCL_OK);
	  }
	  if (strcmp(argv[1], "getSampleRate") == 0) {
	    sprintf(tcl.result(), "%d", sample_rate);
	    return (TCL_OK);
	  }
	}
	if (argc == 3) {
	  if (strcmp(argv[1], "set_sample_rate") == 0) {
	    sample_rate = atoi(argv[2]);
	    return (TCL_OK);
	  }
	}
	return (Audio::command(argc,argv));
}


