/*
 * postdct.cc --
 *
 * Post-DCT Deblocking Algorithms.
 *
 * Derived from original code provided by Jim Chou. For more information see
 * http://www-wavelet.eecs.berkeley.edu/~jimchou/deblock.html. Published in:
 *
 * <J. Chou, M. Crouse & K. Ramchandran "A simple algorithm for
 * removing blocking artifacts in block-transform coded images,"
 * Proc. of International Conference on Image Processing, ICIP,
 * Chicago, IL, Oct. 1998.>
 *
 * <J. Chou, M. Crouse & K. Ramchandran "A simple algorithm for
 * removing blocking artifacts in block-transform coded images," IEEE
 * Signal Processing Letters, Feb. 1998, Vol. 5, No. 2., pp.33-35.>
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/postdct.cc,v 1.11 2002/02/03 03:13:33 lim Exp $
 */

#include <math.h>
#include <stdlib.h>

#include "config.h"
#include "postdct.h"

double get_threshold(u_int8_t *image, int h_size, int v_size, const short *quant) {
    int index = 0;
    double t = 0;
    double pixel_mean[64], pixel_energy[64], accumulated_mean[64], accumulated_energy[64];

    // Initialize the 8x8 mean and energy matrices to zero.
    {
        for(int i = 0; i < 64; i++) {
          pixel_mean[i] = 0;
          pixel_energy[i] = 0;
          accumulated_mean[i] = 0;
          accumulated_energy[i] = 0;
        }
    }

    // Process each 8x8 block of the image.
    {
        for(int i = 0; i < v_size; i += 8)
	        for (int j = 0; j < h_size; j += 8) {
	            // Increment the index. Used to scale down the normalized
	            // energy for coefficients farther into the process.
	            index++;

	            // Reset the block's total estimated quantization error to
	            // zero.
	            double block_euv = 0;

	            // For each coefficient in this 8x8 block, calculate an
	            // estimated quantization error Euv. Do every fourth pixel
	            // instead of every pixel, because computers are still too
	            // slow. Then multiply by 4 since we need an Euv for the
	            // entire block, not a fourth of it.
	            for (int k = 0; k < 8; k += 2)
		        for (int l = 0; l < 8; l += 2) {
		            // Grab the quantized image value.
		            double pixel = image[(i+k)*h_size+j+l];

		            // Set this coefficient's estimated quantization
		            // error to zero.
		            double euv = 0;

		            // Acculumate the unnormalized energy values
		            // for the k,l pixel across all blocks.
		            accumulated_energy[k*8+l] += pixel*pixel;
		            accumulated_mean[k*8+l] += pixel;

		            // Calculate the normalized energy and mean values
		            // for this coefficient.
		            pixel_energy[k*8+l] = accumulated_energy[k*8+l] / index;
		            pixel_mean[k*8+l] = accumulated_mean[k*8+l] / index;

		            // If this is the upper-left pixel of the 8x8
		            // block, then we're dealing with the DC
		            // coefficient. We model this coefficient as a
		            // Gaussian random variable.
		            if (k == 0 && l == 0) {
			        // Calculate Euv of this coefficient. Divide
		                // by 12 because that's the expression for
		                // calculating the variance of the quantization
		                // error, assuming a uniform distribution.
			        euv = quant[0]*quant[0] / 12;
		            }

		            // Otherwise we're dealing with an AC
		            // coefficient. We model these coefficients as
		            // zero-mean Laplacian random variables. Only
		            // estimate the quantization error if the energy
		            // of this coefficient is not zero. Otherwise, we
		            // can't predict what's going on so don't estimate
		            // any quantization error for this
		            // coefficient. These calculations are for the
		            // closed form of the Laplacian integral.
		            else if (pixel_energy[k*8+l] != 0) {
			        double alpha = 2 / pixel_energy[k*8+l];
			        
			        if (pixel != 0) {
			            double pixel_magnitude = (pixel < 0) ?
				        -pixel : pixel;

			            // For very small x, tanh(x) is
			            // approximately x. And it seems that we
			            // always get very small x here.
			            double tanh_aq = alpha*quant[k*8+l]/2;
			            
			            double postmean = 1 / alpha + pixel_magnitude -
				        quant[k*8+l] / 2 / tanh_aq;
			            euv = pixel_magnitude*pixel_magnitude +
				        quant[k*8+l]*quant[k*8+l]/4 -
			                pixel_magnitude * quant[k*8+l] /
				        tanh_aq + 2 * postmean / alpha;
			            euv -= postmean*postmean;
			        } else {
			            // For x close to zero, exp(x) is
			            // approximately x+1. And it seems that we
			            // always get x close to zero here.
			            double exp_aq = -alpha*quant[k*8+l]/4 + 1;

			            // For very small x, sinh(x) is
			            // approximately x. And it seems that we
			            // always get very small x here.
			            double sinh_aq = alpha*quant[k*8+l]/4;

			            euv = 2 / (alpha*alpha) - exp_aq * 
			                (quant[k*8+l]/alpha+quant[k*8+l]*quant[k*8+l]/4) /
				        (2*sinh_aq);
			        }
		            }
		            
		            // Add this coefficient's estimated quantization
		            // error to this block's current total estimated
		            // quantization error.
		            block_euv += 4 * euv;
		        }

	            // Add this block's estimated quantization error to the
	            // image's total estimated quantization error. T =
	            // 1/n^2*(sum from u=0 to n-1)(sum from v=0 to n-1)Euv. We
	            // take the square root of this since we are going to
	            // return sqrt(T).
	            t += sqrt(block_euv/64);
	    }
    }
	
    // Divide the image's total estimated quantization error by the
    // number of blocks processed.
    t /= h_size * v_size / 64;

    // Return the threshold.
    return t;
}

double get_threshold(int codec, int quality) { 
    // JPEG frames should be deblocked at a non-linear rate inverse to
    // the quality of the compression. We use a linear formula that is
    // something of a best-fit curve to the actual integer thresholds
    // based on calculations done on the Lena Sjblom image. In
    // reality, this linear forumla is a compromise between the
    // calculated PSNR values for quality levels between 30 and 55 and
    // actual perceived quality for quality levels between 55 and 80.
    // The calculated ideal threshold value for quality levels above
    // 55 was zero, but the image requires deblocking up to quality
    // level 80.
    if (codec == CODEC_JPEG) {
        return (quality < 80) ? (-18.0/50.0) * quality + 29.8 : 0;
    }
 
    // H.261 frames have a threshold which corresponds to no deblocking
    // at highest quality of 1 and maximum deblocking at quality
    // 30. This formula is an approximation based on the formula used
    // for JPEG frames.
    if (codec == CODEC_H261) {
        return (quality - 1) / 1.2;
    }
 
    // Un-recognized frames should not be deblocked.
    return 0;
}

void deblock(u_int8_t *image, double v_thresh, double h_thresh, double
vis_thresh, int h_size, int v_size) {
    deblock(image, v_thresh, h_thresh, vis_thresh, h_size, v_size, 0, h_size);
}

void deblock(u_int8_t *image, double v_thresh, double h_thresh, double vis_thresh, int h_size, int v_size, int offset, int strip_length) {
    // Horizontal and vertical alpha values. The alpha values are used
    // to modify the boundary values. See section 3 of Chou's paper.
    double h_alpha = (h_thresh > 0 && h_thresh >= vis_thresh)
	? (h_thresh - vis_thresh) / (2 * h_thresh)
	: 0;
    double v_alpha = (v_thresh > 0 && v_thresh >= vis_thresh)
	? (v_thresh - vis_thresh) / (2 * v_thresh)
	: 0;
    if (h_alpha == 0 && v_alpha == 0) return;

    // When deblocking the entire image, start_col is 0 and end_col is
    // h_size, start_row is 0 and end_row is v_size.
    int start_col = offset % h_size;
    int end_col = start_col + strip_length;
    int start_row = (h_size != 0) ? offset / h_size : 0;
    int end_row = start_row + v_size;

    // Verify that the starting values are multiples of 8. Otherwise we
    // won't be deblocking at the block lines.
    start_col -= start_col % 8;
    start_row -= start_row % 8;

    // Remove artificial discontinuities across horizontal block
    // boundaries.
    if (v_alpha != 0) {
        for(int i = start_col; i < end_col; i++) {
	        for(int j = start_row + 8; j < end_row; j += 8) {
	            // Pre-calculate the image indexes which will be used
	            // multiple times.
	            /*
	            int xy_above4 = (j-4)*h_size+i;
	            int xy_above3 = (j-3)*h_size+i;
	            */
	            int xy_above2 = (j-2)*h_size+i;
	            int xy_above  = (j-1)*h_size+i;
	            int xy        = j*h_size+i;
	            int xy_below  = (j+1)*h_size+i;
	            /*
	            int xy_below2 = (j+2)*h_size+i;
	            int xy_below3 = (j+3)*h_size+i;
	            */

	            // Calculate the difference across the horizontal
	            // boundary.
	            double v_diff_boundary = image[xy_above] - image[xy];

	            // Calculate the difference across the pixels adjacent to
	            // the horizontal boundary to compensate for
	            // discontinuities introducted when the pixels at the
	            // horizontal boundary are altered.
	            /*
	            double v_diff_above3 = image[xy_above4] - image[xy_above3];
	            double v_diff_above2 = image[xy_above3] - image[xy_above2];
	            */
	            double v_diff_above = image[xy_above2] - image[xy_above];
	            double v_diff_below = image[xy] - image[xy_below];
	            /*
	            double v_diff_below2 = image[xy_below] - image[xy_below2];
	            double v_diff_below3 = image[xy_below2] - image[xy_below3];
	            */

	            // For each column, compare the difference across the
	            // horizontal boundary to the vertical threshold t =
	            // 2*sqrt(T). Only alter the boundary pixels by
		    // alpha*d if the difference is between the +/-
		    // visible threshold. Chou's code also altered
		    // boundary pixels outside this range by alpha*t,
		    // but that blurred text and other sharp edges so
		    // we decided not to do that.
	            double alter_amt;
		    if (v_diff_boundary < v_thresh && v_diff_boundary > -v_thresh)
		        alter_amt = v_alpha * v_diff_boundary;
		    else
		        alter_amt = 0;

	            // If this is the first or second column of an 8x8 block,
	            // scale down the alteration amount. This is done to help
	            // account for the fact that corner pixels will get
	            // smoothed over both horizontally and vertically.
	            if (i % 8 == 0 || i % 8 == 1)
		        alter_amt /= 1.2;

	            // Apply the boundary alteration. Since image is a short,
	            // add 0.5 to the float alter_amt in order to round it
	            // off.
	            image[xy_above] += (u_int8_t)(0.5 - alter_amt);
	            image[xy] += (u_int8_t)(0.5 + alter_amt);

	            // Calculate the new difference across pixels adjacent to
	            // the boundary.
	            double new_diff_above = image[xy_above2] - image[xy_above];
	            double new_diff_below = image[xy] - image[xy_below];
	            
	            // If the previous difference was zero, and now there is a
	            // difference, replace the adjacent pixel by an average of
	            // itself and the altered boundary pixel. Since image is a
	            // short, add 0.5 to the calculation to round it off. Note
	            // that in Chou's original code, the divide by 2 was done
	            // with 2, and not 2.0.
	            if (v_diff_above == 0 && new_diff_above != 0)
		        image[xy_above2] = (u_int8_t)(0.5 + (image[xy_above2] +
						          image[xy_above]) /
					              2.0);
	            if (v_diff_below == 0 && new_diff_below != 0)
		        image[xy_below] = (u_int8_t)(0.5 + (image[xy_below] +
						         image[xy]) / 2.0);

	            /*
	            // Calculate the new difference across pixels twice
	            // removed from the boundary.
	            double new_diff_above2 = image[xy_above3] - image[xy_above2];
	            double new_diff_below2 = image[xy_below] - image[xy_below2];

	            // If the previous difference was zero, and now there is a
	            // difference, replace the pixel twice removed by an
	            // average of itself and the altered adjacent pixel. Since
	            // image is a short, add 0.5 to the calculation to round
	            // it off. Note that in Chou's original code, the divide
	            // by 2 was done with 2, and not 2.0.
	            if (v_diff_above2 == 0 && new_diff_above2 != 0)
		        image[xy_above3] = (u_int8_t)(0.5 + (image[xy_above3] +
						          image[xy_above2]) /
					              2.0);
	            if (v_diff_below2 == 0 && new_diff_below2 != 0)
		        image[xy_below2] = (u_int8_t)(0.5 + (image[xy_below2] +
						          image[xy_below]) /
					              2.0);

	            // Calculate the new difference across pixels thrice
	            // removed from the boundary.
	            double new_diff_above3 = image[xy_above4] - image[xy_above3];
	            double new_diff_below3 = image[xy_below2] - image[xy_below3];

	            // If the previous difference was zero, and now there is a
	            // difference, replace the pixel trice removed by an
	            // average of itself and the altered twice removed
	            // pixel. Since image is a short, add 0.5 to the
	            // calculation to round it off. Note that in Chou's
	            // original code, the divide by 2 was done with 2, and not
	            // 2.0.
	            if (v_diff_above3 == 0 && new_diff_above3 != 0)
		        image[xy_above4] = (u_int8_t)(0.5 + (image[xy_above4] +
					                  image[xy_above3]) /
					              2.0);
	            if (v_diff_below3 == 0 && new_diff_below3 != 0)
		        image[xy_below3] = (u_int8_t)(0.5 + (image[xy_below3] +
						          image[xy_below2]) /
					              2.0);
	            */
	        }
        }
    }

    // Remove artificial discontinuities across vertical block
    // boundaries.
    if (h_alpha != 0) {
        for(int i = start_col + 8; i < end_col; i += 8) {
	        for(int j = start_row; j < end_row; j++) {
	            // Pre-calculate the image indexes which will be used
	            // multiple times.
	            /*
	            int xy_left4  = j*h_size+i-4;
	            int xy_left3  = j*h_size+i-3;
	            */
	            int xy_left2  = j*h_size+i-2;
	            int xy_left   = j*h_size+i-1;
	            int xy        = j*h_size+i;
	            int xy_right  = j*h_size+i+1;
	            /*
	            int xy_right2 = j*h_size+i+2;
	            int xy_right3 = j*h_size+i+3;
	            */

	            // Calculate the difference across the vertical boundary.
	            double h_diff_boundary = image[xy_left] - image[xy];

	            // Calculate the difference across the pixels adjacent to
	            // the vertical boundary to compensate for discontinuities
	            // introduced when the pixels are the vertical boundary
	            // are altered.
	            /*
	            double h_diff_left3 = image[xy_left4] - image[xy_left3];
	            double h_diff_left2 = image[xy_left3] - image[xy_left2];
	            */
	            double h_diff_left = image[xy_left2] - image[xy_left];
	            double h_diff_right = image[xy] - image[xy_right];
	            /*
	            double h_diff_right2 = image[xy_right] - image[xy_right2];
	            double h_diff_right3 = image[xy_right2] - image[xy_right3];
	            */

	            // For each row, compare the difference across the
	            // vertical boundary to the horizontal threshold t =
	            // 2*sqrt(T). Only alter the boundary pixels by
		    // alpha*d if the difference is between the +/-
		    // visible threshold. Chou's code also altered
		    // boundary pixels outside this range by alpha*t,
		    // but that blurred text and other sharp edges so
		    // we decided not to do that.
	            double alter_amt;
	            if (h_diff_boundary < h_thresh && h_diff_boundary > -h_thresh)
		        alter_amt = h_alpha * h_diff_boundary;
		    else
		        alter_amt = 0;

	            // If this is the first or second row of an 8x8 block,
	            // scale down the alteration amount. This is done to help
	            // account for the fact that corner pixels will get
	            // smoothed over both horizontally and vertically.
	            if (j % 8 == 0 || j % 8 == 1)
		        alter_amt /= 1.2;

	            // Apply the boundary alteration. Since image is a short,
	            // add 0.5 to the float alter_amt in order to round it
	            // off.
	            image[xy_left] += (u_int8_t)(0.5 - alter_amt);
	            image[xy] += (u_int8_t)(0.5 + alter_amt);
	            
	            // Calculate the new difference across pixels adjacent to
	            // the boundary.
	            double new_diff_left = image[xy_left2] - image[xy_left];
	            double new_diff_right = image[xy] - image[xy_right];

	            // If the previous difference was zero, and now there is a
	            // difference, replace the adjacent pixel by an average of
	            // itself and the altered boundary pixel. Since image is a
	            // short, add 0.5 to the calculation to round it off. Note
	            // that in Chou's original code, the divide by 2 was done
	            // with 2, and not 2.0.
	            if (h_diff_left == 0 && new_diff_left != 0)
		        image[xy_left2] = (u_int8_t)(0.5 + (image[xy_left2] +
						         image[xy_left])
						        / 2);
	            if (h_diff_right == 0 && new_diff_right != 0)
		        image[xy_right] = (u_int8_t)(0.5 + (image[xy_right] +
						         image[xy])
						        / 2);

	            /*
	            // Calculate the new difference across pixels twice
	            // removed from the boundary.
	            double new_diff_left2 = image[xy_left3] - image[xy_left2];
	            double new_diff_right2 = image[xy_right] - image[xy_right2];

	            // If the previous difference was zero, and now there is a
	            // difference, replace the pixel twice removed by an
	            // average of itself and the altered adjacent pixel. Since
	            // image is a short, add 0.5 to the calculation to round
	            // it off. Note that in Chou's original code, the divide
	            // by 2 was done with 2, and not 2.0.
	            if (h_diff_left2 == 0 && new_diff_left2 != 0)
		        image[xy_left3] = (u_int8_t)(0.5 + (image[xy_left3] +
						         image[xy_left2])
						        / 2);
	            if (h_diff_right2 == 0 && new_diff_right2 != 0)
		        image[xy_right2] = (u_int8_t)(0.5 + (image[xy_right2] +
						          image[xy_right])
						        / 2);

	            // Calculate the new difference across pixels thrice
	            // removed from the boundary.
	            double new_diff_left3 = image[xy_left4] - image[xy_left3];
	            double new_diff_right3 = image[xy_right2] - image[xy_right3];

	            // If the previous difference was zero, and now there is a
	            // difference, replace the pixel trice removed by an
	            // average of itself and the altered twice removed
	            // pixel. Since image is a short, add 0.5 to the
	            // calculation to round it off. Note that in Chou's
	            // original code, the divide by 2 was done with 2, and not
	            // 2.0.
	            if (h_diff_left3 == 0 && new_diff_left3 != 0)
		        image[xy_left4] = (u_int8_t)(0.5 + (image[xy_left4] +
						         image[xy_left3])
						        / 2);
	            if (h_diff_right3 == 0 && new_diff_right3 != 0)
		        image[xy_right3] = (u_int8_t)(0.5 + (image[xy_right3] +
						          image[xy_right2])
						        / 2);
	            */
	        }
        }
    }
}
