/*
 * mbv2-cmd.h --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *  @(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-cmd.h,v 1.8 2002/02/03 03:17:08 lim Exp $
 */

#ifndef MASH_MBV2_CMD_H
#define MASH_MBV2_CMD_H

#include "mbv2-item.h"
#include "mtrace.h"


class MBv2Page;

class MBv2Cmd {
public:
	MBv2Cmd(MBv2CmdType type) : cmdType_(type), isComplete_(0),
		id_(MBv2InvalidCmdId), tag_(MBv2InvalidCmdId) { }
	MBv2Cmd(MBv2CmdType type, const MBv2Time &ts)
		: cmdType_(type), isComplete_(0), id_(MBv2InvalidCmdId)
	{ timestamp_ = ts; }
	virtual ~MBv2Cmd();
	inline MBv2CmdId id() { return id_; }
	inline void id(MBv2CmdId i) { id_ = i; }
	inline MBv2CmdType type() { return (MBv2CmdType)cmdType_; }
	inline const MBv2Time &timestamp() { return timestamp_; }
	inline MBv2CmdId tag() { return tag_; }
	inline void tag(MBv2CmdId tagid) { tag_ = tagid; }

	Bool execute(MBv2Page *page, const MBv2Time &currentTime,
		     const MBv2Time &targetTime);
	virtual Bool apply(MBv2Page *page)=0;
	virtual Bool reverse(MBv2Page * /*page*/,
			     const MBv2Time& /*targetTime*/) { return TRUE; };
	virtual Bool incomplete(MBv2Page* /*page*/, MBv2Dependency* /*dep*/=0)
	{ return 0; }

	static MBv2Cmd *create(MBv2CmdId id, const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb)=0;
	virtual int adu_size()=0;

	virtual MBv2Item *make_copy(MBv2Page* /*page*/) { return NULL; }
	virtual const MBv2Time &raise_timestamp(MBv2Page* /*page*/)
	{ return timestamp_; }

	virtual Bool target(MBv2Page * /*myPage*/, MBv2Page *& /*page*/,
			    MBv2Cmd *& /*cmd*/){
		MTrace(trcMB, ("no target for this command %u", id_));
		return FALSE;
	}

	static MBv2Cmd * create(MBv2CmdId id, const srm_adu_info *info,
				const Byte *pb, int len);
private:
	virtual Bool extract(const Byte *pb, int len)=0;
	Bool active_at(const MBv2Time &t) {
		return (t >= timestamp_);
	}

protected:
	u_int16_t cmdType_;   // don't use an entire int here
	u_int8_t  isComplete_;/* used by incomplete() to avoid multiple
			       * checks once we know that the cmd is complete*/
	MBv2CmdId id_, tag_;  /* all commands that operate on the same "object"
			       * have the same tag */
	MBv2Time timestamp_;
};


class MBv2CmdCreate : public MBv2Cmd {
public:
	MBv2CmdCreate() : MBv2Cmd(MBv2Create), item_(NULL) { }
	MBv2CmdCreate(const MBv2Time &ts, MBv2Item *item)
		: MBv2Cmd(MBv2Create, ts), item_(item) { }

	virtual ~MBv2CmdCreate() { if (item_) delete item_; }

	MBv2Item *item() { return item_; }
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return sizeof(ADU_CmdCreate) + item_->adu_size(); }
	virtual MBv2Item *make_copy(MBv2Page* page);
	virtual Bool apply(MBv2Page *page);

	MBv2Item *item_;
};


class MBv2CmdGroup : public MBv2Cmd {
public:
	MBv2CmdGroup(MBv2ItemType type) : MBv2Cmd(MBv2Group), item_(NULL),
		groupType_(type),
		start_(MBv2InvalidCmdId), end_(MBv2InvalidCmdId) { }
	MBv2CmdGroup(MBv2ItemType type, const MBv2Time &ts,
		     MBv2CmdId start, MBv2CmdId end)
		: MBv2Cmd(MBv2Group, ts), item_(NULL), groupType_(type),
		  start_(start), end_(end) { }
	virtual ~MBv2CmdGroup() { if (item_) delete item_; }

	MBv2Item *item() { return item_; }
	inline MBv2CmdId start_cmdid() { return start_; }

	static MBv2Cmd *create(const Byte *pb, int len);
	static MBv2Cmd *create(MBv2ItemType type, const MBv2Time &ts,
			       MBv2CmdId start, MBv2CmdId end);
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return sizeof(ADU_CmdGroup); }
	virtual Bool incomplete(MBv2Page* page, MBv2Dependency* dep=0);
	virtual MBv2Item *make_copy(MBv2Page* page);

protected:
	MBv2Item *item_;
	MBv2ItemType groupType_;
	MBv2CmdId start_, end_;
};


class MBv2CmdFreeHand : public MBv2CmdGroup {
public:
	MBv2CmdFreeHand() : MBv2CmdGroup(MBv2Line) { }
	MBv2CmdFreeHand(const MBv2Time &ts, MBv2CmdId start, MBv2CmdId end)
		: MBv2CmdGroup(MBv2Line, ts, start, end) { }
private:
	virtual Bool apply(MBv2Page *page);
};


/*
 * A text object consists of an MBv2CmdCreate, followed by a bunch of
 * MBv2CmdChars, followed by an MBv2CmdGroup
 */
class MBv2CmdText : public MBv2CmdGroup {
public:
	MBv2CmdText() : MBv2CmdGroup(MBv2Text) { }
	MBv2CmdText(const MBv2Time &ts, MBv2CmdId start, MBv2CmdId end)
		: MBv2CmdGroup(MBv2Text, ts, start, end) { }

	static MBv2Item *item(MBv2Cmd *cmd);
private:
	virtual Bool apply(MBv2Page *page);
};


class MBv2CmdCharOrChars : public MBv2Cmd {
public:
	MBv2CmdCharOrChars(MBv2CmdType type) : MBv2Cmd(type),
		createCmd_(MBv2InvalidCmdId), index_(0xFFFF) { }
	MBv2CmdCharOrChars(MBv2CmdType type, const MBv2Time &ts,
			   MBv2CmdId createCmd, u_int16_t index)
		: MBv2Cmd(type, ts), createCmd_(createCmd), index_(index) { }
	virtual ~MBv2CmdCharOrChars() { }

	u_int16_t index() { return index_; }
	virtual char *create_text(MBv2Page *page,
				  MBv2CmdId start=MBv2InvalidCmdId);
protected:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return sizeof(ADU_CmdChars); }
	virtual Bool incomplete(MBv2Page* page, MBv2Dependency* dep=0);
	virtual void insert_chars(char *&text, u_int16_t &text_len,
				  u_int16_t &max_chars)=0;
	void grow_text(char *&text, u_int16_t text_len);

	MBv2CmdId createCmd_;
	u_int16_t index_;
};


class MBv2CmdChar : public MBv2CmdCharOrChars {
public:
	MBv2CmdChar() : MBv2CmdCharOrChars(MBv2Char), char_(0) { }
	MBv2CmdChar(const MBv2Time &ts, MBv2CmdId createCmd, u_int16_t index,
		    char chr) : MBv2CmdCharOrChars(MBv2Char, ts, createCmd,
						   index), char_(chr) { }
	virtual ~MBv2CmdChar() { }

	char get_char() { return char_; }
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return MBv2CmdCharOrChars::adu_size()+1; }
	virtual Bool apply(MBv2Page *page);
	virtual void insert_chars(char *&text, u_int16_t &text_len,
				  u_int16_t &max_chars);
	char char_;
};


class MBv2CmdChars : public MBv2CmdCharOrChars {
public:
	MBv2CmdChars() : MBv2CmdCharOrChars(MBv2Chars), chars_(0) { }
	MBv2CmdChars(const MBv2Time &ts, MBv2CmdId createCmd, u_int16_t index,
		     const char *chars)
		: MBv2CmdCharOrChars(MBv2Chars, ts, createCmd, index),chars_(0)
	{ chars_=new char [strlen(chars)+1]; strcpy(chars_, chars); }
	virtual ~MBv2CmdChars() { if (chars_) delete [] chars_; }

	const char *get_chars() { return chars_; }
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return MBv2CmdCharOrChars::adu_size() + strlen(chars_) + 1; }
	virtual Bool apply(MBv2Page *page);
	virtual void insert_chars(char *&text, u_int16_t &text_len,
				  u_int16_t &max_chars);
	char *chars_;
	// index of (u_int16_t)-1 implies that we are replacing the
	// entire original text
};


class MBv2CmdMove : public MBv2Cmd {
public:
	MBv2CmdMove() : MBv2Cmd(MBv2Move), targetCmd_(MBv2InvalidCmdId),
		dx_(0), dy_(0) { }
	MBv2CmdMove(const MBv2Time &ts, MBv2CmdId targetCmd,
		    int32_t dx, int32_t dy) : MBv2Cmd(MBv2Move,ts),
			targetCmd_(targetCmd), dx_(dx), dy_(dy) { }
	virtual ~MBv2CmdMove() { }

	inline int32_t dx() { return dx_; }
	inline int32_t dy() { return dy_; }

	virtual Bool target(MBv2Page *myPage, MBv2Page *&page, MBv2Cmd *&cmd) {
		page = myPage;
		cmd  = page->cmd(targetCmd_);
		if (!cmd) {
			MTrace(trcMB, ("command %u doesn't exist, "
				       "shouldn't happen", targetCmd_));
			return FALSE;
		} else return TRUE;
	}
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return sizeof(ADU_CmdMove); }
	virtual MBv2Item *make_copy(MBv2Page* page);
	virtual Bool apply(MBv2Page *page);
	virtual Bool incomplete(MBv2Page* page, MBv2Dependency* dep=0);
	virtual const MBv2Time &raise_timestamp(MBv2Page* page);

	MBv2CmdId targetCmd_;
	MBv2Time  raiseTS_;
	int32_t dx_, dy_;
};


class MBv2CmdCopy : public MBv2Cmd {
public:
	MBv2CmdCopy(MBv2CmdType type) : MBv2Cmd(type), targetCid_(0),
		targetCmd_(MBv2InvalidCmdId), item_(NULL) { }

	MBv2CmdCopy(MBv2CmdType type, const MBv2Time &ts, u_int32_t targetCid,
		    MBv2CmdId targetCmd)
		: MBv2Cmd(type, ts), targetCid_(targetCid),
		  targetCmd_(targetCmd), item_(NULL) { }

	virtual ~MBv2CmdCopy() { if (item_) delete item_; }
	MBv2Item *item() { return item_; }

	virtual Bool target(MBv2Page *myPage, MBv2Page *&page, MBv2Cmd *&cmd) {
		page = myPage->source()->find_page(targetCid_);
		if (!page) {
			MTrace(trcMB, ("page %u doesn't exist; "
				       "shouldn't happen", targetCid_));
			return FALSE;
		}
		cmd = page->cmd(targetCmd_);
		if (!cmd) {
			MTrace(trcMB, ("command %u doesn't exist, "
				       "shouldn't happen", targetCmd_));
			return FALSE;
		} else return TRUE;
	}
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return sizeof(ADU_CmdCopy); }
	virtual MBv2Item *make_copy(MBv2Page* page);
	virtual Bool apply(MBv2Page *page);
	virtual Bool incomplete(MBv2Page* page, MBv2Dependency* dep=0);
	virtual const MBv2Time &raise_timestamp(MBv2Page *page);

protected:
	u_int32_t targetCid_;
	MBv2CmdId targetCmd_;
	MBv2Item  *item_;
	MBv2Time  raiseTS_;
};


class MBv2CmdDelete : public MBv2Cmd {
public:
	MBv2CmdDelete() : MBv2Cmd(MBv2Delete), targetCmd_(MBv2InvalidCmdId) {}
	MBv2CmdDelete(const MBv2Time &ts, MBv2CmdId targetCmd)
		 : MBv2Cmd(MBv2Delete, ts), targetCmd_(targetCmd) { }
	virtual ~MBv2CmdDelete() { }

	inline MBv2CmdId target_cmd() { return targetCmd_; }

	virtual Bool target(MBv2Page *myPage, MBv2Page *&page, MBv2Cmd *&cmd) {
		page = myPage;
		cmd  = page->cmd(targetCmd_);
		if (!cmd) {
			MTrace(trcMB, ("command %u doesn't exist, "
				       "shouldn't happen", targetCmd_));
			return FALSE;
		} else return TRUE;
	}
private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size() { return sizeof(ADU_CmdDelete); }
	virtual Bool apply(MBv2Page *page);
	virtual Bool incomplete(MBv2Page* page, MBv2Dependency* dep=0);

	MBv2CmdId targetCmd_;
};


inline MBv2Item
*MBv2CmdText::item(MBv2Cmd *cmd)
{
	switch (cmd->type()) {
	case MBv2Create:   return ((MBv2CmdCreate*)cmd)->item();
	case MBv2EditText: return ((MBv2CmdCopy  *)cmd)->item();
	default:      return NULL;
	}
}

#endif /* #ifdef MASH_MBV2_CMD_H */
