/*
 * mbv2-session.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-session.cc,v 1.16 2002/02/03 03:17:08 lim Exp $";
#endif


#include "mbv2-obj.h"
#include "mbv2-cmd.h"
#include "mbv2-canv.h"
#include <mtrace.h>


DEFINE_OTCL_CLASS(MBv2Session, "MBv2Session") {
	INSTPROC_PUBLIC(reset);
	INSTPROC_PUBLIC(sender);
	INSTPROC_PUBLIC(find_owner);
	INSTPROC_PUBLIC(drop_probability);
}


MBv2Session::MBv2Session() : srm_session_(NULL), sender_(NULL),
	adu_buf_(NULL), adu_buf_size_(0), lastActivePage_(NULL),
	lastActiveCmd_(NULL), lastActiveTimerId_(NULL)
{
	Tcl_InitHashTable(&htSrcs_, TCL_ONE_WORD_KEYS);
}


MBv2Session::~MBv2Session()
{
	// FIXME: must clean up per-source state first!
	if (lastActiveTimerId_) Tcl_DeleteTimerHandler(lastActiveTimerId_);
	Tcl_DeleteHashTable(&htSrcs_);
}


int
MBv2Session::init(int argc, const char * const *argv)
{
	const char *addrstr;
	unsigned int addr, sport, rport, ttl;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(addrstr);
	ARG(sport);
	ARG(rport);
	ARG(ttl);
	END_PARSE_ARGS;

	addr = inet_addr(addrstr);
	if (addr==(unsigned int)-1) {
		Tcl::instance().resultf("invalid address %s", addrstr);
		return TCL_ERROR;
	}

	srm_session_ = srm_create_session(addr, sport, rport, ttl);
	srm_callbacks cb = {
		MBv2Session::recv,
		MBv2Session::should_recover,
		MBv2Session::read_adu,
		MBv2Session::source_update,
		MBv2Session::recv_cid,
		this
	};
	srm_set_callbacks(srm_session_, &cb);
	return TCL_OK;
}


int
MBv2Session::drop_probability(int argc, const char * const *argv)
{
	double prob;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG_DEFAULT(prob, -1.0);
	END_PARSE_ARGS;

	if (prob < 0.0) {
		prob = srm_get_drop_probability(srm_session_);
	} else {
		prob = srm_set_drop_probability(srm_session_, prob);
	}
	Tcl::instance().resultf("%g", prob);
	return TCL_OK;
}


int
MBv2Session::reset(int argc, const char * const *argv)
{
	unsigned int addr, sport, rport, ttl;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(addr);
	ARG(sport);
	ARG(rport);
	ARG(ttl);
	END_PARSE_ARGS;

	srm_reset_session(srm_session_, addr, sport, rport, ttl);
	return TCL_OK;
}


int
MBv2Session::sender(int argc, const char * const *argv)
{
	TclObject *s;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(s);
	END_PARSE_ARGS;
	sender_ = (MBv2Source*) s;
	return TCL_OK;
}


int
MBv2Session::find_owner(int argc, const char * const *argv)
{
	int canvid_int;
	const char *pageid_str;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(pageid_str);
	ARG(canvid_int);
	END_PARSE_ARGS;

	MBv2PageId pageid(pageid_str);
	MBv2CanvId canvid = (MBv2CanvId) canvid_int;
	MBv2CmdId  cmdid;
	MBv2Source *src;

	Tcl_HashSearch hsearch;
	Tcl_HashEntry *entry = Tcl_FirstHashEntry(&htSrcs_, &hsearch);
	while (entry) {
		src = (MBv2Source*)Tcl_GetHashValue(entry);
		MBv2Page *page = src->find_page(pageid);
		if (page) {
			cmdid = page->get_cmdid(canvid);
			if (cmdid != MBv2InvalidCmdId) {
				// found the owner
				char srcid[48];
				srm_srcid2str(srm_get_source_id(
					src->srm_source()), srcid);
				Tcl::instance().resultf("%s", srcid);
				return TCL_OK;
			}
		}
		entry = Tcl_NextHashEntry(&hsearch);
	}

	// didn't find it
	Tcl::instance().result("");
	return TCL_OK;
}


MBv2Canvas *
MBv2Session::get_canvas(const MBv2PageId &pageid)
{
	char pageidStr[48];
	pageid.to_string(pageidStr);

	if (Invoke("get_canvas", pageidStr, NULL)!=TCL_OK) {
		MTrace(trcMB, ("tcl error: %s\n%s", Tcl::instance().result(),
			       Tcl_GetVar(Tcl::instance().interp(),
					  "errorInfo", TCL_GLOBAL_ONLY)));
		return NULL;
	}

	Tcl &tcl = Tcl::instance();
	TclObject *canv = tcl.lookup(tcl.result());
	return (MBv2Canvas*)canv;
}


void
MBv2Session::activity(MBv2Page *page, MBv2Cmd *cmd)
{
	// notify the tcl code only every 250ms
	// that way we don't slow down the drawing mode

	MBv2Time now = current_time();

	if (lastActiveTimerId_) Tcl_DeleteTimerHandler(lastActiveTimerId_);

	if (lastActivePage_==page) {
		u_int32_t diff = 0xFFFFFFFF;
		if (now.lower < lastActiveTime_.lower) {
			diff = (0xFFFFFFFF - lastActiveTime_.lower) +
				now.lower + 1;

			if (now.upper > lastActiveTime_.upper + 1)
				diff = 0xFFFFFFFF;
		} else {
			diff = now.lower - lastActiveTime_.lower;
			if (now.upper > lastActiveTime_.upper)
				diff = 0xFFFFFFFF;
		}

		/* 250ms in NTP representation is
		 * 250000 * 2^32 / 10^6 = 1073741824 */
		if (diff < 1073741824) {
			/* set a Tcl timer, so that do_activity does get called
			 * eventually */
			lastActiveTimerId_ = Tcl_CreateTimerHandler(
				300, MBv2Session::delayed_activity,
				(ClientData)this);
			lastActiveCmd_  = cmd;
			return;
		}
	}

	lastActivePage_ = page;
	lastActiveCmd_  = cmd;
	lastActiveTime_ = now;

	do_activity();
}


void
MBv2Session::do_activity()
{
	char srcid[48], pageid[48], cmdid[20], canvid[20];
	MBv2CanvId id;
	srm_srcid2str(srm_get_source_id(
		lastActivePage_->source()->srm_source()), srcid);
	lastActivePage_->id().to_string(pageid);
	sprintf(cmdid, "%u", lastActiveCmd_->id());
	id = lastActivePage_->get_canvas_item(lastActiveCmd_->id());
	if (id==MBv2InvalidCanvId) {
		*canvid = '\0';
	} else {
		sprintf(canvid, "%u", id);
	}

	if (Invoke("activity", srcid, pageid, cmdid, canvid,
		   (lastActivePage_->is_local() ? "1" : "0"), NULL)!=TCL_OK) {
		MTrace(trcMB, ("tcl error: %s\n%s", Tcl::instance().result(),
			       Tcl_GetVar(Tcl::instance().interp(),
					  "errorInfo", TCL_GLOBAL_ONLY)));
	}
}


void
MBv2Session::delayed_activity(ClientData cd)
{
	((MBv2Session*)cd)->do_activity();
}


void
MBv2Session::rearrange(MBv2Page *page, MBv2Cmd *cmd)
{
	// change the display order of the targeted item to reflect
	// the timestamp
	MBv2CanvId targetId = page->get_canvas_item(cmd->id());
	if (targetId!=MBv2InvalidCanvId) {
		MBv2CanvId canvId = find_most_recent(page->id(), cmd->
						     raise_timestamp(page));
		// canvId may be MBv2InvalidCanvId
		if (canvId != MBv2InvalidCanvId)
			page->canvas()->raise_after(canvId, targetId);
	}
}


// finds the most recent item just before the given timestamp
//   - loops thru all sources and asks each source to return
//     the most recent before timestamp
//
MBv2CanvId
MBv2Session::find_most_recent(MBv2PageId pageid, const MBv2Time &timestamp)
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *entry = Tcl_FirstHashEntry(&htSrcs_, &hsearch);
	MBv2Time currMostRecentTS;
	MBv2CanvId canvId, canvIdMostCurr=MBv2InvalidCanvId;
	while (entry) {
		MBv2Source *src = (MBv2Source*)Tcl_GetHashValue(entry);
		MBv2Page *page = src->find_page(pageid);
		if (page) {
			canvId = page->find_most_recent(currMostRecentTS,
							timestamp);
			if (canvId != MBv2InvalidCanvId) canvIdMostCurr=canvId;
		}
		entry = Tcl_NextHashEntry(&hsearch);
	}
	return canvIdMostCurr;
}


MBv2Source *
MBv2Session::source(srm_source_t source)
{
	MBv2Source *src;
	int created;
	Tcl_HashEntry *entry=Tcl_CreateHashEntry(&htSrcs_, (char*)source,
						 &created);
	if (created) {
		char str[32];
		sprintf(str, "%d", (int)source);
		// the cname field is a dummy in the call to "new MBv2Source"
		// when creating a remote source
		src = (MBv2Source*) TclObject::New("MBv2Source", name(), "",
						   (char*)str, NULL);
		Tcl_SetHashValue(entry, src);
		// make a dummy call to source_update
		source_update(src->srm_source(), (unsigned char*)"", 0);
	} else {
		src = (MBv2Source*) Tcl_GetHashValue(entry);
	}
	return src;
}


Bool
MBv2Session::source(srm_source_t s1, MBv2Source *s2)
{
	int created;
	Tcl_HashEntry *entry=Tcl_CreateHashEntry(&htSrcs_, (char*)s1,
						 &created);
	if (created) {
		Tcl_SetHashValue(entry, s2);
		return TRUE;
	} else return FALSE;
}


MBv2Source*
MBv2Session::get_source(srm_source_t source)
{
	srm_session_t s = srm_get_session(source);
	const srm_callbacks *cb = srm_get_callbacks(s);
	MBv2Session *session = (MBv2Session*) (cb->user_hook);
	return session->source(source);
}


void
MBv2Session::recv(srm_source_t source,
		  unsigned int cid, unsigned int seqno,
		  const unsigned char *data, int len,
		  const srm_adu_info *info)
{
	MBv2Source *s = get_source(source);
	if (s) s->recv(cid, seqno, data, len, info);
}


int
MBv2Session::should_recover(srm_source_t source, unsigned int cid,
			    unsigned int /*sseq*/, unsigned int /*eseq*/)
{
	// FIXME: srm_recover isn't implemented yet; so for now
	// recover everything
	return 1;

	MBv2Source *s = get_source(source);
	if (!s) return 1;
	MBv2Session *sess = s->session();
	if (!sess) return 1;

	// check if the current page is the same as this one
	// if we are currently viewing this page, we must recover
	// otherwise, check if something on the current page is
	// waiting for stuff on this page
	//
	// FIXME:if we are actively following a speaker, we should always
	// recover data for that speaker

	MBv2Page *page = s->find_page(cid);
	if (!page) return 1;

	if (page->is_visible())
		return 1;
	else
		return 0;
}


void
MBv2Session::read_adu(srm_source_t source,
		      unsigned int cid, unsigned int seqno,
		      unsigned char **data_ptr, unsigned int *len_ptr,
		      srm_free_proc *free_proc_ptr,
		      srm_adu_info *info)
{
	MBv2Source *s = get_source(source);
	if (s) s->read_adu(cid, seqno, data_ptr, len_ptr, free_proc_ptr, info);
}


void
MBv2Session::source_update(srm_source_t source,
			   const unsigned char *info,
			   int /*info_len*/)
{
	MBv2Source *s = get_source(source);
	MBv2Session *sess;
	if (!s || (sess = s->session())==NULL) return;

	if (sess->Invoke("source_update", s->name(),
			 (info ? info : (unsigned char*)""), NULL) != TCL_OK) {
		MTrace(trcMB, ("tcl error: %s\n%s", Tcl::instance().result(),
			       Tcl_GetVar(Tcl::instance().interp(),
					  "errorInfo", TCL_GLOBAL_ONLY)));
	}
}


void
MBv2Session::recv_cid(srm_source_t source, unsigned int cid,
		      unsigned int /*parent*/,
		      const unsigned char *name, int name_len)
{
	MBv2Source *s = get_source(source);
	if (s && cid) s->recv_cid(cid, name, name_len);
}

