/*
 * source.cc --
 *
 *      C++ implementation for two objects:
 *
 *        - Source (Source/RTP in tcl) represents the sources belonging 
 *          to an RTP session
 *
 *        - SourceManager is a structure containing a list of all the 
 *          sources that belong to an RTP session.
 *
 * Copyright (c) 1994-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/rtp/source.cc,v 1.34 2002/02/03 04:15:47 lim Exp $";

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "sys-time.h"
#include "source.h"
#include "inet.h"
#include "tclcl.h"
#include "ntp-time.h"

/* gray out src if no ctrl msgs for this many consecutive update intervals */
#define CTRL_IDLE 8.

#define SHASH(a) ((int)((((a) >> 20) ^ ((a) >> 10) ^ (a)) & (SOURCE_HASH-1)))

/*FIXME*/
PacketHandler::~PacketHandler() { }

/*FIXME: there is some confusion here between Source and Source/RTP */

/*
 * <otcl> Class Source
 * Source is the
 * base class for objects that represent the source of a media, data,
 * or control flow flow within some underlying communication session.
 * It represents the demultiplexing granularity for this layer,
 * e.g., packets are dispatched from an aggregate session object to
 * a fine-grained source object based on some source identifier,
 * e.g., an IP address, an RTP SRCID, or an SRM global ID.
 *
 * <otcl> Class Source/RTP -superclass Source
 * Source/RTP is the
 * base class for objects that represent the source of an RTP
 * flow within the underlying RTP session.  Whenever a new source
 * is detected, a source object is created and is registered by
 * invoking the register method on the RTP_Agent class that manages
 * the corresponding session.
 *
 * <otcl> Source/RTP private init { srcid ssrc addr }
 * The constructor of an rtp source object.  <i>srcid</i> is
 * the source's RTP SRCID, <i>ssrc</i> is the RTP syncronization srcid,
 * and <i>addr</i> is the IP address of the packet flow.  Typically,
 * srcid and ssrc are the same and addr is the address of the sending
 * host.  But if a gateway or mixer is involved, then ssrc and addr
 * correspond to the gateway while srcid represents the original
 * packet source located elsewhere in the network.
 */
static class SourceClass : public TclClass {
public:
	SourceClass() : TclClass("Source/RTP") {}
	TclObject* create(int argc, const char*const* argv) {
		if (argc != 7)
			/*FIXME*/
			abort();
		u_int32_t srcid = strtoul(argv[4], NULL, 10);
		u_int32_t ssrc = strtoul(argv[5], NULL, 10);
		u_int32_t addr = strtoul(argv[6], NULL, 10);
		return (new Source(srcid, ssrc, addr));
	}
} source_class;

static class SourceLayerClass : public TclClass {
public:
	SourceLayerClass() : TclClass("SourceLayer/RTP") {}
	TclObject* create(int, const char*const*) {
		return (new Source::Layer());
	}
} sourcelayer_class;

Source::Layer::Layer()
	: fs_(0),
	  cs_(0),
	  np_(0),
	  nf_(0),
	  nb_(0),
	  nm_(0),
	  snp_(0),
	  sns_(0),
	  ndup_(0),
	  nrunt_(0),
	  sts_data_(0),
	  sts_ctrl_(0),
	  ntp_ts_sec_(0),
	  ntp_ts_fsec_(0),
	  mts_(0),
	  ref_ntp_sec_(0),
	  ref_ntp_fsec_(0),
	  ref_mts_(0)
{
	bind("nrunt_", (int*)&nrunt_);
	bind("ndup_", (int*)&ndup_);
	bind("fs_", (int*)&fs_);
	bind("cs_", (int*)&cs_);
	bind("np_", (int*)&np_);
	bind("nf_", (int*)&nf_);
	bind("nb_", (int*)&nb_);
	bind("nm_", (int*)&nm_);
	bind("np_", (int*)&np_);
	bind("ntp_ts_sec_", (unsigned int *)&ntp_ts_sec_);
	bind("ntp_ts_fsec_", (unsigned int *)&ntp_ts_fsec_);
	bind("mts_", (unsigned int *)&mts_);
	bind("ref_ntp_sec_", (unsigned int *)&ref_ntp_sec_);
	bind("ref_ntp_fsec_", (unsigned int *)&ref_ntp_fsec_);
	bind("ref_mts_", (unsigned int *)&ref_mts_);


	/*
	 * Put an invalid seqno in each slot to guarantee that
	 * we don't count any initial dups by mistake.
	 */
	int i;
	for (i = 0; i < SOURCE_NSEQ; ++i)
		seqno_[i] = i + 1;

	/*
	 * Initialize the timevals to zero
	 */
	lts_data_.tv_sec  = 0;
	lts_data_.tv_usec = 0;
	lts_ctrl_.tv_sec  = 0;
	lts_ctrl_.tv_usec = 0;
}

void Source::Layer::clear_counters()
{
	np_ = 0;
	nf_ = 0;
	nb_ = 0;
	nm_ = 0;
	snp_ = 0;
	sns_ = 0;
	ndup_ = 0;
	nrunt_ = 0;

	lts_data_.tv_sec = 0;
	lts_data_.tv_usec = 0;
	lts_ctrl_.tv_sec = 0;
	lts_ctrl_.tv_usec = 0;
}

Source::Source(u_int32_t srcid, u_int32_t ssrc, u_int32_t addr)
	: TclObject(),
	  next_(0),
	  hlink_(0),
	  handler_(0),
	  ctrl_handler_(0),
	  srcid_(srcid),
	  ssrc_(ssrc),
	  addr_(addr),
	  badsesslen_(0),
	  badsessver_(0),
	  badsessopt_(0),
	  badsdes_(0),
	  badbye_(0),
	  format_(-1),
	  mute_(0),
	  lost_(0),
	  trigger_(1),
	  ismixer_(0),
	  reportLoss_(0)
{
	bind("reportLoss_", (int*)&reportLoss_);
	bind("badsesslen_", (int*)&badsesslen_);
	bind("badsessver_", (int*)&badsessver_);
	bind("badsessopt_", (int*)&badsessopt_);
	bind("badsdes_", (int*)&badsdes_);
	bind("badbye_", (int*)&badbye_);

	lts_done_.tv_sec = 0;
	lts_done_.tv_usec = 0;

	int i;
	for (i = 0; i <= RTCP_SDES_MAX; ++i)
		sdes_[i] = 0;

	nlayer_ = 0;
	/*FIXME*/
	for (i = 0; i < NLAYER; ++i)
		layer_[i] = 0;

//	printf("source created, this=0x%x, srcid_=0x%x, ssrc_=0x%x\n", this, srcid_, ssrc_);
}

Source::~Source()
{
//	if (handler_ != 0)
//		deactivate();
//	/*FIXME*/
//	Tcl::instance().evalf("%s unregister", TclObject::name());
	int i;
	for (i = 0; i <= RTCP_SDES_MAX; ++i)
		delete[] sdes_[i];
}

void Source::trigger_media()
{
	trigger_ = 0;
	Tcl::instance().evalf("%s trigger_media", TclObject::name());
}

void Source::notify(Source::Layer* layer)
{
	int n = 0;
	while (layer_[n] != layer && n < NLAYER)
		n++;
	Tcl::instance().evalf("%s notify %d", TclObject::name(), n);
}

PacketModule* Source::activate(int format)
{
	format_ = format;
	Tcl::instance().evalf("%s activate", TclObject::name());
	if (handler_ == 0)
		/* tcl should install a handler */
		abort();
	return (handler_);
}

PacketModule* Source::change_format(int format)
{
	format_ = format;
	Tcl::instance().evalf("%s trigger_format", TclObject::name());
	return (handler_);
}

void Source::deactivate()
{
	Tcl::instance().evalf("%s deactivate", TclObject::name());
	/*FIXME tcl deletes it */
	handler_ = 0;
}

void Source::sdes(int t, const char* s)
{
	char** p = &sdes_[t];
	if (*p != 0 && strcmp(*p, s) == 0)
		/* no change */
		return;

	delete[] *p;
	int n = strlen(s);
	if (n > 254)
		n = 254;
	*p = new char[n + 1];
	strncpy(*p, s, n + 1);
	/*FIXME*/
	Tcl::instance().evalf("%s trigger_sdes", TclObject::name());
}

void Source::trigger_sr()
{
  Tcl::instance().evalf("%s trigger_sr", TclObject::name());
}

/*FIXME*/
int Source::missing() const
{
	int s = 0;
	for (int i = 0; i < nlayer_; ++i) {
		int nm = layer(i).ns() - layer(i).np();
		if (nm < 0)
			nm = 0;
		s += nm;
	}
	return (s);
}

#ifdef notdef
char* Source::stats(char* cp) const
{
	cp = onestat(cp, "Kilobits", nb() >> (10-3));
	cp = onestat(cp, "Frames", nf());
	cp = onestat(cp, "Packets", np());
	cp = onestat(cp, "Missing", missing());
	cp = onestat(cp, "Misordered", nm());
	cp = onestat(cp, "Runts", runt());
	cp = onestat(cp, "Dups", dups());
	cp = onestat(cp, "Bad-S-Len", badsesslen());
	cp = onestat(cp, "Bad-S-Ver", badsessver());
	cp = onestat(cp, "Bad-S-Opt", badsessopt());
	cp = onestat(cp, "Bad-Sdes", badsdes());
	cp = onestat(cp, "Bad-Bye", badbye());
	*--cp = 0;
	return (cp);
}
#endif

static int sdes_atoi(const char* s)
{
	if (strcasecmp(s, "cname") == 0)
		return (RTCP_SDES_CNAME);
	if (strcasecmp(s, "name") == 0)
		return (RTCP_SDES_NAME);
	if (strcasecmp(s, "email") == 0)
		return (RTCP_SDES_EMAIL);
	if (strcasecmp(s, "phone") == 0)
		return (RTCP_SDES_PHONE);
	if (strcasecmp(s, "loc") == 0 || strcasecmp(s, "location") == 0)
		return (RTCP_SDES_LOC);
	if (strcasecmp(s, "tool") == 0)
		return (RTCP_SDES_TOOL);
	if (strcasecmp(s, "note") == 0)
		return (RTCP_SDES_NOTE);
	return (-1);
}

int Source::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	char* wrk = tcl.buffer();

	if (argc == 2) {
		/*
		 * <otcl> Source/RTP public deactivate {}
		 * Dispatch a deactivate event on this source.
		 * This causes the session object to remove the
		 * source from its internal state and notify
		 * agent observers accordingly.
		 */
		if (strcmp(argv[1], "deactivate") == 0) {
			deactivate();
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public addr {}
		 * Return the IP address host responsible for this RTP flow.
		 * Note that this might be the originating source or
		 * an intermediary gateway.
		 */
		if (strcmp(argv[1], "addr") == 0) {
			strcpy(wrk, intoa(addr_));
			tcl.result(wrk);
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public srcid {}
		 * Return the SRCID of this RTP flow.
		 */
		if (strcmp(argv[1], "srcid") == 0) {
			sprintf(wrk, "%lu", (u_long)ntohl(srcid_));
			tcl.result(wrk);
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public ssrc {}
		 * Return the syncronization source for this RTP flow.
		 * In the case of intermediary gateway, the SSRC
		 * is the gateway's SRCID rather than the original source's.
		 */
		if (strcmp(argv[1], "ssrc") == 0) {
			sprintf(wrk, "%lu", (u_long)ntohl(ssrc_));
			tcl.result(wrk);
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public format {}
		 * Return the most recent RTP format of the underlying flow.
		 * Note that the format can change dynamically.
		 */
		if (strcmp(argv[1], "format") == 0) {
			sprintf(wrk, "%d", format_);
			tcl.result(wrk);
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public lastdata {}
		 * Return the system time of the last data/media packet
		 * received on this flow.
		 * Return 0 if no such packet has been received.
		 */
		if (strcmp(argv[1], "lastdata") == 0) {
			/*FIXME layer-0?*/
			time_t now = layer(0).lts_data().tv_sec;
			if (now != 0) {
				char* cp = tcl.buffer();
				tcl.result(cp);
				strcpy(cp, ctime(&now));
				cp = strrchr(cp, '\n');
				if (cp != 0)
					*cp = 0;
			}
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public lastctrl {}
		 * Return the system time of the last control packet
		 * received on this flow.
		 * Return 0 if no such packet has been received.
		 */
		if (strcmp(argv[1], "lastctrl") == 0) {
			/*FIXME layer-0?*/
			time_t now = layer(0).lts_ctrl().tv_sec;
			if (now != 0) {
				char* cp = tcl.buffer();
				tcl.result(cp);
				strcpy(cp, ctime(&now));
				cp = strrchr(cp, '\n');
				if (cp != 0)
					*cp = 0;
			}
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public last-data {}
		 * Return the NTP representation for the time
		 * of the last data/media packet
		 * received on this flow.
		 * Return 0 if no such packet has been received.
		 * FIXME: fix this.  shouldn't distinguish between
		 * sys time and NTP time via last-data and lastdata!
		 */
		if (strcmp(argv[1], "last-data") == 0) {
			const timeval& tv = layer(0).lts_data();/*FIXME layer-0*/
			sprintf(tcl.buffer(), "%u", ntptime(tv));
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public lost {}
		 * Return a non-zero value iff the corresponding source has
		 * been inactive for a sufficient time to consider
		 * it either disconnected from the session
		 * or otherwise terminated.  A source that sends
		 * a valid will not result in true here.
		 */
		if (strcmp(argv[1], "lost") == 0) {
			tcl.result(lost_ ? "1" : "0");
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public handler {}
		 * Alias for the data-handler method for
		 * backward compatibility.
		 */
		if (strcmp(argv[1], "handler") == 0) {
		  if (handler_ != 0) {
		    tcl.result(((TclObject*)handler_)->name());
		  }
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public data-handler handler
		 * If called with the <i>handler</i> argument, sets
		 * the packet handler, derived from the PacketHandler
		 * base class, for media/data packets from this
		 * particular source.  A data packet
		 * handler might be an audio or video decoder.
		 * <p>
		 * If called without any arguments, returns the
		 * OTcl object name of the current data handler.
		 */
		if (strcmp(argv[1], "data-handler") == 0) {
			if (handler_ != 0)
				tcl.result(((TclObject*)handler_)->name());
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public ctrl-handler handler
		 * If called with the <i>handler</i> argument, sets
		 * the packet handler, derived from the PacketHandler
		 * base class, for control packets from this
		 * particular source.  A control packet
		 * handler might process RTCP packets as part
		 * of a media gateway algorithm.
		 * <p>
		 * If called without any arguments, returns the
		 * OTcl object name of the current data handler.
		 */
		if (strcmp(argv[1], "ctrl-handler") == 0) {
			if (ctrl_handler_ != 0)
			       tcl.result(((TclObject*)ctrl_handler_)->name());
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public enable_trigger {}
		 * Enables a one-shot call to the data_trigger method
		 * on the arrival of the next media packet.
		 * When this occurs, the trigger is immediately
		 * disabled.  This mechanism might be used in tandem
		 * with programmable timers by
		 * a user interface componet to highlight the
		 * current speaker.
		 */
		if (strcmp(argv[1], "enable_trigger") == 0) {
			trigger_ = 1;
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP public mute v
		 * If <i>v</i> is non-zero, ``mute'' the media from
		 * this particular source, i.e., ignore RTP data packets
		 * from this source.  If <i>v</i> is zero, unmute
		 * the source.  If <i>v</i> is not given, return the
		 * current setting.
		 */
		if (strcmp(argv[1], "mute") == 0) {
			sprintf(tcl.buffer(), "%u", mute_);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
	} else if (argc == 3) {
		/*
		 * <otcl> Source/RTP public sdes which val
		 * Query or set the RTP source-description information
		 * for this source.  If <i>val</i> is present,
		 * sets the parameter specified by <i>which</i> to
		 * that value.  Modifying an sdes value in this fashion
		 * is defined only for the local source.  If <i>val</i>
		 * is not specified, return the current value
		 * for the indicated item.  Valid options for <i>which</i>
		 * include:
		 * <ul>
		 * <li> cname - canonical name
		 * <li> name - nick name
		 * <li> email - email address
		 * <li> phone - POTS phone number
		 * <li> loc - geographical location (i.e., mailing address)
		 * <li> tool - application tool in use (e.g., vat)
		 * <li> note - descriptive text, e.g., on coffee break
		 * </ul>
		 * For example,
		 * <pre>
		 *   $src sdes name
		 * </pre>
		 * returns the RTP name attribute for the underlying source.
		 */
		if (strcmp(argv[1], "sdes") == 0) {
			int item = sdes_atoi(argv[2]);
			if (item >= 0 && sdes_[item] != 0) {
				strcpy(wrk, sdes_[item]);
				tcl.result(wrk);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "mute") == 0) {
			int v = atoi(argv[2]);
			mute(v);
			return (TCL_OK);
		}
		/* FIXME for backward compat */
		if (strcmp(argv[1], "handler") == 0) {
			const char* o = argv[2];
			if (*o == 0)
			{
				handler_ = 0;
			}
			else
			{
				handler_ = (PacketModule*)TclObject::lookup(o);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-handler") == 0) {
			const char* o = argv[2];
			if (*o == 0)
			{
				handler_ = 0;
			}
			else
			{
				handler_ = (PacketModule*)TclObject::lookup(o);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-handler") == 0) {
			const char* o = argv[2];
			if (*o == 0)
				ctrl_handler_ = 0;
			else {
				ctrl_handler_ = (PacketModule*)TclObject::lookup(o);
				//printf("assigning ctrl_handler_ to 0x%lx, tcl is %s\n", (unsigned long) ctrl_handler_, o);
			}
			return (TCL_OK);
		}

	} else if (argc == 4) {
		if (strcmp(argv[1], "sdes") == 0) {
			int item = sdes_atoi(argv[2]);
			if (item >= 0) {
				/*FIXME error?*/
				sdes(item, argv[3]);
			}
			return (TCL_OK);
		}
		/*
		 * <otcl> Source/RTP private layer n o
		 * For a layered media stream, set layer number <i>n</i>
		 * to point to the SourceLayer object <i>o</i>
		 */
		if (strcmp(argv[1], "layer") == 0) {
			int k = atoi(argv[2]);
			/*FIXME*/
			if (k >= NLAYER)
				abort();
			Layer* p = (Layer*)TclObject::lookup(argv[3]);
			layer_[k] = p;
			k += 1;
			if (k > nlayer_)
				nlayer_ = k;
			/*FIXME never delete layers?*/

			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void Source::lost(int v)
{
	if (lost_ != v) {
		Tcl& tcl = Tcl::instance();
		lost_ = v;
		tcl.evalf("%s trigger_idle", TclObject::name(), lost_);
	}
}

void Source::clear_counters()
{
	/*FIXME*/
	for (int i = 0; i < nlayer_; ++i)
		if (layer_[i] != 0)
			layer_[i]->clear_counters();

	lts_done_.tv_sec = 0;
	lts_done_.tv_usec = 0;
}

/*
 * <otcl> Class SourceManager
 * <i>SourceManager</i> is the base class for manipulating
 * source objects in multimedia sessions.  It is best thought of
 * as a hash table of demultiplexing end-points.  It handles
 * the creation and deletion of Source objects and has
 * methods for iterator over all of the sources in a
 * session using a generator API.
 */
static class SourceManagerClass : public TclClass {
    public:
	SourceManagerClass() : TclClass("SourceManager") {}
	TclObject* create(int, const char*const*) {
		return (new SourceManager);
	}
} source_manager_class;

SourceManager::SourceManager() :
	nsources_(0),
	sources_(0),
	clock_(0),
	keep_sites_(0),
	site_drop_time_(0),
	localsrc_(0),
	generator_(0)
{
	memset((char*)hashtab_, 0, sizeof(hashtab_));
}

int SourceManager::command(int argc, const char *const*argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		/*
		 * <otcl> SourceManager public local {}
		 * Return the OTcl object name of the
		 * Source object that represents the local user
		 * in this particular table.  Throw an error
		 * if the local source has not been installed
		 * in the table.
		 */
		if (strcmp(argv[1], "local") == 0) {
			if (localsrc_ != 0) {
				tcl.result(localsrc_->name());
				return (TCL_OK);
			} else {
				tcl.result("Not yet initialized");
				return (TCL_ERROR);
			}
		}
		/*
		 * <otcl> SourceManager public gen-init {}
		 * Reset the internal generator for enumerating
		 * the source objects maintained in the table.
		 * Results are undefined if objects are inserted
		 * or deleted while the generator is being manipulated.
		 */
		if (strcmp(argv[1], "gen-init") == 0) {
			generator_ = sources_;
			return (TCL_OK);
		}
		/*
		 * <otcl> SourceManager public gen-next {}
		 * Return the object name of the next source object
		 * to be enumerated by the generator abstraction
		 * and advance the internal generator state.
		 */
		if (strcmp(argv[1], "gen-next") == 0) {
			Source* s = generator_;
			if (s != 0) {
				tcl.result(s->name());
				generator_ = s->next_;
			}
			return (TCL_OK);
		}
	} else if (argc == 3) {
		/*
		 * <otcl> SourceManager public keep-sites v
		 * Set the keep-sites attribute to <i>v</i>,
		 * which should be an integer interpreted as a boolean.
		 * If true, keep-sites means that the SourceManager
		 * will never delete a Source object because of
		 * inactivity.  Instead, it will mark such sources
		 * as ``lost'', so that the lost method returns true.
		 * This allows a user-interface, for example, to
		 * display inactive/unreachable participants in a
		 * disabled mode rather than dropping them altogether.
		 */
		if (strcmp(argv[1], "keep-sites") == 0) {
			keep_sites_ = atoi(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> SourceManager public site-drop-time sec
		 * Set the amount of time to wait after the last
		 * session packet received before considering
		 * that a site has become inactive or unreachable.
		 * This value is in seconds.
		 */
		if (strcmp(argv[1], "site-drop-time") == 0) {
			site_drop_time_ = atoi(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> SourceManager public delete src
		 * Explicitly remove the Source object, named
		 * by the <i>src</i> argument, from the SourceManger's
		 * table.  Also, deletes the object.
		 */
		if (strcmp(argv[1], "delete") == 0) {
			Source* s = (Source*)TclObject::lookup(argv[2]);
			if (s != 0)
				remove(s);
			return (TCL_OK);
		}
	} else if (argc == 4) {
		/*
		 * <otcl> SourceManager public create-local srcid addr
		 * Create a new Source object to represent the local
		 * participant in the underlying multimedia session;
		 * initialize the SRCID to <i>srcid</i> and the
		 * IP address to <i>addr</i>.
		 * The local IP address is stored here so it
		 * can be conventiently re-feteched later
		 * but also so the loop detection algorithm
		 * can make use of it.
		 */
		if (strcmp(argv[1], "create-local") == 0) {
			u_int32_t srcid = strtoul(argv[2], 0, 0);
			srcid = htonl(srcid);
			u_int32_t addr = inet_addr(argv[3]);
			init(srcid, addr);
			tcl.result(localsrc_->name());
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void SourceManager::remove_from_hashtable(Source* s)
{
	/* delete the source from hash table */
	u_int32_t srcid = s->srcid();
	int h = SHASH(srcid);
	Source** p = &hashtab_[h];
	while ((*p != s) && (*p != 0)) {
		p = &(*p)->hlink_;
	}
	if (*p != 0) {
		// Source object found
		*p = (*p)->hlink_;
	}
}

/*FIXME this is now more convoluted than it has to be...*/
void SourceManager::init(u_int32_t localid, u_int32_t localaddr)
{
	/*
	 * create the local object.  remove it from the hash
	 * table since collision resolution changes local id
	 * (we special case detection of our own loopbed back packets)
	 */
	/* FIXME */
	localsrc_ = create_source(0, localid, localid, localaddr);
	enter(localsrc_);
	remove_from_hashtable(localsrc_);
	/*
	 * hack to prevent local source from turning gray at startup.
	 * we don't need to do a similar thing for external sources,
	 * because they are only created when a packet arrives.
	 */
	localsrc_->layer(0).lts_ctrl(unixtime());/*FIXME layer-0*/
}

Source* SourceManager::enter(Source* s)
{
	s->next_ = sources_;
	sources_ = s;

	int h = SHASH(s->srcid());
	s->hlink_ = hashtab_[h];
	hashtab_[h] = s;

	++nsources_;

	return (s);
}

// this expects a net order srcid
Source* SourceManager::consult(u_int32_t srcid)
{
	int h = SHASH(srcid);
	for (Source* s = hashtab_[h]; s != 0; s = s->hlink_) {
		/*FIXME pulling these values into variable seems
		  to work around a DEC c++ bug */
		u_int32_t id = s->srcid();
		if (id == srcid)
			return (s);
	}
	return (0);
}

Source* SourceManager::create_source(TclObject* srcsess, u_int32_t srcid,
				     u_int32_t ssrc, u_int32_t addr)
{
	/*FIXME should have helper method-invoker */
	Tcl& tcl = Tcl::instance();
	const char* sname = (srcsess == 0)? "localsrc" : srcsess->name();
	tcl.evalf("%s create-source %u %u %u %s", name(), srcid, ssrc, addr,
		  sname);
	return ((Source*)tcl.lookup(tcl.result()));
}


// this expects a net order srcid
Source* SourceManager::lookup(TclObject* srcsess, u_int32_t srcid,
			      u_int32_t ssrc, u_int32_t addr)
{
	// look for the source id in the SSRC list
	Source* s = consult(srcid);

	if (s == 0) {
		if (srcid == ssrc)
			/*
			 * Heuristic: handle application re-starts
			 * gracefully.  We have a new source that's
			 * not via a mixer.  Try to find an existing
			 * entry from the same host.
			 */
			s = lookup_duplicate(srcid, addr);

		if (s == 0) {
			// new ssrc and address: register the source entity as a new, 
			//	valid source. FYI, RFC1889, section 6.2.1, suggests that 
			//	"New entries may not be considered valid until multiple 
			//	packets carrying the new SSRC have been received (see
			//	Appendix A.1).", but we are too impatient and don't wait 
			//	so much
			s = create_source(srcsess, srcid, ssrc, addr);
			Tcl& tcl = Tcl::instance();
			tcl.evalf("%s notify_observers register %s", name(),
				  s->name());
			enter(s);
		}
	}
	return (s);
}

/*
 * Demux data packet to its source table entry.  (We don't want an extra
 * SSRC arg here because CSRC's via a mixer don't have their own data
 * packets.)  If we haven't seen this source yet, allocate it but
 * wait until we see two in-order packets accepting the flow.
 */
// this expects srcid to be in net order
Source* SourceManager::demux(TclObject* srcsess, u_int32_t srcid,
			     u_int32_t addr, u_int16_t seq, int layer)
{
	Source* s = consult(srcid);
	if (s == 0) {
		s = lookup_duplicate(srcid, addr);
		if (s == 0) {
			/* CSRC=SSRC for data stream */
			s = create_source(srcsess, srcid, srcid, addr);
			Tcl& tcl = Tcl::instance();
			tcl.evalf("%s notify_observers register %s", name(),
				  s->name());
			enter(s);
		}
		/* it takes two in-seq packets to activate source */
		s->layer(layer).fs(seq);
		s->layer(layer).cs(seq, s);
		return (0);
	} else {
		Source::Layer& sl = s->layer(layer);
		/*
		 * check for a srcid conflict or loop:
		 *  - believe the new guy if the old guy said he's done.
		 *  - otherwise, don't believe the new guy if we've heard
		 *    from the old guy 'recently'.
		 */
		if (s->addr() != addr) {
#ifdef notdef
			u_int32_t t = s->lts_done().tv_sec;
			if (t == 0) {
				t = sl.lts_data().tv_sec;
				u_int32_t now = unixtime().tv_sec;
				if (t && int(now - t) <= 2)
					return (0);
				t = sl.lts_ctrl().tv_sec;
				if (t && int(now - t) <= 30)
					return (0);
			}
#endif
			s->addr(addr);
#ifdef notdef
			s->clear_counters();
			s->lost(0);
#endif
		}
		if (sl.np() == 0 && sl.nb() == 0) {
			/*
			 * make sure we get 2 in-seq packets before
			 * accepting source/layer.
			 */
			if ((u_int32_t)((u_int32_t)seq - sl.cs() + 31) > 63) {
				sl.fs(seq);
				sl.cs(seq, s);
				return (0);
			}
		}
	}
	return (s);
}

/*
 * Try to find an entry in the source table with the same network
 * address (i.e.,  a "duplicate entry") but possibly different srcid.
 * As a side effect, refile the source under the new srcid.
 *
 * The idea here is to gracefully handle sites that restart (with
 * a new srcid).  If we assume that it takes a couple seconds to
 * restart
 *
 */
Source* SourceManager::lookup_duplicate(u_int32_t srcid, u_int32_t addr)
{

	/*FIXME - should eventually be conditioned on cname not ipaddr */

	/*
	 * Disable this for now.  It wreaks havoc with gateways from which
	 * streams all come from same IP addr.  Eventually condition on CNAME.
	 */
	return (0);

	/*
	 * could use hashing here, but this is rarely called.
	 */
	register Source* s;
	for (s = sources_; s != 0; s = s->next_) {
		/*
		 * if addresses match, take old entry if:
		 *  - it sent a 'done', or
		 *  - it hasn't sent any data for 2 seconds and
		 *    and any control for 30 seconds.
		 */
		if (s->addr() == addr) {
			if (s->lts_done().tv_sec != 0)
				break;
			u_int32_t now = unixtime().tv_sec;
			u_int32_t t = s->layer(0).lts_data().tv_sec;
			if (t == 0 || int(now - t) > 2) {
				t = s->layer(0).lts_ctrl().tv_sec;
				if (t == 0 || int(now - t) > 30)
					break;
			}
		}
	}
	if (s) {
		remove_from_hashtable(s);
		s->srcid(srcid);
		s->ssrc(srcid);
		int h = SHASH(srcid);
		s->hlink_ = hashtab_[h];
		hashtab_[h] = s;
		s->clear_counters();
		s->lost(0);
	}
	return (s);
}

void SourceManager::remove(Source* s)
{
	--nsources_;

	// if the source is not the local one, remove it from the hash table
	if (s != localsrc_) {
		remove_from_hashtable(s);
	}

	/* delete the source from list */
	Source** p = &sources_;
	while (*p != s)
		p = &(*p)->next_;
	*p = (*p)->next_;

	//delete s;
	// Note: this would only delete the Source object C++ part, not the 
	// otcl one. A better way to destroy a Source object is to call up 
	// Source/RTP::destroy{} as follows:
	//Tcl::instance().evalf("%s destroy", s->name());

	if (s == generator_)
		generator_ = 0;
}

/*
 * Go through all the sources and see if any should be "grayed out"
 * or removed altogether.  'msgint' is the current "report interval"
 * in ms.
 */
void SourceManager::CheckActiveSources(double msgint)
{
	u_int32_t now = unixtime().tv_sec;
	u_int max_idle = u_int(msgint * (CTRL_IDLE / 1000.));
	if (max_idle == 0)
		max_idle = 1;

	Source* n;
	for (Source* s = sources_; s != 0; s = n) {
		n = s->next_;
		u_int32_t t = s->lts_done().tv_sec;

		if (t != 0) {
			// this source has received an RTP BYE packet
			if (keep_sites_)
				s->lost(1);
			else {
				// ensure we don't try to delete the object twice
				timeval tv_zero;
				tv_zero.tv_sec = 0;
				tv_zero.tv_usec = 0;
				s->lts_done(tv_zero);

				// destroy the Source object associated to the source
				Tcl::instance().evalf("%s destroy", s->name());
			}
			continue;
		}

		t = s->layer(0).lts_ctrl().tv_sec;
		if (t == 0) {
			/*
			 * No session packets?  Probably ivs or nv sender.
			 * (or we haven't seen any yet)
			 * Revert to the data time stamp.
			 */
			t = s->layer(0).lts_data().tv_sec;
			if (t == 0) {
				/*
				 * No data time stamp on layer 0.
				 * Look through the other layers.
				 */
				for (int i = 1; i < s->nlayer_; ++i) {
					t = s->layer(i).lts_data().tv_sec;
					if (t != 0)
						break;
				}
			}
		}
		if (t == 0) {
			/* didn't find any timestamps (shouldn't happen) */
			continue;
		}

		if (u_int(now - t) > max_idle) {
			if (keep_sites_ || site_drop_time_ == 0 ||
			    u_int(now - t) < site_drop_time_)
				s->lost(1);
			else {
				// ensure we don't try to delete the object twice
				timeval tv_zero;
				tv_zero.tv_sec = 0;
				tv_zero.tv_usec = 0;
				s->lts_done(tv_zero);

				// destroy the Source object associated to the source
				Tcl::instance().evalf("%s destroy", s->name());
			}
		} else
			s->lost(0);
	}
}

int SourceManager::compare(const void* v0, const void* v1)
{
	const Source* x = *(Source**)v0;
	const Source* y = *(Source**)v1;

	const char* yn = y->sdes(RTCP_SDES_NAME);
	if (yn == 0) {
		yn = y->sdes(RTCP_SDES_CNAME);
		if (yn == 0)
			return (-1);
	}
	const char* xn = x->sdes(RTCP_SDES_NAME);
	if (xn == 0) {
		xn = x->sdes(RTCP_SDES_CNAME);
		if (xn == 0)
			return (1);
	}
	return (strcmp(xn, yn));
}

/*
 * Sort the sources by name and format a corresponding list
 * of ascii srcid's in the input buffer.
 */
void SourceManager::sortactive(char* cp) const
{
	static int ntab;
	static Source** srctab;
	if (srctab == 0) {
		ntab = 2 * nsources_;
		if (ntab < 32)
			ntab = 32;
		srctab = new Source*[ntab];
	} else if (ntab < nsources_) {
		ntab = 2 * nsources_;
		delete srctab;
		srctab = new Source*[ntab];
	}
	int n = 0;
	for (Source* s = sources_; s != 0; s = s->next_)
		if (s->handler() != 0)
			/* source is active if it has a PacketHandler */
			srctab[n++] = s;

	if (n == 0) {
		*cp = 0;
		return;
	}
	qsort(srctab, n, sizeof(*srctab), compare);
	for (int i = 0; i < n; ++i) {
		strcpy(cp, srctab[i]->TclObject::name());
		cp += strlen(cp);
		*cp++ = ' ';
	}
	/* nuke trailing space */
	cp[-1] = 0;
}
