# source.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/collaborator/source.tcl,v 1.15 2002/02/03 04:21:46 lim Exp $


import WidgetClass MultiSourceIcons VideoWidget Rank DragNDrop CompoundButton

Class MultiSource
WidgetClass UIMultiSource -configspec {
	{ -value       value       Value       {}            config_option }
	{ -select      select      Select      0             config_option }
	{ -highlight   highlight   Highlight   0             config_option }

	{ -relief relief Relief flat config_option }
	{ -normalbackground normalBackground NormalBackground \
			WidgetDefault(-background) config_option }
	{ -normalforeground normalForeground NormalForeground \
			WidgetDefault(-foreground) config_option }
	{ -normalrelief normalRelief NormalRelief flat config_option }
	{ -selectbackground selectBackground SelectBackground WidgetDefault \
			config_option }
	{ -selectforeground selectForeground SelectForeground WidgetDefault \
			config_option }
	{ -selectrelief selectRelief SelectRelief sunken config_option }
	{ -highlightrelief highlightRelief HighlightRelief raised \
			config_option }
} -default {
	{ *activeColor white }
	{ *defaultColor WidgetDefault(-background) }
	{ *disabledColor WidgetDefault(-disabledforeground) }
}



WidgetClass UISource -default {
	{ *defaultColor WidgetDefault(-background) }
	{ *disabledColor WidgetDefault(-disabledforeground) }
}



UIMultiSource public init { args } {
	$self instvar config_
	set config_(-value) {}
	set config_(-select) 0
	set config_(-highlight) 0
	set config_(-normalbackground) Black
	set config_(-normalforeground) Black
	set config_(-normalrelief)     flat
	set config_(-selectbackground) Black
	set config_(-selectforeground) Black
	set config_(-selectrelief)     sunken
	set config_(-highlightrelief)  raised

	eval [list $self] next $args
}


UIMultiSource public config_option { option args } {
	$self instvar config_
	if { [llength $args]==0 } {
		return $config_($option)
	} else {
		set value [lindex $args 0]
		set config_($option) $value
		if { $option == "-relief" } {
			$self widget_proc configure $option $value
		}
	}
}



WidgetClass UISource/RTP     -superclass UISource
WidgetClass UISource/Audio   -superclass UISource/RTP
WidgetClass UISource/Video   -superclass UISource/RTP
WidgetClass UISource/Mb      -superclass UISource
WidgetClass UISource/Mbv2    -superclass UISource
WidgetClass UISource/Unknown -superclass UISource


UISource/Audio   set image_ MultiSourceIcons(audio)
UISource/Video   set image_ MultiSourceIcons(video)
UISource/Mb      set image_ MultiSourceIcons(mb)
UISource/Mbv2    set image_ MultiSourceIcons(mb)
UISource/Unknown set image_ MultiSourceIcons(unknown)


UISource public init { widget observer src msrc args } {
	$self set src_ $src
	$self set msrc_ $msrc
	$self set observer_ $observer
	eval $self next $widget $args
}


UISource public destroy { } {
	$self next
}


UISource public src {} { return [$self set src_] }


UISource public create_root_widget {path} {
	[$self info class] instvar image_
	button $path -image $image_ -bd 1 -relief raised -padx 1000 \
			-highlightthickness 0 -takefocus 0
	$path configure -activebackground [option get $path defaultColor \
			UISource]
}


UISource public msrc {msrc} {
	$self set msrc_ $msrc
}


UISource public save_state { state } {
}


UISource public restore_state { state } {
}


UISource public tip {} {
	return {}
}


UISource/Audio public tip {} {
	return Audio
}


UISource/Video public tip {} {
	return Video
}


UISource/Mb public tip {} {
	return Mediaboard
}


UISource/Video public name { } {
	$self instvar msrc_
	return [$msrc_ name]
}


UISource/Video public save_state { state_var } {
	upvar $state_var state
	set state(have_thumbnail) [$self have_thumbnail]
}


UISource/Video public restore_state { state_var } {
	$self instvar msrc_ src_
	upvar $state_var state
	if $state(have_thumbnail) {
		$msrc_ add_thumbnail video $src_
	}
}


UISource/RTP public destroy {} {
	$self instvar info_win_ rtp_win_ decoder_win_
	if [info exists info_win_] {
		delete $info_win_
	}
	if [info exists rtp_win_] {
		delete $rtp_win_
	}
	if [info exists decoder_win_] {
		delete $decoder_win_
	}
	$self next
}


UISource/Video public destroy {} {
	$self instvar src_ thumbnail_ thumbnail_path_ info_path_ observer_ src_

	$observer_ detach_windows $src_
	if [info exists thumbnail_] {
		$self destroy_thumbnail
	}
	if [info exists info_path_] {
		$self destroy_info
	}

	$self instvar scuba_win_
	if [info exists scuba_win_] {
		$self delete-scuba-window
	}

	$self next
}


UISource/Video public build_widget {path} {
	$self tkvar mute_ color_
	$self instvar observer_ src_ userwindows_
	$path configure -command "tk_popup $path.menu \[winfo pointerx $path\]\
			\[winfo pointery $path\]"

	set userwindows_ {}
	set mute_ [$observer_ mute_new_sources]
	$src_ mute $mute_
	set color_ 1

	menu $path.menu
	$path.menu add checkbutton -label "Mute" \
			-variable [$self tkvarname mute_] \
			-command "$self mute \$\{[$self tkvarname mute_]\}"
	$path.menu add checkbutton -label "Color" \
			-variable [$self tkvarname color_] \
			-command "$self color \$\{[$self tkvarname color_]\}"
	$path.menu add cascade -label "Info..." -menu $path.menu.info
	$self build_info_menu $path.menu.info
}


UISource/Video public have_thumbnail {} {
	$self instvar thumbnail_
	return [info exists thumbnail_]
}


UISource/Video public create_thumbnail {path} {
	$self instvar src_ thumbnail_ thumbnail_path_
	set thumbnail_path_ $path.thumbnail$src_
	frame $thumbnail_path_ -bd 1 -relief raised
	set thumbnail_ [new VideoWidget $thumbnail_path_.src 80 60]
	$thumbnail_ set is_slow_ 1
	pack $thumbnail_path_.src -fill both -expand 1
	pack $thumbnail_path_ -in $path -side left

	$self bind_thumbnail $thumbnail_path_.src
	$self attach_thumbnail
}


UISource/Video public destroy_thumbnail {} {
	$self instvar src_ thumbnail_ thumbnail_path_ dragdrop_
	if [info exists thumbnail_] {
		$self detach_thumbnail
		delete $dragdrop_
		delete $thumbnail_
		destroy $thumbnail_path_
		unset thumbnail_
		unset thumbnail_path_
		unset dragdrop_
	}
}


UISource/Video private bind_thumbnail t {
	$self instvar dragdrop_
	bind $t <Double-ButtonPress-1> "$self select_thumbnail"
	set dragdrop_ [new DragNDrop $t "$self dragNdrop_thumbnail"]
	$dragdrop_ proc select_widget  w "$self build_dragdrop \$self \$w"
	$dragdrop_ proc release_widget w "$self delete_dragdrop \$self \$w"
}


UISource/Video private build_dragdrop { dragdrop w } {
	$self instvar src_
	set vw [new VideoWidget $w.video 80 60]
	pack $w.video -fill both -expand 1

        $vw attach-decoder $src_ [UISource/Video set colorModel_] \
			[UISource/Video set use_hw_decode_]
	$vw set is_slow_ 1
	$dragdrop set vw_ $vw
}


UISource/Video private delete_dragdrop { dragdrop w } {
	$dragdrop instvar vw_
	$self instvar src_
	$vw_ detach-decoder $src_
	destroy $w.video
	delete $vw_
	unset vw_
}


#
# Called when user double-clicks on thumbnail video window.
# Create a new window only if the window already
# isn't being displayed (in locked mode).  In this
# case, delete the existing window (i.e., clicking
# on the thumbnail gives a toggle action, but not
# for voice-switched or browse-mode windows).
#
UISource/Video public select_thumbnail {} {
	foreach uw [$self user_windows] {
		if { [$uw attached-source] == "$self" && ![$uw is-switched] &&\
				[[$uw info class] info heritage VideoWidget] \
				== {} } {
			delete $uw
			return
		}
	}

	$self create_user_window
}


#
# Create a UserWindow for this ActiveSource.
#
UISource/Video private create_user_window { } {
	$self instvar observer_
	new UserWindow $observer_ $self [$self yesno useCues] \
			[$observer_ can_voiceswitch] {} 1
}


#
# Returns true if the represented source is h261 format.
#
UISource/Video public isCIF {} {
	$self instvar src_
	return [expr [string compare [$src_ format_name] h261] == 0]
}


UISource/Video private dragNdrop_thumbnail { dragndrop x y } {
	$self instvar observer_
	if ![[$observer_ video_container] drop_thumbnail $dragndrop $self \
			$x $y] {
		$dragndrop zoom_back
	}
}


UISource/Video proc.public decoder_params { colorModel use_hw_decode } {
	$self set colorModel_ $colorModel
	$self set use_hw_decode_ $use_hw_decode
}


#
UISource/Video public attach_thumbnail {} {
        $self instvar src_ thumbnail_
        $thumbnail_ attach-decoder $src_ [UISource/Video set colorModel_] \
			[UISource/Video set use_hw_decode_]
}

#
UISource/Video public detach_thumbnail {} {
        $self instvar src_ thumbnail_
        $thumbnail_ detach-decoder $src_
}


UISource/Video public create_info {path} {
	$self instvar src_ msrc_ timer_id_ info_path_
	$self tkvar ftext_ btext_ ltext_ info_
	set ftext_ "0.0 f/s"
	set btext_ "0.0 kb/s"
	set ltext_ "(0%)"
	set info_ [$msrc_ cname]

	set path $path.info$src_
	set info_path_ $path
	frame $path -bd 2 -relief groove
	frame $path.frame
	label $path.info -textvariable [$self tkvarname info_] \
			-anchor w -pady 0
	label $path.ftext -textvariable [$self tkvarname ftext_] \
			-anchor w -pady 0
	label $path.btext -textvariable [$self tkvarname btext_] \
			-anchor w -pady 0
	label $path.ltext -textvariable [$self tkvarname ltext_] \
			-anchor w -pady 0
	pack $path.ftext $path.btext $path.ltext -fill both -expand 1 \
			-pady 0 -side left -in $path.frame
	pack $path.info -side top -pady 0 -fill x -expand 1
	pack $path.frame -side top -fill x -expand 1
	pack $path -fill both -expand 1

	set timer_id_ [after 1000 "$self update_info"]
}


UISource/Video public destroy_info { } {
	$self instvar info_path_ timer_id_
	if ![info exists info_path_] return

	$self tkvar info_ ftext_ btext_ ltext_
	if [info exists timer_id_] {
		after cancel $timer_id_
		unset timer_id_
	}

	destroy $info_path_
	unset info_path_
	unset info_ ftext_ btext_ ltext_
}


UISource/Video private update_info {} {
	$self tkvar ftext_
	$self instvar src_ msrc_ timer_id_
	if ![info exists ftext_] {
		return
	}

	$self update_rate
	set timer_id_ [after 1000 "$self update_info"]
}


#
# Using the specified video source, <i>src_</i>, as an index, update the
# global arrays bpshat(), fpshat(), shat(), lhat(), ltext(), ftext(), btext().
#
UISource/Video private update_rate {} {
	$self instvar src_
	$self tkvar ftext_ btext_ ltext_
	global fpshat bpshat lhat shat

	set key $src_
	if [string match Session/* [$src_ info class]] {
		set bpshat($key) [expr 8 * [$src_ set nb_]]
		set fpshat($key) [$src_ set nf_]
	} else {
		# only compute loss statistic for network side
		set p [$src_ layer-stat np_]
		set s [$src_ ns]
		set shat($key) $s
		set lhat($key) [expr $s-$p]
		if {$shat($key) <= 0.} {
			set loss 0
		} else {
			set loss [expr 100*$lhat($key)/$shat($key)]
		}
		if {$loss < .1} {
			set ltext_ (0%)
		} elseif {$loss < 9.9} {
			set ltext_ [format "(%.1f%%)" $loss]
		} else {
			set ltext_ [format "(%.0f%%)" $loss]
		}
		set bpshat($key) [expr 8 * [$src_ layer-stat nb_]]
		set fpshat($key) [$src_ layer-stat nf_]
	}

	set fps $fpshat($key)
	set bps $bpshat($key)

	if { $fps < .1 } {
		set fps "0 f/s"
	} elseif { $fps < 10 } {
		set fps [format "%.1f f/s" $fps]
	} else {
		set fps [format "%2.0f f/s" $fps]
	}
	if { $bps < 1 } {
		set bps "0 bps"
	} elseif { $bps < 1000 } {
		set bps [format "%3.0f bps" $bps]
	} elseif { $bps < 1000000 } {
		set bps [format "%3.0f kb/s" [expr $bps / 1000]]
	} else {
		set bps [format "%.1f Mb/s" [expr $bps / 1000000]]
	}
	set ftext_ $fps
	set btext_ $bps

	# FIXME: need this to update the video ctrlmenu
	$self instvar observer_
	$observer_ instvar agent_
	if { $src_ == [$agent_ local] } {
		global ftext btext
		set key [$agent_ set session_]
		set ftext($key) $fps
		set btext($key) $bps
	}
}


UISource/Video public change_info { info } {
	$self tkvar info_
	set info_ $info
}


UISource/Video public get_info {} {
	$self tkvar info_
	if [info exists info_] { return $info_ }  else { return "" }
}



#
# Create a menu for accessing information and data associated with the represented source: <br>
#    <dd> "Site Info"
#    <dd> "RTP Stats"
#    <dd> "Decoder Stats"
#    <dd> "Mtrace from"
#    <dd> "Mtrace to"
#    <dd> and possibly "Scuba Info"
#
UISource/RTP private build_info_menu { m } {
	$self instvar observer_
	menu $m
	set f [$self get_option smallfont]
	$m add command -label "Site Info" \
		-command "$self create-info-window" -font $f
	$m add command -label "RTP Stats"\
		-command "$self create-rtp-window" -font $f
	$m add command -label "Decoder Stats" \
		-command "$self create-decoder-window" -font $f
	if [in_multicast [[$observer_ agent] session-addr]] {
		$m add command -label "Mtrace from" \
			-command "$self create-mtrace-window from" -font $f
		$m add command -label "Mtrace to" \
			-command "$self create-mtrace-window to" -font $f
	}
}


UISource/Video private build_info_menu { m } {
	$self instvar observer_
	$self next $m
	if { [$observer_ scuba_session] != {} } {
		set f [$self get_option smallfont]
		$m add command -label "Scuba Info" -font $f \
				-command "$self create-scuba-window"
	}
}


#
# If a window exists for displaying the Scuba Votes for this source,
# delete it.  Else, create one.
#
UISource/Video private create-scuba-window {} {
	UISource/Video instvar scubaInfoCnt_
	if ![info exists scubaInfoCnt_] { set scubaInfoCnt_ 0 }
	$self instvar scuba_win_ src_ observer_
	if [info exists scuba_win_] {
		$self delete-scuba-window
	} else {
		set scuba_sess [$observer_ scuba_session]
		set scuba_win_ [new ScubaInfoWindow \
				.scubainfo_for_uisrc_video$scubaInfoCnt_ \
				$src_ $self $scuba_sess]
		incr scubaInfoCnt_
		[$scuba_sess source-manager] attach $scuba_win_
		$scuba_win_ timeout
	}
}

#
# Delete the Scuba Votes window for this source.
#
UISource/Video private delete-scuba-window {} {
	$self instvar scuba_win_ observer_
	set scuba_sess [$observer_ scuba_session]
	[$scuba_sess source-manager] detach $scuba_win_
	delete $scuba_win_
	unset scuba_win_
}


#
# If an InfoWindow exists for this source, delete it.  Else, create one.
#
UISource/RTP instproc create-info-window {} {
	$self instvar src_ info_win_
	if [info exists info_win_] {
		$self delete-info-window
	} else {
		set info_win_ [new InfoWindow .info$src_ $src_ $self]
	}
}

#
# Delete the InfoWindow for this source.
#
UISource/RTP instproc delete-info-window {} {
	$self instvar info_win_
	delete $info_win_
	unset info_win_
}

#
# For the represented source, return a string
# describing its network statistics.
#
UISource/RTP instproc stats {} {
	$self instvar src_
	return "Kilobits [expr [$src_ layer-stat nb_] >> (10-3)] \
		Frames [$src_ layer-stat nf_] \
		Packets [$src_ layer-stat np_] \
		Missing [$src_ missing] \
		Misordered [$src_ layer-stat nm_] \
		Runts [$src_ layer-stat nrunt_] \
		Dups [$src_ layer-stat ndup_] \
		Bad-S-Len [$src_ set badsesslen_] \
		Bad-S-Ver [$src_ set badsessver_] \
		Bad-S-Opt [$src_ set badsessopt_] \
		Bad-Sdes [$src_ set badsdes_] \
		Bad-Bye [$src_ set badbye_]"
}

#
# For the represented source, return the decoder statistics.
#
UISource/RTP instproc decoder-stats {} {
	$self instvar src_
	set d [$src_ handler]
	#FIXME
	return [$d stats]
}

#
# If a window exists for displaying the RTP Statistics of this source, delete it.  Else, create one.
#
UISource/RTP instproc create-rtp-window {} {
	$self instvar src_ rtp_win_
	if [info exists rtp_win_] {
		$self delete-rtp-window
	} else {
		set rtp_win_ [new RtpStatWindow .rtp$src_ $src_ \
					"RTP Statistics" \
					"$self stats" \
					"$self delete-rtp-window"]
	}
}

#
# Delete the RTP Statisctics window for this source.
#
UISource/RTP instproc delete-rtp-window {} {
	$self instvar rtp_win_
	delete $rtp_win_
	unset rtp_win_
}

#
# If a window exists for displaying the Decoder Statistics of this source, delete it.  Else, create one.
#
UISource/RTP instproc create-decoder-window {} {
	$self instvar src_ decoder_win_
	if [info exists decoder_win_] {
		$self delete-decoder-window
	} else {
		if { "[$src_ handler]" == "" } {
			new ErrorWindow "no decoder stats yet"
			return
		}
		set decoder_win_ [new RtpStatWindow .decoder$src_ $src_  \
				"Decoder Statistics" \
				"$self decoder-stats" \
				"$self delete-decoder-window"]
	}
}

#
# Delete the Decoder Statisctics window for this source.
#
UISource/RTP instproc delete-decoder-window {} {
	$self instvar decoder_win_
	if [info exists decoder_win_] {
		delete $decoder_win_
		unset decoder_win_
	}
}


UISource public change_info { info } {
	# FIXME
}


UISource public get_info {} {
	# FIXME
	return ""
}


UISource/Video public user_windows {} {
	return [$self set userwindows_]
}


UISource/Video public detach_windows {} {
	$self instvar userwindows_
	foreach uw $userwindows_ {
		$self detach-window $uw
	}
}



#
# Bind a source to a window so that the video stream from the
# represented source appears in UserWindow <i>uw</i>.
#
UISource/Video public attach_window { uw } {
	$self instvar src_ observer_
	UISource/Video instvar colorModel_ use_hw_decode_
	[$uw video-widget] attach-decoder $src_ $colorModel_ $use_hw_decode_
	$self instvar userwindows_
	lappend userwindows_ $uw
	$uw set-name [$src_ getid]

	set scuba_sess [$observer_ scuba_session]
	if { $scuba_sess != {} } {
		$scuba_sess scuba_focus $src_
	}
}


#
# Discontinue the representation of this ActiveSource in UserWindow <i>uw</i>.
#
UISource/Video public detach_window uw {
	$self instvar userwindows_ src_ observer_
	set scuba_sess [$observer_ scuba_session]
	if { $scuba_sess != {} } {
		$scuba_sess scuba_unfocus $src_
	}
	[$uw video-widget] detach-decoder $src_
	# there must be an easier way to do this
	set k [lsearch -exact $userwindows_ $uw]
	if { $k < 0 } {
		puts "[$self get_option appname]: detach-window: FIXME"
		exit 1
	}
	set userwindows_ [lreplace $userwindows_ $k $k]
}


# FIXME: required for UserWindow
UISource/Video public attach-window {w} {
	$self attach_window $w
}
UISource/Video public detach-window {w} {
	$self detach_window $w
}


UISource/Video public color { { c {} } } {
	$self instvar src_ userwindows_ thumbnail_
	$self tkvar color_
	if { $c == {} } { return $color_ }
	set h [$src_ handler]
	if { $h != {} } {
		$h color $c
		foreach uw $userwindows_ {
			[$uw video-widget] redraw
		}
		if [info exists thumbnail_] {
			$thumbnail_ redraw
		}
	}
	set color_ $c
}


UISource/Video public mute { { m {} } } {
	$self instvar src_
	$self tkvar mute_
	if { $m == {} } { return $mute_ }
	$src_ mute $m
	set mute_ $m
}


UISource/Audio public mute { { m {} } } {
	$self instvar src_
	$self tkvar mute_
	if { $m == {} } { return $mute_ }
	$src_ mute $m
	set mute_ $m
	$self draw_indicator
}


UISource/Audio public create_root_widget {path} {
        [$self info class] instvar image_
	# need the container frame becoz the root widget cannot be
	# another WidgetObject; this'll horribly confuse the
	# WidgetClass library
        frame $path
        set path [CompoundButton $path.b -bd 1 -relief raised \
                        -highlightthickness 0]
        $path add button icon -image $image_ -bd 0 -relief flat \
                        -highlightthickness 0 -takefocus 0
        $path.icon configure -activebackground [option get $path defaultColor \
                        UISource]
        $path add canvas indicator -width 0 -height 0 -bg \
                        [option get $path defaultColor UISource] \
                        -bd 0 -relief sunken
        pack $path.icon -side left
        pack $path.indicator -padx 6 -side left
        pack $path -fill both -expand 1
        $self set_subwidget indicator $path.indicator
        $self set_subwidget icon $path.icon


        $path configure -command "tk_popup $path.menu \[winfo pointerx $path\]\
                        \[winfo pointery $path\]"
        menu $path.menu
        $path.menu add checkbutton -label "Mute" \
                        -variable [$self tkvarname mute_] \
                        -command "$self mute \$\{[$self tkvarname mute_]\}"
        $path.menu add cascade -label "Info..." -menu $path.menu.info
        $self build_info_menu $path.menu.info
}


UISource/Audio public create_root_widget1 {path} {
	frame $path -bd 1 -relief raised
	[$self info class] instvar image_
	button $path.icon -image $image_ -bd 0 -relief flat \
			-highlightthickness 0 -takefocus 0
	$path.icon configure -activebackground [option get $path defaultColor \
			UISource]
	frame $path.indicator -width 0 -height 0 -bg [option get $path \
			defaultColor UISource] -bd 0 -relief sunken
	pack $path.icon -side left
	pack $path.indicator -padx 7 -side left
}


UISource/Audio public rank { {rank {}} } {
	# FIXME
	$self instvar rank_
	if ![info exists rank_] { set rank_ 4 }
	if { $rank == {} } { return $rank_ } else {
		set rank_ $rank
		$self draw_indicator
	}
}


UISource/Audio private draw_indicator {} {
	$self instvar src_ disabled_

	set indicator [$self subwidget indicator]
	if { [info exists disabled_] && $disabled_ } {
		$indicator configure -width 10 -height 10 -bd 1 \
				-bg [option get [$self info path] \
				disabledColor UISource/Audio]
		pack configure $indicator -padx 1
	} elseif [$src_ mute] {
		$indicator configure -width 9 -height 9 -bd 1 \
				-bg red
		pack configure $indicator -padx 1
	} else {
		switch [$self rank] {
			0 {
				$indicator configure -width 9 -height 9 \
						-bg green -bd 1
				pack configure $indicator -padx 1
			}
			1 {
				$indicator configure -width 7 -height 7 \
						-bg green -bd 1
				pack configure $indicator -padx 2
			}
			2 {
				$indicator configure -width 3 -height 3 \
						-bg green -bd 1
				pack configure $indicator -padx 5
			}
			default {
				$indicator configure -width 0 -height 0 -bg \
						[option get [$self info path]\
						defaultColor UISource] -bd 0
				pack configure $indicator -padx 6
			}
		}
	}
}


UISource/Audio public disable {flag} {
	$self instvar disabled_
	set disabled_ $flag
	$self draw_indicator
}


UISource public highlight {flag} {
	$self instvar msrc_
	if $flag { set relief sunken } else { set relief raised }
	$self set_relief $relief
	$msrc_ highlight $flag
}


UISource public set_relief {relief} {
	[$self info path] configure -relief $relief
}


UISource/Audio public set_relief {relief} {
	$self subwidget b configure -relief $relief
}


MultiSource public init {sm cname} {
	$self next
	$self instvar sm_ widget_ cname_ highlight_
	set list [$sm list_ui]
	$list insert end [list -id $self ""]
	set widget_ [$list info widget -id $self]
	set sm_ $sm
	set cname_ $cname
	set highlight_ 0

	$self build_widget $widget_
}


MultiSource public destroy {} {
	$self instvar sm_
	set list [$sm_ list_ui]
	$list delete -id $self
	$self next
}


MultiSource public build_widget { path } {
	$path configure -bd 1 -relief sunken
	frame $path.info_and_labels
	frame $path.labels
	label $path.name -anchor w -pady 0
	pack $path.name -fill both -expand 1 -side right -anchor w \
			-in $path.labels
	pack $path.labels -fill x -side bottom -in $path.info_and_labels
	pack $path.info_and_labels -fill x -expand 1 -side right -anchor sw
}


MultiSource public highlight {flag} {
	$self instvar widget_ highlight_
	if $flag { incr highlight_ } else { incr highlight_ -1 }
	if { $highlight_ > 0 } {
		set color [option get $widget_ activeColor UIMultiSource]
	} else {
		set color [option get $widget_ defaultColor UIMultiSource]
	}
	$widget_ subwidget name configure -bg $color
}


MultiSource public cname {} {
	return [$self set cname_]
}


MultiSource public name {} {
	$self instvar widget_
	return [$widget_ subwidget name cget -text]
}


MultiSource public change_name { uisrc name {do_mb_hack 1} } {
	$self instvar widget_ sources_ sm_
	$widget_ subwidget name configure -text $name
	if $do_mb_hack { $sm_ mb_cname_hack $self $uisrc }
}


MultiSource public add_source {media src observer} {
	$self instvar widget_ sources_

	set m [string toupper [string index $media 0]][string tolower \
			[string range $media 1 end]]
	if { [WidgetClass info instances UISource/$m] != "UISource/$m" } {
		set m Unknown
	}

	set sources_($media,$src) [UISource/$m $widget_.source$src \
			$observer $src $self]
	pack $sources_($media,$src) -side left -fill y -side left \
			-in $widget_.labels
}


MultiSource public remove_source {media src} {
	$self instvar sources_ widget_
	destroy $sources_($media,$src)
	unset sources_($media,$src)

	if { [winfo exists $widget_.thumbnails] && \
			[pack slaves $widget_.thumbnails] == {} } {
		destroy $widget_.thumbnails
		destroy $widget_.info
	}

	if { [llength [pack slaves $widget_.labels]] <= 1 } {
		# only the name label is left
		delete $self
		return 0
	}

	return 1
}


MultiSource public add_thumbnail {media src} {
	$self instvar widget_ sources_
	if ![winfo exists $widget_.thumbnails] {
		frame $widget_.thumbnails
		pack $widget_.thumbnails -fill y -side left
		frame $widget_.info
		pack $widget_.info -fill both -side top \
				-in $widget_.info_and_labels
	}

	$sources_($media,$src) create_thumbnail $widget_.thumbnails
	$sources_($media,$src) create_info $widget_.info
}


MultiSource public remove_thumbnail {media src} {
	$self instvar widget_ sources_

	$sources_($media,$src) destroy_thumbnail
	$sources_($media,$src) destroy_info
	if { [winfo exists $widget_.thumbnails] && \
			[pack slaves $widget_.thumbnails] == {} } {
		destroy $widget_.thumbnails
		destroy $widget_.info
	}
}


MultiSource public num_sources {} {
	$self instvar sources_
	return [llength [array names sources_]]
}


MultiSource public change_cname {cname} {
	$self set cname_ $cname
}


MultiSource public get_uisource {media src} {
	$self instvar sources_
	if [info exists sources_($media,$src)] {
		return $sources_($media,$src)
	} else { return "" }
}


Class MultiSourceManager


MultiSourceManager public init {list {cl MultiSource}} {
	$self instvar list_ class_
	set list_ $list
	set class_ $cl
}


MultiSourceManager public list_ui {} {
	return [$self set list_]
}


MultiSourceManager public register {media src cname observer} {
	$self instvar msrcs_ class_
	if [info exists msrcs_($media,$src)] {
		puts stderr "source object already exists for $media,$src"
		return $msrcs_($media,$src)
	}

	if [info exists msrcs_(cname,$cname)] {
		set s $msrcs_(cname,$cname)
	} else {
		set s [new $class_ $self $cname]
		set msrcs_(cname,$cname) $s
	}

	$s add_source $media $src $observer
	set msrcs_($media,$src) $s
	return $s
}


MultiSourceManager public unregister {media src} {
	$self instvar msrcs_
	if ![info exists msrcs_($media,$src)] return

	set cname [$msrcs_($media,$src) cname]
	if ![$msrcs_($media,$src) remove_source $media $src] {
		# this multi-source is empty and has been deleted
		unset msrcs_(cname,$cname)
	}
	unset msrcs_($media,$src)
}


MultiSourceManager public change_cname {media src cname} {
	$self instvar msrcs_
	if ![info exists msrcs_($media,$src)] return

	if { [$msrcs_($media,$src) num_sources] > 1 || \
			[info exists msrcs_(cname,$cname)] } {
		set uisrc [$msrcs_($media,$src) get_uisource $media $src]
		$uisrc save_state state_var
		set o [$uisrc set observer_]
		$self unregister $media $src
		set s [$self register $media $src $cname $o]
		set uisrc [$msrcs_($media,$src) get_uisource $media $src]
		$uisrc restore_state state_var
		return $s
	} else {
		# update in place
		unset msrcs_(cname,[$msrcs_($media,$src) cname])
		$msrcs_($media,$src) change_cname $cname
		set msrcs_(cname,$cname) $msrcs_($media,$src)
		return $msrcs_($media,$src)
	}
}


MultiSourceManager public enable_mb_cname_hack { flag } {
	$self instvar enable_hack_ msrcs_
	if { $flag } {
		set enable_hack_ 1
		# run through the whole list
		foreach n [array names msrcs_ cname,*] {
			# a previous pass thru the list may have deleted this
			# msrc
			if ![info exists msrcs_($n)] continue

			set msrc $msrcs_($n)
			if { [llength [$msrc mb_hack_get_mb_uisources]]==0 } {
				$self mb_cname_hack $msrc {}
			}
		}
	} else {
		# run through the whole list
		foreach n [array names msrcs_ cname,*] {
			# a previous pass thru the list may have deleted this
			# msrc
			if ![info exists msrcs_($n)] continue

			set msrc $msrcs_($n)
			if { [llength [$msrc mb_hack_get_mb_uisources]] > 0 } {
				$self mb_cname_hack $msrc {} 1
			}
		}

		unset enable_hack_
	}
}


MultiSourceManager public mb_cname_hack { msrc uisrc {only_kickout 0} } {
	$self instvar enable_hack_ msrcs_

	if ![info exists enable_hack_] return

	# first check if there are any mb sources in this msrc
	# if so, kick them out
	foreach u [$msrc mb_hack_get_mb_uisources] {
		if { $u == $uisrc } continue
		$self mb_hack_move_mb_src $u $msrc {}
	}

	if $only_kickout return

	if { $uisrc != {} && [$uisrc info class] == "UISource/Mb" } {
		set ismb 1
	} else {set ismb 0}
	# now run thru all the msrcs; if you find an msrc with the same
	# name as the one that just changed, then look for mb objects
	# inside that msrc and group them with our msrc
	set name [$msrc name]
	foreach n [array names msrcs_ cname,*] {
		# a previous pass thru the list may have deleted this msrc
		if ![info exists msrcs_($n)] continue

		set m $msrcs_($n)
		if { $m == $msrc } continue
		if { [$m name] != $name } continue

		if $ismb {
			# copy us into $m
			$self mb_hack_move_mb_src $uisrc $msrc $m
			break
		} else {
			# copy all the mb source from $m into us
			foreach u [$m mb_hack_get_mb_uisources] {
				if { $u == $uisrc } continue
				$self mb_hack_move_mb_src $u $m $msrc
			}
		}
	}
}


MultiSource private mb_hack_get_mb_uisources { } {
	$self instvar sources_
	set list {}
	foreach n [array names sources_ mb,*] {
		lappend list $sources_($n)
	}
	return $list
}


MultiSourceManager private mb_hack_move_mb_src { uisrc old_msrc new_msrc } {
	set o [$uisrc set observer_]
	set src [$uisrc src]
	if { $new_msrc != {} } {
		set cname [$new_msrc cname]
	} else {
		set cname [$src set mb_hack_cname_]
	}

	$uisrc save_state state_var
	$self unregister mb $src
	set msrc [$self register mb $src $cname $o]
	set uisrc [$msrc get_uisource mb $src]
	$uisrc restore_state state_var

	$src instvar mb_hack_name_
	if { $new_msrc == {} && [info exists mb_hack_name_] } {
		$msrc change_name $uisrc $mb_hack_name_ 0
	}

	$o set msrcs_($src) $msrc
}



#MultiSourceManager public change_name {media src name} {
#	$self instvar sources_
#	if ![info exists sources_($media,$src)] return
#
#	$sources_($media,$src) change_name $name
#}


MultiSourceManager public activate {media src} {
}


MultiSourceManager public deactivate {media src} {
}


Class Observer/RTP -superclass Observer
Class Observer/Video -superclass Observer/RTP -configuration {
	filterGain 0.25
	vain false
}

Class Observer/Audio -superclass Observer/RTP
Class Observer/Mb    -superclass Observer
Class Observer/Mbv2  -superclass Observer


Observer/RTP public init { sm agent } {
	$self next
	$self instvar sm_ agent_
	set sm_ $sm
	set agent_ $agent

	$agent_ attach $self
}


Observer/RTP public agent { } {
	$self instvar agent_
	return $agent_
}


Observer/Video public scuba_session { } {
	$self instvar scuba_sess_
	return $scuba_sess_
}


Observer/Audio public init { sm agent panel controlMenu } {
	$self next $sm $agent
	$self instvar controlMenu_ rank_ panel_
	set controlMenu_ $controlMenu
	set panel_ $panel
	set rank_ [new Rank]
}


Observer/Audio public destroy {} {
	$self cancel_talk_monitor
	$self next
}


Observer/RTP private create_multi_source {media src} {
	$self instvar sm_ msrcs_
	set cname [$src sdes cname]
	if { $cname == "" } { set cname [$src addr] }
	set msrcs_($src) [$sm_ register $media $src $cname $self]

	$self trigger_sdes $src
}


Observer/Audio private create_multi_source src {
	$self next audio $src
}


Observer/Audio public register src {
	#
	# if we display everyone (i.e., non-mixers and mixers),
	# then just create the name in the sitebox.  otherwise,
	# it gets deferred until we receive an SDES packet
	# (which never happens for a mixer) and trigger_sdes
	# gets called (or activate gets called because that site
	# is speaking in which case we have no choice but to
	# put up the non-sdes-style name up).
	#
	$self instvar agent_
	if { [$self yesno displayMixers] || "$src" == [$agent_ local] } {
		$self create_multi_source $src
	}
}


Observer/Audio public unregister src {
	$self instvar sm_ msrcs_
	$self cancel_talk_monitor $src
	$sm_ unregister audio $src
	unset msrcs_($src)
}


Observer/RTP public trigger_sdes src {
	$self instvar msrcs_ sm_
	if ![info exists msrcs_($src)] {
		#
		# create_multi_source will call us back (with the name set)
		# so just return here
		#
		$self create_multi_source $src
		return
	}
	#
	# Figure out best presentation from available information.
	#
	set name [$src sdes name]
	set cname [$src sdes cname]
	set addr [$src addr]
	if { $name == "" } {
		if { $cname == "" } {
			set src_nickname $addr
			set info $addr/[$src format_name]

		} else {
			set src_nickname $cname
			set info "$addr/[$src format_name]"
		}
	} elseif [cname_redundant $name $cname] {
		set src_nickname $name
		set info $addr/[$src format_name]
	} else {
		set src_nickname $name
		set info $cname/[$src format_name]
	}
	set src_info $cname/[$src format_name]

	set msg [$src sdes text]
	if { $msg != "" } {
		set info $msg
	}
	set src_info $info

	# figure out our media
	set cl [$self info class]
	set media [string tolower [lindex [split $cl /] 1]]
	set uisrc [$msrcs_($src) get_uisource $media $src]

	#
	# only call change_name when name really changes
	# all this does is change the name in the on-screen site display
	#
	if { [$msrcs_($src) name] != $src_nickname } {
		$msrcs_($src) change_name $uisrc $src_nickname
	}
	$self change_name $uisrc $src_nickname

	if { [$uisrc get_info] != $src_info } {
		$uisrc change_info $src_info
	}

	if { [$msrcs_($src) cname] != $cname } {
		$self change_cname $media $src $cname
		set msrcs_($src) [$sm_ change_cname $media $src $cname]
	}
}


Observer/RTP public change_cname { media src cname } {
}


Observer/Audio public change_cname { media src cname } {
	$self instvar rank_ msrcs_
	set uisrc [$msrcs_($src) get_uisource audio $src]
	$rank_ clear $uisrc
}


Observer/RTP public change_name { uisrc name } {
}


Observer/Video public change_name { uisrc name } {
	$self instvar videoContainer_
	$videoContainer_ change_name $uisrc $name
}


#
# Delete the statistics associated with the source, <i>src</i>.
#
Observer/Audio public deactivate src {
	$src handler ""
}

#
# Called from our AudioAgent base class the first time
# a source sends data packets.  AudioAgent::activate creates
# the decoder; we do additional ui specific initialization here.
#
Observer/Audio public activate src {
# FIXME this only sets lecture-mode when a new source arrives...
	set decoder [$src handler]
	$self instvar controlMenu_
	$decoder lecture-mode [$controlMenu_ query lectureMode]
}


#
# Create an entry for the provided source, <i>src</i>, and begin monitoring
# their activity.
#
Observer/Audio public trigger_media src {
	$self instvar rank_ msrcs_ id_
	if ![info exists msrcs_($src)] {
		$self create_multi_source $src
	}
	set uisrc [$msrcs_($src) get_uisource audio $src]
	$uisrc highlight 1
	$rank_ touch $uisrc

	set id_($src) [after 500 "$self monitor_talk_spurt $src"]

	#
	# if we don't have the audio, request.
	# (but don't demand it since this happens on
	#  network input not user intervention)
	#
	#FIXME does this belong in AudioAgent?
	#
	$self instvar panel_
	$panel_ action
}


#
# Every 1/2 second, check how recently the source, <i>src</i> has sent
# data.  If the source is active, keep control of the audio device, else
# unhilight their name.
#
Observer/Audio private monitor_talk_spurt src {
	$self instvar agent_ panel_ msrcs_ id_
	if [info exists msrcs_($src)] {
		set uisrc [$msrcs_($src) get_uisource audio $src]
		set delta [expr [$agent_ ntp_time] - [$src last-data]]
		#FIXME delta is in ntp time units.  (65536 Hz)
		if { $delta > 20000 } {
			$uisrc highlight 0
			$src enable_trigger
			unset id_($src)
		} else {
			$panel_ action
			set id_($src) [after 500 \
					"$self monitor_talk_spurt $src"]
		}
	}
}

#
# Cancels monitor_talk_spurt.
#
Observer/Audio private cancel_talk_monitor { {src {}} } {
        $self instvar id_
	if { $src != {} } {
		if [info exists id_($src)] {
			after cancel $id_($src)
			unset id_($src)
		}
	} else {
		foreach src [array names id_] {
			after cancel $id_($src)
			unset id_($src)
		}
	}
}


#
Observer/Audio public trigger_idle src {
	$self instvar msrcs_
	if [info exists msrcs_($src)] {
		set uisrc [$msrcs_($src) get_uisource audio $src]
		$uisrc disable [$src lost]
	}
}


#FIXME
Observer/Audio public trigger_format src {
	$self instvar agent_
	$agent_ deactivate $src
	$agent_ activate $src
}



Observer/Video public init { sm agent videoContainer {can_voiceswitch 0} \
		{ scuba_sess {} } } {
	$self set videoContainer_ $videoContainer
	$self set can_voiceswitch_ can_voiceswitch
	$self set scuba_sess_ $scuba_sess
	$self next $sm $agent
}


Observer/Video public register {src} {
	$self create_multi_source $src
}


Observer/Video public unregister src {
	$self instvar sm_ msrcs_
	$sm_ unregister video $src
	unset msrcs_($src)
}


Observer/Video private create_multi_source src {
	$self next video $src
}


#
# Add a <i>src</i> to the active senders list.  E.g., make a postage
# stamp window appear, stats, etc. so that the user can select
# the video stream.
#
Observer/Video public activate src {
	#
	# give the VideoAgent a chance to create and install
	# a decoder and for that decoder to see a packet so it can
	# determine the output geometry and color decimation.
	# we shouldn't have to do this (e.g., resize will
	# take care of a geometry change), but currently
	# decoders can't trigger a renderer realloation
	# when the decimation changes.FIXME fix this
	# Note that trigger_format allocates the decoder object
	# is called before the event loop goes idle
	#
	after idle "$self really_activate $src"
}


Observer/Video private really_activate src {
	$self instvar msrcs_ videoContainer_
	$msrcs_($src) add_thumbnail video $src

	$self update_decoder $src
	$videoContainer_ activate [$msrcs_($src) get_uisource video $src]

	#
	# Someone became activated, so we have to change
	# the switchable menu to include this participant
	#
	Switcher rebuild_switch_list_menu
}


Observer/Video public detach_windows {src} {
	$self instvar msrcs_ videoContainer_
	if [info exists msrcs_($src)] {
		set uisrc [$msrcs_($src) get_uisource video $src]

		$videoContainer_ deactivate $uisrc \
				[$self active-sources $uisrc 1]
		# delete all userwindows
		foreach uw [$uisrc user_windows] {
			#FIXME should check if we're voice-switched
			# and if so, bump window
			if { [[$uw info class] info heritage VideoWidget] \
					== {} } { delete $uw }
		}

		# detach all user windows and thumbnails
		$uisrc detach_windows
		$msrcs_($src) remove_thumbnail video $src
		$uisrc delete-decoder-window
	}
}


#
# Remove a <i>src</i> from the active senders list.
#
Observer/Video public deactivate {src} {
	$self instvar msrcs_ videoContainer_
	$self detach_windows $src
	$src handler ""

	#
	# Someone became de-activated, so we have to change
	# the switchable menu to un-include this participant
	#
	Switcher rebuild_switch_list_menu

	#FIXME
	global fpshat bpshat lhat shat
	unset fpshat($src)
	unset bpshat($src)
	unset lhat($src)
	unset shat($src)
}


#
# For the video source, <i>src</i>, set the rate variables.
#
Observer/Video private set_rate_vars {src} {
	global fpshat bpshat lhat shat
	if [info exists fpshat($src)] {
		unset fpshat($src)
		unset bpshat($src)
		unset lhat($src)
		unset shat($src)
	}
	set gain [$self get_option filterGain]
	set fpshat($src) 0
	rate_variable fpshat($src) $gain
	set bpshat($src) 0
	rate_variable bpshat($src) $gain
	set lhat($src) 0
	rate_variable lhat($src) $gain
	set shat($src) 0
	rate_variable shat($src) $gain

#FIXME set guys in stat window too!
}


#
# For the source specified by <i>src</i>, update its rate variables and
# its source description.
#
Observer/Video private update_decoder src {
	$self set_rate_vars $src
	$self trigger_sdes $src
}



#
Observer/Video public trigger_format {src} {
	$self instvar msrcs_ videoAgent_

	if ![info exists msrcs_($src)] {
		#
		# if we get a change format before really_activate
		# was called (i.e., so we don't even have a thumbnail yet),
		# don't do anything
		#
		return
	}

	set uisrc [msrcs_($src) get_uisource video $src]

	set L [$uisrc user_windows]
	$uisrc detach_windows
	$uisrc detach_thumnail
	#FIXME
	set extoutList [extout_detach_src $src]

	set d [$videoAgent_ reactivate $src]

	$self update_decoder $src
	$d color [$uisrc color]

	foreach uw $L {
		$uisrc attach_window $uw
		[$uw video-widget] redraw
	}
	$uisrc attach_thumbnail
	#FIXME
	extout_attach_src $src $extoutList
}


Observer/Video public trigger_format_all { } {
	$self instvar msrcs_
	foreach s [array names msrcs_] {
		if [$s have_thumbnail] { $self trigger_format $s }
	}
}


Observer/Video public mute_new_sources {} {
	# FIXME
	return 0
}


Observer/Video public get_uisource src {
	$self instvar msrcs_
	if ![info exists msrcs_($src)] { return "" }
	return [$msrcs_($src) get_uisource video $src]
}


# FIXME: required for UserWindow
Observer/Video public get_activesource src {
	return [$self get_uisource $src]
}


Observer/Video public active-sources { {not_this {}} {list_uisrcs 0} } {
	$self instvar msrcs_
	set list {}
	if $list_uisrcs { set var uisrc } else { set var src }
	foreach src [array names msrcs_] {
		set uisrc [$msrcs_($src) get_uisource video $src]
		if { $uisrc != $not_this && [$uisrc have_thumbnail] } {
			lappend list [set $var]
		}
	}
	return $list
}


Observer/Video public video_container { } {
	$self instvar videoContainer_
	return $videoContainer_
}


Observer/Video public can_voiceswitch { } {
	return [$self set can_voiceswitch_]
}


#
# Dispatch focus method on switcher windows.
#
Observer/Video public focus_speaker { infoname msg } {
	foreach s [$self active-sources] {
		if { [$s sdes cname] == $msg } {
		        Switcher focus $s
		}
        }
}
