# fca-fcd.tcl --
#
#       Floor control dynamics
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


########### FCAFloorInstance ########
Class FCAFloorInstance
FCAFloorInstance instproc init {} {
    $self next
    $self set grantSeq_ 0
    $self set srcId_ ""
    $self set requestId_ ""
}


FCAFloorInstance instproc evacuate {} {
    $self set grantSeq_ 0
    $self set srcId_ ""
    $self set requestId_ ""
}


FCAFloorInstance instproc srcId {} {
    return [$self set srcId_]
}


FCAFloorInstance instproc requestId {} {
    return [$self set requestId_]
}


FCAFloorInstance instproc isGrant {} {
    #return [$self set isGrant_]
    if { [$self set srcId_]=={} } {
	return 0
    } else {
	return 1
    }
}


FCAFloorInstance instproc grantSeq {} {
    return [$self set grantSeq_]
}


########### FCAFloor ################
Class FCAFloor -superclass FCAFloorConfig
FCAFloor instproc init {floorType maxHldrs} {
    $self next $floorType $maxHldrs
    $self instvar floorInstances_
    for {set i 0} {$i < $maxHldrs} {incr i 1 } {
	set floorInstances_($i) [new FCAFloorInstance]
    }
}


#FCAFloor instproc evacuate { instance } {
#    $self instvar floorInstances_
#    $floorInstances_($instance) evacuate
#}

FCAFloor instproc evacuate {} {
    $self instvar floorInstances_ maxHldrs_
    for {set i 0} {$i < $maxHldrs_ } {incr i 1} {
	$floorInstances_($i) evacuate
    }
}

# return free instance number if one is avail, else return -1
FCAFloor instproc getFreeInstance {} {
    $self instvar floorInstances_ maxHldrs_
    for {set i 0} {$i < $maxHldrs_} {incr i 1} {
	if { [info exist floorInstances_($i)]} {
	    return $i
	}
    }
    return -1
}


FCAFloor instproc floorInstances {} {
    return [$self set floorInstances_]
}


#FCAFloor instproc getLastGrantSeq {instance} {
#    $self instvar floorInstances_
#    return [$floorsInstances_($instance) set grantSeq_]
#}

#FCAFloor instproc getIsGrant {instance} {
#    $self instvar floorInstances_
#    return [$floorsInstances_($instance) set isGrant_]
#}

# return floorInstance of srcId, -1 for not existing
#FCAFloor instproc floorInstance { srcId } {
#    $self instvar floorInstances_ maxHldrs_
#    for {set i 0} {$i < $maxHldrs_} {incr i 1} {
#	if {[$hldrIds_($i) set srcId_] == $srcId}{
#	    return $i
#	}
#    }
#    return -1
#}


FCAFloor instproc getFloorInstance { instance } {
    if { ($instance < [$self set maxHldrs_]) && ($instance >= 0)} {
	return [$self set floorInstances_($instance)]
    } else {
	return ""
    }
}


FCAFloor instproc get_holder {instanceNum srcIdVar requestIdVar} {
    $self instvar floorInstances_ maxHldrs_
    upvar $srcIdVar srcId
    upvar $requestIdVar requestId

    if { ($maxHldrs_ > $instanceNum) && ($instanceNum >= 0) } {
	set srcId [$floorInstances_($instanceNum) set srcId_]
	set requestId [$floorInstances_($instanceNum) set requestId_]
    } else {
	set srcId ""
	set requestId ""
    }
}


# returns the floor instance if the request exists; -1 otherwise
FCAFloor instproc request_exists {srcId requestId} {
    $self instvar floorInstances_ maxHldrs_
    for { set i 0 } { $i < $maxHldrs_ } { incr i } {
	if { [$floorInstances_($i) set srcId_]==$srcId &&
	[$floorInstances_($i) set requestId_]==$requestId } {
	    return $i
	}
    }
    return -1
}


# return true(1) if successful
FCAFloor instproc grant_floor {instanceNum srcId requestId grantSeq} {
    $self instvar floorInstances_ maxHldrs_
    if { ($maxHldrs_ > $instanceNum) && ($instanceNum >= 0) } {
	$floorInstances_($instanceNum) set srcId_ $srcId
	$floorInstances_($instanceNum) set requestId_ $requestId
	$floorInstances_($instanceNum) set grantSeq_ $grantSeq
	#$floorInstances_($instanceNum) set isGrant_  1
	return 1
    }
    DbgOut "Error, no such instance ($instanceNum) exists for this floor"
    return 0
}


FCAFloor instproc release_floor {instanceNum} {
    $self instvar floorInstances_ maxHldrs_
    if { ($maxHldrs_ > $instanceNum) && ($instanceNum >= 0) } {
	$floorInstances_($instanceNum) set srcId_ ""
	$floorInstances_($instanceNum) set requestId_ ""
	#$floorInstances_($instanceNum) set isGrant_ 0
    }
}



# return 0 if successful
#FCAFloor instproc releaseHolder {instance grantSeq} {
#    $self instvar floorInstances_ maxHldrs_
#    if { ($maxHldrs_ > $instanceNum) && ( $instanceNum >=0 ) } {
#	if {[$floorInstances_($instanceNum) set grantSeq_] != $grantSeq} {
#	    $floorInstances_($instanceNum) set isGrant_ 0
#	    return 1
#	} else {
#	    DbgOut "this proc shouldn't be called, seq no mismatch"
#	}
#    }
#    DbgOut "Error, no such instance ($instanceNum) exists for this floor"
#    return 0
#}









Class FCA_RequestList


FCA_RequestList instproc init { } {
    $self next
    $self set list_ {}
}


FCA_RequestList instproc add { srcId requestId } {
    $self instvar list_
    lappend list_ "$srcId+$requestId"
    DbgOut "Appending $srcId:$requestId to list_: $list_"
}


FCA_RequestList instproc remove_top { srcIdVar requestIdVar } {
    upvar $srcIdVar srcId
    upvar $requestIdVar requestId
    $self instvar list_

    set id [lindex $list_ 0]
    set list_ [lreplace $list_ 0 0]
    set id [split $id "+"]
    set srcId [lindex $id 0]
    set requestId [lindex $id 1]
}


FCA_RequestList instproc get_list {} {
    $self instvar list_
    set returnlist ""
    foreach item $list_ {
	set newItem [split $item "+"]
	lappend returnlist $newItem
    }
    return $returnlist
}


FCA_RequestList instproc remove { srcId requestId } {
    $self instvar list_
    set idx [lsearch $list_ $srcId+$requestId]
    if { $idx!=-1 } {
	set list_ [lreplace $list_ $idx $idx]
	return 1
    } else {
	return 0
    }
}



FCA_RequestList instproc exists { srcId requestId } {
    $self instvar list_
    DbgOut "Searching for existence of $srcId:$requestId in list_: $list_"
    if { [lsearch $list_ "$srcId+$requestId"] != -1 } {
	return 1
    } else {
	return 0
    }
}


FCA_RequestList instproc count { } {
    $self instvar list_
    return [llength $list_]
}


Class FCA_RequestList/Bounded -superclass FCA_RequestList


FCA_RequestList/Bounded instproc init { fcp } {
    $self next
    $self set fcp_ $fcp
}


FCA_RequestList/Bounded instproc is_full { } {
    $self instvar list_ fcp_
    if { [llength $list_] < [$fcp_ maxRqs] } {
	return 0
    }
    return 1
}


FCA_RequestList/Bounded instproc avail_space {} {
    $self instvar list_ fcp_
    return [expr [$fcp_ maxRqs] - [llength $list_]]
}















# DATA MEMBERS of Class FCAFloorDynamics
# mgr_    : FCA manager for the instance of FCAFloorDynamics
# floors_    : of class FCAFloor -- list of floors
# admittedRequests_: list of FCA_RequestList/Bounded
# pendingRequests_: list of FCA_RequestList
# fcp_    : floor control policy for this session
# moderatorIds_: list of chairs ids (list of uid and addr) { {uid addr} ... }
# MEMBER FUNCTIONS:
# isChair $uid $addr, setRqQsize $qsize, matchfid $fid, fl $fid,
# removeHolder $fid $uid $addr
# newRq $fid $rq
# replaceHolder $fid $olduid $oldaddr $newuid $newaddr

Class FCAFloorDynamics
FCAFloorDynamics instproc init {mgr fcp moderatorIds} {
    $self next
    $self instvar mgr_ fcp_ floors_ admittedRequests_ pendingRequests_ \
	    moderatorIds_
    #set curStateNum_ 0
    set topGrantSeq_ 0
    set mgr_ $mgr
    set fcp_ $fcp
    set moderatorIds_ $moderatorIds
    set admittedRequests_ [new FCA_RequestList/Bounded $fcp]
    set pendingRequests_  [new FCA_RequestList]
    #set canceledRequests_ [new FCA_RequestList]

    foreach i [$fcp_ floorConfigs] {
	puts "$i"
	set floors_([$i floorType]) [new FCAFloor [$i floorType] [$i maxHldrs]]
    }


    $self set curModeratorState_ 0
}


FCAFloorDynamics instproc reset {} {
    $self instvar floors_ fcp_
    set ftypes [$fcp_ floorTypes]
    foreach f $ftypes {
	$floors_($f) evacuate
    }
}


FCAFloorDynamics instproc admitted_requests {} {
    return [$self set admittedRequests_]
}


FCAFloorDynamics instproc maxInstanceNum { floorType } {
    $self instvar fcp_
    return [$fcp_ getMaxHdlrs $floorType]
}


# return fcaPkt of type PKT_GRANT_UPDATE
FCAFloorDynamics instproc make_grant_pkt {} {
    $self instvar floors_
    set fcaPkt [new FCA_Packet]
    $fcaPkt set pktType "PKT_GRANT_UPDATE"
    $fcaPkt set moderatorState [$self moderator_state]
    $fcaPkt set pktUpdates ""
    set numUpdates 0
    set ftypes [array names floors_]
    foreach ftype $ftypes {
        set floor $floors_($ftype)
        set maxInstanceNum [$self maxInstanceNum $ftype]

        for {set instanceNum 0} {$instanceNum < $maxInstanceNum } \
		{ incr instanceNum 1 } {
	    set instance [$self getFloorInstanceObj $ftype $instanceNum]
	    if {[$instance srcId] != ""} {
		incr numUpdates 1
		set pktUpdate [new FCA_Packet]
		$pktUpdate set srcId [$instance srcId]
		$pktUpdate set grantSeqno [$instance grantSeq]
		$pktUpdate set requestId [$instance requestId]
		$pktUpdate set floorType [$floor floorType]
		$pktUpdate set floorInstance $instanceNum
		$fcaPkt lappend pktUpdates $pktUpdate
	    }
	}
    }
    $fcaPkt set numUpdates $numUpdates
    return $fcaPkt
}


FCAFloorDynamics instproc make_queue_pkt {} {
    $self instvar admittedRequests_

    set fcaPkt [new FCA_Packet]
    $fcaPkt set pktType "PKT_QUEUE_UPDATE"
    $fcaPkt set pktUpdates ""
    $fcaPkt set moderatorState [$self moderator_state]
    set admittedList [$admittedRequests_ get_list]
    $fcaPkt set numUpdates [llength admittedList]
    foreach rq $admittedList {
	set pktUpdate [new FCA_Packet]
	$pktUpdate set srcId [lindex $rq 0]
	$pktUpdate set requestId [lindex $rq 1]
	$pktUpdate set isAdd 1
	$fcaPkt lappend pktUpdates $pktUpdate
    }
    return $fcaPkt
}



FCAFloorDynamics instproc getFloorInstanceObj { ftype instance } {
    $self instvar floors_ fcp_
    if { ![info exist floors_($ftype)] } {
	DbgOut "$ftype floor type doesn't exist"
	return ""
    }

    set instObj [$floors_($ftype) getFloorInstance $instance]
    if { $instObj!={} } {
	return $instObj
    }
    DbgOut "Floor instance doesn't exist"
    return ""
}


FCAFloorDynamics instproc isRequestQFull {} {
    $self instvar admittedRequests_
    return [$admittedRequests_ is_full]
}


FCAFloorDynamics instproc numRequests {} {
    $self instvar admittedRequests_
    return [$admittedRequests_ count]
}



FCAFloorDynamics instproc avail_requestQ_space {} {
    $self instvar admittedRequests_
    return [$admittedRequests_ avail_space]
}


# return boolean indicating whether successful
FCAFloorDynamics instproc admit_request { srcId requestId } {
    $self instvar admittedRequests_

    DbgOut "Admitting request $srcId:$requestId"
    $admittedRequests_ add $srcId $requestId

    # add to the ui as well, if we have the actual request
    $self instvar mgr_
    set allRequests [[$mgr_ getRcvr $srcId] get_requests]
    if { [$allRequests have $requestId] } {
	[$mgr_ uiMgr] add_admit [$allRequests get $requestId]
    }
}


# return boolean to indicate whether changed fcd
FCAFloorDynamics instproc remove_admitted_request {srcId requestId} {
    $self instvar admittedRequests_
    if { ![$admittedRequests_ remove $srcId $requestId] } {
	DbgOut "Could not remove $srcId:$requestId from admitted requests"
	return 0
    }

    # remove from the ui as well, if we have the actual request
    $self instvar mgr_
    set allRequests [[$mgr_ getRcvr $srcId] get_requests]
    if { [$allRequests have $requestId] } {
	[$mgr_ uiMgr] remove_admit [$allRequests get $requestId]
    } else {
	DbgOut "Don't have $srcId:$requestId"
    }

    return 1
}


# get floor request that ask for ftype and instance, not removed from list
#FCAFloorDynamics instproc getFloorRequest { ftype instance } {
#    $self instvar admittedRequests_
#    foreach i $admittedRequests_ {
#	if {($ftype == [$i set floorType_]) && \
#		($instance == [$i set floorInstance_])} {
#	    return $i
#	}
#    }
#    return ""
#}

# first item in the queue is returned and dequeued
#FCAFloorDynamics instproc nextFloorRequest {} {
#    $self instvar admittedRequests_
#    if { [llength $admittedRequests_] == 0 } {
#	DbgOut "There is no floor requests. "
#	return ""
#    }
#    set rqst [lindex $admittedRequests_ 0 0]
#    set admittedRequests_ [lreplace $admittedRequests_ 0 0]
#    return $rqst
#}


# first item in the queue is returned and dequeued
#FCAFloorDynamics instproc next_pending_request {} {
#    $self instvar pendingRequests_
#
#    $pendingRequests_ remove_head srcId requestId
#    if { $srcId=={} || $requestId=={} } {
#	DbgOut "Could not get any request from the pending queue"
#	return ""
#    }
#
#    set rcvr [$mgr_ getRcvr $srcId]
#    if { $rcvr=={} } {
#	DbgOut "Could not find receiver object for $srcId"
#	return ""
#    }
#    return [$rcvr get_request $requestId]
#}



FCAFloorDynamics instproc isPendingRequest {srcId rqId} {
    $self instvar pendingRequests_
    return [$pendingRequests exists $srcId $rqId]
}


# return 1 if changed fcd
FCAFloorDynamics instproc new_pending_request {rq} {
    $self instvar pendingRequests_

    $pendingRequests_ add [$rq srcId] [$rq requestId]

    $self instvar mgr_
    [$mgr_ uiMgr] add_pending $rq
}


# return 1 if changed fcd
FCAFloorDynamics instproc delete_pending_request {srcId rqId} {
    $self instvar pendingRequests_ mgr_
    $self instvar remove $srcId $rqId

    if { ![$pendingRequests_ remove $srcId $rqId] } {
	DbgOut "Could not remove $srcId:$rqId from admitted requests"
	return 0
    }

    set allRequests [[$mgr_ getRcvr $srcId] get_requests]
    if { [$allRequests have $rqId] } {
	[$mgr_ uiMgr] remove_pending [$allRequests get $rqId]
    }
    return 1
}


#FCAFloorDynamics instproc getLastGrantSeq {ftype instance} {
#    $self instvar floors_
#    return [$floors_($ftype) getLastGrantSeq $instance]
#}

#FCAFloorDynamics instproc getIsGrant {instance} {
#    $self instvar floors_
#    return [$floors_($ftype) getIsGrant $instance]
#}

# return instance number or -1 if none avail
FCAFloorDynamics instproc getFreeInstance {ftype} {
    $self instvar floors_
    return [$floors_($ftype) getFreeInstance]
}


FCAFloorDynamics instproc cancel_request_if_done { srcId requestId } {
    $self instvar mgr_ fcp_ floors_

    foreach fconfig [$fcp_ floorConfigs] {
	if { [$floors_([$fconfig floorType]) request_exists $srcId \
		$requestId] != -1 } {
	    # this request still exists
	    return
	}
    }

    set allRequests [[$mgr_ getRcvr $srcId] get_requests]
    $allRequests cancel $requestId
}


# return 1 if successful
FCAFloorDynamics instproc grant_floor {ftype instance srcId requestId \
	grantSeq} {
    $self instvar floors_ mgr_
    $floors_($ftype) get_holder $instance oldSrcId oldRequestId

    DbgOut "About to grant the floor"
    if { ![$floors_($ftype) grant_floor $instance $srcId $requestId \
	    $grantSeq]} {
	return 0
    }

    if { $oldSrcId!="" && $oldRequestId!="" } {
	$self cancel_request_if_done $oldSrcId $oldRequestId
    }
    DbgOut "Trying to remove admitted request"
    $self remove_admitted_request $srcId $requestId

    [$mgr_ uiMgr] grant_received $ftype [$self getFloorInstanceObj \
	    $ftype $instance]
    return 1
}



FCAFloorDynamics instproc release_floor {ftype instance} {
    $self instvar floors_ mgr_

    $floors_($ftype) evacuate $instance
    [$mgr_ uiMgr] release_received $ftype $instance
}


# return 0 if successful
FCAFloorDynamics instproc releaseHolder {ftype instance grantSeq} {
    $self instvar floors_
    return [$floors_($ftype) releaseHolder $instance $grantSeq]
}





Class FCAFloorDynamics/Moderator -superclass FCAFloorDynamics

FCAFloorDynamics/Moderator instproc init { mgr fcp moderatorIds } {
    $self next $mgr $fcp $moderatorIds
}





FCAFloorDynamics instproc moderator_state { } {
    return [$self set curModeratorState_]
}


FCAFloorDynamics instproc incr_moderator_state { } {
    $self instvar curModeratorState_
    incr curModeratorState_
    return $curModeratorState_
}


FCAFloorDynamics instproc set_moderator_state { value } {
    $self set curModeratorState_ $value
}


# GR--Grant or Release
FCAFloorDynamics instproc gr_is_stale {floorType instance grantSeq} {
    set instObj [$self getFloorInstanceObj $floorType $instance]

    set lastGrant [$instObj grantSeq]
    if { $grantSeq > $lastGrant } {
	# definitely not stale
	return 0
    }
    if {$grantSeq < $lastGrant} {
	return 1
    }

    # the two seqnos are the same
    if { [$instObj isGrant] } {
	return 0
    } else {
	return 1
    }
}




FCAFloorDynamics instproc handle_floor_request { request } {
    # I am a participant; check if this request has been admitted
    # if so, add it to the ui

    $self instvar admittedRequests_
    if { [$admittedRequests_ exists [$request srcId] [$request requestId]] } {
	$self instvar mgr_
	[$mgr_ uiMgr] add_admit $request
    }
}


FCAFloorDynamics/Moderator instproc handle_floor_request { request } {
    $self admit_request [$request srcId] [$request requestId]

    # this request gets admitted
    set state [$self incr_moderator_state]

    $self instvar mgr_
    set localRcvr [$mgr_ localRcvr]
    set isAdd 1
    $localRcvr sendQueueUpdate $isAdd [subst {{[$request srcId] \
	    [$request requestId]}}] $state
}


FCAFloorDynamics instproc handle_floor_cancel { srcId requestId \
	needQueueUpdate } {
    # the cancel might be for an already granted request
    # in that case release the grant!

    $self instvar mgr_ fcp_ floors_

    foreach fconfig [$fcp_ floorConfigs] {
	set instance [$floors_([$fconfig floorType]) request_exists $srcId \
		$requestId]
	if { $instance!=-1 } {
	    # this request exists; release this floor
	    $floors_([$fconfig floorType] release_floor $instance
	}
    }
}


FCAFloorDynamics/Moderator instproc handle_floor_cancel { srcId requestId \
	needQueueUpdate } {
    $self instvar mgr_
    $self next $srcId $requestId $needQueueUpdate
    if { $needQueueUpdate==1 } {
	set isAdd 0
	set state [$self incr_moderator_state]
	[$mgr_ localRcvr] sendQueueUpdate $isAdd \
		[subst {{$srcId $requestId}}] $state
    }
}


# return boolean indicating whether the request was deleted or not
FCAFloorDynamics instproc delete_request { srcId requestId } {
    $self instvar admittedRequests_ pendingRequests_ mgr_
    if { [$admittedRequests_ exists $srcId $requestId] } {
	$self remove_admitted_request $srcId $requestId
    } elseif { [$pendingRequests_ exists $srcId $requestId] } {
	$self delete_pending_request $srcId $requestId
    } else {
	DbgOut "Huh! We are trying to cancel a request that is neither in the"
	DbgOut "admitted list nor in the pending list"
	return 0
    }

    if { [[$mgr_ localRcvr] srcId] == $srcId } {
	# I am cancelling my own request; I should buffer this event until the
	# moderator gets it
	$canceledRequests_ add $srcId $requestId
    }
    return 1
}
