# alloc.tcl --
#
#       Defines AddressAllocator, which basically tries to classify a
#       multicast address within a certain range.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


# Abstract base class for classes that do multicast address allocation.
# Different subclasses can be defined (e.g., AddressAllocator/MDHCP,
# /AAP, etc..) and then other objects that need to allocate an address
# can simply use [AddressAllocator instance] and not worry about the
# particular algorithm being used.
Class AddressAllocator

#
AddressAllocator public init {} {
    AddressAllocator set instance_ $self
}

#
AddressAllocator proc instance {} {
    return [AddressAllocator set instance_]
}

# The entry point for request an address.  Right now, allocates an
# address from the ScopeZone <i>zone</i>.  If <i>n</i> is specified,
# then that many consecutive addresses should be allocated.
# This API needs to be expanded to include address lifetime, etc..
AddressAllocator public alloc {zone {n 1}} {
    $self fatal "AddressAllocator::alloc called -- should be overridden in child"
}

# Implements the AddressAllocator api to allocate a unique addresse
# that does not clash with another addresse allocated in a SAP
# announcement.
Class AddressAllocator/SAP -superclass AddressAllocator

# Instantiate a new SAP address allocator.  The parameter <i>source</i>
# is a ProgramSource/SAP object that will be used to extract a list of
# SAP announcements to check for address conflicts.
AddressAllocator/SAP public init {source} {
    $self next
    $self set source_ $source
}

#
AddressAllocator/SAP public alloc {zone {n 1}} {
    $self instvar source_
    set conflict 1
    while {$conflict != 0} {
	set conflict 0
	set base [$self random-addr $zone]
	set addr $base
	for {set i 0} {$i<$n} {incr i} {
	    incr conflict [$self conflict $addr $zone $source_]
	    set addr [$self inet_ntoa [expr [$self inet_addr $addr] + 1]]
	}
    }
    return $addr
}

# Generates a random address from the range of addresses in the
# ScopeZone <i>zone</i>.
AddressAllocator/SAP private random-addr {zone} {
    set range [$zone range]
    set l [split $range /]
    set bits [lindex $l 1]
    set mask [expr ~((-1)<<(32 - $bits))]
    set base [expr [$self inet_addr [lindex $l 0]] &~ $mask]
    set addr [expr $base + ([random] & $mask)]
    return [$self inet_ntoa $addr]
}

# Determines whether the address <i>addr</i> clashes with any
# addresses already allocated in programs visible to the
# ProgramSource/SAP object <i>source</i>.
AddressAllocator/SAP private conflict {addr zone source} {
    set conflict 0

    #FIXME should have a way to quickly examine only programs
    #    in this scope zone
    $source instvar progs_
    foreach p [array names progs_] {
	foreach msg [$progs_($p) set msgs_] {
	    if {[$msg have_field c] && $addr == [$msg set caddr_]} {
		set conflict 1
		break
	    }
	    foreach media [$msg set allmedia_] {
		if {[$media have_field c] && $addr == [$media set caddr_]} {
		    set conflict 1
		    break
		}
	    }
	}
    }
    return $conflict
}

# Same as the C library <tt>inet_addr</tt> routine.  Converts
# the IP address <i>a</i> written in dotted decimal notation
# (e.g., 224.2.127.254) to a 32 bit integer.
AddressAllocator/SAP private inet_addr {a} {
    set l [split $a .]
    set addr [expr [lindex $l 0] <<24]
    incr addr [expr [lindex $l 1] <<16]
    incr addr [expr [lindex $l 2] <<8]
    incr addr [lindex $l 3]
    return $addr
}

# Same as the C library <tt>inet_ntoa</tt> routine.  Converts the
# 32 bit integer <i>n</i> to dotted decimal notation.
AddressAllocator/SAP private inet_ntoa {n} {
    set a [expr ($n>>24) & 0xff]
    set b [expr ($n>>16) & 0xff]
    set c [expr ($n>>8) & 0xff]
    set d [expr $n & 0xff]
    return "$a.$b.$c.$d"
}
