# ui-nsdr.tcl --
#
#       Defines the primary nsdr UI
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import notebook NewList SDPEditWindow ProgramWindow HelpWindow

# Implements the main user interface for the Nsdr session
# directory tool.
Class NsdrUI

# Builds the main user interface in the window <i>w</i>
NsdrUI public init {w app} {
    $self set app_ $app
    $self set w_ $w

    $self instvar notebook_
    set notebook_ [notebook $w.n -tabheight 27 -tabpadx 2 -bd 2 ]
    $self set sources_ {}

    set helpWindow [new NsdrHelpWindow .help]

    set b "$w.bar"
    frame $b -relief ridge -bd 2
    label $b.title -text "nsdr v[version]" -relief flat -justify left
    button $b.new -relief raised -highlightthickness 1 \
	-text "New" -command "$self new-program"
    button $b.help -relief raised -highlightthickness 1 \
	-text "Help" -command "$helpWindow toggle"
    button $b.quit -relief raised -highlightthickness 1 \
	-text "Quit" -command "$app exit"
    pack $b.title -side left -fill both -expand yes
    pack $b.new $b.help $b.quit -side left -padx 1 -pady 1

    pack $b -side bottom -fill x -expand no
    pack $notebook_ -fill both -expand yes -padx 2 -pady 2
}

# Creates a new tab for this source.  Returns an integer that can
# be used in subsequent calls to the UI to add/remove programs.
NsdrUI public addsource s {
    $self instvar sources_
    set n [llength $sources_]
    lappend sources_ $s
    $self set orders_($n) {}

    $self instvar notebook_
    $notebook_ addTab [$s name]
    set f [frame $notebook_.f$n]
    $notebook_ addFrame $f $n -expand yes -fill both -pady 2

    $self set lists_($n) [new NewList $f]

    if {$n == 0} {
	$notebook_ tabPress 0
    }
}

#
NsdrUI private lookup {src} {
    $self instvar sources_ progs_

    set n [lsearch -exact $sources_ $src]
    if {$n == -1} {
	$self fatal "NsdrUI inconsistency in sources_"
    }
    return $n
}

# Adds the Program <i>prog</i> to the user interface in the frame
# corresponding to the ProgramSource <i>src</i>.
NsdrUI public addprog {src prog} {
    $self instvar sources_ progs_ orders_ lists_

    set n [$self lookup $src]

    # create the new Program object and update local state
    set o [$prog unique_key]
    set p [new ProgramWindow $prog]
    set progs_($n:$o) $p

    # update the user interface
    set title [string tolower [$p title]]
    set order $orders_($n)
    set i 0
    set pr [lindex $order 0]
    while {$pr != ""} {
	set title2 [string tolower [$pr title]]
	if {[string compare $title $title2] <= 0} break
	incr i
	set pr [lindex $order $i]
    }
    if {$pr == ""} {set i "end" }
    set orders_($n) [linsert $order $i $p]
    $self instvar lists_
    $lists_($n) insert $i [$p title] "$p toggle-window"
}

# Removes the Program object <i>prog</i> from the user interface
# frame corresponding to the ProgramSource <i>src</i>.
NsdrUI public removeprog {src prog} {
    $self instvar sources_ progs_ orders_ lists_

    set n [$self lookup $src]

    # find the corresponding program
    set o [$prog unique_key]
    if ![info exists progs_($n:$o)] {
	$self fatal "NsdrUI::removeprog: inconsistency in progs_!"
    }

    set p $progs_($n:$o)

    set i [lsearch -exact $orders_($n) $p]
    if {$i < 0} {
	$self fatal "NsdrUI::removeprog: insconsistency in orders_!"
    }

    set orders_($n) [lreplace $orders_($n) $i $i]
    $lists_($n) delete $i

    delete $p
}

# describe
NsdrUI public updateprog {src prog} {
    $self instvar sources_ progs_

    set n [$self lookup $src]
    set o [$prog unique_key]
    if ![info exists progs_($n:$o)] {
	$self fatal "NsdrUI inconsistency in progs_"
    }

    $progs_($n:$o) updateprog $prog
}

# describe
NsdrUI public current-source {} {
    $self instvar notebook_ sources_
    set i [$notebook_ index]
    return [lindex $sources_ $i]
}

# describe
NsdrUI private new-program {} {
    new SDPEditWindow "" [$self current-source]
}

#
# A toggle-able dismissable toplevel window for displaying a
# bulleted list of tips to help the user.
#
Class NsdrHelpWindow -superclass HelpWindow

#
# Instantiate, but do not yet display or iconify, a dismissable toplevel
# window using the provided widgetpath, <i>w</i>.  Within <i>w</i>,
# create a bulleted list of tips to help the vic user.  (This method is
# called when the user toggles this window for the first time.)
#
NsdrHelpWindow instproc build w {

	$self create-window $w "nsdr: Help" {

"Click on a program name to display detailed information \
and launch media tools."

"Use ``New'' button to create new session announcements."

"If the user interface looks peculiar, you might \
have X resources that conflict with tk.  A common problem is \
defining ``*background'' and/or ``*foreground''."

"Bugs and suggestions to openmash-users@openmash.org.  Thanks."
	}
    }
