# ui-timeedit.tcl --
#
#       Defines a widget that allows for editing a time and date
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Combobox

# Constructs and manages a window that is used to edit the times
# at which an SDP session description is active.
Class SDPTimeEditWindow

SDPTimeEditWindow set times_(minute) 60
SDPTimeEditWindow set times_(hour) 3600
SDPTimeEditWindow set times_(day) 86400
SDPTimeEditWindow set times_(week) 604800

SDPTimeEditWindow set numbers_(one) 1
SDPTimeEditWindow set numbers_(two) 2
SDPTimeEditWindow set numbers_(three) 3
SDPTimeEditWindow set numbers_(four) 4
SDPTimeEditWindow set numbers_(five) 5
SDPTimeEditWindow set numbers_(six) 6
SDPTimeEditWindow set numbers_(seven) 7
SDPTimeEditWindow set numbers_(eight) 8
SDPTimeEditWindow set numbers_(nine) 9
SDPTimeEditWindow set numbers_(ten) 10
SDPTimeEditWindow set numbers_(eleven) 11
SDPTimeEditWindow set numbers_(twelve) 12

# should redo as:
#
# +------------+------------+
# |            |            |
# |  calendar  |   repeat   |
# |            |            |
# +------------+------------+
#

# Creates the user interface elements needed for this window.  As
# with SDPEditWindow and SDPMediaEditWindow, the times described in
# the SDPMessage object <i>msg</i> will be displayed initially or
# defaults will be displayed if <i>msg</i> is an empty string.
SDPTimeEditWindow public init {w msg} {
    $self set win_ $w
    $self set msg_ $msg

    if [$self yesno simpleInterface] {
	$self build-simple
    } else {
	$self build-advanced
    }
}

SDPTimeEditWindow private build-simple {} {
    $self instvar win_
    catch {
	foreach w [winfo children $win_] { destroy $w }
    }
    pack [label $win_.l -text "NEED TO IMPLEMENT SIMPLE INTERFACE"]
}

SDPTimeEditWindow private build-advanced {} {
    $self instvar win_ msg_ src_

    catch {
	foreach w [winfo children $win_] { destroy $w }
    }

    $self instvar startentry_ durationbox_ repeatsbox_

    # start time
    label $win_.startl -text "Start Time:" -anchor e
    grid $win_.startl -row 0 -column 0 -sticky ew
    if {$msg_ == ""} {
	# round to nearst half hour
	set now [clock format [expr ([clock seconds]+1)/1800 * 1800] \
		     -format "%I:%M %p %b %d"]
    } else {
	if [$msg_ have_field r] {
            set l [$msg_ field_value r]
	}
	set now "FIXME"
    }
    set startentry_ [entry $win_.starte]
    $startentry_ insert 0 $now
    grid $startentry_ -row 0 -column 1 -sticky ew
#    button $win_.startb -text "Edit..." -command "$self edit-start"
#    grid $win_.startb -row 0 -column 2 -sticky ew

    # duration
    label $win_.durl -text "Duration:" -anchor e
    grid $win_.durl -row 1 -column 0 -sticky ew
    if {$msg_ == ""} {
	set duration "2 hours"
    } else {
	set duration "FIXME"
    }
    #FIXME
    set durationbox_ [new Combobox $win_.durc $duration "$self valid-duration" "1 hour" "1 day" "1 week"]
    grid $win_.durc -row 1 -column 1 -sticky ew

    # repeats
    label $win_.repl -text "Repeat:" -anchor e
    grid $win_.repl -row 2 -column 0 -sticky ew
    #FIXME get current value from msg
    menubutton $win_.repm -text "No" -anchor w -menu $win_.repm.m \
	-indicatoron yes -relief raised -bd 2
    grid $win_.repm -row 2 -column 1 -sticky ew
    menu $win_.repm.m
    foreach r {"No" "Daily" "Weekly" "Every Two Weeks"} {
	$win_.repm.m add command -label $r \
	    -command "$self set-repeat \"$r\""
    }

    label $win_.repsl -text "for:" -anchor e
    grid $win_.repsl -row 3 -column 0 -sticky ew
    #FIXME
    set repeatsbox_ [new Combobox $win_.repsc "1 Week" "$self valid-duration" \
			 "1 Week" "2 Weeks" "1 Month" "2 Months"]
    $repeatsbox_ disable
    grid $win_.repsc -row 3 -column 1 -sticky ew
}

#
SDPTimeEditWindow private duration2secs {s} {
    $class instvar times_ numbers_
    set l [split $s]
    if {[llength $l] != 2} {
	return -1
    }

    # check if first component is a number
    set n [lindex $l 0]
    if {![regexp {[0-9]+} $n]} {
	# not a literal number, check if it is a spelled out
	# number (e.g., "one")
	set n [string tolower $n]
	if ![info exists numbers_($n)] {
	    return -1
	}
	set n $numbers_($n)
    }

    # check if second component is duration (e.g., "days", "months", etc.)
    set u [string tolower [lindex $l 1]]
    set i [string last "s" $u]
    if {$i != -1} {
	set u [string range $u 0 [expr $i-1]]
    }
    if ![info exists times_($u)] {
	return -1
    }

    set n [expr $n * $times_($u)]
    return $n
}

# Used as a callback for TextEntry widgets.  Determines if the
# string <i>t</i> is a valid duration (i.e., that is consists of
# a number and a unit of time) so that invalid durations cannot be
# entered.
SDPTimeEditWindow private valid-duration {t} {
    set s [$self duration2secs $t]
    if {$s < 0} {
	return 1
    }
    return 0

}

# Called when a selection is made in the repeat menu.  Enables or
# disables the repeat duration menu as necessary.
SDPTimeEditWindow private set-repeat {r} {
    $self instvar win_ repeatsbox_
    $win_.repm configure -text $r
    if {$r == "No"} {
	$repeatsbox_ disable
    } else {
	$repeatsbox_ enable
    }

    #FIXME set menu items on repeat for how long menu appropriately
}

#
SDPTimeEditWindow public buildmsg {} {
    $self instvar startentry_ durationbox_ win_ repeatsbox_

    set start [unix_to_ntp [clock scan [$startentry_ get]]]
    set duration [$self duration2secs [$durationbox_ get]]
    set end [format %u [expr $start + $duration]]
    set text "t=$start $end\n"

    # FIXME handle repeat

    return $text
}
