# desc-services.tcl --
#
#       Wrappers around RecorderClient and DcClient objects for safe use in
#       Broadcast Description
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import RecorderClient
import MessageBox

# The idea of this class is to allow robustness so that if the recorder app
#    is not up, the Broadcast Description will still work and the record and
#    stop commands will be NOPs
#
#

Class BroadcastDescription_Recorder

BroadcastDescription_Recorder instproc init {hostname port} {
    $self instvar rc_ ok_

    if {[catch {set rc_ [new RecorderClient $hostname $port]} result]} {
	set ok_ 0
	# FIXME - should have a "more info" button that allows them to get
	#   additional info about why it failed
	#
	# FIXME - we don't want this dialog box to prevent the rest of the app
	#   from executing, which it currently does
	set msg "Warning: cannot connect to recorder app at $hostname/$port"
	tk_messageBox -message $msg -type ok -title "BroadcastDescription_Recorder: Warning" -icon "warning"
    } else {
	# success
	set ok_ 1
    }
}

BroadcastDescription_Recorder instproc record {} {
    $self instvar rc_ ok_

    if {$ok_} {
	if {[catch {$rc_ record} result]} {
	    puts stderr "Warning: BroadcastDescription_Recorder::record: failed: $result"
	}
	return 1
    }
    return 0
}

BroadcastDescription_Recorder instproc stop {} {
    $self instvar rc_ ok_

    if {$ok_} {
	if {[catch {$rc_ stop} result]} {
	    puts stderr "Warning: BroadcastDescription_Recorder::stop: failed: $result"
	}
	return 1
    }
    return 0
}

################################################################

Class BroadcastDescription_VD

BroadcastDescription_VD instproc init {handle} {
    $self instvar handle_

    set handle_ $handle
}

BroadcastDescription_VD instproc enableQM {} {
    $self instvar handle_

    if {$handle_ != ""} {
	if {[catch {$handle_ enableQM} result]} {
	    puts stderr "Warning: BroadcastDescription_VD::enableQM: failed: $result"
	}
    }
}

BroadcastDescription_VD instproc disableQM {} {
    $self instvar handle_

    if {$handle_ != ""} {
	if {[catch {$handle_ disableQM} result]} {
	    puts stderr "Warning: BroadcastDescription_VD::disableQM: failed: $result"
	}
    }
}

################################################################

# this abstraction just tells you what current source is being displayed
#   on the overhead projector using the "source" command

Class BroadcastDescription_Projector

BroadcastDescription_Projector instproc init {} {
    $self instvar source_

    # FIXME - we can't use this b/c we don't know if the projector is using
    #   the RGB or video input
    #
    # right now we assume that it is an RGB source to start with; this is
    #   only a problem at init time since the matrixSwitch events will
    #   specify the current input
    set source_ [$BroadcastDescription::405Control matrix_getInputSource "projectorRGB"]

    set callback "$self switchCallback"
    set filter [$self getFilter "projector"]
    $BroadcastDescription::405Control callback_register $callback $filter
    $BroadcastDescription::405Control callback_enable $callback
}

BroadcastDescription_Projector public destroy {} {
    set callback "$self switchCallback"
    $BroadcastDescription::405Control callback_unregister $callback
}

# there are two ways the projector source can change, matrixSwitch events
#   or the speaker pushing a button on the panel
BroadcastDescription_Projector public switchCallback {amxMsg} {
    $self instvar source_

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
    } else {
	return ""
    }

    if {$type == "matrixSwitch"} {
	set input [lindex $eventData 0]
	set output [lindex $eventData 1]
	if {$output == "projector"} {
	    if {$input != $source_} {
		# the stream was switched
		set source_ $input
	    }
	}
    }

    if {$type == "amxResponse"} {
	set cmd [lindex $eventData 0]
	set dev [lindex $eventData 1]
	set chan_str [lindex $eventData 2]
	if {$dev == "remotePanel" || $dev == "touchPanel"} {
	    if {$cmd == "push"} {
		# it is a user pushing a button on the touch panel
		#		puts stdout "matrix command"
		switch -exact -- $chan_str {
		    pcToProj {
			set source_ "frontPC"
		    }
		    vcrToProj {
			set source_ "frontVCR"
		    }
		    elmoToProj {
			set source_ "elmo"
		    }
		    laptopToProj {
			set source_ "laptop"
		    }
		    sgiToProj {
			set source_ "sgiPC"
		    }
		}
	    }
	}
    }

    return ""
}

BroadcastDescription_Projector public source {} {
    $self instvar source_

    return $source_
}

BroadcastDescription_Projector instproc getFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput or amxResponse events that have are pushes
    #   of the remote or touch panels
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter set output \[lindex \$data 1\]\n"
    set filter "$filter if \{\$output == \"$watchOutput\"\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    set filter "$filter if \{\$type == \"amxResponse\"\} \{\n"
    set filter "$filter set cmd \[lindex \$data 0\]\n"
    set filter "$filter set dev \[lindex \$data 1\]\n"
    set filter "$filter set chan \[lindex \$data 2\]\n"
    set filter "$filter if \{\$dev == \"remotePanel\" || \$dev == \"touchPanel\"\} \{\n"
    set filter "$filter if \{\$cmd == \"push\"\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"

    return $filter
}