# statemachine.tcl --
#
#       The state machine part of the QuestionMonitor.  It takes in mic levels
#       and decides which view to show.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

# use the Tcl namespace facility for global variables to prevent conflicts
namespace eval QMStateMachine {
    variable enabled
    variable debugOn
}

Class QMStateMachine

QMStateMachine instproc init {inputs gaugeList timeSlice callback stateFileName getIndexCallback {spoofAfter 0} {myAfter ""} {callback2 ""} {onlevel 10}} {
    $self instvar timeSlice_           # amount of time to wait between polling
                                       #    of microphone active levels
    $self instvar inputs_              # list of indexes into gaugeTransducer_
    $self instvar gaugeTransducer_     # array of gauge transducers to get
                                       #    activeLevels from
    $self instvar defaultMode_         # speaker is default
    $self instvar state_               # type of state we're in: recent or
                                       #    discussion
    $self instvar lastAfter_           # id of last call to tcl "after" procedure
    $self instvar recentDuration_      # how long to stay in the recent state
    $self instvar onLevel_             # level of activeSum for a mic input to
                                       #    cause a call to handleInput and
                                       #    possibly change state
    $self instvar discussionTimeout_   # amount of time to stay in a discussion
                                       #    state if all mic inputs are below
                                       #    $onLevel_
    $self instvar callback_            # procedure to be called when a new mode
                                       #    is selected
    $self instvar getIndexCallback_
    $self instvar stateFile_

    # to facilitate simulation
    $self instvar spoofAfter_
    $self instvar myAfter_
    $self instvar callback2_

    set QMStateMachine::debugOn 1

    # record into instance variables
    set callback_ $callback
    set callback2_ $callback2
    set timeSlice_ $timeSlice
    set inputs_ $inputs
    array set gaugeTransducer_ $gaugeList
    set spoofAfter_ $spoofAfter
    set getIndexCallback_ $getIndexCallback

    if {$QMStateMachine::debugOn} {
	set stateFile_ [open $stateFileName "w"]
    }

    # make sure lastAfter_ is defined, this is used by state machine
    set lastAfter_ ""
    set defaultMode_ "speaker"
    set recentDuration_ 3000
    # FIXME - what is a good value for this?  2000, 5000, 7000 seem ok
    #   longer values allow people asking a long question to keep the camera
    #   longer, but in periods of silence after a question, the camera will be
    #   delayed from going back to the speaker
    set discussionTimeout_ 7000
    set onLevel_ $onlevel
    set QMStateMachine::enabled 0

    # to facilitate simulation
    set myAfter_ $myAfter
#    puts stdout "myAfter_ is $myAfter_"

    $self changeState "discussion" $defaultMode_
    # start the timeout
    $self doAfter $timeSlice_ "$self checkLevels"
}

QMStateMachine instproc destroy {} {
    $self instvar stateFile_

    catch {close $stateFile_}
}

QMStateMachine public doAfter {delay callback} {
    $self instvar spoofAfter_ myAfter_

    if {$delay == "cancel"} {
	if {!$spoofAfter_} {
	    return [after cancel $callback]
	} else {
	    return [$myAfter_ cancelProc $callback]
	}
    }
    # it's a regular after call
    if {!$spoofAfter_} {
	return [after $delay $callback]
    } else {
	return [$myAfter_ afterProc $delay $callback]
    }
}

QMStateMachine public enable {} {
    set QMStateMachine::enabled 1
}

QMStateMachine public disable {} {
    $self instvar lastAfter_

    set QMStateMachine::enabled 0
    $self doAfter cancel $lastAfter_
}

#
# this is called every $timeSlice
#
# it figures out the dominant audio input and calls handleInput
#
QMStateMachine instproc checkLevels {} {
    $self instvar timeSlice_ inputs_ gaugeTransducer_ onLevel_ callback2_
    $self instvar curMode_ defaultMode_

    # reschedule here so that there will be no drift
    $self doAfter $timeSlice_ "$self checkLevels"

    if {!$QMStateMachine::enabled} {
	#  do nothing
	return ""
    }

    set max 0
    set maxMic speaker


    foreach input $inputs_ {
	set actives($input) [$gaugeTransducer_($input) getActiveSum]
#	puts stdout "$input == $actives($input)"
	set levels($input) [$gaugeTransducer_($input) getLevelSum]
    }

    # FIXME - should we amplify the current modes input if it's an
    #   audience question to make staying on the audience member easier
    #   until they really stop speaking?
    # How much should we boost by?
    #
    # amplify the current mode's input if we're in audience mode
    if {$curMode_ != "$defaultMode_"} {
	set boost 5
    	set actives($curMode_) [expr $actives($curMode_) + $boost]
    }

    foreach index [array names actives] {
#	puts "actives($index) = $actives($index)"
#	puts "levels($index) = $levels($index)"
	if {$actives($index) > $onLevel_} {
#	    puts "levels($index) = $levels($index)"
	}
	if {$actives($index) > $max} {
	    set max $actives($index)
	    set maxMic $index
	}
    }

    if {$callback2_ != ""} {
	eval $callback2_
    }

    if {$maxMic != "speaker"} {
	# make sure that the audience mic is at least OFFSET_THRESHOLD
	#   above the speaker level
	set spkrLevel $actives(speaker)
	set audLevel $actives($maxMic)
	set diff [expr $audLevel - $spkrLevel]
	# FIXME - shouldn't set it every time
	set OFFSET_THRESHOLD 8
	if {$diff < $OFFSET_THRESHOLD} {
	    set maxMic "speaker"
	}
    }

    if {$actives($maxMic) > $onLevel_} {
	$self handleInput $maxMic
    }
}

# the "mode" input is the dominant audio input
#
# the default mode should be speaker
QMStateMachine instproc handleInput {mode} {
    $self instvar state_ curMode_ lastAfter_ defaultMode_ discussionTimeout_

    switch -exact -- $state_ {
	discussion {
	    if {$curMode_ != $mode} {
		$self changeState "recent" $mode
	    } else {
		# reset timeout if not in the default mode
		#
		# if you are in the default mode, there is no need for a
		#    timeout
		if {$curMode_ != $defaultMode_} {
		    $self doAfter cancel $lastAfter_
		    set lastAfter_ [$self doAfter $discussionTimeout_ "$self changeState recent $defaultMode_"]
		}
	    }
	}
	default {
	    # recent is included here
	    # do nothing
	}
    }
}

QMStateMachine instproc changeState {newState newMode} {
    $self instvar state_ curMode_ lastAfter_ defaultMode_
    $self instvar discussionTimeout_ recentDuration_ callback_

    if {$QMStateMachine::debugOn} {
	$self recordStateChange $newState $newMode
    }

    set state_ $newState
    set curMode_ $newMode
    $self doAfter cancel $lastAfter_

#    puts stdout "entering $newState state, $newMode mode"

    switch -exact -- $newState {
	recent {
	    set cmd "$callback_ $newMode"
	    eval $cmd
	    $self doAfter $recentDuration_ "$self changeState discussion $curMode_"
	}
	discussion {
	    if {$curMode_ != $defaultMode_} {
		set lastAfter_ [$self doAfter $discussionTimeout_ "$self changeState recent $defaultMode_"]
	    }
	}
	default {
	    puts stderr "QMStateMachine::changeState: error, unknown state $newState"
	}
    }
}

QMStateMachine instproc recordStateChange {newState newMode} {
    $self instvar getIndexCallback_ stateFile_

    set index [eval $getIndexCallback_]
    puts $stateFile_ "$index $newState $newMode"
}

