# ui-switcher.tcl --
#
#       This module handles the video switcher abstraction, a virtual base
#       class for switching among media flows (sources).
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/ui-switcher.tcl,v 1.15 2002/02/03 04:39:55 lim Exp $


import Configuration

#
# This module handles the video switcher abstraction, a virtual base
# class for switching among media flows (sources).  Generic objects are
# registered as switchable with a method to call back when switching
# occurs.  By using a generic callback mechanism, this code is shared
# between the on screen windows and external video outputs.
# <p>
#
# Subclasses must fill in the following methods: <br>
# <dd>	switch
# <dd>	prev_active_src
# <dd>	next_active_src
#
Class Switcher -configuration {
	switchInterval 5
}

#
# Create a Switcher for the source, <i>src</i>.
#
Switcher instproc init src {
	$self next
	$self instvar src_ switch_list_
	set src_ $src
	Switcher set all_($self) 1
}

#
Switcher instproc destroy {} {
	$self cancel_timer
	Switcher unset all_($self)
}

#
# Enable switching.
#
Switcher instproc enable {} {
	$self touch
}

#
# Returns true if this Switcher is switching.  For example, a UserWindow
# is "enabled" if it is timer-switched or voice-switched.
#
Switcher instproc enabled {} {
	$self instvar ts_
	return [info exists ts_]
}

#
# Disable switching.
#
Switcher instproc disable {} {
	$self instvar ts_
 	unset ts_
}

#
# Start the timer-base cycling of sources.  Continue to switch to the
# next active source every <i>switchInterval</i> seconds, (an option in
# the resource database), until we <i>cancel_timer</i>.
#
Switcher public set_timer {} {
	$self sched
}

#
# Cancel the timer-base cycling of sources.
#
Switcher instproc cancel_timer {} {
	$self instvar timer_id_
	if [info exists timer_id_] {
		after cancel $timer_id_
		unset timer_id_
	}
}

#
# If the source, <i>src</i>, is selected in the switchable list, switch to it.
#
Switcher instproc switch_to src {
	$self instvar src_ switch_list_
	if { $src != $src_  && [lsearch $switch_list_ $src] != -1 } {
		$self switch $src
		set src_ $src
	}
}

#
# Switch to the next active source.
#
Switcher instproc forward {} {
	$self instvar src_
	$self switch_to [$self next_active_src $src_]
}

#
# Switch to the previous active source.
#
Switcher instproc reverse {} {
	$self instvar src_
	$self switch_to [$self prev_active_src $src_]
}

Switcher set clock_ 1

#
# No longer used... <br>
# Increment the clock_ (essentially an ordering stamp) for use by the LRU algorithm.
#
Switcher instproc touch {} {
	Switcher instvar clock_
	$self instvar ts_
	set ts_ $clock_
	incr clock_
}

Switcher proc focus src {
	Switcher instvar ignore_
	if [info exists ignore_($src)] {
		return
	}

	Switcher instvar all_ speaker_

	#
	# Remember the current speaker
	#
	set speaker_ $src

	#
	# This new version is not doing the LRU mechanism.
	# Instead, find all switcher windows that are enabled and
	# contains src in its switchable list, then switch all of them
	#
	foreach o [array names all_] {
		if { [$o enabled] &&
	             [lsearch [$o set switch_list_] $src] != -1 } {
			 $o switch_to $src
		}
	}
}

#
# After <i>switchInterval</i> seconds, (an option in the resource database), expire the timer.
#
Switcher private sched {} {
	$self instvar timer_id_
	set ms [expr 1000 * [$self get_option switchInterval]]
	set timer_id_ [after $ms "$self timeout"]
}

#
# When the timer expires, switch to the next active source and reset the timer.
#
Switcher private timeout {} {
	$self instvar timer_id_
	if [info exists timer_id_] {
		$self forward
		$self sched
	}
}

#
# Rebuild the switchable menu for all existing switcher
# windows.
#
Switcher proc rebuild_switch_list_menu { } {
        Switcher instvar all_
        foreach o [array names all_] {
	        $o rebuild_switch_list_menu
	}
}

