/*
 * transducer-log.cc --
 *
 *      Writes a binary file with the raw microphone input values.
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "transducer-log.h"
#include "tclcl.h"
#include "qm-files.h"

static class LogTransducerClass : public TclClass {
public:
	LogTransducerClass() : TclClass("Transducer/Log") {}
	TclObject* create(int argc, const char*const* argv) {
		return(new LogTransducer(argc, argv));
	}
} log_transducer_class;

LogTransducer::LogTransducer(int argc, const char* const* argv)
{
  struct MicInputHeader hdr;
  // args passed in tcl start at argv[4]

  enabled = 0;
  index = 0;

  // get the filename from argument 5
  logfile = NULL;
  if(argc > 4)
  {
    sprintf(filename, "%s", argv[4]);
    logfile = fopen(argv[4], "w+");
    if(logfile == NULL)
    {
      fprintf(stderr, "LogTransducer::LogTransducer: file %s could not be opened for writing\n", argv[4]);
    }
    else
    {
      fwrite(&hdr, sizeof(struct MicInputHeader), 1, logfile);
    }
  }
  else
  {
    fprintf(stderr, "LogTransducer::LogTransducer: no filename specified; will not log\n");
  }
}

LogTransducer::~LogTransducer()
{
  struct MicInputHeader hdr;

  if(logfile != NULL)
  {
    hdr.numSamples = index;
    fseek(logfile, 0, SEEK_SET);
    fwrite(&hdr, sizeof(struct MicInputHeader), 1, logfile);
    fflush(logfile);
    fclose(logfile);
  }
}

// by experimentation, this gets called on the average every 20 ms =>
//    50 samples/sec
//
// by experimentation, it also appears that this "double" is always of the
//    form 60.00000, 32.0000, etc, i.e., converting it to an int loses no
//    precision
//
// for logging, 50 samples/second => 3000 samples/minute
//    3000 samples/min * 16 bytes/sample = 48000 bytes/min
//    48000 bytes/min = 2,880,000 bytes/hour
void LogTransducer::set(double d)
{
  //printf("LogTransducer::set: %lf => %d\n", d, (int)d);
  int result;
  struct MicInputInfo info;

  if(logfile != NULL)
  {
    if(enabled)
    {
      gettimeofday(&(info.sysTime), NULL);
      info.val = (int)d;
      info.index = index;
      result = fwrite(&info, sizeof(struct MicInputInfo), 1, logfile);
//      printf("%d %d\n", index, val);
      index++;
      if(result <= 0)
      {
	fprintf(stderr, "LogTransducer::set: unable to log to file %s\n", filename);
      }
    }
  }
}

int LogTransducer::getIndex()
{
  return(index);
}

int LogTransducer::command(int argc, const char* const* argv)
{
  const char* cmd = argv[1];
  Tcl& tcl = Tcl::instance();

  if(argc == 2)
  {
    if(!strcmp(cmd, "enableLogging"))
    {
      enabled = 1;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "disableLogging"))
    {
      enabled = 0;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "getIndex"))
    {
      tcl.resultf("%d", index);
      return(TCL_OK);
    }
  }

  // default to TclObject
  return(TclObject::command(argc, argv));
}

