/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: source-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/source-srmv2.cc,v 1.20 2001/08/02 22:26:33 lim Exp $
 * 
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "srmv2.h"
#include "source-srmv2.h"
#include "session-srmv2.h"
#include "ns-srmv2.h"


static int lsrc_cnt_=0;


SRMv2_Source::SRMv2_Source(const char *srcid, int islocal)
	: sid_(NULL), info_(NULL), info_len_(0), namespace_(NULL),
	  session_(NULL), ipaddr_(0), ch_(NULL)
{
	char default_srcid[100];

	if (!srcid) {
		sprintf(default_srcid, "%s:%ld:%d", intoa(LookupLocalAddr()),
			(long) getpid(), lsrc_cnt_++);
		lsrc_cnt_++;
		srcid = default_srcid;
	}
	
	sid_ = new int [4];
	ipaddr_ = 0;
	md5_digest(srcid, (unsigned char *) sid_);

	srm_trace(srmSRM, ("source id %s: %08x %08x %08x %08x\n", srcid,
			   sid_[0], sid_[1], sid_[2], sid_[3]));
	namespace_ = new SRMv2_NameSpace;

	if (islocal) {
		ch_ = new SRMv2_LocalSrcCtrlHandler(this);
		ch_->start_timer();
	}
}


SRMv2_Source::SRMv2_Source(const int *srcid, int islocal)
	: sid_(NULL), info_(NULL), info_len_(0), namespace_(NULL),
	  session_(NULL), ipaddr_(0), ch_(NULL)
{
	sid_ = new int [4];
	for (int i = 0; i < 4; i++)
		sid_[i] = srcid[i];
	namespace_ = new SRMv2_NameSpace();

	if (islocal) {
		ch_ = new SRMv2_LocalSrcCtrlHandler(this);
		ch_->start_timer();
	}
}


void
SRMv2_Source::session(SRMv2_Session* s)
{
	session_ = s;
	if (ch_) ch_->manager(s);
}


unsigned int
SRMv2_Source::calloc(unsigned int parent, unsigned int offset,
		     const unsigned char *container_name, int name_len)
{
	int created = 0;
	unsigned int cid = 0;
	naddr *path;
	unsigned int count;

	/* Need to advertise the newly created container. */
	cid  = namespace_->calloc(parent, offset, &created);
	if (created) {
		NS_Node *node = namespace_->getnode(cid);
		if (node) node->name(container_name, name_len);
		namespace_->ancestors(cid, &path, &count);
#ifndef NDEBUG
		srm_trace(srmSRM, ("\tTransmitting namespace update"));
		namespace_->display(1);
		srm_trace(srmSRM|srmVerbose,
			  ("%s <--> %d\n", container_name, cid));
#endif
		session_->send_announcement(this, 0, path, count);
	}
	return cid;
}

int
SRMv2_Source::update_appinfo(const unsigned char *dp)
{
	int newlen = ntohs(*((u_int16_t*)dp));
	const unsigned char *newinfo = (dp + sizeof(u_int16_t));

	if ( !(newlen==0xFFFF || (newlen==0 && info_len_==0) ||
	       (newlen==info_len_ && memcmp(newinfo, info_, info_len_)==0)) ) {
		/* the new app info is different from the old one
		 * notify the application that we received a new app_info
		 */
		set_app_info(newinfo, newlen);
		srm_source_update_proc proc;
		proc = session_->callbacks()->source_update_proc;
		if (proc) {
			(*proc)((srm_source_t)this, info_, info_len_);
		}
	}

	if (newlen==0xFFFF) newlen = 0;
	if ((newlen+2)%4) newlen += (2 - newlen%4);
	return sizeof(u_int16_t) + newlen;
}


void
SRMv2_Source::update_namespace(naddr_plus_name *path, int plen)
{
	unsigned int pid, cid, offset;
	SRMv2_Signature sign;
	int created = 0, name_len;
	unsigned char *name;
	NS_Node *node;
	int olen = plen;
	naddr_plus_name *opath = path;
	int count = 0;

	while (((unsigned int)plen) >= sizeof(naddr_plus_name)) {
		count ++;
		offset = ntohl(path->addr.offset);
		cid = ntohl(path->addr.cid);
		pid = ntohl(path->addr.pcid);
		namespace_->calloc(pid, offset, &created, cid);

		/* extract the container name */
		name_len = ntohl(path->name_len);
		name = path->name;
		srm_trace(srmSRM, ("{%d.%d}", cid, offset));

		if (created) {
		  /* This is a new node, we must set its name */
		  node = namespace_->getnode(cid);
		  if (node) node->name(name, name_len);
		  
		  /* pass this info on to the application */
		  srm_recv_cid_proc proc = session()->callbacks()->recv_cid_proc;
		  if (proc) {
		    (*proc)(this, cid, pid, name, name_len);
		  }
		}

		/* move the path pointer forward */
		if (name_len%4) name_len += (4 - name_len%4);
		path = (naddr_plus_name*) (((char*)path) +
					   sizeof(naddr_plus_name) + name_len);
		plen -= sizeof(naddr_plus_name) + name_len;
	}
	
	naddr *comps = new naddr[count];
	namespace_->compute_signature();
	namespace_->display(1);
	int i = 0;

	/* Check for signature mismatches in top-down order. 
	 * First, accumulate all cid, signature pairs.
	 * Then check in reverse order. 
	 */
	while (((unsigned int)olen) >= sizeof(naddr_plus_name)) {
		sign.set(ntohl(opath->addr.childsig),
			 ntohl(opath->addr.adu),
			 ntohl(opath->addr.byte));
		cid      = ntohl(opath->addr.cid);
		name_len = ntohl(opath->name_len);
	  
		comps[i].cid      = cid;
		comps[i].childsig = sign.childsig;
		comps[i].adu      = sign.edge.adu;
		comps[i].byte     = sign.edge.byte;
		i++;

		if (name_len%4) name_len += (4 - name_len%4);
		opath = (naddr_plus_name*) (((char*)opath) +
					    sizeof(naddr_plus_name) +name_len);
		olen -= sizeof(naddr_plus_name) + name_len;
	}

	for (i = count-1; i >= 0; i--) {
		SRMv2_Signature have;
		cid  = comps[i].cid;
		sign.set(comps[i].childsig, comps[i].adu, comps[i].byte);
		node = namespace_->getnode(cid);
		if (!node) continue;
		have = node->signature();
		if (! (sign == have) ) {
			srm_trace(srmSRM, ("R sig: %d 0x%x,%d:%d (have) "
					   "0x%x,%d:%d (got)", cid,
					   have.childsig, have.edge.adu,
					   have.edge.byte,sign.childsig,
					   sign.edge.adu, sign.edge.byte));
			/* look for holes in the data that we've received */
			SRMv2_Edge edge(sign.edge);
			node->check_holes(this, session_, edge);

			if (sign.childsig != have.childsig && 
			    session_->srm_should_recover(this, cid,
							 (unsigned int)-1,
							 (unsigned int)-1)
			    != 0) {
				/* Internal node mismatch. Try to generate
				 * namespace request. */
				SRMv2_Request* req = new SRMv2_Request(cid);
				req->session(session_);
				req->source(this);
				node->sched_nsrequest(req);
				srm_trace(srmSRM, ("S NSR (%p) CID = %d\n",
						   req, cid));
			}
			
			node->signature(sign.childsig, sign.edge.adu,
					sign.edge.byte);
			node->compute_signature_bottom_up();
		}
	}
	delete [] comps;
}

/*
 * Application triggered loss recovery.
 * Recover cid:sseq, ..., cid:(eseq-1).
 */
void
SRMv2_Source::recover(unsigned int cid, unsigned int sseq, unsigned int eseq)
{
	NS_Node *node = namespace_->getnode(cid);

	for (unsigned int i=sseq; i<=eseq; i++) {
	        if (!node->ignore_loss(i)) {
			node->ignore_loss(i, 1);
			/*
			 * Schedule a separate repair request for each ADU.
			 * This can be optimized if successive ADUs are missing.
			 */
			node->request_missing(this, session_, i);
		}
	}
}


void
SRMv2_Source::announce()
{
	SRMv2_NameSpace* nsp;
	naddr *comps=0;
	int count;

	nsp = name_space();
	nsp->compute_signature();
	nsp->hiprio(&comps, &count, MAX_ANNOUNCEMENTS);
	if (comps) {
		session_->send_announcement(this, 1, comps, count);
		delete [] comps;
	}
}


SRMv2_Source::~SRMv2_Source()
{
	if (sid_)  delete [] sid_;
	if (info_) delete [] info_;
	if (ch_)   delete ch_;
}

