# Copyright (C) 2000-2001 The OpenRPG Project
#
#    openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: mapper/map.py
# Author: OpenRPG
# Maintainer:
# Version:
#   $Id: map.py,v 1.40 2003/05/28 01:55:12 snowdog_ Exp $
#
# Description:
#
__version__ = "$Id: map.py,v 1.40 2003/05/28 01:55:12 snowdog_ Exp $"

from map_version import MAP_VERSION
from map_msg import *
from min_dialogs import *
from map_prop_dialog import *
import orpg.dirpath
import random
import os
import thread
import gc
import traceback

from miniatures_handler import *
from whiteboard_handler import *
from background_handler import *
from fog_handler import *

from grid_handler import * 
from map_handler import *

##-----------------------------
## canvas window
##-----------------------------

#MOPEN_TOOL = wxNewId()
#MSAVE_TOOL = wxNewId()
#ZOOM_IN = wxNewId()
#ZOOM_OUT = wxNewId()
#PROP_TOOL = wxNewId()
#LABEL_TOOL = wxNewId()
#LAYER_TOOL = wxNewId()
#MIN_LIST_TOOL = wxNewId()
#MIN_TOOL = wxNewId()
#MIN_URL = wxNewId()
#SERIAL_TOOL = wxNewId()


# Various marker modes for player tools on the map
MARKER_MODE_NONE = 0
MARKER_MODE_MEASURE = 1
MARKER_MODE_TARGET = 2
MARKER_MODE_AREA_TARGET = 3

###

#MAP_REFRESH_MINI_URLS = wxNewId()

MODE_MINI = wxNewId()
MODE_TAPE = wxNewId()
MODE_DRAW = wxNewId()
MODE = wxNewId()
MODE_SUBMENU = wxNewId()

#dummy menus
SET_MAP_FOCUS = wxNewId()

#MIN_LOCK = wxNewId()
#MIN_HIDE = wxNewId()
#miniature_list = []
class map_canvas(wxScrolledWindow):
    def __init__(self, parent, ID, isEditor = 0):
        wxScrolledWindow.__init__(self, parent, ID,style = wxHSCROLL | wxVSCROLL | wxSUNKEN_BORDER )
        self.frame = parent
        #self.memory_bitmap = None
#        self.player_id = player_id
        self.MAP_MODE = 1      #Mode 1 = MINI, 2 = DRAW, 3 = TAPE MEASURE
        self.layers = {}
        self.layers['bg'] = layer_back_ground(self)
        self.layers['grid'] = grid_layer(self)
        self.layers['whiteboard'] = whiteboard_layer(self)
        self.layers['miniatures'] = miniature_layer(self)
        self.layers['fog'] = fog_layer(self)
        EVT_ERASE_BACKGROUND(self, self.on_erase_background)
        EVT_PAINT(self, self.on_paint)
        EVT_LEFT_DOWN(self, self.on_left_down)
        EVT_LEFT_DCLICK(self, self.on_left_dclick)
        EVT_LEFT_UP(self, self.on_left_up)
        EVT_RIGHT_DOWN(self, self.on_right_down)
        EVT_MOTION(self, self.on_motion)
        EVT_SCROLLWIN(self, self.on_scroll)
        EVT_CHAR(self, self.on_char)
        EVT_SIZE(self, self.on_resize)
        self.set_size((1000,1000))
        self.root_dir = os.getcwd()
        self.size_change = 0
        self.isEditor = isEditor
        self.map_version = MAP_VERSION

        # Create the marker mode attributes for the map
        self.markerMode = MARKER_MODE_NONE
        self.markerStart = wxPoint( -1, -1 )
        self.markerStop = wxPoint( -1, -1 )
        self.markerMidpoint = wxPoint( -1, -1 )
        self.markerAngle = 0.0

        # Optimization of map refreshing during busy map load
        self.lastRefreshValue = 0
        self.requireRefresh = 0
        self.lastRefreshTime = 0
        self.zoom_display_timer = mytimer(self.better_refresh)
        random.seed( time.time() )
        self.image_timer = mytimer( self.processImages )
        self.image_timer.Start( 100 )

        # Used to check if we've used the user cache size value
        self.cacheSizeSet = 0
        self.inside = 0
        
    def better_refresh(self):
        self.Refresh(false)

    def pre_destory_cleanup(self):
        self.layers["miniatures"].del_all_miniatures()

    def processImages(self):
        global IMAGE_CACHE_SIZE

        if not self.cacheSizeSet:
            self.cacheSizeSet = 1
            cacheSize = self.frame.settings.get_setting( "ImageCacheSize" )
            if len(cacheSize):
                IMAGE_CACHE_SIZE = int(cacheSize)
            else:
                print "Default cache size being used."

            print "Current image cache size is set at " + str(IMAGE_CACHE_SIZE) + " images, using random purge."

        if not imageQueue.empty():
            (path, image, image_type, object_type, object) = imageQueue.get()
            img = wxBitmap( image, image_type )

            # Now, apply the image to the proper object
            if object_type == "miniature":
                min = self.layers['miniatures'].get_miniature_by_id(object)
                min.set_bmp(img)
            elif object_type == "background" or object_type == "texture":
                self.layers['bg'].bg_bmp = img
                if object_type == "background":
                    self.set_size([img.GetWidth(),img.GetHeight()])

            # Flag that we now need to refresh!
            self.requireRefresh += 1

            # Only update if item is not already in the cache!
            if not imageCache.has_key( path ):
                imageCache[path] = img

            # Randomly purge an item from the cache, while this is lamo, it does
            # keep the cache from growing without bounds, which is pretty important!
            if len(imageCache) >= IMAGE_CACHE_SIZE:
                keyList = imageCache.keys()
                delIndex = random.randrange( 0, len(keyList) )
                print "Cache purge: ", imageCache[keyList[delIndex]]
                del imageCache[keyList[delIndex]]

        else:
            # Now, make sure not only that we require a refresh, but that enough time has
            # gone by since our last refresh.  This keeps back to back refreshing occuring during
            # large map loads.  Of course, we are now trying to pack as many image refreshes as
            # we can into a single cycle.
            if self.requireRefresh and (self.requireRefresh == self.lastRefreshValue):
                if (self.lastRefreshTime) < time.time():
                    self.requireRefresh = 0
                    self.lastRefreshValue = 0
                    self.lastRefreshTime = time.time()
                    self.Refresh(false)

            else:
                self.lastRefreshValue = self.requireRefresh
            pass

    def on_scroll(self,evt):
        if self.frame.settings.get_setting("AlwaysShowMapScale") == "1":
            self.printscale()
        evt.Skip()
#        self.Refresh(false)

    def on_char(self,evt):
        if self.frame.settings.get_setting("AlwaysShowMapScale") == "1":
            self.printscale()
        evt.Skip()


    def printscale(self):
        wxBeginBusyCursor()
        dc = self.preppaint()
        dc.SetOptimization( true )
        self.showmapscale(dc)
        dc.SetOptimization( false )
        self.Refresh(false)
        wxEndBusyCursor()

    def send_map_data(self,action = "update"):
        wxBeginBusyCursor()
        send_text = self.toxml(action)

        if send_text:
            if not self.isEditor:
                self.frame.session.send(send_text)
            ##print "Sent:  %s" %  send_text
        else:
            print "Send of no data in send_map_data() ignored"
        wxEndBusyCursor()


    def get_size(self):
        return self.size

    def set_size(self,size):
        if size[0] < 300:
            size = (300,size[1])
        if size[1] < 300:
            size = (size[0],300)
        self.size_changed = 1
        self.size  = size
        self.fix_scroll()
        self.layers['fog'].resize(size)

    def fix_scroll(self):
        scale=self.layers['grid'].mapscale
        pos=self.GetViewStart()
        unit=self.GetScrollPixelsPerUnit()
        pos=[pos[0]*unit[0],pos[1]*unit[1]]
        size=self.GetClientSize()
        unit=[10*scale,10*scale]
        if (unit[0]==0 or unit[1]==0):
            return        
        pos[0]/=unit[0]
        pos[1]/=unit[1]
        mx=[int(self.size[0]*scale/unit[0])+1,int(self.size[1]*scale/unit[1]+1)]
        self.SetScrollbars(unit[0],unit[1],mx[0],mx[1],pos[0],pos[1])

    def on_resize(self,evt):
        self.fix_scroll()
        evt.Skip()

    def on_erase_background(self, evt):
        evt.Skip()
#        dc = evt.GetDC()
#        if not dc:
#            dc = wxClientDC(self)
        # if self.layers['bg'].type==BG_TEXTURE:
#             self.layers['bg'].draw(dc,1)
#replaced self.layers['grid'].mapscale with one
#        else:
#            evt.Skip()

    def on_paint(self, evt):

        # I don't know thje purpose of the code below.  Commenting it out for now.
        #self.inside += 1
        #if (self.inside>1):
        #    try:
        #        evt.Skip()
        #    except:
         #       pass
         #   return


        scale = self.layers['grid'].mapscale
        scrollsize = self.GetScrollPixelsPerUnit()
        clientsize = self.GetClientSize()
        topleft1 = self.GetViewStart()
        topleft = [topleft1[0]*scrollsize[0],topleft1[1]*scrollsize[1]]
        
        if (clientsize[0] > 1) and (clientsize[1] > 1):
            dc = wxMemoryDC()
            dc.BeginDrawing()
            dc.SetOptimization(true)

            bmp = wxEmptyBitmap(clientsize[0]+1,clientsize[1]+1)
            dc.SelectObject(bmp)
            dc.SetPen(wxTRANSPARENT_PEN)
            dc.SetBrush(wxBrush(self.GetBackgroundColour(),wxSOLID))
            dc.DrawRectangle(0,0,clientsize[0]+1,clientsize[1]+1)
            dc.SetDeviceOrigin(-topleft[0],-topleft[1])
            dc.SetUserScale(scale,scale)
            self.layers['bg'].draw(dc,scale,topleft,clientsize)
            self.layers['grid'].draw(dc,[topleft[0]/scale,topleft[1]/scale],[clientsize[0]/scale,clientsize[1]/scale])
            self.layers['miniatures'].draw(dc,[topleft[0]/scale,topleft[1]/scale],[clientsize[0]/scale,clientsize[1]/scale])
            self.layers['whiteboard'].draw(dc)
            self.layers['fog'].draw(dc,topleft,clientsize)
            dc.SetPen(wxNullPen)
            dc.SetBrush(wxNullBrush)
            dc.SelectObject(wxNullBitmap)
            dc.EndDrawing()
            del dc
            
            wdc = self.preppaint()
            wdc.DrawBitmap(bmp,topleft[0],topleft[1])
            if self.frame.settings.get_setting("AlwaysShowMapScale") == "1":
                self.showmapscale(wdc)

        try:
            evt.Skip()
        except:
            pass
        #self.inside = 0 #see above
        

    def preppaint(self):
        dc = wxPaintDC(self)        
        self.PrepareDC(dc)
        return (dc)

    def showmapscale(self,dc):
        scalestring = "Scale x" + `self.layers['grid'].mapscale`[:3]
        (textWidth,textHeight) = dc.GetTextExtent(scalestring)
        dc.SetUserScale(1,1)
        dc.SetPen(wxLIGHT_GREY_PEN)        
        dc.SetBrush(wxLIGHT_GREY_BRUSH)
        x = dc.DeviceToLogicalX(0)
        y = dc.DeviceToLogicalY(0)
        dc.DrawRectangle(x,y,textWidth+2,textHeight+2)
        dc.SetPen(wxRED_PEN)
        dc.DrawText(scalestring,x+1,y+1)
        dc.SetPen(wxNullPen)
        dc.SetBrush(wxNullBrush)


    def snapMarker( self, snapPoint ):
        """Based on the position and unit site, figure out where we need to snap to.  As is, on
        a square grid, there are four possible places to snap.  On a hex gid, there are 6 or 12 snap
        points."""

        # If snap to grid is disabled, simply return snapPoint unmodified
        if self.layers['grid'].snap:
            # This means we need to determine where to snap our line.  We will support
            # snapping to four different snapPoints per square for now.
            # TODO!!!
            if self.layers['grid'].mode == GRID_HEXAGON:
                size = self.layers['grid'].unit_size_y

            else:
                size = int(self.layers['grid'].unit_size)

                # Find the uppper left hand corner of the grid we are to snap to
                offsetX = (snapPoint.x / size) * size
                offsetY = (snapPoint.y / size) * size

                # Calculate the delta value between where we clicked and the square it is near
                deltaX = snapPoint.x - offsetX
                deltaY = snapPoint.y - offsetY

                # Now, figure our what quadrant (x, y) we need to snap to
                snapSize = size / 2

                # Figure out the X snap placement
                if deltaX <= snapSize:
                    quadXPos = offsetX

                else:
                    quadXPos = offsetX + size

                # Now, figure out the Y snap placement
                if deltaY <= snapSize:
                    quadYPos = offsetY

                else:
                    quadYPos = offsetY + size

                # Create our snap snapPoint and return it
                snapPoint = wxPoint( quadXPos, quadYPos )

        return snapPoint



# Bunch of math stuff for marking and measuring
    def calcSlope( self, start, stop ):
        """Calculates the slop of a line and returns it."""

        if start.x == stop.x:
            s = 0.0001

        else:
            s = float((stop.y - start.y)) / float((stop.x - start.x))

        return s



    def calcSlopeToAngle( self, slope ):
        """Based on the input slope, the angle (in degrees) will be returned."""

        # See if the slope is neg or positive
        if slope == abs(slope):
            # Slope is positive, so make sure it's not zero
            if slope == 0:
                a = 0

            else:
                # We are positive and NOT zero
                a = 360 - atan(slope) * (180.0/pi)

        else:
            # Slope is negative so work on the abs of it
            a = atan(abs(slope)) * (180.0/pi)

        return a



    def calcLineAngle( self, start, stop ):
        """Based on two points that are on a line, return the angle of that line."""

        a = self.calcSlopeToAngle( self.calcSlope( start, stop ) )

        return a



    def calcPixelDistance( self, start, stop ):
        """Calculate the distance between two pixels and returns it.  The calculated
        distance is the Euclidean Distance, which is:
        d = sqrt( (x2 - x1)**2 + (y2 - y1)**2 )"""

        return sqrt( abs((stop.x - start.x)**2 - (stop.y - start.y)**2) )



    def calcUnitDistance( self, start, stop, lineAngle ):

        distance = self.calcPixelDistance( start, stop )
        ln = "%0.2f" % lineAngle
        if self.layers['grid'].mode == GRID_HEXAGON:
            if ln == "0.00" or ln == "359.99":
                ud = distance / self.layers['grid'].unit_size_y
            else:
                ud = (sqrt(abs((stop.x - start.x)**2 + (stop.y - start.y)**2))) / self.layers['grid'].unit_size_y

        else:
            if ln == "0.00" or ln == "359.99":
                ud = distance / self.layers['grid'].unit_size
            else:
                ud = (sqrt(abs((stop.x - start.x)**2 + (stop.y - start.y)**2))) / self.layers['grid'].unit_size
            #ud = sqrt( abs((stop.x - start.x)**2 - (stop.y - start.y)**2) )
        return ud



    def on_tape_motion(self, evt):
        """Track mouse motion so we can update the marker visual every time it's moved"""

        # Make sure we have a mode to do anything, otherwise, we ignore this
        if self.markerMode:
            # Grap the current DC for all of the marker modes
            dc = wxClientDC( self )
            self.PrepareDC( dc )
            dc.SetUserScale(self.layers['grid'].mapscale,self.layers['grid'].mapscale)

            # Grab the current map position
            pos = self.snapMarker( evt.GetLogicalPosition( dc ) )

            # Enable brush optimizations
            dc.SetOptimization( true )

            # Set up the pen used for drawing our marker
            dc.SetPen( wxPen(wxRED, 1, wxLONG_DASH) )

            # Now, based on the marker mode, draw the right thing
            if self.markerMode == MARKER_MODE_MEASURE:
                if self.markerStop.x != -1 and self.markerStop.y != -1:
                    # Set the DC function that we need
                    dc.SetLogicalFunction( wxINVERT )

                    # Erase old and Draw new marker line
                    dc.BeginDrawing()
                    dc.DrawLine( self.markerStart.x, self.markerStart.y, self.markerStop.x, self.markerStop.y )
                    dc.DrawLine( self.markerStart.x, self.markerStart.y, pos.x, pos.y )
                    dc.EndDrawing()

                    # Restore the default DC function
                    dc.SetLogicalFunction( wxCOPY )

                # As long as we are in marker mode, we ned to update the stop point
                self.markerStop = pos

            dc.SetPen( wxNullPen )

            # Disable brush optimizations
            dc.SetOptimization( false )

            del dc

        else:
            pass



    def on_tape_down( self, evt ):
        """Greg's experimental tape measure code.  Hopefully, when this is done, it will all be
        modal based on a toolbar."""

        dc = wxClientDC( self )
        self.PrepareDC( dc )
        dc.SetUserScale(self.layers['grid'].mapscale,self.layers['grid'].mapscale)
        pos = evt.GetLogicalPosition( dc )

        # If grid snap is enabled, then snap the tool to a proper position
        pos = self.snapMarker( evt.GetLogicalPosition( dc ) )

        # Maker mode should really be set by a toolbar
        self.markerMode = MARKER_MODE_MEASURE

        # Erase the old line if her have one
        if self.markerStart.x != -1 and self.markerStart.y != -1:
            # Enable brush optimizations
            dc.SetOptimization( true )

            # Set up the pen used for drawing our marker
            dc.SetPen( wxPen(wxRED, 1, wxLONG_DASH) )

            # Set the DC function that we need
            dc.SetLogicalFunction( wxINVERT )

            # Draw the marker line
            dc.BeginDrawing()
            dc.DrawLine( self.markerStart.x, self.markerStart.y, self.markerStop.x, self.markerStop.y )
            dc.EndDrawing()

            # Restore the default DC function and pen
            dc.SetLogicalFunction( wxCOPY )
            dc.SetPen( wxNullPen )

            # Disable brush optimizations
            dc.SetOptimization( false )

        # Save our current start and reset the stop value
        self.markerStart = pos
        self.markerStop = pos

        del dc



    def on_tape_up( self, evt ):
        """When we release the middle button, disable any marking updates that we have been doing."""

        # If we are in measure mode, draw the actual UNIT distance
        if self.markerMode == MARKER_MODE_MEASURE:
            dc = wxClientDC( self )
            self.PrepareDC( dc )
            dc.SetUserScale(self.layers['grid'].mapscale,self.layers['grid'].mapscale)
            # Draw the measured distance on the DC.  Since we want
            # the text to match the line angle, calculate the angle
            # of the line.
            lineAngle = self.calcLineAngle( self.markerStart, self.markerStop )
            distance = self.calcUnitDistance( self.markerStart, self.markerStop, lineAngle )
            midPoint = (self.markerStart + self.markerStop)
            midPoint.x /= 2
            midPoint.y /= 2

            # Adjust out font to be bigger & scaled
            font = dc.GetFont()
            #scaleSize = dc.GetUserScale()
            #scale = (scaleSize[0] + scaleSize[1]) / 2
            #newPointSize = 2 + font.GetPointSize() + scale
            #font.SetPointSize( newPointSize )

            # Optimize the brushes
            dc.SetOptimization( true )

            # Set the DC function that we need
            dc.SetLogicalFunction( wxINVERT )

            # Set the pen we want to use
            dc.SetPen( wxBLACK_PEN )

            # Now, draw the text at the proper angle on the canvas
            self.markerMidpoint = midPoint
            self.markerAngle = lineAngle
            dText = "%0.2f Units" % (distance)
            dc.BeginDrawing()
            dc.DrawRotatedText( dText, midPoint.x, midPoint.y, lineAngle )
            dc.EndDrawing()

            # Restore the default font and DC
            dc.SetFont( wxNullFont )
            dc.SetLogicalFunction( wxCOPY )
            del font

            # Disable brush optimization
            dc.SetOptimization( false )
            del dc

        self.markerMode = MARKER_MODE_NONE

    # MODE 1 = MOVE, MODE 2 = whiteboard, MODE 3 = Tape measure
    def on_left_down(self,evt):
        if evt.ShiftDown():
            self.on_tape_down (evt)
        else:
            self.frame.on_left_down(evt)

    def on_right_down(self,evt):
        if evt.ShiftDown():
            pass
        else:
            self.frame.on_right_down(evt)


    def on_left_dclick(self,evt):
        if evt.ShiftDown():
            pass
        else:
            self.frame.on_left_dclick(evt)

    def on_left_up(self, evt):
        if evt.ShiftDown():
            self.on_tape_up(evt)
        else:
            self.frame.on_left_up(evt)

    def on_motion(self,evt):
        if evt.ShiftDown():
            self.on_tape_motion(evt)
        else:
            self.frame.on_motion(evt)


    def on_zoom_out(self, evt):
        if self.layers['grid'].mapscale > 0.6:
            self.layers['grid'].mapscale -= .1
            scalestring = "x" + `self.layers['grid'].mapscale`[:3]
            self.frame.get_current_layer_handler().zoom_out_button.SetToolTip(wxToolTip("Zoom out from " + scalestring) )
            self.frame.get_current_layer_handler().zoom_in_button.SetToolTip(wxToolTip("Zoom in from " + scalestring) )

            self.set_size(self.size)
            dc = wxClientDC(self)
            dc.BeginDrawing()
            scalestring = "Scale x" + `self.layers['grid'].mapscale`[:3]
            (textWidth,textHeight) = dc.GetTextExtent(scalestring)
            dc.SetPen(wxLIGHT_GREY_PEN)
            dc.SetBrush(wxLIGHT_GREY_BRUSH)
            dc.DrawRectangle(dc.DeviceToLogicalX(0),dc.DeviceToLogicalY(0),textWidth,textHeight)
            dc.SetPen(wxRED_PEN)
            dc.DrawText(scalestring,dc.DeviceToLogicalX(0),dc.DeviceToLogicalY(0))
            dc.SetPen(wxNullPen)
            dc.SetBrush(wxNullBrush)
            dc.EndDrawing()
            del dc
            self.zoom_display_timer.Start(500,1)



    def on_zoom_in(self, evt):
        self.layers['grid'].mapscale += .1
        scalestring = "x" + `self.layers['grid'].mapscale`[:3]
        self.frame.get_current_layer_handler().zoom_out_button.SetToolTip(wxToolTip("Zoom out from " + scalestring) )
        self.frame.get_current_layer_handler().zoom_in_button.SetToolTip(wxToolTip("Zoom in from " + scalestring) )

        self.set_size(self.size)

        dc = wxClientDC(self)
        dc.BeginDrawing()
        scalestring = "Scale x" + `self.layers['grid'].mapscale`[:3]
        (textWidth,textHeight) = dc.GetTextExtent(scalestring)
        dc.SetPen(wxLIGHT_GREY_PEN)
        dc.SetBrush(wxLIGHT_GREY_BRUSH)
        dc.DrawRectangle(dc.DeviceToLogicalX(0),dc.DeviceToLogicalY(0),textWidth,textHeight)
        dc.SetPen(wxRED_PEN)
        dc.DrawText(scalestring,dc.DeviceToLogicalX(0),dc.DeviceToLogicalY(0))
        dc.SetPen(wxNullPen)
        dc.SetBrush(wxNullBrush)
        dc.EndDrawing()
        del dc
        self.zoom_display_timer.Start(500,1)

    def on_prop(self, evt):
        session=self.frame.session
        if (session.my_role() <> session.ROLE_GM):
            self.frame.myopenrpg.get_component("chat").InfoPost("You must be a GM to use this feature")
            return
        dlg = general_map_prop_dialog(self.frame.GetParent(),self.size,self.layers['bg'],self.layers['grid'])
        if dlg.ShowModal() == wxID_OK:
            self.set_size(dlg.size)
            self.send_map_data()
            self.Refresh(false)
        dlg.Destroy()
        os.chdir(self.root_dir)


    def add_miniature( self, min_url, min_label = '', min_unique = -1 ):
        if min_unique == -1:
            min_unique = not self.use_serial
        if min_url == "" or min_url == "http://":
            return
        if min_url[:7] != "http://" :
            min_url = "http://"+min_url
        # make label
        wxBeginBusyCursor()
        if self.auto_label:
            if min_label == '':
                min_label = self.get_label_from_url( min_url )
            if not min_unique and self.use_serial:
                #self.layers['miniatures'].serial_number = self.layers['miniatures'].serial_number + 1 # we're 1 indexed
                min_label = '%s %d' % ( min_label, self.layers['miniatures'].next_serial() )
        else:
            min_label = ""
        if self.frame.min_url.FindString(min_url) == -1:
            self.frame.min_url.Append(min_url)
        try:
            self.layers['miniatures'].add_miniature(self.frame.session.get_next_id(),min_url,label=min_label)
        except:
            # When there is an exception here, we should be decrementing the serial_number for reuse!!
            print "Unable to load/resolve URL: " + min_url + "on resource \"" + min_label + "\"!!!\n\n"
            self.layers['miniatures'].rollback_serial()

        wxEndBusyCursor()
        self.send_map_data()
        self.Refresh(false)

    def get_label_from_url( self, url='' ):
        if url == '':
            return ''
        start = url.rfind("/") + 1
        label = url[ start : len( url ) -4 ]
        return label



    def toxml(self, action = "update"):

        if action == "new":
            self.size_changed = 1

        xml_str = "<map version='" + self.map_version + "'"
        changed = self.size_changed
        if self.size_changed:
            xml_str += " sizex='" + str(self.size[0]) + "'"
            xml_str += " sizey='" + str(self.size[1]) + "'"

        s = ""
        keys = self.layers.keys()
        for k in keys:
            if (k!="fog" or action!="update"):
                s += self.layers[k].toxml(action)

        self.size_changed = 0

        if s:
            return xml_str + " action='" + action + "'>" + s + "</map>"
        else:
            if changed:
                return xml_str + " action='" + action + "'/>"
            else:
                return ""


    def takexml(self,xml):
        #
        # Added Process Dialog to display during long map parsings
        # as well as a try block with an exception traceback to try
        # and isolate some of the map related problems users have been
        # experiencing --Snowdog 5/15/03
        #
        # Apparently Process Dialog causes problems with linux.. commenting it out. sheez.
        #  --Snowdog 5/27/03
        try:
            #create (and show) a process dialog
            ##proc_dialog=wxProgressDialog( "Map Loading", "Map Loading, Please Wait...", 5, self )
            ##proc_dialog.Update( 0 )


            #parse the map DOM
            xml_dom = parseXml(xml)
            ##proc_dialog.Update( 1 )
            
            
            #print "number of uncollectables = " + str(gc.collect())
            node_list = xml_dom.getElementsByTagName("map")

            if len(node_list) < 1:
                print "Invalid XML format for mapper"
            else:
                # set map version to incoming data so layers can convert
                self.map_version = node_list[0].getAttribute("version")


                action = node_list[0].getAttribute("action")
                if action == "new":
                    self.layers = {}
                    try:
                        self.layers['bg'] = layer_back_ground(self)
                    except:
                        pass
                    try:
                        self.layers['grid'] = grid_layer(self)
                    except:
                        pass
                    try:
                        self.layers['miniatures'] = miniature_layer(self)
                    except:
                        pass
                    try:
                        self.layers['whiteboard'] = whiteboard_layer(self)
                    except:
                        pass
                    try:                    
                        self.layers['fog'] = fog_layer(self)
                    except:
                        pass

                sizex = node_list[0].getAttribute("sizex")
                if sizex <> "":
                    sizex = int(float(sizex))
                    sizey = self.size[1]
                    self.set_size((sizex,sizey))
                    self.size_changed = 0

                sizey = node_list[0].getAttribute("sizey")
                if sizey <> "":
                    sizey = int(float(sizey))
                    sizex = self.size[0]
                    self.set_size((sizex,sizey))
                    self.size_changed = 0

                children = node_list[0]._get_childNodes()

        #fog layer must be computed first, so that no data is inadvertently revealed

                for c in children:
                    name = c._get_nodeName()
                    if name == "fog":
                        self.layers[name].takedom(c)

                        
                for c in children:
                    name = c._get_nodeName()
                    if name != "fog":
                        self.layers[name].takedom(c)

                # all map data should be converted, set map version to current version
                self.map_version = MAP_VERSION

                self.Refresh(false)  
            ##proc_dialog.Update( 4 )

            xml_dom.unlink()  # eliminate circular refs
            ##proc_dialog.Update( 5 )
            ##proc_dialog.Show(FALSE)
            ##proc_dialog.Destroy()
        except:
            print "EXCEPTION: Critical Error Loading Map!!!"
            traceback.print_exc()
        #print "Explicit garbage collection shows %s undeletable items." % str(gc.collect())

    def re_ids_in_xml(self, xml):
        new_xml = ""
        tmp_map = map_msg()
        xml_dom = parseXml(str(xml))
        node_list = xml_dom.getElementsByTagName("map")
        if len(node_list) < 1:
            print "Invalid XML format for mapper"
        else:
            tmp_map.init_from_dom(node_list[0])
            miniatures_layer = tmp_map.children["miniatures"]
            if miniatures_layer:
                minis = miniatures_layer.get_children().keys()
                if minis:
                    for mini in minis:
                        m = miniatures_layer.children[mini]
                        m.init_prop("id",self.frame.session.get_next_id())

            # This allows for backward compatibility with older maps which do not
            # have a whiteboard node.  As such, if it's not there, we'll just happily
            # move on and process as always.
            if tmp_map.children.has_key( "whiteboard" ):
                whiteboard_layer = tmp_map.children["whiteboard"]
                if whiteboard_layer:
                    lines = whiteboard_layer.get_children().keys()
                    if lines:
                        for line in lines:
                            l = whiteboard_layer.children[line]
                            l.init_prop("id",self.frame.session.get_next_id())
            new_xml = tmp_map.get_all_xml()
        if xml_dom:
            xml_dom.unlink()
        return str(new_xml)



LAYER_TABS = wxNewId()
MENU_LAYER_BG = wxNewId()
MENU_LAYER_MIN = wxNewId()
MENU_LAYER_WHITEBOARD = wxNewId()
MENU_LAYER_GRID = wxNewId()
MENU_FLUSH_CACHE = wxNewId()
MENU_LAYER_MAP = wxNewId()
MENU_LAYER_FOG = wxNewId()

class map_wnd(wxPanel):
    def __init__(self, parent, id, openrpg):
        wxPanel.__init__(self, parent, id)
        self.canvas = map_canvas(self, -1)
        self.myopenrpg = openrpg
        self.session = openrpg.get_component('session')
        self.settings = openrpg.get_component('settings')
        self.top_frame = self.myopenrpg.get_component('frame')
        self.root_dir = os.getcwd()

        self.current_layer = 0
        self.layer_tabs = wxNotebook(self, LAYER_TABS, style=wxNB_BOTTOM)
        self.layer_handlers = []
        self.layer_handlers.append(background_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[0],"Background")
        self.layer_handlers.append(grid_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[1],"Grid")
        self.layer_handlers.append(miniatures_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[2],"Miniatures")
        self.layer_handlers.append(whiteboard_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[3],"Whiteboard")
        self.layer_handlers.append(fog_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[4],"Fog")
        self.layer_handlers.append(map_handler(self.layer_tabs,-1,self.canvas))
        self.layer_tabs.AddPage(self.layer_handlers[5],"General")
#        self.layer_handlers.append(fog_handler(self.layer_tabs,-1,self.canvas))
#        self.layer_tabs.AddPage(self.layer_handlers[2],"Fog")
        
        EVT_NOTEBOOK_PAGE_CHANGED(self.layer_tabs, LAYER_TABS, self.on_layer_change)
        EVT_SIZE(self, self.on_size)
        
        self.load_default()

        # size of tabs is diffrent on windows and linux :(
        
        if wxPlatform == '__WXMSW__':
            self.toolbar_height = 50
        else:
            self.toolbar_height = 60
            
        self.layer_tabs.SetSelection(2)

    def load_default(self):
        session=self.session
        if session.is_connected() and (session.my_role() <> session.ROLE_GM) and (session.use_roles()):
            self.top_frame.myopenrpg.get_component("chat").InfoPost("You must be a GM to use this feature")
            return        
        f = open(orpg.dirpath.dir_struct["template"]+"default_map.xml")
        self.new_data(f.read())
        f.close()
        self.canvas.send_map_data("new")
        if not session.is_connected() and (session.my_role() <> session.ROLE_GM):
            session.update_role("GM")

    def new_data(self,data):
        self.canvas.takexml(data)
        self.update_tools()
        
    def on_save(self,evt):
        session=self.session
        if (session.my_role() <> session.ROLE_GM):
            self.top_frame.myopenrpg.get_component("chat").InfoPost("You must be a GM to use this feature")
            return
        
        d =wxFileDialog(self.GetParent(),"Save map data",orpg.dirpath.dir_struct["user"],"","*.xml",wxSAVE)
        if d.ShowModal() == wxID_OK:
            f = open(d.GetPath(),"w")
            data = "<nodehandler class=\"min_map\" icon=\"compass\" module=\"core\" name=\"miniature Map\">"
            data += self.canvas.toxml("new")
            data += "</nodehandler>"
            data = data.replace(">",">\n")
            f.write(data)
            f.close()
        d.Destroy()
        os.chdir(self.root_dir)

    def on_open(self,evt):
        session=self.session
        if session.is_connected() and (session.my_role() <> session.ROLE_GM) and (session.use_roles()):
            self.top_frame.myopenrpg.get_component("chat").InfoPost("You must be a GM to use this feature")
            return
        
        d =wxFileDialog(self.GetParent(),"Select a file",orpg.dirpath.dir_struct["user"],"","*.xml",wxOPEN)
        if d.ShowModal() == wxID_OK:
            f = open(d.GetPath())
            map_string = f.read()
            new_xml = self.canvas.re_ids_in_xml(map_string)
            if new_xml:
                self.canvas.takexml(new_xml)
                self.canvas.send_map_data("new")
                self.update_tools()
                if not session.is_connected() and (session.my_role() <> session.ROLE_GM):
                    session.update_role("GM")

        d.Destroy()
        os.chdir(self.root_dir)

    def get_current_layer_handler(self):
        return self.layer_handlers[self.current_layer]

    def on_layer_change(self,evt):
        self.current_layer = evt.GetSelection()
        bg=self.layer_handlers[0]        
        session=self.session
        if (session.my_role() <> session.ROLE_GM):
            bg.url_path.Show(false)
        else:
            bg.url_path.Show(true)
        self.canvas.Refresh(false)
        evt.Skip()

    def on_left_down(self,evt):
        self.layer_handlers[self.current_layer].on_left_down(evt)

    #double click handler added by Snowdog 5/03
    def on_left_dclick(self,evt):
        self.layer_handlers[self.current_layer].on_left_dclick(evt)

    def on_right_down(self,evt):
        self.layer_handlers[self.current_layer].on_right_down(evt)

    def on_left_up(self, evt):
        self.layer_handlers[self.current_layer].on_left_up(evt)

    def on_motion(self,evt):
        self.layer_handlers[self.current_layer].on_motion(evt)
        
    def MapBar(self,id,data):
        self.canvas.MAP_MODE = data
        if id == 1:
            self.canvas.MAP_MODE = data

    def set_map_focus(self,evt):
        self.canvas.SetFocus()

    def pre_exit_cleanup(self):
        # do some pre exit clean up for bitmaps or other objects
        flushImageCache()
        self.canvas.pre_destory_cleanup()
        
        
    def update_tools(self):
        for h in self.layer_handlers:
            h.update_info()

    def on_hk_map_layer(self,evt):
        id = evt.GetId()
        if id == MENU_LAYER_BG:
            self.current_layer = 0
        if id == MENU_LAYER_GRID:
            self.current_layer = 1
        if id == MENU_LAYER_MIN:
            self.current_layer = 2
        elif id == MENU_LAYER_WHITEBOARD:
            self.current_layer = 3
        elif id == MENU_LAYER_FOG:
            self.current_layer = 4
        elif id == MENU_LAYER_MAP:
            self.current_layer = 5
        self.layer_tabs.SetSelection(self.current_layer)

    def on_flush_cache(self,evt):
        flushImageCache()

    def build_menu(self):
        top_frame = self.myopenrpg.get_component('frame')
        # temp menu
        menu = wxMenu()
        #menu.Append(SET_MAP_FOCUS, 'Map\tCtrl-M', 'Map')
        menu.Append(MOPEN_TOOL,"&Load Map")
        menu.Append(MSAVE_TOOL,"&Save Map")
        menu.AppendSeparator()
        menu.Append(MENU_LAYER_BG, 'Background Layer\tCtrl-1', 'Background Layer')
        menu.Append(MENU_LAYER_GRID, 'Grid Layer\tCtrl-2', 'Grid Layer')
        menu.Append(MENU_LAYER_MIN, 'Miniature Layer\tCtrl-3', 'Miniature Layer')
        menu.Append(MENU_LAYER_WHITEBOARD, 'Whiteboard Layer\tCtrl-4', 'Whiteboard Layer')
        menu.Append(MENU_LAYER_FOG, 'Fog Layer\tCtrl-5', 'Fog Layer')
        menu.Append(MENU_LAYER_MAP, 'General Properties\tCtrl-6', 'General Properties')
        menu.AppendSeparator()
        menu.Append(PROP_TOOL,"&Properties")
        menu.AppendSeparator()
        menu.Append(MENU_FLUSH_CACHE, "&Flush Image Cache\tCtrl-F" )
        top_frame.mainmenu.Insert(2, menu, '&Map')

        EVT_MENU(top_frame, SET_MAP_FOCUS, self.set_map_focus)
        EVT_MENU(top_frame, MENU_LAYER_BG, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_LAYER_GRID, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_LAYER_MAP, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_LAYER_FOG, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_LAYER_MIN, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_LAYER_WHITEBOARD, self.on_hk_map_layer)
        EVT_MENU(top_frame, MENU_FLUSH_CACHE, self.on_flush_cache)

        EVT_MENU(top_frame, PROP_TOOL, self.canvas.on_prop)
        EVT_MENU(top_frame, MOPEN_TOOL, self.on_open)
        EVT_MENU(top_frame, MSAVE_TOOL, self.on_save)

    def get_hot_keys(self):
        self.build_menu()
        entries = []
        entries.append((wxACCEL_CTRL,ord('M'),SET_MAP_FOCUS))
        entries.append((wxACCEL_CTRL,ord('1'),MENU_LAYER_BG))
        entries.append((wxACCEL_CTRL,ord('2'),MENU_LAYER_GRID))
        entries.append((wxACCEL_CTRL,ord('3'),MENU_LAYER_MIN))
        entries.append((wxACCEL_CTRL,ord('4'),MENU_LAYER_WHITEBOARD))
        entries.append((wxACCEL_CTRL,ord('5'),MENU_LAYER_FOG))
        entries.append((wxACCEL_CTRL,ord('6'),MENU_LAYER_MAP))
        entries.append((wxACCEL_CTRL,ord('F'),MENU_FLUSH_CACHE))
        return entries

    def on_size(self,evt):
        s = self.GetClientSizeTuple()
        self.canvas.SetDimensions(0,0,s[0],s[1]-self.toolbar_height)
        self.layer_tabs.SetDimensions(0,s[1]-self.toolbar_height,s[0],self.toolbar_height)
