/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996-2000 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    support@pftp.de
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#include <sys/types.h>
#endif
#if defined SunOS_SOLARIS || defined unicos
#include <fcntl.h>
#endif
#include <ctype.h>
#include <dirent.h>
#include <sys/time.h>
#if defined Linux || defined NEXTSTEP
#include <sys/dir.h>
#endif
#if defined OSF1
#include <sys/stat.h>
extern int mknod(const char *, int, dev_t);
#endif
#include "main.h"

extern long dlastentry;
extern long tlastentry;
extern char *tfiles[PFTP_MAX_PFM_ENTRY]; /* holds the selected files */
extern char *dfiles[PFTP_MAX_PFM_ENTRY]; /* holds the selected files */
extern long set_entry_in_files(char **, char *, long *);
extern void rm_entry_from_files(char **, char *, long *);
extern void display_pa(char **, char *, char *, int);
extern char *tabdir(char *);
extern short mycmp(char *, char *);
extern void find_free_name(char *, char *);
extern void free_vec(char **);

extern char *headline;
extern short CLEAR_TAG;


void print_error_mesg(char *func, char *str)
{
   fprintf(stderr, "\n** %s: %s: %s!\n", func, str, _PFTP_ERROR_ARRAY_);
   fprintf(stderr, "*** Type any key to continue ***");
   fgetc(stdin);
}


short create_dir(char *s, char *isdir)
{
   char c, *tmp=NULL, *cdir=NULL;
   struct stat buf;
   short last=0;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));

   for (tmp=cdir; (*tmp = *s); tmp++, s++);
   if (*--tmp == '/' && tmp != cdir) *tmp = '\0';
   else tmp++;
   if (!isdir) {
      for (; *tmp != '/' && tmp != cdir; tmp--);
      if (tmp == cdir) ++tmp;
      *tmp = '\0';
   }
   if (stat(cdir, &buf)) {
      fprintf(stderr, "\n** `%s' does not exist!\n** Create that directory (y|n)? ", cdir);
      while((c = fgetc(stdin)) != 'y' && c != 'Y' && c != 'n' && c != 'N');
      if (c == 'n' || c == 'N') {
         free(cdir);
         return 0;
      }
   }
   if (isdir) strcat(cdir, isdir);
   
   for (tmp=cdir+1; *(tmp-1); tmp++) {
      if (*tmp == '/' || !*tmp) {
         if (!*tmp) last = 1;
         *tmp = '\0';
         if (stat(cdir, &buf)) {
            if (mkdir(cdir, 0755) < 0) {
               print_error_mesg("mkdir", cdir);
               free(cdir);
               return 0;
            }
         }
         if (!last) *tmp = '/';
      }
   }
   free(cdir);
   return 1;
}


int set_status_information(char *file, struct stat *buf)
{
   pftp_tv tv[2];

   if (chmod(file, buf->st_mode) < 0) {
      print_error_mesg("chmod", file);
      return 0;
   }
   if (!getuid() && chown(file, buf->st_uid, buf->st_gid)) {
      print_error_mesg("chown", file);
      return 0;
   }
   tv[0].tv_sec = buf->st_mtime;
   tv[1].tv_sec = tv[0].tv_sec;
   tv[0].tv_usec = tv[1].tv_usec = 0;
   if (utimes(file, (struct timeval *)tv)) {
      print_error_mesg("utimes", file);
      return 0;
   }
   return 1;
}


short copy_file(char *s, char *d)
{
   int sfd=0, dfd=0, ptrun=(*statstr)->OVERWRITE ? O_TRUNC: 0;
   ssize_t n=0, check=0;
   extern char *saving_str;
   char str[BUFSIZE], c=0;
   char name[HUNAME];
   char *strptr=NULL, *stmp=NULL;
   struct stat buf;

   stmp = d;
   while (!access(stmp, F_OK) && !(*statstr)->OVERWRITE) {
      if ((*statstr)->_SKIP_) return 1;
      ptrun = 0;
      fprintf(stderr, "** File %s exists!\n** Overwrite(y|n) all(Y), rename(r), skip(s) all(S), cancel(c) ? ", stmp);
      c = fgetc(stdin);
      fprintf(stderr, "\n%s", saving_str);
      if(c == 'Y' || c == 'y') {
         if (lstat(stmp, &buf)) {
            print_error_mesg("stat", stmp);
            return 0;
         }
         else if ((buf.st_mode & 0200) != 0200) {
            if (chmod(stmp, buf.st_mode|0200) < 0) {
               print_error_mesg("chmod", stmp);
               return 0;
            }
         }
         if (c == 'Y') (*statstr)->OVERWRITE = BIT_ONE;
         ptrun = O_TRUNC;
         break;
      }
      else if (c == 'r') {
         find_free_name(d, name);
         stmp = name;
      }
      else if (c == 's') return 1;
      else if (c == 'S') {
         (*statstr)->_SKIP_ = BIT_ONE;
         return 1;
      }
      else if (c == 'c') return 0;
      else {
         strcpy(str, stmp);
         if (!(stmp = tabdir(str))) return 0;
         fprintf(stderr, "\n%s", saving_str);
      }   
   }

   /*
    * Get information about the file.
    */
   if (lstat(s, &buf)) {
      print_error_mesg("fstat", s);
      return 0;
   }

   if (!buf.st_size && !getuid()) {
      if (mknod(stmp, buf.st_mode, buf.st_rdev) < 0) {
         print_error_mesg("mknod", stmp);
         return 0;
      }
   }
   else {
      if ((buf.st_mode & S_IFMT) == S_IFIFO) {
         if (mkfifo(stmp, buf.st_mode) < 0) {
            print_error_mesg("mknod", stmp);
            return 0;
         }
      }
      else if ((buf.st_mode & S_IFMT) == S_IFLNK) {
         memset(str, '0', BUFSIZE);
         if (readlink(stmp, str, BUFSIZE) < 0) {
            print_error_mesg("readlink", stmp);
            return 0;
         }
         if (symlink(str, stmp) < 0) {
            print_error_mesg("symlink", stmp);
            return 0;
         }
      }
      else {
         /*
          * Open the source and the destination files.
          */
         if ((sfd = open(s, O_RDONLY)) < 0) {
            print_error_mesg("open", s);
            return 0;
         }

         if ((dfd = open(stmp, O_CREAT|O_WRONLY|ptrun, 0200)) < 0) {
            print_error_mesg("open", stmp);
            close(sfd);
            return 0;
         }
         
         /*
          * Copy source file.
          */
         for (n=0; (n = read(sfd, str, BUFSIZE)) > 0;) {
            for (check=0, strptr=str; n > 0; n-=check, strptr += check) {
               if ((check = write(dfd, strptr, n)) < 0) {
                  print_error_mesg("write", stmp);
                  close(sfd);
                  return 0;
               }
            }
         }
         close(dfd);
         close(sfd);
      }
   }

   /*
    * Set information from the source file.
    */
   if (!set_status_information(stmp, &buf)) return 0;

   /*
    * Mark files for being removed.
    */
   set_entry_in_files(dfiles, s, &dlastentry);

   return 1;
}


short save_dir(char *s, char *d)
{
   char *sdir=NULL;
   char *ddir=NULL;
   char *str[HUNAME];
   char **tmp=NULL;
   int ret=0;
   struct stat orig_buf, buf;
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   tmp = str;

   if (!stat(s, &buf)) {
      MEM_CHECK((ddir = (char *)calloc(LONAME, sizeof(char))));
      MEM_CHECK((sdir = (char *)calloc(LONAME, sizeof(char))));

      if ((buf.st_mode & S_IFMT) == S_IFDIR) {
         if (!(dp = opendir(s))) {
            print_error_mesg("opendir", s);
            free(sdir);
            free(ddir);
            return 0;
         }
         while ((dir = readdir(dp)))
         {
            if (dir->d_ino && mycmp(dir->d_name, "..") && mycmp(dir->d_name, ".") \
               && mycmp(PFTP_INFO_FILE, dir->d_name) && mycmp(PFTP_LOCK_FILE, dir->d_name)) {
               MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
               strcpy(*tmp++, dir->d_name);
            }
         }
         closedir(dp);
         *tmp = NULL;
      }
      else {
         ret = copy_file(sdir, ddir);
         free(sdir);
         free(ddir);
         return ret;
      }
      for (ret=1, tmp--; ret && tmp+1 != str; tmp--) {
         sprintf(sdir, "%s/%s", s, *tmp);
         sprintf(ddir, "%s/%s", d, *tmp);
         free(*tmp);
         *tmp = NULL;
         if (!stat(sdir, &orig_buf) && ((orig_buf.st_mode & S_IFMT) == S_IFDIR)) {
            if (stat(ddir, &buf)) {
               if (mkdir(ddir, 0755) < 0) {
                  free_vec(tmp);
                  free(sdir);
                  print_error_mesg("stat", ddir);
                  free(ddir);
                  return 0;
               }
            }
            else if (!((buf.st_mode & S_IFMT) == S_IFDIR)) {
               free_vec(tmp);
               free(sdir);
               fprintf(stderr, "\n** `%s' is not a directory!\n", ddir);
               fprintf(stderr, "*** Type any key to continue ***");
               fgetc(stdin);
               free(ddir);
               return 0;
            }
            save_dir(sdir, ddir);
            ret = set_status_information(ddir, &orig_buf);
         }
         else ret = copy_file(sdir, ddir);
      }
      free_vec(tmp);
      free(sdir);
      free(ddir);
   }
   return ret;
}


short save_tags(char *d)
{
   int i=0;
   char **tag=NULL, *tmp=NULL;
   char *cdir=NULL;
   struct stat buf;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));

   CLEAR_TAG = 1;
   for (tag=tfiles; tag-tfiles <= tlastentry; tag++) {
      if (*tag) {
         if (!stat(*tag, &buf) && (buf.st_mode & S_IFREG)) {
            for (i=0, tmp=*tag; i < 2 && *tmp; tmp++) if (*tmp == '/') i++;
            sprintf(cdir, "%s/%s", d, tmp);
            for (tmp=cdir+1; *tmp; tmp++) {
               if (*tmp == '/') {
                  *tmp = '\0';
                  if (stat(cdir, &buf)) {
                     if (mkdir(cdir, 0755) < 0) {
                        print_error_mesg("mkdir", cdir);
                        free(cdir);
                        return 0;
                     }
                  }
                  *tmp = '/';
               }
            }
            if (!copy_file(*tag, cdir)) {
               free(cdir);
               return 0;
            }
         }
         rm_entry_from_files(tfiles, *tag, &tlastentry);
      }
   }
   free(cdir);
   return 1;
}


char *get_subject(char *subj, char *user, char *date)
{
   FILE *fp=NULL;
   char *str=NULL;
   struct stat buf;

   MEM_CHECK((str = (char *)calloc(LONAME, sizeof(char))));
   sprintf(str, "%s/%s/%s", user, date, PFTP_INFO_FILE);
   if (!stat(str, &buf)) {
      if (!(fp = fopen(str, "r"))) {
         free(str);
         return NULL;
      }
      while (fgets(str, SONAME, fp)) {
         if (!strncmp("Subject:", str, 8)) {
            strcpy(subj, str+9);
            free(str);
            fclose(fp);
            return subj;
         }
      }
      fclose(fp);
   }
   free(str);
   return NULL;
}


int is_pftp_mesg(char *file)
{
   FILE *fp=NULL;
   char str[LONAME];
   int count=0;

   if ((fp = fopen(file, "r")) == NULL) {
      print_error_mesg("fopen", file);
      return 0;
   }
   for (count=1; count; count--) {
      if (!fgets(str, LONAME, fp)) break;
      *(str+8) = 0;
      if (!mycmp(str, "Subject:")) count += 2;
   }
   fclose(fp);
   if (count) return 0;
   return 1;
}
