/*****
*
* Copyright (C) 2001, 2002 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "detect.h"

#include <libprelude/list.h>
#include <libprelude/common.h>
#include <libprelude/idmef-tree.h>
#include <libprelude/variable.h>

#include "rules.h"
#include "rules-operations.h"
#include "rules-parsing.h"
#include "rules-default.h"

#include "snort-keys.h"

typedef struct {
        const char *rule;
        int (*handler)(const char *file, int line, int level);
} parse_hook_t;



int snortrules_parse(void);
static int parse_signature_file(const char *file, int level);


static int n_rules, n_ignored;
static plugin_detect_t plugin;
static subscribtion_t subscribtion[] = {
        { p_end, NULL },
};


extern int data_msg_id;
extern int data_sid_id;
extern int data_revision_id;
extern int data_classtype_id;
extern int data_reference_id;
static char *rulesetdir = NULL;



/*
 * This function is called by the signature engine
 * when a signature match a packet.
 */
static void signature_matched_cb(packet_container_t *packet, data_t *data)
{
        data_msg_t *msg;
        classtype_t *class;
        nids_alert_t alert;
        data_reference_t *ref;
        idmef_impact_t impact;
        idmef_additional_data_t *ad;
        
        nids_alert_init(&alert);

        ad = signature_engine_get_data_by_id(data, data_sid_id);
        if ( ad ) 
                list_add_tail(&ad->list, &alert.additional_data_list);

        ad = signature_engine_get_data_by_id(data, data_revision_id);
        if ( ad )
                list_add_tail(&ad->list, &alert.additional_data_list);
        
        class = signature_engine_get_data_by_id(data, data_classtype_id);
        if ( class ) {
                impact.severity = class->severity;
                impact.type = class->type;
                impact.completion = class->completion;                
                impact.description.string = class->desc;
                impact.description.len = class->desclen;
                alert.impact = &impact;
        }
        
        ref = signature_engine_get_data_by_id(data, data_reference_id);
        if ( ref ) {
                alert.classification.url.string = ref->url;
                alert.classification.url.len = ref->url_len;
                alert.classification.origin = ref->origin;
        }
                
        msg = signature_engine_get_data_by_id(data, data_msg_id);
        if ( msg ) {
                alert.classification.name.string = msg->msg;
                alert.classification.name.len = msg->len;
        }
        
        nids_alert((plugin_generic_t *)&plugin, packet, &alert, NULL);
}



static rules_node_t *get_protocol_node(const char *file, int line) 
{
        const char *param;
        rules_node_t *root = NULL;
        
        param = strtok(NULL, " ");
        if ( ! param ) {
                log(LOG_INFO, "%s (%d) Missing protocol for alert rule\n", file, line);
                return NULL;
        }

        if ( strcasecmp(param, "ip") == 0 )
                root = signature_engine_get_ip_root();
        
        else if ( strcasecmp(param, "tcp") == 0 )
                root = signature_engine_get_tcp_root();

        else if ( strcasecmp(param, "udp") == 0 )
                root = signature_engine_get_udp_root();
        
        else if ( strcasecmp(param, "icmp") == 0 )
                root = signature_engine_get_icmp_root();

        else 
                log(LOG_INFO, "%s (%d) Unknown or unsupported protocol %s\n", file, line, param);

        return root;
}



/*
 * yacc/lex parser use this global variable to
 * communicate with us.
 */
rules_t *rule_parsed;


static int parse_signature(const char *file, int line, int level, run_f_t match) 
{
        int ret;
        const char *value;        
        rules_node_t *root;

        root = get_protocol_node(file, line);
        if ( ! root ) {
                log(LOG_ERR, "couldn't get protocol node.\n");
                return -1;
        }

        value = strtok(NULL, "");
        if ( ! value ) {
                log(LOG_INFO, "%s (%d) Missing test\n", file, line);
                return -1;
        }

        set_parsing_buffer(value);

        ret = snortrules_parse();
        if ( ret != 0 ) {
                signature_parser_set_error("Syntax Error [fix grammar.y to report correctly this error]");
                log(LOG_INFO, "%s (%d) Parse error: %s\n", file, line, signature_parser_get_error_buffer());
                n_ignored++;
                return -1;
        }

        if ( ! rule_parsed ) {
                log(LOG_INFO, "%s (%d) Parse error: incomplete rule\n", file, line);
                n_ignored++;
                return -1;
        }

        ret = signature_parser_post_processing(rule_parsed);
        if ( ret < 0 ) {
                log(LOG_INFO, "%s (%d) Parse error: %s\n", file, line, signature_parser_get_error_buffer());
                return -1;
        }

        ret = signature_engine_add_rules(root, rule_parsed, match);
        if ( ret < 0 ) {
                delete_rules(rule_parsed);
                return -1;
        }

        delete_rules(rule_parsed);
        
        n_rules++;
        
        return 0;
}




static int parse_include(const char *file, int line, int level) 
{
        char buf[1024];
        const char *param;
        
        param = strtok(NULL, "");
        if ( ! param ) {
                log(LOG_INFO, "%s (%d) Missing include filename.\n", file, line);
                return -1;
        }

        if ( rulesetdir && *param != '/' ) {
                snprintf(buf, sizeof(buf), "%s/%s", rulesetdir, param);
                param = buf;
        }
        
        return parse_signature_file(param, level + 1);
}




static int parse_var(const char *file, int line, int level) 
{
        const char *param, *value;
        
        param = strtok(NULL, " ");
        if ( ! param ) {
                log(LOG_INFO, "%s (%d) Missing variable name.\n", file, line);
                return -1;
        }

        value = strtok(NULL, "");
        if ( ! value ) {
                log(LOG_INFO, "%s (%d) Missing value of variable \"%s\".\n", file, line, param);
                return -1;
        }
        
        variable_set(strdup(param), strdup(value));

        return 0;
}



static int parse_preprocessor(const char *file, int line, int level) 
{
        /*
         * preprocessor configuration, just ignore the line
         */
        return -1;
}



static int parse_output(const char *file, int line, int level)
{
        /*
         * output configuration, just ignore the line
         */
        return -1;
}



static int parse_pass(const char *file, int line, int level)
{
        return parse_signature(file, line, level, NULL);
}



static int parse_log(const char *file, int line, int level) 
{
        /* 
         * not implemented
         * i dont know if it can be implemented :-)
         */
        n_ignored++;

        log(LOG_INFO, "%s (%d) Log rule not implemented yet\n", file, line);

        return -1;
}



static int parse_dynamic(const char *file, int line, int level) 
{
        /*
         * not implemented and this is a special log rule, so...
         */
        n_ignored++;

        log(LOG_INFO, "%s (%d) Dynamic rule not implemented yet.\n", file, line);
        
        return -1;
}



static int parse_alert(const char *file, int line, int level) 
{
	return parse_signature(file, line, level, signature_matched_cb);
}



static const char *skip_space(const char *string) 
{
        if ( ! string )
                return NULL;

        while ( *string == ' ' )
                string++;

        return string;
}



static int parse_reference(const char *file, int line, int level) 
{
        const char *name, *url;

        name = strtok(NULL, " ");
        if ( ! name ) {
                log(LOG_INFO, "%s (%d) Couldn't get reference name.\n", file, line);
                return -1;
        }

        url = strtok(NULL, " ");
        if ( ! url ) {
                log(LOG_INFO, "%s (%d) Couldn't get reference url.\n", file, line);
                return -1;
        }
        
        return add_reference(skip_space(name), skip_space(url));
}



static int parse_class(const char *file, int line, int level) 
{
        const char *shortname, *desc, *priority, *completion, *type;
                
        shortname = strtok(NULL, ",");
        if ( ! shortname ) {
                log(LOG_INFO, "%s (%d) Couldn't get classtype short name.\n", file, line);
                return -1;
        }

        desc = strtok(NULL, ",");
        if ( ! desc ) {
                log(LOG_INFO, "%s (%d) Couldn't get classtype descriprion.\n", file, line);
                return -1;
        }

        priority = strtok(NULL, ",");
        if ( ! priority ) {
                log(LOG_INFO, "%s (%d) Couldn't get classtype priority.\n", file, line);
                return -1;
        }

        type = strtok(NULL, ",");
        if ( ! type ) {
                log(LOG_INFO, "%s (%d) Couldn't get classtype type.\n", file, line);
                return -1;
        }
        
        completion = strtok(NULL, ",");
        if ( ! completion ) {
                log(LOG_INFO, "%s (%d) Couldn't get classtype completion.\n", file, line);
                return -1;
        }
        
        return add_classtype(skip_space(shortname), skip_space(desc),
                             skip_space(priority), skip_space(type), skip_space(completion));
}




static int parse_config(const char *file, int line, int level) 
{
        const char *param;

        param = strtok(NULL, " ");
        if ( ! param ) {
                log(LOG_INFO, "%s (%d) Missing config name.\n", file, line);
                return -1;
        }
        
        if ( strcasecmp(param, "classification:") == 0 )
                return parse_class(file, line, level);

        else if ( strcasecmp(param, "reference:") == 0 )
                return parse_reference(file, line, level);
        
        return 0;
}



static int replace_str(char **str, const char *needle, const char *replacement) 
{
        char *ptr, *out;
        int off, new_len, replacement_len, needle_len;
        
        ptr = strstr(*str, needle);
        if ( ! ptr ) {
                log(LOG_ERR, "couldn't find %s!\n", needle);
                return -1;
        }

        needle_len = strlen(needle);
        replacement_len = strlen(replacement);

        /*
         * compute the offset where needle start.
         * (idmef string count \0 in len, that's the reason of the + 1).
         */
        off = strlen(*str) - (strlen(ptr) + 1);
        new_len = strlen(*str) + replacement_len - needle_len;
        
        out = malloc(new_len);
        if ( ! out ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
        
        memcpy(out, *str, off);
        memcpy(out + off, replacement, replacement_len);
        strcpy(out + off + replacement_len, ptr + needle_len);

        free(*str);
        *str = out;
        
        return 0;
}





static int resolve_variable(const char *file, int linenum, char **line) 
{
        int ret;
        char *value, c;
        char outvar[100], *str = *line;
        int i = 0, escaped = 0, is_variable = 0;
        
        while ( (c = *str++) != '\0' ) {

                if ( escaped ) {
                        escaped = 0;
                        continue;
                }
                
                if ( ! is_variable && c == '\\' )
                        escaped = 1;

                else if ( c == '$' && ! escaped ) {
                        is_variable = 1;
                        outvar[i++] = c;
                        continue;
                }

                if ( ! is_variable )
                        continue;

                if ( i >= sizeof(outvar) ) {
                        log(LOG_INFO, "%s:%d: variable name exceed buffer size.\n", file, linenum);
                        is_variable = 0;
                        continue;
                }
                
                if ( isalnum(c) || c == '_' ) 
                        outvar[i++] = c;
                else {
                        is_variable = 0;
                        outvar[i] = '\0';
                        i = 0;

                        value = variable_get(outvar + 1);
                        if ( ! value ) {
                                log(LOG_INFO, "%s:%d: unknow variable '%s'.\n", file, linenum, outvar);
                                return -1;
                        }
                        
                        ret = replace_str(line, outvar + 1, value);
                        if ( ret < 0 )
                                return -1;

                        str = *line;
                }
        }

        return 0;
}




static void parse_line(const char *file, char **str, int linenum, int level)
{
        int i, ret;
        char *cmd, *line = *str;
        parse_hook_t rtbl[] = {
                { "include", parse_include           },
                { "var", parse_var                   },
                { "preprocessor", parse_preprocessor },
                { "output", parse_output             },
                { "pass", parse_pass                 },
                { "log", parse_log                   },
                { "dynamic", parse_dynamic           },
                { "alert", parse_alert               },
                { "config", parse_config             },
                { NULL, NULL                         },
        };
        
        /*
         * find the beginning of the line
         */
        while( (*line == ' ' || *line == '\t') && *line != '\0' )
                line++;

        /*
         * ignore comment or empty line
         */
        if ( *line == '\0' || *line == '#' || *line == ';' || *line == 0x0a )
                return;
        
        /*
         * delete the last space of the line
         */
        i = strlen(line) - 1;
        while ( i > 0 && (line[i] == ' ' || line[i] == 0x0a)) 
                line[i--] = '\0';

        /*
         * this is a dirty hack : only try to resolve variable
         * if this file was not included (default conf file).
         * (ruleset variable are still valid, but handled directly by
         *  the flex/bison parser).
         */
        if ( level == 0 ) {
                ret = resolve_variable(file, linenum, str);
                if ( ret < 0 )
                        return;
        }
        
        /*
         * split the line
         */
        cmd = strtok(*str, " ");
        if ( ! cmd )
                return;
        
        for ( i = 0; rtbl[i].rule != NULL; i++ ) {
                if ( strcasecmp(rtbl[i].rule, cmd) == 0 ) {
                        rtbl[i].handler(file, linenum, level);
                        return;
                }
        }

        log(LOG_INFO, "%s (%d) Unknow command %s.\n", file, linenum, cmd);
}





static int parse_signature_file(const char *file, int level)
{
	FILE *fd;
        int linenum = 0;
        char buf[1024], *line;
        
        fd = fopen(file, "r");
        if ( ! fd ) {		
		log(LOG_ERR, "error opening '%s'.\n", file);
		return -1;
	}
        
	while ( prelude_read_multiline(fd, &linenum, buf, sizeof(buf)) == 0 ) {
                
                line = strdup(buf);

                parse_line(file, &line, linenum, level);

                    //free(line);
	}

	fclose(fd);

        return 0;
}
	


static int load_signature_file(const char *filename)
{
        int ret;

	n_rules = n_ignored = 0;
	ret = parse_signature_file(filename, 0);

        
        log(LOG_INFO, "- Signature engine added %d and ignored %d signature.\n", n_rules, n_ignored);

        return ret;
}




static int set_ruleset(prelude_option_t *opt, const char *arg) 
{
        int ret;
        char *ptr;
        
        rulesetdir = strdup(arg);

        ptr = strrchr(rulesetdir, '/');
        if ( ptr )
                *ptr = '\0';
        else {
                free(rulesetdir);
                rulesetdir = NULL;
        }
        
        init_key_parser();
        
        ret = load_signature_file(arg);
        
        if ( rulesetdir )
                free(rulesetdir);
        
        if ( ret < 0 ) 
                return prelude_option_error;
        
        return prelude_option_success;
}



plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;
        
        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 0, "snortrules",
                                 "Set SnortRules plugin options", no_argument,
                                 NULL, NULL);

        /*
         * Not a wide option for now. We need to ensure we don't leak anything
         * when freeing the whole currently existing tree.
         */
        opt = prelude_option_add(opt, CLI_HOOK|CFG_HOOK, 'r', "ruleset",
                                 "Specify a ruleset to use", required_argument, set_ruleset, NULL);

        /*
         * we want this option to be called last, so that capture
         * variable ($interface_ADDRESS) are already set.
         */
        prelude_option_set_priority(opt, option_run_last);
        
	plugin_set_name(&plugin, "SnortRules");
        plugin_set_author(&plugin, "The Prelude Team");
        plugin_set_contact(&plugin, "prelude-devel@prelude-ids.org");
        plugin_set_desc(&plugin, "Snort signature parser.");
        plugin_set_subscribtion(&plugin, subscribtion);
        
        return (plugin_generic_t *) &plugin;
}      





