/*****
*
* Copyright (C) 2001 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/

#ifndef RULES_TYPE_H
#define RULES_TYPE_H

/*
 * abstract type
 */
typedef void *(*copy_f_t)(void *set);
typedef void (*delete_f_t)(void *set);
typedef int (*equal_f_t)(void *set1, void *set2);
typedef int (*match_f_t)(void *set, void *subset);
typedef int (*intersection_f_t)(void *set1, void *set2);
typedef void (*split_f_t)(void *set1, void *set2, void **inter, void **subset1, void **subset2);
typedef void (*print_f_t)(void *set);



typedef struct {
	int id;
	int priority;
	copy_f_t copy;
	delete_f_t delete;
	equal_f_t equal;
	match_f_t match;
	intersection_f_t intersection;
	split_f_t split;
	match_packet_f_t match_packet;
	print_f_t print;
} generic_type_t; 



/*
 * IP
 */
typedef struct {
	uint32_t ip;
        uint32_t netmask;
} ip_t;


ip_t *copy_ip(ip_t *ip);
void print_ip(ip_t *ip);
void delete_ip(ip_t *ip);
int equal_ip(ip_t *ip1, ip_t *ip2);
int match_ip(ip_t *ip, ip_t *subip);
int intersection_ip(ip_t *ip1, ip_t *ip2);
void split_ip(ip_t *ip1, ip_t *ip2, ip_t**inter, ip_t**r_ip1, ip_t**r_ip2);



/*
 * Segment
 */
typedef struct {
	unsigned int lo;
        unsigned int hi;
} segment_t;



void print_segment(segment_t *segment);
void delete_segment(segment_t *segment);
segment_t *copy_segment(segment_t *segment);
int equal_segment(segment_t *segment1, segment_t *segment2);
int match_segment(segment_t *segment, segment_t *subsegment);
int intersection_segment(segment_t *segment1, segment_t *segment2);
void split_segment(segment_t *segment1, segment_t *segment2,
                   segment_t **inter, segment_t **r_segment1, segment_t **r_segment2);



/*
 * Flags
 */
typedef struct {
	unsigned int mask;
	unsigned int flags;
} flags_t;



void delete_flags(flags_t *flags);
flags_t *copy_flags(flags_t *flags);
int equal_flags(flags_t *flags1, flags_t *flags2);
int match_flags(flags_t *flags, flags_t *subflags);
int intersection_flags(flags_t *flags1, flags_t *flags2);
void split_flags(flags_t *flags1, flags_t *flags2,
                 flags_t **inter, flags_t **r_flags1, flags_t **r_flags2);
void print_flags(flags_t *flags);



/*
 * Integer
 */
typedef struct {
	int num;
} integer_t;



integer_t *copy_integer(integer_t *integer);
void delete_integer(integer_t *integer);
int equal_integer(integer_t *integer1, integer_t *integer2);
int match_integer(integer_t *integer, integer_t *subinteger);
int intersection_integer(integer_t *integer1, integer_t *integer2);
void split_integer(integer_t *integer1, integer_t *integer2,
                   integer_t **inter, integer_t **r_integer1, integer_t **r_integer2);
void print_integer(integer_t *integer);



/*
 * Basic Parsing
 */
ip_t *parse_ip(const char *str);
segment_t *parse_segment(const char *str);
integer_t *parse_integer(const char *str);




/*
 * type registering function
 */
generic_type_t *signature_engine_get_type_by_id(int id);



int signature_engine_register_type(int priority, copy_f_t copy,
                                   delete_f_t delete, equal_f_t equal,
                                   match_f_t match, intersection_f_t intersection,
                                   split_f_t split, print_f_t print,
                                   match_packet_f_t match_packet);


/*
 * This macro suppose existing copy_fun delete_fun etc...
 */
#define REGISTER_TYPE(priority, fun, match_packet)           \
        signature_engine_register_type((priority),           \
                      (copy_f_t) copy_##fun,                 \
                      (delete_f_t) delete_##fun,             \
                      (equal_f_t) equal_##fun,               \
                      (match_f_t) match_##fun,               \
                      (intersection_f_t) intersection_##fun, \
                      (split_f_t) split_##fun,               \
                      (print_f_t) print_##fun,               \
		      (match_packet_f_t) match_packet)


#endif
