/* Copyright (C) 1999-2000 Bernhard Trummer
 * Copyright (C) 2003      Benjamin Drieu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: presentation.cc,v $
 * Revision 1.14  2003/04/24 14:07:10  benj
 * Adding Debian stuff
 * Some update with autotool-dev
 *
 * Revision 1.13  2003/03/06 14:54:29  benj
 * Update copyrigth where it is necessary
 *
 * Revision 1.12  2003/02/26 12:28:33  benj
 * Clean up options a bit
 * Fix a crash when images are not found
 * Command option '-l' now has an optional argument
 * .tex files are now based on xml presentation name
 *
 * Revision 1.11  2003/02/25 15:54:19  benj
 * - Fix prestimel DTD where it had some flaws
 * - Create a <titlenode> environment because these informations has
 *   nothing to do in a slide <environment>
 *
 * Revision 1.10  2003/02/25 14:11:20  benj
 * Changed a little the writeTOC mechanism
 *
 * Revision 1.9  2003/02/25 13:27:55  benj
 * Table of contents is now much more beautiful
 *
 * Revision 1.8  2003/02/24 17:05:26  benj
 * ChangeLog is now a real changelog
 * Beautify TOC
 *
 * Revision 1.7  2003/02/18 13:17:31  slash
 * Applied Benjamin's JavaScript patch.
 * Applied some g++-3.2 fixes (mainly missing std::'s).
 *
 * Revision 1.6  2003/02/18 12:56:50  bdrieu
 * - Update to make prestimel compile with g++-3.2
 * - Use Magick++ instead of Magick
 *
 * Revision 1.5  2002/05/02 09:32:32  slash
 * Changes to make PresTiMeL compilable with g++ 3.0.
 * (Thanks to Benjamin Drieu)
 *
 * Revision 1.4  2002/01/21 09:31:31  slash
 * The HTML-output is now declared as utf-8 charset.
 *
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:34  slash
 * Initial release.
 *
 */

#include <stdio.h>

#include <libintl.h>
#define _(String) gettext(String)

#include <string>
#include <fstream>

#include <libxml/parser.h>

#include "main.h"
#include "commentnode.h"
#include "htmlnode.h"
#include "latexnode.h"
#include "titlenode.h"
#include "slidenode.h"
#include "sectionnode.h"
#include "parameters.h"
#include "presentation.h"


//---------------------------------------------------------------------------
Presentation::Presentation() : Node()
{
    xml_tree_ = NULL;
    title = NULL;
    toc_nodes_.clear();
}

//---------------------------------------------------------------------------
Presentation::~Presentation()
{
    toc_nodes_.clear();
}


//---------------------------------------------------------------------------
int Presentation::parseXMLFile()
{
    std::ifstream xml_file(Parameters_.xml_file_.c_str());

    if (!xml_file) {
        ENDL;
        printf(_("The XML-file \"%s\" doesn't exist"),
               Parameters_.xml_file_.c_str());
        ENDL;
        return -1;
    }

    xml_tree_ = xmlParseFile(Parameters_.xml_file_.c_str());
    if (xml_tree_ == NULL) {
        return -1;
    }

    // Find the XML-node, where the presentation starts
    xmlNodePtr node = xml_tree_->xmlRootNode;
    while ((node->type != XML_ELEMENT_NODE) ||
           strcmp((char*)node->name, "presentation")) {
        node = node->next;
    }

    // Extract the child-nodes.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        if (!strcmp((char*)node->name, "html")) {
            child_nodes_.push_back(new HTMLNode(node));

        } else if (!strcmp((char*)node->name, "latex")) {
            child_nodes_.push_back(new LaTeXNode(node));

        } else if (!strcmp((char*)node->name, "titleslide")) {
            TitleNode *title = new TitleNode(node);
            child_nodes_.push_back(title);
            toc_nodes_.push_back((TOCInterface*)title);

        } else if (!strcmp((char*)node->name, "slide")) {
            SlideNode *slide = new SlideNode(node);
            child_nodes_.push_back(slide);
            toc_nodes_.push_back((TOCInterface*)slide);

        } else if (!strcmp((char*)node->name, "section")) {
            SectionNode *section = new SectionNode(node);
            child_nodes_.push_back(section);
            toc_nodes_.push_back((TOCInterface*)section);

        } else if (node->type == XML_COMMENT_NODE) {
            child_nodes_.push_back(new CommentNode(node));

        } else {
            Number_Of_Errors_++;
        }
    }

    SlideNode::initFilenameOfFirstAndLastSlide();

    return 0;
}


//---------------------------------------------------------------------------
void Presentation::writeHTML(int parameter) const
{
    copyThemeFiles();

    // Write the table of contents.
    printf(_("Creating the table of contents ... "));
    writeTOC();
    printf(_("finished."));
    ENDL;

    // Write the slides.
    printf(_("Creating %d slides ... "), SlideNode::All_Slides_.size());
    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeHTML();
    }
    printf(_("finished."));
    ENDL;

    // Write misc files.
    printf(_("Creating misc files ... "));
    writeMisc();
    printf(_("finished."));
    ENDL;
}


//---------------------------------------------------------------------------
void Presentation::writeLaTeX() const
{
    // Create the LaTeX-file.

    // Write the LaTeX-header.
    if (Parameters_.isOutlineEnabled()) {
      Output_.open(Parameters_.latex_outline_file_.data());
        Output_ << "\\documentclass[12pt]{article}" << std::endl
                << "\\usepackage{graphicx}" << std::endl
//              << "\\usepackage[pdftex]{graphicx}" << std::endl
                << "\\usepackage{tabularx}" << std::endl
                << "\\usepackage{fancyheadings}" << std::endl
                << "\\usepackage{latexsym}" << std::endl
                << "\\usepackage{isolatin1}" << std::endl
                << "\\usepackage{url}" << std::endl
//              << "\\usepackage[pdftex,plainpages=false]{hyperref}" << std::endl
                << std::endl;

        Output_ << "\\renewcommand{\\rmdefault}{phv}" << std::endl
                << "\\renewcommand{\\sfdefault}{phv}" << std::endl
                << "\\renewcommand{\\ttdefault}{pcr}" << std::endl << std::endl;

        Output_ << "\\newcommand{\\PreserveBackslash}[1]{\\let\\temp=\\\\#1\\let\\\\=\\temp}" << std::endl
                << "\\let\\PBS=\\PreserveBackslash" << std::endl
<< "\\newcolumntype{L}{>{\\PBS\\raggedright\\hspace{0pt}}l}" << std::endl
                << "\\newcolumntype{R}{>{\\PBS\\raggedright\\hspace{0pt}}r}" << std::endl
                << "\\newcolumntype{C}{>{\\PBS\\raggedright\\hspace{0pt}}c}" << std::endl
                << "\\newcolumntype{Y}{>{\\PBS\\raggedright\\hspace{0pt}}X}" << std::endl << std::endl;

        Output_ << "\\textwidth=6.5in" << std::endl
                << "\\textheight=8.5in " << std::endl
                << "\\oddsidemargin=-0.00in" << std::endl
                << "\\evensidemargin=-0.00in" << std::endl
                << "\\topmargin=-0.25in" << std::endl
                << "\\parskip=0.15in" << std::endl
                << "\\parindent=0in" << std::endl
                << "\\pagenumbering{arabic}" << std::endl;

    } else {
        Output_.open(Parameters_.latex_file_.data());
        Output_ << "\\documentclass{seminar}" << std::endl
                << "\\usepackage{slidesec}" << std::endl
                << "\\usepackage{semcolor}" << std::endl
                << "\\usepackage{graphicx}" << std::endl
                << "\\usepackage[latin1]{inputenc}" << std::endl
                << "\\usepackage[T1]{fontenc}" << std::endl
                << "\\usepackage{latexsym}" << std::endl
                << "\\usepackage{url}" << std::endl;
//      if (Parameters_.output_format_ == PDFLATEX) {
//          Output_ << "\\usepackage{ae,aecompl}" << std::endl;
//      }
    }
    

    Output_ << std::endl << "\\begin{document}" << std::endl << std::endl;

    // Write the content of all slides to the LaTeX-file.
    for (unsigned int i=0; i<child_nodes_.size();i++) {
        child_nodes_[i]->writeLaTeX();
    }

    // Write the LaTeX-footer.
    Output_ << "\\end{document}" << std::endl;
    Output_.close();
}

//---------------------------------------------------------------------------
void Presentation::writeXML() const
{
    Output_.open("presentation.xml");

    Output_ << "<?xml version=\"1.0\"?>" << std::endl
            << "<!DOCTYPE presentation SYSTEM \"/usr/local/share/prestimel/prestimel.dtd\">" << std::endl;

    Output_ << "<presentation>" << std::endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</presentation>" << std::endl;

    Output_.close();
}


//---------------------------------------------------------------------------
void Presentation::copyThemeFiles() const
{
    std::string source, destination;

    // Copy the next-button to the target-directory.
    if (Parameters_.next_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_NEXT;
        destination = BUTTON_NEXT;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the prev-button to the target-directory.
    if (Parameters_.prev_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_PREV;
        destination = BUTTON_PREV;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the first-button to the target-directory.
    if (Parameters_.first_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_FIRST;
        destination = BUTTON_FIRST;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the last-button to the target-directory.
    if (Parameters_.last_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_LAST;
        destination = BUTTON_LAST;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the note-button to the target-directory.
    if (Parameters_.note_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_NOTE;
        destination = BUTTON_NOTE;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the stylesheet to the target-directory.
    source = Parameters_.full_path_to_theme_ + STYLES;
    switch (Parameters_.getResolution()) {
      case 640:  source += "_640"; break;
      case 800:  source += "_800"; break;
      case 1024: source += "_1024"; break;
    }
    source += STYLES_EXT;
    destination = std::string(STYLES) + STYLES_EXT;
    fcopy(source.c_str(), destination.c_str());

    // Copy the background-image for the slide to the target-directory.
    if (Parameters_.isSlideBackgroundEnabled()) {
        source = Parameters_.full_path_to_theme_ + BG_SLIDE + JPG_EXT;
        destination = std::string(BG_SLIDE) + JPG_EXT;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the background-graphic for the header to the target-directory.
    if (Parameters_.isHeaderBackgroundEnabled()) {
        source = Parameters_.full_path_to_theme_ + BG_HEADER;
        switch (Parameters_.getResolution()) {
          case 640:  source += "_640"; break;
          case 800:  source += "_800"; break;
          case 1024: source += "_1024"; break;
        }
        source += PNG_EXT;
        destination = std::string(BG_HEADER) + PNG_EXT;
        fcopy(source.c_str(), destination.c_str());
    }
}


//---------------------------------------------------------------------------
void Presentation::writeTOC() const
{
    unsigned int i=0;
  
    Output_.open("toc.html");

    // First, write the HTML-header.
    Output_ << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">" << std::endl
            << "<HTML>" << std::endl
            << "<HEAD>" << std::endl
            << "<META"
            << " http-equiv=\"Content-Type\""
            << " content=\"text/html; charset=utf-8\""
            << ">" << std::endl
            << "<META"            << " name=\"Generator\""
            << " content=\"PresTiMeL " << VERSION << "\""
            << ">" << std::endl
            << "<TITLE>" << Presentation_.getTitle() << " : "
	    << _("Table of contents") << "</TITLE>" << std::endl
	    << "<LINK"
	    << " href=\"styles.css\""
	    << " rel=\"stylesheet\""
	    << " type=\"text/css\""
	    << ">" << std::endl
            << "</HEAD>" << std::endl
            << "<BODY class=\"graphic\">" << std::endl
            << "<H2>" << Presentation_.getTitle() << "</H2>" << std::endl
            << "<H3>" << _("Table of contents") << "</H3>" << std::endl;

    for (i=0; i<toc_nodes_.size(); i++) {
        toc_nodes_[i]->writeTOCEntry();
    }

    Output_ << "</BODY>" << std::endl
            << "</HTML>" << std::endl;

    Output_.close();
}

//---------------------------------------------------------------------------
void Presentation::writeMisc() const
{
    Output_.open("managekeys.js");

    Output_ << "// Some variables to autodetect browser type" << std::endl;
    Output_ << "var ns=(document.layers);" << std::endl;
    Output_ << "var ie=(document.all);" << std::endl;
    Output_ << "var w3=(document.getElementById && !ie);" << std::endl;
    Output_ << "" << std::endl;
    Output_ << "// Return an object" << std::endl;
    Output_ << "function get_object(id)" << std::endl;
    Output_ << "{ " << std::endl;
    Output_ << " if(!ns && !ie && !w3) " << std::endl;
    Output_ << "   return null;" << std::endl;
    Output_ << " if (ie)" << std::endl;
    Output_ << "   e=eval(\"document.all.\" + id);" << std::endl;
    Output_ << " else if (ns)" << std::endl;
    Output_ << "   e=eval('document.links[id]');" << std::endl;
    Output_ << " else if (w3)" << std::endl;
    Output_ << "   e=eval('document.getElementById(id)');" << std::endl;
    Output_ << " return e;" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "// Change the current page to the id found in the page" << std::endl;
    Output_ << "function jumpto(id)" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  e = get_object(id);" << std::endl;
    Output_ << "  if ((e != null) && (e.href != null)) {" << std::endl;
    Output_ << "      location.href = e.href;" << std::endl;
    Output_ << "  }" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "" << std::endl;
    Output_ << "// Change the current page" << std::endl;
    Output_ << "function next_page()" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  jumpto('next_slide');" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "" << std::endl;
    Output_ << "function previous_page()" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  jumpto('previous_slide');" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "" << std::endl;
    Output_ << "// Event Handler that receive Key Event" << std::endl;
    Output_ << "function getkey(e)" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  if (e == null)" << std::endl;
    Output_ << "   { // IE" << std::endl;
    Output_ << "     kcode = window.event.keyCode;" << std::endl;
    Output_ << "   } " << std::endl;
    Output_ << "  else" << std::endl;
    Output_ << "   { // Mozilla" << std::endl;
    Output_ << "     kcode = e.which;" << std::endl;
    Output_ << "   }" << std::endl;
    Output_ << "  key = String.fromCharCode(kcode).toLowerCase();" << std::endl;
    Output_ << "  switch(key)" << std::endl;
    Output_ << "   {" << std::endl;
    Output_ << "     case \"n\":" << std::endl;
    Output_ << "     case \" \":" << std::endl;
    Output_ << "       next_page();" << std::endl;
    Output_ << "       return false;" << std::endl;
    Output_ << "     case \"p\":" << std::endl;
    Output_ << "       previous_page();" << std::endl;
    Output_ << "       return false;" << std::endl;
    Output_ << "   }" << std::endl;
    Output_ << "  switch(kcode)" << std::endl;
    Output_ << "   {" << std::endl;
    Output_ << "     case 8:" << std::endl;
    Output_ << "       previous_page();" << std::endl;
    Output_ << "       return false;" << std::endl;
    Output_ << "   }" << std::endl;
    Output_ << "  return true;" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "" << std::endl;
    Output_ << "if(w3 || ie)" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  document.onkeypress = getkey;" << std::endl;
    Output_ << "} " << std::endl;
    Output_ << "else" << std::endl;
    Output_ << "{" << std::endl;
    Output_ << "  document.captureEvents(Event.KEYUP);" << std::endl;
    Output_ << "  document.onkeyup = getkey;" << std::endl;
    Output_ << "  document.captureEvents(Event.KEYPRESS);" << std::endl;
    Output_ << "  document.onkeypress = getkey;" << std::endl;
    Output_ << "}" << std::endl;
    Output_ << "" << std::endl;
    Output_.close();
}

