/***************************************************************************
                          database_mysql.cpp  -  description
                             -------------------
    begin                : Thu May 3 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "database_mysql.h"

#include <stdlib.h>
#include <stdio.h>

#include <sstream>

#include <iostream>
#include <qmessagebox.h>


using namespace std;

// *** constructor ***
DataBase_MySQL::DataBase_MySQL() {
	db = QSqlDatabase::addDatabase("QMYSQL3");
	isConnected = false;
	error = 0;
}

// ##############################################
// # establishes connection to database
// ##############################################
void DataBase_MySQL::establishConnection(QString host, QString dbname, QString user, QString pass)
{
    if(!QSqlDatabase::contains()) {
	error = 1;
	isConnected = false;
	cout << _("Qt3 MYSQL plugin not available") << endl;
	return; 
    }

    error = 0;
    if(db->isOpen()) db->close();
    db->setDatabaseName(dbname);
    db->setUserName(user);
    db->setPassword(pass);
    db->setHostName(host);
    if(db->open()) {
	isConnected = true;
    	QSqlQuery query("SHOW tables;", db);
    	if(query.isActive()) {
	    bool artists_exists = false, mediums_exists = false, tracks_exists = false;
	    bool playlists_exists = false, playlist_tracks_exists = false, version_history_exists = false;
	    bool q1 = true, q2 = true, q3 = true;
	    bool q4 = true, q5 = true, q6 = true;
	    while(query.next()) {
		if(query.value(0).toString() == "artists") artists_exists = true;
	    else if(query.value(0).toString() == "mediums") mediums_exists = true;
	    else if(query.value(0).toString() == "tracks") tracks_exists = true;
		else if(query.value(0).toString() == "playlists") playlists_exists = true;
		else if(query.value(0).toString() == "playlist_tracks") playlist_tracks_exists = true;
	    else if(query.value(0).toString() == "version_history") version_history_exists = true;
	    }
	    if(!artists_exists) {
	    	q1 = query.exec("CREATE TABLE artists (id int(11) NOT NULL default '0', name varchar(255) default NULL, total int(11) default NULL, local int(11) default NULL, isFavourite tinyint(4) default NULL, PRIMARY KEY (id)) TYPE=MyISAM;");
	    } else if ( !checkIndex( "artists", "name_index" ) )  // creates index if missing  
	                 query.exec("ALTER TABLE artists ADD INDEX name_index (name);");
	    if(!mediums_exists) {
		q2 = query.exec("CREATE TABLE mediums (id int(11) NOT NULL default '0', type int(11) default NULL, label varchar(255) default NULL, path varchar(255) default NULL, checksum int(11) default NULL, PRIMARY KEY (id)) TYPE=MyISAM;");
	    }
	    if(!tracks_exists) {
		q3 = query.exec("CREATE TABLE tracks (id int(11) NOT NULL default '0', path varchar(255) default NULL, filename varchar(255) default NULL, medium int(11) default NULL, lastModified datetime default NULL, hasChanged tinyint(4) default NULL, mimetype int(11) default NULL, version int(11) default NULL, layer int(11) default NULL, mode int(11) default NULL, bitrate int(11) default NULL, samplerate int(11) default NULL, length int(11) default NULL, size int(11) default NULL, artist varchar(255) default NULL, title varchar(255) default NULL, album varchar(255) default NULL, tracknumber int(11) default NULL, year varchar(255) default NULL, genre int(11) default NULL, comment varchar(255) default NULL, PRIMARY KEY (id), INDEX artist_index (artist)) TYPE=MyISAM;");
	    } else if ( !checkIndex( "tracks", "artist_index" ) )  // creates index if missing  
	                 query.exec("ALTER TABLE tracks ADD INDEX artist_index (artist);");
	    if(!playlists_exists) {
		q4 = query.exec("CREATE TABLE playlists (id int(11) NOT NULL default '0', name varchar(255) default NULL, PRIMARY KEY (id)) TYPE=MyISAM;");
	    }
	    if(!playlist_tracks_exists) {
		q5 = query.exec("CREATE TABLE playlist_tracks (id int(11) NOT NULL auto_increment, track_id int(11) default NULL, playlist_id int(11) default NULL, position_in_playlist int(11) default NULL, PRIMARY KEY(id)) TYPE=MyISAM;");
	    }
	    if(!version_history_exists) {  // table created before 0.9.2
	      q6 = query.exec("CREATE TABLE version_history (version varchar(255) NOT NULL, date datetime NOT NULL) TYPE=MyISAM;");
	    }
        
        if (!checkVersion()) {
            QString sqlquery;
            sqlquery.sprintf("INSERT INTO version_history (version, date) VALUES ('%s', NOW());", VERSION);
            query.exec(sqlquery);
        }

        if(!checkExtra("playlist_tracks", "id", "auto_increment")) {
          query.exec("ALTER TABLE playlist_tracks CHANGE id id INT(11) NOT NULL AUTO_INCREMENT;");
        }
        
	    if(q1 && q2 && q3 && q4 && q5 && q6) {
		maxID_artists         = getMaxIDFromTable("artists");
		maxID_tracks          = getMaxIDFromTable("tracks");
		maxID_mediums         = getMaxIDFromTable("mediums");
		maxID_playlists       = getMaxIDFromTable("playlists");
		// maxID_playlist_tracks = getMaxIDFromTable("playlist_tracks");
		deltamap = new QMap<int, QMap<int, DELTAINFO*> >;
	    } else {
		error = 1;
		isConnected = false;
	    }
	} else {
	    error = 1;
	    isConnected = false;
	}
    } else {
	error = 1;
	isConnected = false;
    }

    if(error) cout << _("ERROR CONNECTION") << endl;
}

// ##############################################
// # creates new database
// ##############################################
QString DataBase_MySQL::createNewDatabase(QString host, QString dbname, QString user, QString pass, QString su, QString su_pass)
{
    // TODO: replace localhost by real hostname or IP if database
    // is located on another host.
    QSqlDatabase *dbase = QSqlDatabase::addDatabase("QMYSQL3", "newdb");
    dbase->setHostName(host);
    dbase->setDatabaseName("mysql");
    dbase->setUserName(su);
    dbase->setPassword(su_pass);
    if(dbase->open())
    {
	   QSqlQuery query("CREATE DATABASE " + dbname + ";", dbase);
       query.exec("GRANT USAGE ON * TO " + user + "@localhost IDENTIFIED BY '" + pass + "' ;");
       query.exec("GRANT ALTER, CREATE, DELETE, INDEX, INSERT, SELECT, UPDATE ON " + dbname + ".* TO " + user + "@localhost;");
	   query.exec("FLUSH PRIVILEGES;");
	   dbase->close();
	   QSqlDatabase::removeDatabase("newdb");
	   if(query.lastError().type() == QSqlError::None) return "";
	   else return query.lastError().databaseText();
    }
    else
    {
	   QSqlDatabase::removeDatabase("newdb");
	   return _("Cannot connect to server (maybe wrong user/pass ?!)");
    }
}


// ##############################################
// # Get the base sampler list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getSamplerBasis()
{
    return getArtistAlbumBasisByQuery("SELECT comment,album,count(*),count(NULLIF(medium=0,0)),0,0,medium FROM tracks WHERE comment='Sampler' OR comment='Soundtrack' GROUP BY comment, BINARY album;");
}

// ##############################################
// # Get the base favourites list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getFavouritesBasis()
{
    return getArtistAlbumBasisByQuery("SELECT artist,album,count(*),count(NULLIF(medium=0,0)),isFavourite,artists.id,medium FROM tracks,artists WHERE isFavourite=1 AND BINARY artist=name GROUP BY BINARY artist, BINARY album;");
}

// ##############################################
// # Get the base artist and album list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getArtistAlbumBasis()
{
    return getArtistAlbumBasisByQuery("SELECT artist,album,count(*),count(NULLIF(medium=0,0)),isFavourite,artists.id,medium FROM tracks,artists WHERE BINARY artist=name GROUP by BINARY artist, BINARY album;");
}

QList<ARTISTALBUM> *DataBase_MySQL::getArtistAlbumBasisByQuery(QString sql)
{
    QList<ARTISTALBUM>  *list;

    QSqlQuery query(sql, db);

    if(query.isActive())
    {
        list = new QList<ARTISTALBUM>;
        while(query.next())
        {
            ARTISTALBUM *item = new ARTISTALBUM();
            item->artist      = query.value(0).toString();
            item->album       = query.value(1).toString();
            item->total       = query.value(2).toInt();
            item->local       = query.value(3).toInt();
            item->isFavourite = query.value(4).toBool();
            item->artistId    = query.value(5).toInt();
            item->medium      = query.value(6).toInt();
            list->append(item);
        }
        return list;
    }

    error = 1;
    return 0;
}

// ##############################################
// # Get the base medium list
// ##############################################
QList<MEDIUM> *DataBase_MySQL::getMediumBasis()
{
  QList<MEDIUM>	*list;
  
  QSqlQuery query("SELECT id, type, label, path, checksum FROM mediums;", db);
  
  if(query.isActive()) {
    list = new QList<MEDIUM>;
    MEDIUM *item = new MEDIUM();
    item->id = 0;
    item->type = MEDIUM_HARDDISK;
    item->label = "Harddisk";
    item->path = "";
    item->checksum = 0;
    list->append(item);
    while(query.next()) {
      MEDIUM *item      = new MEDIUM();
      item->id          = query.value(0).toInt();
      item->type        = query.value(1).toInt();
      item->label       = query.value(2).toString();
      item->path        = query.value(3).toString();
      item->checksum    = query.value(4).toInt();
      list->append(item);
    }
    return list;
  } else {
    error = 1;
    return 0;
  }
}

// ##############################################
// # Get the base playlist list
// ##############################################
QList<PLAYLIST> *DataBase_MySQL::getPlaylistBasis()
{
  QList<PLAYLIST> *list;
  
  QSqlQuery query("SELECT id, name FROM playlists ORDER BY name;", db);
  
  if(query.isActive()) {
    list = new QList<PLAYLIST>;
    while(query.next()) {
      PLAYLIST *item = new PLAYLIST();
      item->id       = query.value(0).toInt();
      item->name     = query.value(1).toString();
      list->append(item);
    }
    return list;
  } else {
    error = 1;
    return 0;
  }
}

// ##############################################
// # Get the base playlist tracks list
// ##############################################
QList<PLAYLIST_TRACK> *DataBase_MySQL::getPlaylistTracksBasis(int playlistID)
{
  QList<PLAYLIST_TRACK> *list;

  QSqlQuery query;
  stringstream sqlquery;
  sqlquery << "SELECT t1.id, t1.track_id, t2.medium, t1.playlist_id, t1.position_in_playlist, t2.artist, t2.title, t2.path, t2.filename"
	  	   << " FROM playlist_tracks AS t1, tracks AS t2"
		   << " WHERE t1.playlist_id = " << playlistID << " AND t1.track_id = t2.id"
		   << " ORDER BY t1.position_in_playlist;";
  query.exec(sqlquery.str().c_str());
  if(query.isActive()) {
    list = new QList<PLAYLIST_TRACK>;
    while(query.next()) {
      PLAYLIST_TRACK *item = new PLAYLIST_TRACK();
      item->id                   = query.value(0).toInt();
      item->track_id             = query.value(1).toInt();
      item->medium_id		 = query.value(2).toInt();
      item->playlist_id          = query.value(3).toInt();
      item->position_in_playlist = query.value(4).toInt();
      item->artist               = query.value(5).toString();
      item->title                = query.value(6).toString();
      item->path                 = query.value(7).toString();
      item->filename             = query.value(8).toString();
      list->append(item);
    }
    return list;
  } else {
    error = 1;
    return 0;
  }
}

// ##############################################
// # Generate a playlist
// ##############################################
void DataBase_MySQL::generatePlaylist(int playlistID, int numberTracks, int minTrackLength, int maxTrackLength, bool onlyFavouriteArtist)
{
  QSqlQuery query;
  stringstream sqlquery;
  sqlquery << "INSERT INTO playlist_tracks (track_id,playlist_id,position_in_playlist) "
           << "SELECT DISTINCT tracks.id," << playlistID << ",0 "
           << "FROM tracks "
           << ((onlyFavouriteArtist) ? "LEFT JOIN artists ON artist=name " : "")
           << "WHERE medium=0 "
           << ((onlyFavouriteArtist) ? "AND isFavourite=1 " : "")
           << "AND length BETWEEN " << minTrackLength << " AND " << maxTrackLength << " "
           << "ORDER BY RAND() "
           << "LIMIT 0," << numberTracks << ";";
  query.exec(sqlquery.str().c_str());
  if(!query.isActive()) {
    error = 1;
  }
}

// ##############################################
// # Test presence of an index
// ##############################################
bool DataBase_MySQL::checkIndex(const QString& table, const QString& indexname)
{
  QString sqlquery;
  sqlquery.sprintf("SHOW INDEX FROM %s", table.latin1());
  QSqlQuery query(sqlquery, db);

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(2).toString() == indexname) return true;
    }
    return false;
  } else {
    error = 1;
    return false;
  }
}

// ##############################################
// # Test presence of an Extra feature of a column of a table
// ##############################################
bool DataBase_MySQL::checkExtra(const QString& table, const QString& column, const QString& extra)
{
  QString sqlquery;
  sqlquery.sprintf("DESC %s", table.latin1());
  QSqlQuery query(sqlquery, db);

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(0).toString() == column &&
          query.value(5).toString() == extra) return true;
    }
    return false;
  } else {
    error = 1;
    return false;
  }
}

// ##############################################
// # Check if database layout fits current version
// ##############################################
bool DataBase_MySQL::checkVersion()
{
  QString sqlquery;
  sqlquery.sprintf("SELECT date FROM version_history WHERE version='%s'", VERSION);
  QSqlQuery query(sqlquery, db);

  if(query.isActive()) {
    if(query.next()) {
      return true;
    }
    return false;
  } else {
    error = 1;
    return false;
  }
}

// ##############################################
// # Returns the deltaMap and creates a new one
// ##############################################
QMap<int, QMap<int, DELTAINFO*> > *DataBase_MySQL::getArtistDelta()
{
  QMap<int, QMap<int, DELTAINFO*> > *t = deltamap;	
  deltamap = new QMap<int, QMap<int, DELTAINFO*> >;
  return t;
}

// ##############################################
// # Returns the deltaMap for non local media
// ##############################################
QMap<int, QMap<int, int> > *DataBase_MySQL::getLocalDeltaBasis()
{
  QMap<int, QMap<int, int> > *deltalocal = new QMap<int, QMap<int, int> >;
 
  QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
  
  if(query.isActive()) {
    while(query.next()) {
      int medium = query.value(0).toInt();
      QSqlQuery subquery("SELECT artist FROM tracks WHERE medium = " + query.value(0).toString() + ";", db);
      if(subquery.isActive()) {
		while(subquery.next()) {
	  	  if((*deltalocal)[medium].contains(artistToID[subquery.value(0).toString()])) (*deltalocal)[medium][artistToID[subquery.value(0).toString()]]++;
	  	  else (*deltalocal)[medium][artistToID[subquery.value(0).toString()]] = 1;
		}
      } else {
		error = 1;
		return 0;
      }
    }
  } else {
    error = 1;
    return 0;
  }
  return deltalocal;
}

// PHF begin
QMap<int, QMap<QString, int> > *DataBase_MySQL::getLocalAlbumDeltaBasis()
{
  QMap<int, QMap<QString, int> > *deltalocal = new QMap<int, QMap<QString, int> >;
 
  QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
  
  if(query.isActive()) {
    while(query.next()) {
      int medium = query.value(0).toInt();
      QSqlQuery subquery("SELECT artist,album FROM tracks WHERE medium = " + query.value(0).toString() + ";", db);
      if(subquery.isActive()) {
		while(subquery.next()) {
	  	  if((*deltalocal)[medium].contains(subquery.value(0).toString()+'\0'+subquery.value(1).toString())) 
		    (*deltalocal)[medium][subquery.value(0).toString()+'\0'+subquery.value(1).toString()]++;
	  	  else (*deltalocal)[medium][subquery.value(0).toString()+'\0'+subquery.value(1).toString()] = 1;
	  	  if((*deltalocal)[medium].contains(subquery.value(0).toString())) 
		    (*deltalocal)[medium][subquery.value(0).toString()]++;
	  	  else (*deltalocal)[medium][subquery.value(0).toString()] = 1;
		}
      } else {
		error = 1;
		return 0;
      }
    }
  } else {
    error = 1;
    return 0;
  }
  return deltalocal;
}
// PHF end

// ##############################################
// # Returns the deltaMap for one non-local media
// ##############################################
QMap<int, int> DataBase_MySQL::getLocalDelta(int id)
{
  QMap<int, int> delta;

  QString sqlquery;
  sqlquery.sprintf("SELECT artist FROM tracks WHERE medium = %d;", id);
	QSqlQuery query(sqlquery, db);

  if(query.isActive()) {
    while(query.next()) {
      if(delta.contains(artistToID[query.value(0).toString()])) delta[artistToID[query.value(0).toString()]]++;
      else delta[artistToID[query.value(0).toString()]] = 1;
    }
  } else error = 1;
  return delta;
}

// PHF begin
QMap<QString, int> DataBase_MySQL::getLocalAlbumDelta(int id)
{
  QMap<QString, int> delta;

  QString sqlquery;
  sqlquery.sprintf("SELECT artist,album FROM tracks WHERE medium = %d;", id);
	QSqlQuery query(sqlquery, db);

  if(query.isActive()) {
    while(query.next()) {
      if(delta.contains(query.value(0).toString()+'\0'+query.value(1).toString())) 
	delta[query.value(0).toString()+'\0'+query.value(1).toString()]++;
      else delta[query.value(0).toString()+'\0'+query.value(1).toString()] = 1;
      if(delta.contains(query.value(0).toString())) 
	delta[query.value(0).toString()]++;
      else delta[query.value(0).toString()] = 1;
    }
  } else error = 1;
  return delta;
}
// PHF ends

// ##############################################
// # append tracks to database
// ##############################################
int DataBase_MySQL::appendTracks(QList<TRACK> *tracklist, int mediumtype, QString mediumlabel, QString mediumpath, unsigned mediumchecksum, bool createNewMedium, int mediumID)
{
  int incrLocal, medium, track;
  
	QSqlQuery query;
  
  switch(mediumtype) {
  case MEDIUM_HARDDISK:
    incrLocal = 1;
    medium    = 0;
    break;
  case MEDIUM_CDROM:
  case MEDIUM_SMB:
  case MEDIUM_NFS:
  case MEDIUM_EXTERN:
    incrLocal = 0;
    if(createNewMedium) {
      medium    = ++maxID_mediums;
      stringstream sqlquery;
      sqlquery << "INSERT INTO mediums (id, type, label, path, checksum) VALUES (\""
	       << medium << "\", \""
	       << mediumtype << "\", \""
	       << adjustString(mediumlabel) << "\", \""
	       << adjustString(mediumpath) << "\", \""
	       << mediumchecksum << "\");";
	  query.exec(sqlquery.str().c_str());
      if(!query.isActive()) {
	error = 1;
	return 0;
      }
    } else {
      medium = mediumID;
    }
  }
  
  
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    track = ++maxID_tracks;
    stringstream sqlquery;
    sqlquery << "INSERT INTO tracks (id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment) VALUES (\""
	     << track << "\", \""
	     << adjustString(curr->path) << "\", \""
	     << adjustString(curr->filename) << "\", \""
	     << medium << "\", \""
	     << dateTimeToString(curr->lastModified) << "\", \""
	     << (curr->hasChanged ? 1 : 0) << "\", \""
	     << curr->mimetype << "\", \""
	     << curr->version << "\", \""
	     << curr->layer << "\", \""
	     << curr->mode << "\", \""
	     << curr->bitrate << "\", \""
	     << curr->samplerate << "\", \""
	     << curr->length << "\", \""
	     << curr->size << "\", \""	
	     << adjustString(curr->artist) << "\", \""
	     << adjustString(curr->title) << "\", \""
	     << adjustString(curr->album) << "\", \""
	     << curr->tracknumber << "\", \""
	     << adjustString(curr->year) << "\", \""
	     << curr->genre << "\", \""
	     << adjustString(curr->comment) << "\");";
	query.exec(sqlquery.str().c_str());
    if(!query.isActive()) {
      error = 1;
      return 0;
    }
    
    int artist;
    stringstream sqlquery2;
    if(!artistToID.contains(curr->artist)) {
      artist = ++maxID_artists;
      artistToID[curr->artist] = artist;
      baselist[artist] = 1;
      sqlquery2 << "INSERT INTO artists (id, name, total, local, isFavourite) VALUES (\""
		<< artist << "\", \"" << adjustString(curr->artist) << "\", \"1\", \"" << incrLocal << "\", \"0\");";
    } else {
      artist = artistToID[curr->artist];
      baselist[artist]++;
      sqlquery2 << "UPDATE artists SET total=total+1, local=local+" << incrLocal << " WHERE id = \"" << artist << "\";";
    }
    query.exec(sqlquery2.str().c_str());
    if(!query.isActive()) {
      error = 1;
      return 0;
    }

    if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(medium)) (*deltamap)[artist][medium] = new DELTAINFO(1, incrLocal, curr->artist);
    else {
      (*deltamap)[artist][medium]->total++;
      if(incrLocal != 0) (*deltamap)[artist][medium]->local++;
    }

    // PHF starts here

    QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artist][medium]->deltaalbum;
    if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
    else {
      (*deltaalbum)[curr->album].total++;
      if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
    };
  }
  if ( verbose == 5 )  {
     qWarning( "DataBase_MySQL::appendTracks" );  
     dumpdeltamap( deltamap );  
  }
  return medium;
}


// ##############################################
// # update tracks in database
// ##############################################
void DataBase_MySQL::updateTracks(QList<TRACK> *tracklist)
{
  QString origArtist;
  QString origAlbum;
  QSqlQuery query;
  
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "SELECT artist FROM tracks WHERE id = \"" << curr->id << "\";";
    if(query.exec(sqlquery.str().c_str())) {
      if(query.next()) origArtist = query.value(0).toString();
    } else {
      error = 1;
      return;
    }

    stringstream sqlquery3;
    sqlquery3 << "SELECT album FROM tracks WHERE id = \"" << curr->id << "\";";
    if(query.exec(sqlquery3.str().c_str())) {
      if(query.next()) origAlbum = query.value(0).toString();
    } else {
      error = 1;
      return;
    }

    stringstream sqlquery2;
    sqlquery2 << "UPDATE tracks SET"
	      << " path = \"" << adjustString(curr->path) << "\","
	      << " filename = \"" << adjustString(curr->filename) << "\","
	      << " medium = \"" << curr->medium << "\","
	      << " lastModified = \"" << dateTimeToString(curr->lastModified) << "\","
	      << " hasChanged = \"" << (curr->hasChanged ? 1 : 0) << "\","
	      << " mimetype = \"" << curr->mimetype << "\","
	      << " version = \"" << curr->version << "\","
	      << " layer = \"" << curr->layer << "\","
	      << " mode = \"" << curr->mode << "\","
	      << " bitrate = \"" << curr->bitrate << "\","
	      << " samplerate = \"" << curr->samplerate << "\","
	      << " length = \"" << curr->length << "\","
	      << " size = \"" << curr->size << "\","
	      << " artist = \"" << adjustString(curr->artist) << "\","
	      << " title = \"" << adjustString(curr->title) << "\","
	      << " album = \"" << adjustString(curr->album) << "\","
	      << " tracknumber = \"" << curr->tracknumber << "\", "
	      << " year = \"" << adjustString(curr->year) << "\","
	      << " genre = \"" << curr->genre << "\","
	      << " comment = \"" << adjustString(curr->comment) << "\""
	      << " WHERE id = \"" << curr->id << "\";";
    if(!query.exec(sqlquery2.str().c_str())) {
      error = 1;
      return;
    }

    int origArtistID, artistID;
    int incrLocal;
    if(curr->medium == MEDIUM_HARDDISK) incrLocal = 1; else incrLocal = 0;

    if(curr->artist != origArtist) {
      int artist = artistToID[origArtist];
      stringstream sqlquery;
      if(baselist[artist] == 1) {
	baselist.remove(artist);
	artistToID.remove(origArtist);
	sqlquery << "DELETE FROM artists WHERE id = \"" << artist << "\";";
      } else {
	baselist[artist]--;
	sqlquery << "UPDATE artists SET total=total-1, local=local-" << incrLocal << " WHERE id = \"" << artist << "\";";
      }
      if(!query.exec(sqlquery.str().c_str())) {
	error = 1;
	return;
      }

      origArtistID = artist;

      if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) (*deltamap)[artist][curr->medium] = new DELTAINFO(-1, -incrLocal, origArtist);
      else {
	(*deltamap)[artist][curr->medium]->total--;
	if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local--;
      }
 
      stringstream sqlquery2;
      if(!artistToID.contains(curr->artist)) {
	artist = ++maxID_artists;
	artistToID[curr->artist] = artist;
	baselist[artist] = 1;
	sqlquery2 << "INSERT INTO artists (id, name, total, local, isFavourite) VALUES (\""
		  << artist << "\", \"" << adjustString(curr->artist) << "\", \"1\", \"" << incrLocal << "\", \"0\");";
      } else {
	artist = artistToID[curr->artist];
	baselist[artist]++;
	sqlquery2 << "UPDATE artists SET total=total+1, local=local+" << incrLocal << " WHERE id = \"" << artist << "\";";
      }
      if(!query.exec(sqlquery2.str().c_str())) {
	error = 1;
	return;
      }
      
      artistID = artist;

      if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) (*deltamap)[artist][curr->medium] = new DELTAINFO(1, incrLocal, curr->artist);
      else {
	(*deltamap)[artist][curr->medium]->total++;
	if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local++;
      }
    } else {
      origArtistID = artistToID[curr->artist];
      artistID = origArtistID;
    }

    //  PHF specific handling for tree view only

    if(  ( artistID != origArtistID) ) {
      QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[origArtistID][curr->medium]->deltaalbum;
      if ( !deltaalbum->contains( origAlbum ) ) (*deltaalbum)[origAlbum] = ALBUMINFO( -1, -incrLocal, origAlbum );
      else {
        (*deltaalbum)[origAlbum].total--;
        if (incrLocal != 0) (*deltaalbum)[origAlbum].local--;
      };

      deltaalbum = (*deltamap)[artistID][curr->medium]->deltaalbum;
      if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
      else {
        (*deltaalbum)[curr->album].total++;
        if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
      };
    };

    if ( ( origAlbum != curr->album ) && ( artistID == origArtistID ) ) {
      if(!deltamap->contains(artistID) || !(*deltamap)[artistID].contains(curr->medium)) (*deltamap)[artistID][curr->medium] = new DELTAINFO(0, 0, curr->artist);
      QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artistID][curr->medium]->deltaalbum;
      if ( !deltaalbum->contains( origAlbum ) ) (*deltaalbum)[origAlbum] = ALBUMINFO( -1, -incrLocal, origAlbum );
      else {
        (*deltaalbum)[origAlbum].total--;
        if (incrLocal != 0) (*deltaalbum)[origAlbum].local--;
      };

      if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
      else {
        (*deltaalbum)[curr->album].total++;
        if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
      };
    };
  }
  if ( verbose == 5 )  {
     qWarning( "DataBase_MySQL::updateTracks" );  
     dumpdeltamap( deltamap );  
  }
}

// ##############################################
// # delete tracks out of database
// ##############################################
void DataBase_MySQL::deleteTracks(QList<TRACK> *tracklist)
{
  QSqlQuery query;
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM tracks WHERE id = \"" << curr->id << "\";";
    if(!query.exec(sqlquery.str().c_str())) {
      error = 1;
      return;
    }
    
    int incrLocal;
    if(curr->medium == MEDIUM_HARDDISK) incrLocal = 1; else incrLocal = 0;

    int artist = artistToID[curr->artist];
    stringstream sqlquery2;
    if(baselist[artist] == 1) {
      baselist.remove(artist);
      artistToID.remove(curr->artist);
      sqlquery2 << "DELETE FROM artists WHERE id = \"" << artist << "\";";
    } else {
      baselist[artist]--;
      sqlquery2 << "UPDATE artists SET total=total-1, local=local-" << incrLocal << " WHERE id = \"" << artist << "\";";
    }
    if(!query.exec(sqlquery2.str().c_str())) {
      error = 1;
      return;
    }
    
    if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) (*deltamap)[artist][curr->medium] = new DELTAINFO(-1, -incrLocal, curr->artist);
    else {
      (*deltamap)[artist][curr->medium]->total--;
      if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local--;
    }
    // PHF starts here
    QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artist][curr->medium]->deltaalbum;
    if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( -1, -incrLocal, curr->album );
    else {
      (*deltaalbum)[curr->album].total--;
      if (incrLocal != 0) (*deltaalbum)[curr->album].local--;
    };
  }
  if ( verbose == 5 )  {
     qWarning( "DataBase_MySQL::deleteTracks" );  
     dumpdeltamap( deltamap );  
  }
  deletePlaylistTracks(tracklist); // also delete tracks from playlist
}

// ##############################################
// # append new playlist to database
// ##############################################
int DataBase_MySQL::appendPlaylist(QString listname)
{
  QSqlQuery query;
  stringstream sqlquery;
  sqlquery << "SELECT id FROM playlists WHERE name = \"" << adjustString(listname) << "\";";
  if(!query.exec(sqlquery.str().c_str())) {
    error = 1;
    return -1;
  }

  if(query.size() > 0) {
    // should only have one playlist named <listname>
    error = 1;
    return -1;
  }

  stringstream sqlquery2;
  int playlist = ++maxID_playlists;
  sqlquery2 << "INSERT INTO playlists (id, name) VALUES (\""
	  		<< playlist << "\", \"" << adjustString(listname) << "\");";
  query.exec(sqlquery2.str().c_str());
  if(!query.isActive()) {
    error = 1;
    return -1;
  }

  return playlist;
}

// ##############################################
// # rename playlist
// ##############################################
void DataBase_MySQL::renamePlaylist(int id, QString newname)
{
  QSqlQuery query;
  stringstream sqlquery;

  stringstream sqlquery2;
  sqlquery2 << "UPDATE playlists SET"
	  		<< " name = \"" << adjustString(newname) << "\""
			<< " WHERE id = \"" << id << "\";";
  if(!query.exec(sqlquery2.str().c_str())) {
    error = 1;
    return;
  }
}

// ##############################################
// # delete playlist from database
// ##############################################
void DataBase_MySQL::deletePlaylist(int id)
{
  QSqlQuery query;
  stringstream sqlquery;

  sqlquery << "DELETE FROM playlists WHERE id = \"" << id << "\";";
  if(!query.exec(sqlquery.str().c_str())) {
    error = 1;
    return;
  }

  sqlquery.str("");
  sqlquery << "DELETE FROM playlist_tracks WHERE playlist_id = \""
	  	   << id << "\";";
  if(!query.exec(sqlquery.str().c_str())) {
    error = 1;
    return;
  }
}

// ##############################################
// # append track(s) to a playlist
// ##############################################
int DataBase_MySQL::appendPlaylistTracks(QList<TRACK> *tracklist, int playlistID, int *position)
{
  QSqlQuery query;

//  int oldID = maxID_playlist_tracks;
  int oldID = getMaxIDFromTable("playlist_tracks");

//  int pl_track,cur_position;
  int cur_position;
  if(position != 0)
	cur_position = *position;
  else
	cur_position = -1;

  // if position is NULL, then  used the next one available  
if(cur_position < 0) {
    stringstream sqlquery;
    sqlquery << "SELECT position_in_playlist FROM playlist_tracks"
			 << " WHERE playlist_id = \"" << playlistID << "\""
			 << " ORDER BY position_in_playlist DESC LIMIT 1;";
    if(!query.exec(sqlquery.str().c_str())) {
      error = 1;
      return -1;
    }
    if(query.next())
      cur_position = query.value(0).toInt()+1;
    else
      cur_position = 1;
    if(position != 0) *position = cur_position;
  }
  
  for(TRACK *curr = tracklist->first(); curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "SELECT id FROM playlist_tracks"
			 << " WHERE playlist_id = \"" << playlistID << "\""
			 << "   AND track_id = \"" << curr->id << "\";";
    if(!query.exec(sqlquery.str().c_str())) {
      error = 1;
      return -1;
    }
    if(query.next()) continue;   // don't allow duplicates in one playlist

//    pl_track = ++maxID_playlist_tracks;
    stringstream sqlquery2;
//    sqlquery2 << "INSERT INTO playlist_tracks (id, track_id, playlist_id, position_in_playlist) VALUES (\"" 
    sqlquery2 << "INSERT INTO playlist_tracks (track_id, playlist_id, position_in_playlist) VALUES (\"" 
//			  << pl_track << "\", \""
			  << curr->id << "\", \""
			  << playlistID << "\", \""
			  << cur_position++ << "\");";
    query.exec(sqlquery2.str().c_str());
    if(!query.isActive()) {
      error = 1;
      return -1;
    }
  }
  return oldID;
}

// ##############################################
// # delete track(s) from a playlist
// ##############################################
void DataBase_MySQL::deletePlaylistTracks(QList<TRACK> *tracklist)
{
  QSqlQuery query;
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM playlist_tracks"
			 << " WHERE track_id = \"" << curr->id << "\";";

    if(!query.exec(sqlquery.str().c_str())) {
      error = 1;
      return;
    }
  }

  stringstream sqlquery;
  sqlquery << "SELECT id FROM playlists;";
  query.exec(sqlquery.str().c_str());
  if(query.isActive()) {
    while(query.next()) {
      int playlistID = query.value(0).toInt();
      QSqlQuery query2;
      stringstream sqlquery2;
      sqlquery2 << "SELECT track_id FROM playlist_tracks"
				<< " WHERE playlist_id = \"" << playlistID << "\""
				<< " ORDER BY position_in_playlist;";
      query2.exec(sqlquery2.str().c_str());
      if(query2.isActive()) {
	int temppos = 1;
	while(query2.next()) {
	  int trackID = query2.value(0).toInt();
	  updatePlaylistTrack(playlistID, trackID, temppos++);
	}
      } else {
	error = 1;
	return;
      }
    }
  } else {
    error = 1;
    return;
  }
}

void DataBase_MySQL::deletePlaylistTracks(QList<PLAYLIST_TRACK> *playlist_tracklist)
{
  if(!playlist_tracklist) return;
  int playlistID = playlist_tracklist->first()->playlist_id;
    
  QSqlQuery query;
  for(PLAYLIST_TRACK *curr = playlist_tracklist->first();
      curr != 0; curr = playlist_tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM playlist_tracks"
   			 << " WHERE playlist_id = \"" << playlistID << "\""
			 << "   AND track_id = \"" << curr->track_id << "\";";
    if(!query.exec(sqlquery.str().c_str())) {
      error = 1;
      return;
    }
  }

  stringstream sqlquery;
  sqlquery << "SELECT track_id FROM playlist_tracks"
		   << " WHERE playlist_id = \"" << playlistID << "\""
		   << " ORDER BY position_in_playlist;";
  query.exec(sqlquery.str().c_str());
  if (query.isActive()) {
    int temppos = 1;
    while(query.next()) {
      int trackID = query.value(0).toInt();
      updatePlaylistTrack(playlistID, trackID, temppos++);
    }
  } else {
    error = 1;
    return;
  }
}

// ##############################################
// # update tracks(s) in playlist
// # (position change)
// ##############################################
void DataBase_MySQL::updatePlaylistTrack(int playlistID, int trackID, int new_pos)
{
  QSqlQuery query;
  stringstream sqlquery;
  sqlquery << "UPDATE playlist_tracks SET"
		   << " position_in_playlist = \"" << new_pos << "\""
		   << " WHERE playlist_id = \"" << playlistID << "\""
		   << "   AND track_id = \"" << trackID << "\";";
  if(!query.exec(sqlquery.str().c_str())) {
    error = 1;
    return;
  }
}

// ##############################################
// # delete entire medium from database
// ##############################################
void DataBase_MySQL::deleteMedium(int id)
{
  deleteTracks(getTracksByMedium(id));

  QSqlQuery query;

  stringstream sqlquery;
  sqlquery << "DELETE FROM mediums WHERE id = " << id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    error = 1;
    return;
  }
}

// ##############################################
// # toggles favourite flag of artist
// ##############################################
void DataBase_MySQL::modifyFavouriteArtist(int id, bool state)
{
	QSqlQuery query;
  stringstream sqlquery;
  sqlquery << "UPDATE artists SET isFavourite=" << ((state) ? "1" : "0") << " WHERE id = " << id << ";";
  if(!query.exec(sqlquery.str().c_str())) error = 1;
}

// ##############################################
// # get list of tracks by medium
// ##############################################
QList<TRACK> *DataBase_MySQL::getTracksByMedium(int id)
{
	stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE medium = \"" << id << "\";";
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by medium
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByMedium(int id)
{
  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE medium = \"" << id << "\";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by artist
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByArtist(int id)
{
  stringstream sqlquery;
  QString artist;
  for(QMap<QString, int>::Iterator it = artistToID.begin(); it != artistToID.end(); ++it) if(it.data() == id) artist = it.key();
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE artist = BINARY \"" << adjustString(artist) << "\";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by artist and album
// ##############################################
  QList<TRACK> *DataBase_MySQL::queryTracksByArtistAlbum( QString artist, QString album )
  {
    stringstream sqlquery;
    sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE artist = BINARY \"" << adjustString(artist) << "\"";
    if (!album.isNull() )
      sqlquery << " AND album = BINARY \"" << adjustString(album) << "\"";
    sqlquery << ";";
    lastquery = sqlquery.str();
    return applySelectTracksQuery(sqlquery.str());
  }

// ##############################################
// # query list of tracks by album
// ##############################################
  QList<TRACK> *DataBase_MySQL::queryTracksByAlbum( QString sampler, QString album )
  {
    stringstream sqlquery;
    sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE comment = '" << sampler << "'";
    if (!album.isNull() )
      sqlquery << " AND album = BINARY '" << adjustString(album) << "'";
    sqlquery << ";";
    lastquery = sqlquery.str();
    return applySelectTracksQuery(sqlquery.str());
  }


// ##############################################
// # query list of tracks by playlist
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByPlaylist(int id)
{
  stringstream sqlquery;
  sqlquery << "SELECT t2.id, t2.path, t2.filename, t2.medium, t2.lastModified, t2.hasChanged, t2.mimetype, t2.version, t2.layer, t2.mode, t2.bitrate, t2.samplerate, t2.length, t2.size, t2.artist, t2.title, t2.album, t2.tracknumber, t2.year, t2.genre, t2.comment"
    	   << " FROM playlist_tracks AS t1, tracks AS t2"
		   << " WHERE t1.playlist_id = \"" << id << "\" AND t1.track_id = t2.id"
		   << " ORDER BY t1.position_in_playlist;";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by phrase
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByPhrase(QString phrase, bool byFilename, bool byArtist, bool byTitle, bool byAlbum, bool byComment)
{
  phrase = adjustString(phrase);

  if(phrase == "" || (!byFilename && !byArtist && !byTitle && !byAlbum && !byComment)) return 0;

  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment FROM tracks WHERE";
  bool isFirst = true;
  if(byFilename) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(filename) REGEXP UCASE(\"" << adjustString(phrase) << "\") OR UCASE(path) REGEXP UCASE(\"" << adjustString(phrase) << "\")";
    isFirst = false;
  }
  if(byArtist) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(artist) REGEXP UCASE(\"" << adjustString(phrase) << "\")";
    isFirst = false;
  }
  if(byTitle) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(title) REGEXP UCASE(\"" << adjustString(phrase) << "\")";
    isFirst = false;
  }
  if(byAlbum) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(album) REGEXP UCASE(\"" << adjustString(phrase) << "\")";
    isFirst = false;
  }
  if(byComment) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(comment) REGEXP UCASE(\"" << adjustString(phrase) << "\")";
    isFirst = false;
  }
  
  sqlquery << ";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # reget last query
// ##############################################
QList<TRACK> *DataBase_MySQL::getLastQuery() {
  if(lastquery == "") return 0; else return applySelectTracksQuery(lastquery);
}

// ##############################################
// # disconnect from database
// ##############################################
void DataBase_MySQL::disconnect() {
  if(isConnected) {
    for(QMap<int, QMap<int, DELTAINFO*> >::Iterator it = deltamap->begin(); it != deltamap->end(); ++it)
      for(QMap<int, DELTAINFO*>::Iterator it2 = it.data().begin(); it2 != it.data().end(); ++it2) delete it2.data();
    delete deltamap;
    baselist.clear();
    artistToID.clear();
    if(db->isOpen()) db->close();
    isConnected = false;
    error = 0;
    lastquery = "";
  }
}

// *** destructor ***
DataBase_MySQL::~DataBase_MySQL() {
  disconnect();
}

// *** get max ID from a table ***
int DataBase_MySQL::getMaxIDFromTable(QString table)
{
    stringstream sqlquery;
    sqlquery << "SELECT MAX(id) FROM " << table <<";";
	QSqlQuery query(sqlquery.str().c_str(), db);
  
	if(query.isActive())
	{
    	if(query.next()) return query.value(0).toInt();
  	}
  	else error = 1;

  	return 0;
}

// *** get number of records from a table ***
int DataBase_MySQL::getCountFromTable(QString table)
{
    stringstream sqlquery;
    sqlquery << "SELECT COUNT(id) FROM " << table <<";";
    QSqlQuery query(sqlquery.str().c_str(), db);

    if(query.isActive())
    {
        if(query.next()) return query.value(0).toInt();
    }
    else error = 1;

    return 0;
}

// *** get number of records from a table ***
int DataBase_MySQL::getCountColumnDistinctFromTable(QString column, QString table)
{
    stringstream sqlquery;
    sqlquery << "SELECT COUNT(DISTINCT " << column << ") FROM " << table <<";";
    QSqlQuery query(sqlquery.str().c_str(), db);

    if(query.isActive())
    {
        if(query.next()) return query.value(0).toInt();
    }
    else error = 1;

    return 0;
}

// *** apply query and convert to list ***
QList<TRACK> *DataBase_MySQL::applySelectTracksQuery(string sqlquery)
{
  error = 0;
  QSqlQuery query(sqlquery.c_str());
  
  if(query.isActive()) {
    QList<TRACK> *list = new QList<TRACK>;
    while(query.next()) {
      TRACK *track = new TRACK;
      track->id           = query.value(0).toInt();
      track->path         = query.value(1).toString();
      track->filename     = query.value(2).toString();
      track->medium       = query.value(3).toInt();
      track->lastModified = query.value(4).toDateTime();
      track->hasChanged   = (query.value(5).toInt() != 0);
      track->mimetype     = query.value(6).toInt();
      track->version      = query.value(7).toInt();
      track->layer        = query.value(8).toInt();
      track->mode         = query.value(9).toInt();
      track->bitrate      = query.value(10).toInt();
      track->samplerate   = query.value(11).toInt();
      track->length       = query.value(12).toInt();
      track->size         = query.value(13).toInt();
      track->artist       = query.value(14).toString();
      track->title        = query.value(15).toString();
      track->album        = query.value(16).toString();
      track->tracknumber  = query.value(17).toInt();
      track->year         = query.value(18).toString();
      track->genre        = query.value(19).toInt();
      track->comment      = query.value(20).toString();
      list->append(track);
    }
    return list;
  } else {
    if(query.lastError().type() == QSqlError::Statement) {
      app->lock();
      QMessageBox::information(0, "SQL error", query.lastError().driverText() + "\n" + query.lastError().databaseText(), QMessageBox::Ok);
      app->unlock();
      error = 2;
    } else  error = 1;
    return 0;
  }
}

// *** adjust strings for use with mysql ***
QString DataBase_MySQL::adjustString(QString s)
{
  for(unsigned i=0; i<s.length(); i++) {
    if(s[i] == '\'' || s[i] == '\"' || s[i] == '\\') s.insert(i++, '\\');
  }
  return s;
}

// *** convert mysql string to datetime ***
QDateTime DataBase_MySQL::stringToDatetime(QString s)
{
  int y, m, d, h, mi, sec;
  sscanf(s.latin1(), "%d-%d-%d %d:%d:%d", &y, &m, &d, &h, &mi, &sec);
  return QDateTime(QDate(y,m,d), QTime(h,mi,sec));
}

// *** convert datetime to mysql string ***
QString DataBase_MySQL::dateTimeToString(QDateTime d)
{
  QString s;
  s.sprintf("%d-%d-%d %d:%d:%d", d.date().year(), d.date().month(), d.date().day(), d.time().hour(), d.time().minute(), d.time().second());
  return s;
}


