/***************************************************************************
                          lvi_track.cpp  -  description
                             -------------------
    begin                : Wed May 23 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lvi_track.h"

#include <qimage.h>

#include <sstream>

#include "tagger.h"

#include "gui.h"

using namespace std;

// ##################################################
// # constructor
// ##################################################
LVI_Track::LVI_Track(QListView *parent, TRACK *_track)
	: QListViewItem(parent), track(_track)
{
  QString s;
  
  setText(2, track->artist);
  setText(3, track->title);
  setText(4, s.sprintf("%d:%02d ", track->length/60, track->length%60));
  if(track->bitrate < 0) setText(5, s.sprintf("%d ", -track->bitrate));
  else setText(5, s.sprintf("%d ", track->bitrate)); 
  if(track->tracknumber == 0) setText(6, track->album);
  else if(track->album == "") setText(6, s.sprintf("[%02d]", track->tracknumber));
  else {
    stringstream ss;
    s.sprintf(" [%02d]", track->tracknumber);
    ss << track->album.latin1() << s.latin1();
    setText(6, ss.str().c_str());
  }
  setText(7, track->comment);
  setText(8, track->year);
  setText(9, Tagger::getGenre(track->genre));
  setText(10, track->path);
  setText(11, track->filename);

  myfilename = track->filename;
  mygenre = track->genre;
  myalbum = track->album;
  mytracknumber = track->tracknumber;
  isShowed = true;
  isAvailable = true;
  setDragEnabled(true);
}

void LVI_Track::adjustIcon()
{
  switch(myType) {
  case MEDIUM_HARDDISK:
    setPixmap(0, QPixmap(PROKYON_HOME + "/images/lvi_hdd.png")); break;
  case MEDIUM_CDROM:
    if(!isAvailable) setPixmap(0, GUI::fadeOutIcon(PROKYON_HOME + "/images/lvi_cdrom.png"));
    else setPixmap(0, QPixmap(PROKYON_HOME + "/images/lvi_cdrom.png"));
    break;
  case MEDIUM_SMB:
    if(!isAvailable) setPixmap(0, GUI::fadeOutIcon(PROKYON_HOME + "/images/lvi_smb.png"));
    else setPixmap(0, QPixmap(PROKYON_HOME + "/images/lvi_smb.png"));
    break;
  case MEDIUM_NFS:
    if(!isAvailable) setPixmap(0, GUI::fadeOutIcon(PROKYON_HOME + "/images/lvi_nfs.png"));
    else setPixmap(0, QPixmap(PROKYON_HOME + "/images/lvi_nfs.png"));
    break;
  }
}

void LVI_Track::applyMedium(int type, QString label) {
  myType = type;
  setText(1, label);
  adjustIcon();
}

void LVI_Track::applyPath(QString path) {
  mypath = path;
}

void LVI_Track::setIsShowed(bool _isShowed)
{
  isShowed = _isShowed;

  if(isShowed) {
    if(myType == MEDIUM_CDROM) {
      if(!isAvailable) setText(10, QString("[" + text(1) + "]" + track->path));
      else setText(10, mypath + track->path);
    } else if(myType == MEDIUM_SMB || myType == MEDIUM_NFS) {
      if(!isAvailable) setText(10, QString("[" + text(1) + "]" + track->path));
      else setText(10, mypath + track->path);
    }
    setText(11, myfilename);
    adjustIcon();
  }
}

void LVI_Track::setAvailability(bool state) {
  isAvailable = state;
  if(isShowed) setIsShowed(true);
    setDragEnabled(isAvailable);
}

void LVI_Track::setNewFilename(QString fname) {
  myfilename = fname;
  track->filename = fname;
  if(isShowed) setText(11, fname);
}

// ##################################################
// # set informations for this track
// ##################################################
void LVI_Track::setFilename(QString filename) {
  setText(11, filename);
  if(filename != track->filename) setPixmap(11, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(11, 0);
  myfilename = filename;
}

void LVI_Track::setArtist(QString artist) {
  // PHF begin
  setText(2, artist.stripWhiteSpace());
  // PHF end
  if(artist != track->artist) setPixmap(2, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(2, 0);
}

void LVI_Track::setTitle(QString title) {
  setText(3, title);
  if(title != track->title) setPixmap(3, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(3, 0);
}

void LVI_Track::setAlbum(QString album) {
  QString s;
  if(mytracknumber == 0) setText(6, album);
  else if(album == "") setText(6, s.sprintf("[%02d]", mytracknumber));
  else {
    stringstream ss;
    s.sprintf(" [%02d]", mytracknumber);
    ss << album.latin1() << s.latin1();
    setText(6, ss.str().c_str());
  }
  if(album != track->album || mytracknumber != track->tracknumber) setPixmap(6, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(6, 0);
  myalbum = album;
}

void LVI_Track::setTracknumber(int tracknumber) {
  QString s;
  if(tracknumber == 0) setText(6, myalbum);
  else if(myalbum == "") setText(6, s.sprintf("[%02d]", tracknumber));
  else {
    stringstream ss;
    s.sprintf(" [%02d]", tracknumber);
    ss << myalbum.latin1() << s.latin1();
    setText(6, ss.str().c_str());
  }
  if(myalbum != track->album || tracknumber != track->tracknumber) setPixmap(6, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(6, 0);
  mytracknumber = tracknumber;
}

void LVI_Track::setYear(QString year) {
  setText(8, year);
  if(year != track->year) setPixmap(8, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(8, 0);
}

void LVI_Track::setComment(QString comment) {
  setText(7, comment);
  if(comment != track->comment) setPixmap(7, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(7, 0);
}

void LVI_Track::setGenre(int genre) {
  setText(9, Tagger::getGenre(genre));
  if(genre != track->genre) setPixmap(9, QPixmap(PROKYON_HOME + "/images/lvi_changed.png")); else setPixmap(9, 0);
  mygenre = genre;
}

// ##################################################
// # return informations for this track
// ##################################################
QString LVI_Track::getFilename() { return myfilename; }
QString LVI_Track::getArtist()   { return text(2); }
QString LVI_Track::getTitle()    { return text(3); }
QString LVI_Track::getAlbum()    { return myalbum; }
int     LVI_Track::getTracknumber() { return mytracknumber; }
QString LVI_Track::getYear()     { return text(8); }
int     LVI_Track::getGenre()    { return mygenre; }
QString LVI_Track::getComment()  { return text(7); }

// ##################################################
// # has some infos changed?
// ##################################################
bool LVI_Track::hasFilenameChanged() { return track->filename != myfilename; }
bool LVI_Track::hasArtistChanged()   { return track->artist != text(2); }
bool LVI_Track::hasTitleChanged()    { return track->title != text(3); }
bool LVI_Track::hasAlbumChanged()    { return track->album != myalbum; }
bool LVI_Track::hasTracknumberChanged() { return track->tracknumber != mytracknumber; }
bool LVI_Track::hasYearChanged()     { return track->year != text(8); }
bool LVI_Track::hasGenreChanged()    { return track->genre != mygenre; }
bool LVI_Track::hasCommentChanged()  { return track->comment != text(7); }

// ##################################################
// # apply the changes
// ##################################################
void LVI_Track::applyChanges()
{
  track->artist   = text(2);
  track->filename = myfilename;
  track->title    = text(3);
  track->album    = myalbum;
  track->tracknumber = mytracknumber;
  track->year     = text(8);
  track->comment  = text(7);
  track->genre    = mygenre;

  setPixmap(3, 0);
  setPixmap(9, 0);
  setPixmap(2, 0);
  setPixmap(11, 0);
  setPixmap(6, 0);
  setPixmap(8, 0);
  setPixmap(7, 0);
}

// ##################################################
// # destructor
// ##################################################
LVI_Track::~LVI_Track() {
  delete track;
}
