/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: opt_proc.c,v 1.22 2004/01/09 20:29:28 type2 Exp $ */

/*
 *******************************************************************************
 *
 *                            PROSPECT PROJECT
 *                        Linux Options Processing Module
 *
 *******************************************************************************
 */

#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: opt_proc.c,v 1.22 2004/01/09 20:29:28 type2 Exp $";
#endif

/*
 * System Header Files
 */
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

/*
 * Prospect Header Files
 */
#include "version.h"
#include "prospect.h"
#include "linux_model.h"
#include "linux_module.h"
#include "opt_proc.h"
#include "xml_report.h"

/*
 * Static global vars.
 */
static int gErrFlag = 0;

/* remember: 'A' is at 65 ('a' is at 97) */
static struct option prospect_command_options[]={
    {"help", 0, 0, 'h'},
    {"usage", 0, 0, 'h'},
    {"version", 0, 0, 'v'},
    {"output", 1, 0, 'o'},
    {"xml-output", 0, 0, 'x'},
    {"xml-dtd", 0, 0, 20},

    {"follow-forks", 0, 0, 'f'},
    {"system-wide", 0, 0, 's'},
    {"kernel-only", 0, 0, 'k'},
    {"merge-profiles", 0, 0, 'm'},
    {"disassemble", 0, 0, 'e'},
    {"force-disassemble", 0, 0, 21},

    {"sampling-frequency", 1, 0, 'H'},
    {"min-time", 1, 0, 'M'},
    {"system-sort", 0, 0, 23},
    {"no-sort", 0, 0, 24},
    {"vmlinux", 1, 0, 25},
    {"system-map", 1, 0, 26},
    {"binary-trace-save", 1, 0, 'B'},
    {"binary-trace-read", 1, 0, 'b'},

    {"debug-level", 1, 0, 'd'},
    {"debug-system", 1, 0, 27},
    {"debug-trace", 1, 0, 28},
    {"no-gettimeofday-in-trace", 0, 0, 29},
    {"buf-flush-rate", 1, 0, 30},
    {"rt-priority", 1, 0, 31},
    {"inherit-priority", 1, 0, 32},
    {"concurrent-gdbs", 1, 0, 33},
    {"ascii-trace", 1, 0, 34},

    {0, 0, 0, 0}
};

static struct option *prospect_cmd_options = prospect_command_options;

/*
 * Static protos
 */
static int Atoi(char *s);
static void not_implemented(char *what);

/*
 * void usage(void)
 *
 * Usage function. No return.
 */
void
usage(void)
{
    char myBasename[256];

    extract_basename(myBasename, gConf.my_name);

    printversion();

    pscr("\
Usage: %s [options ...] command [arg ...]\n\
Profile 'command' and optionally whole system, options are:\n\
\n\
  -h, -?, --help                This information you're reading.\n\
  -v, --version                 Print out version information and exit.\n\
  -o, --output=File             Output results to File, not stdout.\n\
  -x, --xml-output              Output XML not ASCII.\n\
      --xml-dtd                 Output the XML DTD and exit.\n\
\n\
  -f, --follow-forks            Profile 'command' and descendants.\n\
  -s, --system-wide             Profile all processes and kernel.\n\
  -k, --kernel-only             Only profile the kernel.\n\
  -m, --merge-profiles          Merge profiles of same binary images.\n\
  -e, --disassemble             Produce disassembled profiles.\n\
      --force-disassemble       Force disassembly of all symbols.\n\
\n\
  -H, --sampling-frequency=Hz   Set sampling frequency to Hz (default: 200Hz).\n\
  -M, --min-time=Seconds        Set minimum time output cutoff for output,\n\
                                  default is 0.001 seconds.\n\
  --system-sort                 Sort output by system time, not user.\n\
  --no-sort                     Do not sort output.\n\
  --system-map=File             Set System.map file to use for kernel profiles.\n\
  --vmlinux=File                Set uncompressed vmlinux image file to use.\n\
  -B, --binary-trace-save=File  Set File to use for binary trace output.\n\
  -b, --binary-trace-read=File  Use binary trace File as input, not system.\n\
  --ascii-trace=File            Output ASCII translation of trace or run.\n\
\n\
  -d, --debug-level=Level       Set debugging level to Level.\n\
  --debug-system=System         Debug System subsystem (dis, kmod).\n\
  --debug-trace=File            Output debugging trace into File.\n\
  --no-gettimeofday-in-trace    Do not use gettimeofday() calls in -I trace.\n\
  --buf-flush-rate=Rate         Set buffer flushing rate to Rate seconds,\n\
                                  default is 2.0 seconds.\n\
  --rt-priority=Pri             Set real time priority to Pri (default is 0\n\
                                  which is normal timeshare priority).\n\
  --inherit-priority            Let child inherit priority, perhaps useful\n\
                                  when used with the --rt-priority option.\n\
  --concurrent-gdbs=Num         Set Num of concurrent gdb slave pipes for\n\
                                  disassembly, default is 8.\n",
    myBasename);

    prospect_exit(2);
    return;
} /* usage() */

/*
 * int process_options(int argc, char *argv[])
 *
 * Process command line options.
 */
int 
process_options(int argc, char *argv[])
{
    int ii;

    /* 
     * Go execute the 'options' list
     */
    while(
         (ii = 
         getopt_long(argc, argv, 
             "+B:b:d::efH:hkmM:o:svx?::",
             prospect_cmd_options, 0))
         != -1
         )
    {
      switch (ii)    
      {
         case 'B':    /* Specify Binary trace output file */
            /* block */
            {
                struct  stat    StatBuf;

                /* dont' allow overwrite if file exists */
                if (stat(optarg, &StatBuf) == 0) {
                    ferr("file \"%s\" exists and would be overwriten by -B,\n",
                         optarg);
                    ferr("      move it out of the way or delete it first\n");
                    gErrFlag++;
                }
                else {
                    gConf.fh_trace_out = create_fd(optarg);

                    if (gConf.fh_trace_out == -1) {
                        ferr("open of \"%s\" for -B failed\n", optarg);
                        gErrFlag++;
                    }
                }
            }
            break;

         case 'b':    /* Read Binary trace input file */
            //not_implemented("Binary Trace Input");
            gConf.fh_trace_in = open(optarg, O_RDONLY);
            if (gConf.fh_trace_in < 0)  {
               ferr("open of '%s' for -b failed\n",optarg);
               gErrFlag++;
            }
            break;
         
         case 'd':   /* debug level */
            if (optarg && (Atoi(optarg)>0))
                gConf.bug_level = Atoi(optarg);
            else
                gConf.bug_level++;
            mINFORM("Increased gConf.bug_level to %d", gConf.bug_level); 
            break;

         case 'e':    /* disassembled profiles */
            if (gConf.flags.disassemble)
                gConf.flags.force_disassemble = TRUE;
            gConf.flags.disassemble=TRUE;
            break;

         case 'f':
            mINFORM("Follow forks profile option detected.");
            gConf.flags.follow_forks = TRUE;
            break;

         case 'H':    /* sampling frequency in Hz */
            {
                int tmp = Atoi(optarg);
                if (tmp > 0) {
                    gOp.samplehz = tmp;
                    mINFORM("Sampling freq overridden to: %d Hz", gOp.samplehz);
                }
                else 
                    ferr("--sampling-frequency=<%s> not understood\n",
                         optarg);
            }
            break;
         
         case 'h':
            usage();
            break;
         
         case '?':    
            ferr("execute prospect --help for help\n");
            gErrFlag++;
            break;

         case 'k':
            mINFORM("Kernel profile option detected.");
            gConf.flags.kernel_only = TRUE;
            break;
         
         case 'm':   /* Merge profile feature */  
            not_implemented("Merged Profiles");
            mINFORM("Merge profiles feature activated.");
            gConf.flags.merge_profiles = TRUE;
            break;
                
         case 'M':    
            gConf.min_value = (double)atof(optarg);
            mINFORM("Min print cutoff overridden to: '%s'=%f", optarg,
                    gConf.min_value);
            break;

         case 'o':    /* Specify screen output file */
            gConf.strm_scr = create_stream(optarg);
            if (gConf.strm_scr == (FILE *)0) {
                ferr("could not open output file: <%s>\n",
                         optarg);
               gErrFlag++;
            }
            break;

         case 's':
            mINFORM("System wide profile option detected.");
            gConf.flags.system_wide = TRUE;
            break;

         case 'v':    
            printversion();
            break;

         case 'x':    /* XML */
            not_implemented("XML Output");
            if (gConf.flags.xml_output) {
                mINFORM(" File app-perf.dtd asked for.");
                /* make sure we output something */
                if (gConf.strm_scr == NULL) gConf.strm_scr = stdout;
                xml_generate_dtd();
                /* exit without our signature */
                exit(0);
            }
            gConf.flags.xml_output=1;
            mINFORM("XML output chosen.");
            break;

         /* --- Long options codes only follow --- */

         case 20:
            /* make sure we output something */
            if (gConf.strm_scr == NULL) gConf.strm_scr = stdout;
            xml_generate_dtd();
            /* exit without our signature */
            exit(0);
            break;

         case 21:
            gConf.flags.force_disassemble = TRUE;
            gConf.flags.disassemble = TRUE;
            mINFORM("Disassembly forced for all functions");
            break;

         case 23:
            gConf.flags.sort_by_system = TRUE;
            break;

         case 24:
            gConf.flags.no_sort = TRUE;
            break;

         case 25:
            /* block */
            {
                struct  stat    StatBuf;

                if (stat(optarg, &StatBuf) != 0)  /* file not exits */
                {
                    ferr("file \"%s\" for -vmlinux option does not exist\n",
                          optarg);
                    gErrFlag++;
                }
                else
                {
                    if (gConf.kernel_image != NULL)
                    {
                        ferr("file already specified for --vmlinux: \"%s\"\n",
                              gConf.kernel_image);
                        gErrFlag++;
                    }
                    else
                    {
                        gConf.kernel_image = strdup(optarg);
                        mINFORM("Set kernel image to: %s", gConf.kernel_image);
                    }
                }
            }
            break;

         case 26:
            /* block */
            {
                struct  stat    StatBuf;

                if (stat(optarg, &StatBuf) != 0)  /* file not exists */
                {
                    ferr("file \"%s\" for --system-map option "
                         "does not exist\n", optarg);
                    gErrFlag++;
                }
                else
                {
                    if (gConf.system_map != NULL)
                    {
                        ferr("file already specified "
                             "for --system-map: \"%s\"\n", gConf.system_map);
                        gErrFlag++;
                    }
                    else
                    {
                        if (S_ISREG(StatBuf.st_mode)) {
                            gConf.system_map = strdup(optarg);
                            mINFORM("Set system map file to: %s", 
                                    gConf.system_map);
                        }
                        else {
                            ferr("file \"%s\" for --system-map option "
                                 "is not a file\n", optarg);
                            gErrFlag++;
                        }
                    }
                }
            }
            break;
         
         case 27:
            if (strcmp(optarg,"dis")==0) {
                ferr("DBG: Disassembly subsystem flag on.\n");
                gConf.flags.dis_debug=TRUE;
            }
            else if (strcmp(optarg,"kmod")==0) {
                ferr("DBG: Kernel modules subsystem flag on.\n");
                gConf.flags.kmod_debug=TRUE;
            }
            else {
                ferr("--debug-system=<%s> not implemented\n", optarg);
            }
            break;

         case 28:    /* Specify Ascii inform trace output file */
            gConf.strm_info = create_stream(optarg);
            if (gConf.strm_info == (FILE *)0) gErrFlag++;
            mINFORM("Prospect start, detected trace flag");
            break;

         case 29:
            gConf.flags.no_gettimeofday = TRUE;
            break;

         case 30:
            /* block */
            {
                double val = atof(optarg);
                val*=100;
                gOp.flushrate = (int) val;
                mINFORM("Flush rate overridden to: %d", gOp.flushrate);
            }
            break;
                
         case 31:
            mINFORM("Change realtime option: %s", optarg);
            gConf.rt_pri = Atoi(optarg);
            if (gConf.rt_pri <= 0)          /* allow for non-realtime */
                gConf.rt_pri=REALTIME_OFF;
            else if (gConf.rt_pri > 99)     /* above 99 is undefined   */
                gConf.rt_pri=99;
            make_realtime(gConf.rt_pri);    /* update our priority      */
            break;

         case 32:
            mINFORM("Inherit priority activated");
            gConf.inherit_pri = TRUE;
            break;

         case 33:
            mINFORM("Sizing list of disassembler pipes to: %s", optarg);
            {
                int tmp;
                tmp = Atoi(optarg);
                if (tmp <= 0) {
                    ferr("--concurrent-gdbs=<%s> not understood\n",
                         optarg);
                }
                else 
                    gConf.dis_list_size = tmp;
            }
            break;
         
         case 34:    /* Specify Ascii trace output file */
            gConf.strm_rec = create_stream(optarg);
            if (gConf.strm_rec == (FILE *)0)
               gErrFlag++;
            break;
         

         default: 
            gErrFlag++;
         }
         if (gErrFlag) {
             prospect_exit(1);
         }
    }

    return 0;
} /* process_options() */


/*
 * ------------------------------  Static Functions ----------------------- 
 */

/*
 * static int Atoi(char *s)
 *
 * Same as atoi(), but with possible 'kKmM' suffixi.
 */ 
static int  
Atoi(char *s)
{
   char *s1;
   int   numb;

   mINFORM("Enter Atoi('%s')", s);
   /* allow input in any base */
   numb = (int)strtol(s, &s1, 0);
   if (s == s1)   
   {
       mINFORM("  Conversion failed.");
       return (-1);
   }
   mINFORM("  Converted str to (%d)", numb);

   /* Check for kilo modifier */
   if (*s1 == 'K' || *s1 == 'k')   {
      numb *= 1024; 
      mINFORM("  Found k modifier, returning (%d)", numb);
      return(numb);
   }
   /* Check for milo modifier */ 
   if (*s1 == 'M' || *s1 == 'm')   {
      numb *= (1024*1024); 
      mINFORM("  Found m modifier, returning (%d)", numb);
      return(numb);
   }
   /* if not kKmM suffixi, then must be NULL */
   if (*s1 != '\0')  { 
      mINFORM("  '%s' not a legal number, returning error", s);
      ferr("'%s' is not a legal number\n", s);
      return(-1);
   }  
   mINFORM("  Fall thru, returning (%d)", numb);
   return(numb);

} /* Atoi() */

/*
 * void not_implemented(char *msg)
 *
 * Not implemented exit.
 */
static void
not_implemented(char *what)
{
    ferr("My apologies, but \"%s\" is not yet implemented.\n", what);
    prospect_exit(1);

} /* not_implemented() */
