package AtomicData::Time;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Time.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::Time - time.

=head1 SYNOPSIS

see AtomicData.pm for more.

=head1 DESCRIPTION

Implements data encapsulation for times.

=head1 Parameters

see parent class for additional documentation. Will accept
'modern_times', 'past' and 'future' which all take boolean values. If
the values accessed by these keys are true, then they will have the
following effects:

modern_times: requires that the date be within 100 years of the
current date.

future: requires that the date be in the future.

past: requires that the date be in the past.

post_millennium : requires that the date be greater than 1000 (used to
make sure that they input all four characters)

=cut

use POSIX qw(strftime);
use Time::ParseDate qw(parsedate);
use AtomicData::Text;

@AtomicData::Time::ISA = qw(AtomicData::Text);

=head1 METHODS

see AtomicData.pm for further methods.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  # perform text canonicalization.
  $val = AtomicData::Text->_canonicalize($val);

  # blank is already canonical.
  $val eq "" and return $val;

  # here, deal with forms parsedate() does not accept.  clean input.
  if ($val =~ /^(\w+\s+\d+,)\s*(\d\d+)$/) {
    my ($chunk,$year) = ($1,$2);
    if (length($year) == 2) {
      $this->{_two_digit_year}++;
      $year = ($year > 50) ? 1900+$year : 2000+$year;
    } elsif (length($year) != 4) {
      $this->{_can_failed} = ['Unrecognized date string.'];
      return undef;
    } elsif ($year =~ /^0+$/) {
      $this->{_can_failed} = ['Year cannot be 0.'];
      return undef;
    }
    $val = $chunk." ".$year;
  }

  # convert our input value to seconds since epoch.
  if (! defined($val = parsedate($val))) {
    $this->{_can_failed} = ['Unrecognized time string.'];
    return undef;
  }

  # return seconds since epoch.
  return $val;
}

=head2 _verify

 instance/private
 (int $value) _verify ()

DESCRIPTION:

Will accept a value and return the canonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{future} and time() <= $value) {
    push @$problems, "'$value' must be in the future.";
  }
  if ($parameters->{past} and time() >= $value) {
    push @$problems, "'$value' must be in the past.";
  }
  if ($parameters->{modern_times} and
      abs(time()-$value) > 100 / (60 * 60 * 24 * 365)) {
    push @$problems, "'$value' must be within 100 years of the current date.";
  }
  if ($parameters->{post_millennium} and $this->year() < 1000) {
    push @$problems, "'$value' must be after 1000 A.D.";
  }

  @$problems and return (0, $problems);
  return (1, []);
}

=head2 day

 instance
 (int $date) day ()

DESCRIPTION:

Will return the day portion of the date.

=cut

sub day {
  my ($this) = @_;
  return strftime("%d",localtime($this->canonicalize()));
}

=head2 sday

 instance
 (string $day) sday ()

DESCRIPTION:

Will return the string day (e.g., Monday) for the date being
encapsulated.

=cut

sub sday {
  my ($this) = @_;
  return strftime("%A",localtime($this->canonicalize()));
}

=head2 month

 instance
 (int $date) month ()

DESCRIPTION:

Will return the month portion of the date.

=cut

sub month {
  my ($this) = @_;
  return strftime("%m",localtime($this->canonicalize()));
}

=head2 smonth

 instance
 (string $month) smonth ()

DESCRIPTION:

Will return the string name of the month (e.g., December).

=cut

sub smonth {
  my ($this) = @_;
  return strftime("%B",localtime($this->canonicalize()));
}

=head2 year

 instance
 (int $date) year ()

DESCRIPTION:

Will return the year portion of the date.

=cut

sub year {
  my ($this) = @_;
  return strftime("%Y",localtime($this->canonicalize()));
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
