// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "gldevice.h"
#include "mouseobject.h"
#include <boost/python/class.hpp>
#include <boost/python/overloads.hpp>
#include <stdexcept>

namespace visual {

/* Called by clickObject and mouseObject to translate a button click code into
 * a text string.
 */ 
std::string 
mousebase::button_name(int buttons) 
{
	switch (buttons) {
		case 1:
			return "left";
		case 2:
			return "right";
		case 3:
		case 4:
			return "middle";
		default:
			// This should *never* happen.
			throw std::invalid_argument("Button type should be left, right, or middle.");
	}
}

/* Project the cursor's current location onto the plane specified by the normal 
 * vector 'normal' and a perpendicular distance 'dist' from the origin.
 */
vector 
mousebase::py_project1( vector normal, double dist)
{
	double ndc = normal.dot(cam) - dist;
	double ndr = normal.dot(ray);
	double t = -ndc / ndr;
	vector v = cam + ray*t;
	return v;
}

/* Project the cursor's current position onto the plane specified by the normal vector
 * 'normal' rooted at the position vector 'point'.
 */
vector 
mousebase::py_project2( vector normal, vector point)
{
	double dist = normal.dot(point);
	double ndc = normal.dot(cam) - dist;
	double ndr = normal.dot(ray);
	double t = -ndc / ndr;
	vector v = cam + ray*t;
	return v;	
}

std::string 
mousebase::get_buttons()
{
	return button_name( buttons);
}

boost::python::object
mousebase::get_press()
{
	if (is_press())
		return boost::python::object( get_buttons());
	else
		return boost::python::object();
}

boost::python::object
mousebase::get_release()
{
	if (is_release())
		return boost::python::object( get_buttons());
	else
		return boost::python::object();
}

boost::python::object
mousebase::get_drag()
{
	if (is_drag())
		return boost::python::object( get_buttons());
	else
		return boost::python::object();
}

boost::python::object
mousebase::get_drop()
{
	if (is_drop())
		return boost::python::object( get_buttons());
	else
		return boost::python::object();
}

boost::python::object
mousebase::get_click()
{
	if (is_click())
		return boost::python::object( get_buttons());
	else
		return boost::python::object();
}

boost::python::object
mousebase::get_pick()
{
	if (pick)
		return pick->getObject();
	else
		return boost::python::object();
}

/************** clickobject implementation **************/


/************** mouseObject implementation **************/


void mouseObject::clear_events( int i)
{
	if (i != 0) {
		throw std::invalid_argument( "mouse.events can only be set to zero");
	}
	mutex::lock L(mtx);
	if (!clicks.empty()) {
		// Should be clicks.clear(), but that isn't supported by the queue class.
		clicks = std::queue< std::pair< boost::shared_ptr<clickObject>, bool> >();
	}
	clickCount = 0;
	return;
}

/* Returns the next mouse event object in the queue.  Blocks until there is one
 * available (by polling) if the queue is empty.
 */
boost::shared_ptr<clickObject> 
mouseObject::py_getevent() 
{
	while (1) {
		{
			mutex::lock L(mtx);
			if (!clicks.empty()) {
				boost::shared_ptr<clickObject> c = clicks.front().first;
				if ( clicks.front().second) {
					clickCount -= 1;
				}
				clicks.pop();
				return c;
			}
		}
		threaded_sleep(0.010);
	}
}

/* Returns the next click event in the queue.  Blocks until there is one available
 * if the queue is empty.  All non-click events are removed from the queue until
 * a click event is found.
 */
boost::shared_ptr<clickObject> 
mouseObject::py_getclick()
{
	while (1) {
		{ 
			mutex::lock L(mtx);
			if (!clicks.empty()) {
				boost::shared_ptr<clickObject> c = clicks.front().first;
				bool is_click = clicks.front().second;
				clicks.pop();
				if (is_click) {
					clickCount -= 1;
					return c;
				}
			}
		}
		threaded_sleep(0.010);
	}
}


namespace {
BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS( mousebase_project_partial_overloads, mousebase::py_project2, 1, 2);
} 
	
void ui_object_init_type( void)
{
	using namespace boost::python;
	
	class_<mousebase>( "clickbase", no_init)
		.def( "project", &mousebase::py_project2, mousebase_project_partial_overloads( args("normal", "point")
			, "project the mouse pointer to the plane specified by the normal vector 'normal' and passing through either the specified 'point' or the origin.") )
		.def( "project", &mousebase::py_project1, args("normal", "d")
			, "project the mouse pointer to the plane specified by the normal vector 'normal' that is either:\n"
			"'d' distance away from the origin ( click.project( normal=vector, d=scalar)), or\n"
			"includes the point 'point' ( click.project( normal=vector, point=vector))\n"
			"or passes through the origin ( click.project( normal=vector)).")
		.add_property( "pos", &mousebase::get_pos)
		.add_property( "pick", &mousebase::get_pick)
		.add_property( "pickpos", &mousebase::get_pickpos)
		.add_property( "camera", &mousebase::get_camera)
		.add_property( "ray", &mousebase::get_ray)
		.add_property( "button", &mousebase::get_buttons)
		.add_property( "press", &mousebase::get_press)
		.add_property( "release", &mousebase::get_release)
		.add_property( "click", &mousebase::get_click)
		.add_property( "drag", &mousebase::get_drag)
		.add_property( "drop", &mousebase::get_drop)
		.add_property( "shift", &mousebase::is_shift)
		.add_property( "alt", &mousebase::is_alt)
		.add_property( "ctrl", &mousebase::is_ctrl)
		;
		
	class_< clickObject, boost::shared_ptr<clickObject>, bases<mousebase>, boost::noncopyable>( "click_object"
		, "This class provides access to a specific mouse event.", no_init)
		;
	
	class_< mouseObject, boost::shared_ptr<mouseObject>, bases<mousebase>, boost::noncopyable>( "mouse_object"
		, "This class provides access to the mouse.", no_init)
		.def( "getclick", &mouseObject::py_getclick)
		.add_property( "clicked", &mouseObject::get_clicked)
		.def( "getevent", &mouseObject::py_getevent)
		.add_property( "events", &mouseObject::get_events, &mouseObject::clear_events)
		;

	class_<cursorObject, boost::shared_ptr<cursorObject>, boost::noncopyable>( "cursor_object"
		, "This class provides access to the cursor.", no_init)
		// Read-write cursor visibility.  Not available on all platforms
		.add_property( "visible", &cursorObject::get_visible, &cursorObject::set_visible)
		;

	class_<kbObject, boost::shared_ptr<kbObject>, boost::noncopyable>( "kb_object"
		, "This class provides keyboard interaction.", no_init)
		.def( "getkey", &kbObject::pop_next_key, "Returns the next key press value.")
		.add_property( "keys", &kbObject::get_available_keys) // The number of key presses waiting in the queue.
		;
}


} // !namespace visual
