/*
 * QGLViewerXML.h
 * $Id: QGLViewerXML.h,v 1.8 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000, 2001 Michael Meissner, Alexander Buck
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class QGLViewerXML
//  Purpose     : Provides  funcionality


#ifndef __QGLVIEWERXML_H
#define __QGLVIEWERXML_H


// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif


// Own
///////////
#include "CP2D.h"
#include "CV2D.h"
#include "CBoundingBox3D.h"
#include "CCamera.h"
#include "CList.h"
#include "CCameraKeyPathPoint.h"
#include "CCameraKeyPathAttributes.h"

// Qt
///////////
#include <qdom.h>


// forward declarations
/////////////////////////



/** Namespace of serialization functions of different
  * classes into XML documents and other help functions.
  *
  * @author Michael Meissner, Alexander Buck
  */
namespace QGLViewerXML {

   /** Creates a new xml node with parent parent and member attribute
    * as specified in member (optional). The node is added to the
    * parent node. */
   QDomElement addNode(QDomElement& parent,
		       const QString& member = QString::null);

   /** Queries the node with member attribute as specified out of 
    * the children of the parent node. */
   QDomElement queryNode(const QDomElement& parent,
			 const QString& member);

  /** Read a CP2D from a QDomNode. The read CP2D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CP2D&);

  /** Write a CP2D into a QDomNode. The write uses the
   * CP2D and the corresponding QString. */
  bool writeXML(QDomElement, const CP2D&);

  /** Read a CP3D from a QDomNode. The read CP3D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CP3D&);

  /** Write a CP3D into a QDomNode. The write uses the
   * CP3D and the corresponding QString. */
  bool writeXML(QDomElement, const CP3D&);

  /** Read a CP4D from a QDomNode. The read CP4D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CP4D&);

  /** Write a CP4D into a QDomNode. The write uses the
   * CP4D and the corresponding QString. */
  bool writeXML(QDomElement, const CP4D&);


  /** Read a CV2D from a QDomNode. The read CV2D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CV2D&);

  /** Write a CV2D into a QDomNode. The write uses the
   * CV2D and the corresponding QString. */
  bool writeXML(QDomElement, const CV2D&);

  /** Read a CV3D from a QDomNode. The read CV3D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CV3D&);

  /** Write a CV3D into a QDomNode. The write uses the
   * CV3D and the corresponding QString. */
  bool writeXML(QDomElement, const CV3D&);

  /** Read a CV4D from a QDomNode. The read CV4D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CV4D&);

  /** Write a CV4D into a QDomNode. The write uses the
   * CV4D and the corresponding QString. */
  bool writeXML(QDomElement, const CV4D&);


  /** Read a CQuat from a QDomNode. The read CQuat and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CQuat&);

  /** Write a CQuat into a QDomNode. The write uses the
   * CQuat and the corresponding QString. */
  bool writeXML(QDomElement, const CQuat&);


  /** Read a CBoundingBox3D from a QDomNode. The read BoundingBox
   * and the corresponding QString are passed along. */
  bool readXML(const QDomElement&, CBoundingBox3D&);

  /** Write a CBoundingBox3D into a QDomNode. The write uses the
   * CBoundingBox3D and the corresponding QString. */
  bool writeXML(QDomElement, const CBoundingBox3D&);


  /** Read a CMat4D from a QDomNode. The read CMat4D and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CMat4D&);

  /** Write a CMat4D into a QDomNode. The write uses the
   * CMat4D and the corresponding QString. */
  bool writeXML(QDomElement, const CMat4D&);


  /** Read a CCamera from a QDomNode. The read CCamera and the 
   * corresponding QString are passed along. */
  bool readXML(const QDomElement&, CCamera&);

  /** Write a CCamera into a QDomNode. The write uses the
   * CCamera and the corresponding QString. */
  bool writeXML(QDomElement, const CCamera&);

  /** Read a CCameraKeyPathPoint from a QDomNode. */
  bool readXML(const QDomElement&, CCameraKeyPathPoint&);
    
  /** Write a QCameraKeyPathPoint into a QDomNode. The attributes are only saved if 
    * fParams is true.
    */
  bool writeXML(QDomElement, const CCameraKeyPathPoint&,bool fParams=true);
    
  /** Read CCameraKeyPathAttributes from a QDomNode. The read CCameraKeyPathAttributes
    * and the 
    * corresponding QString are passed along. 
    */
  bool readXML(const QDomElement&, CCameraKeyPathAttributes&);
    
  /** Write CCameraKeyPathAttributes into a QDomNode. The write uses the
    * CCameraKeyPathAttributes and the corresponding QString. 
    */
  bool writeXML(QDomElement, const CCameraKeyPathAttributes&);

  /** Deserializes a CList. The members are added to the
    * passed CList. The object type needs to have a default
    * constructor. Returns true on success, false on error. */
  template <class T>
    bool readXML(const QDomElement& domElem, CList<T>& list, QString tagName="CList") {
    if (domElem.nodeName().compare(tagName) != 0)
      return false;

    int l=0;
    QDomNode node = domElem.firstChild();
    while (!node.isNull()) {
      if (node.isElement()) {
	QDomElement elem = node.toElement();
	T cam;

	if (QGLViewerXML::readXML(elem, cam)) {
	  list.insertAsLast(new T(cam));
	  l++;
	}
      }
      node = node.nextSibling();
    }

    return l>0;
  };



  /** Serializes a CList. Object type needs to have a writeXML
    * method with reference passing. Returns true on success,
    * false on error. */
  template <class T>
    bool writeXML(QDomElement domElem, const CList<T>& list, QString tagName="CList") {
    domElem.setTagName(tagName);
    CListContainer<T> *pContainer;
    pContainer = list.getFirst();
    while (pContainer) {
      if (!writeXML(addNode(domElem), *(pContainer->getObject())))
	return false;
      pContainer = pContainer->getNext();
    }
    return true;
  };

}


#endif
