/*
 * QCameraKeyPathEditorImpl.cpp
 * $Id: 
 *
 * Copyright (C) 2001 Alexander Buck, Michael Meissner, Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

// Qt
///////
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qpopupmenu.h>
#include <qfiledialog.h>
#include <qmessagebox.h>


// System
///////////
#include <stdio.h>
#include <math.h>


// QGLViewer
//////////////
#include "CBoundingBox3D.h"
#include "QGLViewerXML.h"
#include "QGLViewerIO.h"
#include "CMat4D.h"
#include "QCameraKeyPathEditorImpl.h"
#include "QIconOptions.h"



// Function  : CCamPathToolImpl
// Parameters: QGLExaminerViewer *viewer
//             QWidget *parent, 
//	       const char *name, bool modal, 
//	       WFlags fl
// Purpose   : default constructor.
// Comments  : See docu for details.
QCameraKeyPathEditorImpl::QCameraKeyPathEditorImpl( QGLViewer *viewer, QWidget* parent, const char* name, WFlags fl) :
    QCameraKeyPathEditor(parent,name,fl)
/*************************************************************/
{
   m_pMyViewer = viewer;

   m_pCameraKeyPath = new CList<CCameraKeyPathPoint>();
   Q_CHECK_PTR(m_pCameraKeyPath);

   setMenuBar(QCameraKeyPathEditorLayout);

   m_pDValEye = new QDoubleValidator(-1000000.0,1000000.0,5,this);
   Q_CHECK_PTR(m_pDValEye);
   m_pDValRef = new QDoubleValidator(-1000000.0,1000000.0,5,this);
   Q_CHECK_PTR(m_pDValRef);
   m_pDValUp = new QDoubleValidator(-1.0,1.0,5,this);
   Q_CHECK_PTR(m_pDValUp);
   m_pDValVAngle = new QDoubleValidator(0.0,180.0,2,this);
   Q_CHECK_PTR(m_pDValVAngle);
   m_pDValIcon = new QDoubleValidator(0.0,100.0,2,this);
   Q_CHECK_PTR(m_pDValIcon);
   m_pDValFrustum = new QDoubleValidator(0.0,100.0,2,this);
   Q_CHECK_PTR(m_pDValFrustum);
   m_pIValFrame = new QIntValidator(1,10000,this);
   Q_CHECK_PTR(m_pIValFrame);
   m_pDValTension = new QDoubleValidator(0.0,100.0,2,this);
   Q_CHECK_PTR(m_pDValTension);
   m_pDValCon = new QDoubleValidator(0.0,100.0,2,this);
   Q_CHECK_PTR(m_pDValCon);
   m_pDValBias = new QDoubleValidator(0.0,100.0,2,this);
   Q_CHECK_PTR(m_pDValBias);
  
   xEyeValue->setValidator(m_pDValEye);
   yEyeValue->setValidator(m_pDValEye);
   zEyeValue->setValidator(m_pDValEye);

   xRefValue->setValidator(m_pDValRef);
   yRefValue->setValidator(m_pDValRef);
   zRefValue->setValidator(m_pDValRef);

   xUpValue->setValidator(m_pDValUp);
   yUpValue->setValidator(m_pDValUp);
   zUpValue->setValidator(m_pDValUp);

   verAngleValue->setValidator(m_pDValVAngle);
 
   frameValue->setValidator(m_pIValFrame);
   tensionValue->setValidator(m_pDValTension);
   conValue->setValidator(m_pDValCon);
   biasValue->setValidator(m_pDValBias);
 
   pathDropSite->setCameraPath(getCamerapathPtr());
   pathDropSite->setEnabled(false);
 
   cameraBox->setSelectionMode(QListBox::Extended);
 
   writeAttributes(m_attributes);
   
   m_fCorrectValues = false;
   m_fDeleteWithSelection = true;
   
   cameraBox->setEnabled(false);
   deleteButton->setEnabled(false);
   attachButton->setEnabled(false);
   replaceButton->setEnabled(false);
 
   cameraSlider->setSteps(1,1);
   setSlider(-1);
   writeCameraText(NULL);
 
   m_fClearSelectedCams = false;
   m_fAttachCamera = false;
   m_fClearSelection = false;
   
   setAcceptDrops(TRUE);
 
   writeCameraText(NULL);
 
   connect(pathDropSite, SIGNAL(sigCameraKeyPathDropped(const CList<CCameraKeyPathPoint> &)),
           this, SLOT(sltSetCameraPath(const CList<CCameraKeyPathPoint> &)));
 
   // create the OpenGl display list for an arrow
   ///////////////////////////////////////////////
   m_rfIconSize = 0.01;           
   m_rfFrustumSize = 0.01;              
   makeArrowDispList();
 
   return;
} 


// Function  : ~CCamPathToolImpl
// Parameters: 
// Purpose   : default destructor.
// Comments  : See docu for details.
QCameraKeyPathEditorImpl::~QCameraKeyPathEditorImpl()
/*************************************************************/
{
   delete m_pqMenuBar;
   delete m_pqOptions;
   delete m_pDValEye;
   delete m_pDValRef;
   delete m_pDValUp;
   delete m_pDValVAngle;
   delete m_pDValIcon;
   delete m_pDValFrustum;
   delete m_pIValFrame;
   delete m_pDValTension;
   delete m_pDValCon;
   delete m_pDValBias;
   delete m_pCameraKeyPath;
}


// Function  : updateBoundingBox
// Parameters: 
// Purpose   : Updates bounding box every time a camera is added or deleted.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::updateBoundingBox()
/*************************************************************/
{
   CBoundingBox3D MyBox;
   CP3D MyPoint;
   MyBox = m_pMyViewer->getCameraPtr()->getBoundingBox();

   for (int j = 0; j<m_pCameraKeyPath->getNumObjects(); j++) {
      MyPoint = CP3D((*m_pCameraKeyPath)[j].getCamera().getEyePos().getX(),
		     (*m_pCameraKeyPath)[j].getCamera().getEyePos().getY(),
		     (*m_pCameraKeyPath)[j].getCamera().getEyePos().getZ());
	
      MyBox.addPoint(MyPoint);
    }
   m_pMyViewer->setBoundingBox(MyBox,false);
}



// Function  : draw
// Parameters: 
// Purpose   : Draws a path with an Open-GL display list.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::draw()
/*************************************************************/
{
   if (m_ViewingMode != hidden) {
      glCallList(m_glPathDispList);
   }
}



// Function  : sltDeleteCamera
// Parameters: 
// Purpose   : Delete marked cameras from the path.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltDeleteCamera()
/*************************************************************/
{
   int loopCounter = 0;
   CList<CCameraKeyPathPoint> copiedPoints;
   int nNumObjects = m_pCameraKeyPath->getNumObjects();
   bool *pfCopiedHighlights;
   int nSliderIndex;

   pfCopiedHighlights = new bool[nNumObjects];
   Q_CHECK_PTR(pfCopiedHighlights);

   //save slider index
   for (int j=0;j<m_pCameraKeyPath->getNumObjects();j++) {
      if (m_highlightedCams[j]){
         nSliderIndex = j;
         break;
      }
   }
  
   for (int k=0; k < nNumObjects; k++) {
      if (m_highlightedCams[k])
         pfCopiedHighlights[k] = true;
      else
         pfCopiedHighlights[k] = false;
   }

   if (nNumObjects > 1) {         // at least 2 elements
      for (int i=0; i < nNumObjects; i++) {
         if(pfCopiedHighlights[i]) {	    
            m_pCameraKeyPath->remove(&((*m_pCameraKeyPath)[i-loopCounter]));
            m_highlightedCams.remove(&(m_highlightedCams[i-loopCounter]));
            loopCounter +=1;
         }
      }
      QString qNumObjects;
      qNumObjects.setNum(m_pCameraKeyPath->getNumObjects());
      sliderEndLabel->setText(qNumObjects);
      formatCamBox();
        
      if (m_fDeleteWithSelection) {
         if (nSliderIndex+1 > m_pCameraKeyPath->getNumObjects()) {
            m_nSliderIndex = m_pCameraKeyPath->getNumObjects()-1;
            setSlider(m_nSliderIndex);
            cameraBox->clearSelection();
            cameraBox->setCurrentItem(m_nSliderIndex);
            cameraBox->setSelected(m_nSliderIndex,true);
         }
         else {
            m_nSliderIndex = nSliderIndex;
            setSlider(m_nSliderIndex);
            cameraBox->clearSelection();
            cameraBox->setCurrentItem(m_nSliderIndex);
            cameraBox->setSelected(m_nSliderIndex,true);
         }
      }
   }
   else if (m_pCameraKeyPath->getNumObjects() == 1) {               //one element  
      m_pCameraKeyPath->remove(&((*m_pCameraKeyPath)[m_nSliderIndex]));
      m_highlightedCams.remove(&(m_highlightedCams[m_nSliderIndex]));
      setSlider(-1);
      writeCameraText(NULL);
      formatCamBox();
      cameraBox->setEnabled(false);
      deleteButton->setEnabled(false);
      replaceButton->setEnabled(false);
      pathDropSite->setEnabled(false);
      attachButton->setEnabled(false);
   }
   updateHighlights();
   updateBoundingBox();
    
   QString qEndValue;
   qEndValue.setNum(m_pCameraKeyPath->getNumObjects()-1);
   sliderEndLabel->setText(qEndValue);

   delete pfCopiedHighlights;
}


// Function  : sltReplaceCamera
// Parameters: 
// Purpose   : Delete marked cameras from the path and insert current camera.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltReplaceCamera()
/*************************************************************/
{
   int insertIndex = 0;
   m_fDeleteWithSelection = false;

   for (int i=0;i<m_pCameraKeyPath->getNumObjects();i++) {
      if (m_highlightedCams[i]) {
         insertIndex = i;
         break;
      }
   }
   CCamera cam = m_pMyViewer->getCamera();
   if (insertIndex == 0) {
      sltDeleteCamera();   
      CCameraKeyPathAttributes atts;
      QString qName("new camera");
      CCameraKeyPathPoint *pCamPoint = new CCameraKeyPathPoint(cam,atts,qName);
      Q_CHECK_PTR(pCamPoint);
      bool *pBool = new bool(false);
      Q_CHECK_PTR(pBool);
      m_pCameraKeyPath->insertAsFirst(pCamPoint);
      m_highlightedCams.insertAsFirst(pBool);
      formatCamBox();
      writeCameraText(&cam);
      cameraBox->setSelected(0,true);
      cameraBox->setCurrentItem(0);
   }
   else {
      sltDeleteCamera();
      if (insertIndex+1 > m_pCameraKeyPath->getNumObjects()) {
         m_nSliderIndex = m_pCameraKeyPath->getNumObjects() - 1;
         setCamera(cam);
      }  
      else {
         m_nSliderIndex = insertIndex-1;
         setCamera(cam);
      }
   }
   if (m_fAttachCamera)
      m_pMyViewer->setCamera((*m_pCameraKeyPath)[m_nSliderIndex].getCamera());
   makePathDispList();
   redraw();
   if (m_pCameraKeyPath->getNumObjects() == 1) {
      m_fDeleteWithSelection = true;
      replaceButton->setEnabled(true);
      attachButton->setEnabled(true);
      pathDropSite->setEnabled(true);
      deleteButton->setEnabled(true);
      cameraBox->setEnabled(true);
   }
   m_fDeleteWithSelection = true;
}


// Function  : sltSetIconSize
// Parameters: 
// Purpose   : Saves current iconsize in m_rfIconSize.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltSetIconSize(double d)
/*************************************************************/
{
   m_rfIconSize = d;
   makeArrowDispList();
   makePathDispList();
   redraw();
}  


// Function  : sltSetFrustumSize
// Parameters: 
// Purpose   : Saves current frustumSize in m_rfFrustumSize.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltSetFrustumSize(double d)
/*************************************************************/
{
   m_rfFrustumSize = d;
   makePathDispList();
   redraw();
}  


// Function  : sltShowOptions
// Parameters: 
// Purpose   : Shows Options
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltShowOptions()
/*************************************************************/
{
   m_pqOptions->show();
}


// Function  : sltSaveInXML
// Parameters: 
// Purpose   : Saves current camera list in XML-file.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltSaveInXML()
/*************************************************************/
{
   QString fileName = QFileDialog::getSaveFileName(QString::null, "*.xml",
						   this);
   m_qSaveName = fileName;
   if ( !m_qSaveName.isEmpty()) {
      QString qOutString(m_qSaveName);
      QFile qOutFile(qOutString);

      for (int i=0; i<m_pCameraKeyPath->getNumObjects();i++) {
	  (*m_pCameraKeyPath)[i].setName(cameraBox->text(i));
      }

      QGLViewerIO::write(qOutFile,*m_pCameraKeyPath);
   }
}


// Function  : sltSetSaveName
// Parameters: const QString& qName
// Purpose   : Saves given qname for the filename.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltSetSaveName(const QString& qName)
/*************************************************************/
{
   m_qSaveName = qName;
}


// Function  : sltLoadFile
// Parameters: 
// Purpose   : Loads XML-file with camerapathpoints.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltLoadFile()
/*************************************************************/
{
  bool fOK = false;
  while (!fOK)
    {
      QString qInString = QFileDialog::getOpenFileName(QString::null,"*.xml",this);
      if (QFile::exists(qInString))
	{
	  if ( !qInString.isEmpty() )
	    {
	      QFile qInFile(qInString);
	      m_pCameraKeyPath->clear(1);
	      m_highlightedCams.clear(1);
	      if (!QGLViewerIO::read(qInFile,*m_pCameraKeyPath)) {
		showError("Could not open file (maybe wrong format?)");
		return;
	      }
	      for (int i=0;i<m_pCameraKeyPath->getNumObjects();i++)
		{
		  m_highlightedCams.insertAsLast(new bool(false));
		}
	      m_highlightedCams[0]=true;
	      
	      //update tool
	      ///////////////
	      attachButton->setEnabled(true);
	      replaceButton->setEnabled(true);
	      deleteButton->setEnabled(true);
	      pathDropSite->setEnabled(true);
	      CCamera Cam = (*m_pCameraKeyPath)[0].getCamera();
	      writeCameraText(&Cam);
	      formatCamBox();
	      setSlider(0);
	      cameraBox->setSelected(0,true);
	      cameraBox->setCurrentItem(0);
	      cameraBox->setEnabled(true);
	      pathDropSite->setEnabled(true);
	      writeAttributes(m_attributes);
	      QString qEndValue;
	      qEndValue.setNum(m_pCameraKeyPath->getNumObjects()-1);
	      sliderEndLabel->setText(qEndValue);
	      if(attachButton->text() == "Attach") {
		updateBoundingBox();
	      }
	      else {
		m_fAttachCamera = true;
		m_detachCamera = m_pMyViewer->getCamera();
		updateBoundingBox();
		m_pMyViewer->setCamera(Cam);
	      }
	      updateHighlights();
	      fOK = true;
	      qInFile.close(); 
	    }
	}
      else if (qInString == NULL)  fOK = true;
      else showError("Could not open file");
    }
  makePathDispList();
  redraw();
}


// Function  : sltChangeCamValue
// Parameters: 
// Purpose   : Saves current values in current camera.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltChangeCamValue()
/*************************************************************/
{
  if (m_fCorrectValues) {
    if (m_pCameraKeyPath->getNumObjects() > 0) {
	for( int i=0; i<m_pCameraKeyPath->getNumObjects(); i++) {
	    if(m_highlightedCams[i]) {
		if (checkEye->isChecked())
		  (*m_pCameraKeyPath)[i].changeCamera()->setEyePos(getCurrentEyePos());
		if (checkRef->isChecked())
		  (*m_pCameraKeyPath)[i].changeCamera()->setRefPoint(getCurrentRefPoint());
		if (checkUp->isChecked())
		  (*m_pCameraKeyPath)[i].changeCamera()->setViewUp(getCurrentViewUp());
		if (checkAngle->isChecked())
		  (*m_pCameraKeyPath)[i].changeCamera()->setFovy(getCurrentFovy());
		if (checkFrames->isChecked())
		(*m_pCameraKeyPath)[i].setAttributes(getCurrentFrames(),
						     (*m_pCameraKeyPath)[i].getAttributes().getTension(),
						     (*m_pCameraKeyPath)[i].getAttributes().getContinuity(),
						     (*m_pCameraKeyPath)[i].getAttributes().getBias());
		if (checkTension->isChecked())
		(*m_pCameraKeyPath)[i].setAttributes((*m_pCameraKeyPath)[i].getAttributes().getFrames(),
						     getCurrentTension(),
						     (*m_pCameraKeyPath)[i].getAttributes().getContinuity(),
						     (*m_pCameraKeyPath)[i].getAttributes().getBias());
		if (checkContinuity->isChecked())
		(*m_pCameraKeyPath)[i].setAttributes((*m_pCameraKeyPath)[i].getAttributes().getFrames(),
						     (*m_pCameraKeyPath)[i].getAttributes().getTension(),
						     getCurrentContinuity(),
						     (*m_pCameraKeyPath)[i].getAttributes().getBias()); 
		if (checkBias->isChecked())
		 (*m_pCameraKeyPath)[i].setAttributes((*m_pCameraKeyPath)[i].getAttributes().getFrames(),
						      (*m_pCameraKeyPath)[i].getAttributes().getTension(),
						      (*m_pCameraKeyPath)[i].getAttributes().getContinuity(),
						      getCurrentBias()); 
	      }
	  }
	CCamera Cam = (*m_pCameraKeyPath)[m_nSliderIndex].getCamera();
	writeCameraText(&Cam);
      }
  }
}


// Function  : sltAddNewCamera
// Parameters: 
// Purpose   : Adds current camera to path.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltAddNewCamera()
/*************************************************************/
{
   m_myCamera.setEyePos(getCurrentEyePos());
   m_myCamera.setRefPoint(getCurrentRefPoint());
   m_myCamera.setViewUp(getCurrentViewUp());
   saveInList("new camera");
   setSlider(m_nSliderIndex);
   formatCamBox();
   updateBoundingBox();
   updateHighlights();
}


// Function  : sltCheckValues
// Parameters: 
// Purpose   : Checks camera fields for correct input every time they 
//             are changed.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltCheckValues()
/*************************************************************/
{
   QString qCamString;
   int q = 2;
   bool fOK = true;
   m_fCorrectValues = true;
  
   qCamString = xEyeValue->text();
   if (m_pDValEye->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = yEyeValue->text();
   if (m_pDValEye->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;
  
   qCamString = zEyeValue->text();
   if (m_pDValEye->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = xRefValue->text();
   if (m_pDValRef->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;
  
   qCamString = yRefValue->text();
   if (m_pDValRef->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;
   
   qCamString = zRefValue->text();
   if (m_pDValRef->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;
 
   qCamString = xUpValue->text();
   if (m_pDValUp->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;
   
   qCamString = yUpValue->text();
   if (m_pDValUp->validate(qCamString,q) != QValidator::Acceptable)
     fOK = false;
  
   qCamString = zUpValue->text();
   if (m_pDValUp->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = verAngleValue->text();
   if (m_pDValVAngle->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = frameValue->text();
   if (m_pIValFrame->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = tensionValue->text();
   if (m_pDValTension->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = conValue->text();
   if (m_pDValCon->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   qCamString = biasValue->text();
   if (m_pDValBias->validate(qCamString,q) != QValidator::Acceptable)
      fOK = false;

   m_fCorrectValues = fOK;
}


// Function  : sltUpdateHighlights
// Parameters: int nIndex
// Purpose   : Marks current highlighted cameras. Sets slider to nIndex.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltUpdateHighlights(int nIndex)
/*************************************************************/
{
  int nCount = 0;

  if (cameraBox->count() > 1)
    {
      m_fClearSelection = false;
      cameraSlider->setValue(nIndex);
      for (int i=0; i<m_highlightedCams.getNumObjects(); i++)
	{
	  if (cameraBox->isSelected(i))
	    {
	      m_highlightedCams[i] = true;
	      nCount++;
	    }
	  else 
             m_highlightedCams[i] = false;
	}
    }
  if (!m_fAttachCamera)
    {
      if (nCount>1) attachButton->setEnabled(false);
      else attachButton->setEnabled(true);
    }
  makePathDispList();
  redraw();
  m_fClearSelection = true;
}


// Function  : sltClearHighlights
// Parameters: int nIndex
// Purpose   : Clears old selection.Selects item nIndex and redraws scene.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltClearHighlights(int nIndex)
/*************************************************************/
{
   if (m_fClearSelection) {
      cameraBox->clearSelection();
   }
   cameraBox->setSelected(nIndex,true);
   cameraBox->setCurrentItem(nIndex);
   updateHighlights();
}


// Function  : sltUpdateTool
// Parameters: int nIndex
// Purpose   : Updates tool with given nIndex. 
//             Prints the correct values on the screen.             
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltUpdateTool(int nIndex)
/*************************************************************/
{
   m_nSliderIndex = nIndex;
   if (!(m_nSliderIndex > m_pCameraKeyPath->getNumObjects()-1)) { 
      CCamera Cam = (*m_pCameraKeyPath)[m_nSliderIndex].getCamera();
      writeCameraText(&Cam);
      m_myCamera = Cam;
      cameraBox->setSelected(m_nSliderIndex,true);
      cameraBox->setCurrentItem(m_nSliderIndex);
      writeAttributes((*m_pCameraKeyPath)[m_nSliderIndex].getAttributes());
      if (m_fAttachCamera) {
	  m_pMyViewer->setCamera(m_myCamera);
	}
    }
}


// Function  : sltAttachCamera
// Parameters: 
// Purpose   : Handles attach/detach feature.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltAttachCamera()
/*************************************************************/
{
   if ( attachButton->text() == "Attach") {
      attachButton->setText("Detach");
      m_fAttachCamera = true;
      m_detachCamera = m_pMyViewer->getCamera();
      m_pMyViewer->setCamera(m_myCamera);
      updateBoundingBox();
      redraw();
   }
   else {
      attachButton->setText("Attach");
      m_pMyViewer->setCamera(m_detachCamera);
      updateBoundingBox();
      m_fAttachCamera = false;
      sltUpdateHighlights(m_nSliderIndex);
   }
}



// Function  : sltAbout
// Parameters: 
// Purpose   : Shows "About" box
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltAbout()
/*************************************************************/
{
  QMessageBox::about(this, tr("About QCameraKeyPathEditor"),
		     tr( "	             QCameraKeyPathEditor Version 1.0\n\n"
			 "                   Copyright (C) 2001 Alexander Buck\n\n"
			 "This program is free software; you can redistribute it "
			 "and/or modify it under \nthe terms of the GNU General Public "
			 "License as published by the Free \nSoftware Foundation; either "
			 "version 2 of the License, or (at your option) any \nlater version."
			 "\n\nThis program is distributed in the hope that it will be useful, "
			 "but WITHOUT \nANY WARRANTY; without even the implied warranty of "
			 "MERCHANTABILITY\n or FITNESS FOR A PARTICULAR PURPOSE. See the GNU "
			 "General Public \nLicense for more details.\n\nYou should have "
			 "received a copy of the GNU General Public License along \nwith this "
			 "program; if not, write to the Free Software Foundation, Inc., \n59 "
			 "Temple Place, Suite 330, Boston, MA 02111-1307 USA" ));
}


// Function  : sltMoveUp
// Parameters: 
// Purpose   : Moves selected items upwards.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltMoveUp()
/*************************************************************/
{
  int nFirst, nLast;
  int nNumObjects = m_pCameraKeyPath->getNumObjects();
  int *pnHighlighted;
  int *pnNewHighlighted;
  int nCurrentItem = cameraBox->currentItem();

  pnHighlighted = new int[nNumObjects];
  Q_CHECK_PTR(pnHighlighted);
  pnNewHighlighted = new int[nNumObjects];
  Q_CHECK_PTR(pnNewHighlighted);

  for (int j=0; j<nNumObjects;j++)
    {
      if (cameraBox->isSelected(j))
	pnHighlighted[j] = 1;
      else
	pnHighlighted[j] = 0;
    }
    
  int i = 0;
  while (i<nNumObjects-1)
    {
      while ((!cameraBox->isSelected(i)) && ( i<nNumObjects-1 ) )
	i++;
      if ( (i==nNumObjects-1) && (!cameraBox->isSelected(i)) ) 
	break;
      nFirst = i;
      while ( (cameraBox->isSelected(i+1)) && (i<nNumObjects-1) )
	i++;
      nLast = i;
      if (nFirst != 0)
	{
	  CCamera cCam = (*m_pCameraKeyPath)[nFirst-1].getCamera();
	  CCameraKeyPathAttributes cAtts = (*m_pCameraKeyPath)[nFirst-1].getAttributes();
	  QString qName = (*m_pCameraKeyPath)[nFirst-1].getName();
	  CCameraKeyPathPoint *qTmp = new CCameraKeyPathPoint(cCam,cAtts,qName);
	  Q_CHECK_PTR(qTmp);
	  m_pCameraKeyPath->remove(&((*m_pCameraKeyPath)[nFirst-1]));
	  if (nLast == nNumObjects-1 ){
	    m_pCameraKeyPath->insertAsLast(qTmp);
	  }
	  else
	    m_pCameraKeyPath->insertAfter((*m_pCameraKeyPath)(nLast-1),qTmp);
	}
      i++;
    }
  formatCamBox();
  pnNewHighlighted[nNumObjects-1] = 0;
  for (int k=0; k<nNumObjects-1;k++)
     pnNewHighlighted[k] = pnHighlighted[k+1];
  if (pnHighlighted[0] == 1)
    {
      int m = 0;
      while (pnHighlighted[m] == 1)
	m++;
      pnNewHighlighted[m-1] = 1;
    }
  for (int l=0;l<nNumObjects;l++)
    {
      if (pnNewHighlighted[l] == 1)
	cameraBox->setSelected(l,true);
    }
  updateHighlights();
  if (nCurrentItem == 0)
    cameraBox->setCurrentItem(nCurrentItem);
  else
    cameraBox->setCurrentItem(nCurrentItem-1);
  
  delete pnHighlighted;
  delete pnNewHighlighted;
}



// Function  : sltMoveDown
// Parameters: 
// Purpose   : Moves selected items downwards.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltMoveDown()
/*************************************************************/
{
  int nFirst, nLast;
  int nNumObjects = m_pCameraKeyPath->getNumObjects();
  int *pnHighlighted;
  int *pnNewHighlighted;
  int nCurrentItem = cameraBox->currentItem();

  pnHighlighted = new int[nNumObjects];
  Q_CHECK_PTR(pnHighlighted);
  pnNewHighlighted = new int[nNumObjects];
  Q_CHECK_PTR(pnNewHighlighted);

  for (int j=0; j<nNumObjects;j++)
    {
      if (cameraBox->isSelected(j))
	pnHighlighted[j] = 1;
      else
	pnHighlighted[j] = 0;
    }
  int i = nNumObjects-1;
  while (i>=0)
    {
      while ((!cameraBox->isSelected(i)) && ( i>0 ) ){
	i--;
      }
      if ( (i==0) && (!cameraBox->isSelected(i)) ) 
	break;
      nLast = i;
      while ( (cameraBox->isSelected(i-1)) && (i>0) )
	i--;
      nFirst = i;
      if (nLast != nNumObjects-1)
	{
	  CCamera cCam = (*m_pCameraKeyPath)[nLast+1].getCamera();
	  CCameraKeyPathAttributes cAtts = (*m_pCameraKeyPath)[nLast+1].getAttributes();
	  QString qName = (*m_pCameraKeyPath)[nLast+1].getName();
	  CCameraKeyPathPoint *qTmp = new CCameraKeyPathPoint(cCam,cAtts,qName);
	  Q_CHECK_PTR(qTmp);
	  m_pCameraKeyPath->remove(&((*m_pCameraKeyPath)[nLast+1]));
	  if (nFirst == 0)
	    m_pCameraKeyPath->insertAsFirst(qTmp);
	  else
	    m_pCameraKeyPath->insertAfter((*m_pCameraKeyPath)(nFirst-1),qTmp);
	}
      i--;
    }
  formatCamBox();
  pnNewHighlighted[0] = 0;
  for (int k=1; k<nNumObjects;k++)
     pnNewHighlighted[k] = pnHighlighted[k-1];
  if (pnHighlighted[nNumObjects-1] == 1)
    {
      int m = nNumObjects-1;
      while (pnHighlighted[m] == 1)
	m--;
      pnNewHighlighted[m+1] = 1;
    }
  for (int l=0;l<nNumObjects;l++)
    {
      if (pnNewHighlighted[l] == 1)
	cameraBox->setSelected(l,true);
    }
  updateHighlights();
  if (nCurrentItem == nNumObjects-1)
    cameraBox->setCurrentItem(nCurrentItem);
  else
    cameraBox->setCurrentItem(nCurrentItem+1);
  delete pnHighlighted;
  delete pnNewHighlighted;
}



// Function  : sltUpdateName
// Parameters: int nIndex
// Purpose   : Updates name in nIndex in the list when a name was changed.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltUpdateName(int nIndex)
/*************************************************************/
{
  (*m_pCameraKeyPath)[nIndex].setName(cameraBox->text(nIndex));
}



// Function  : eventFilter
// Parameters: QObject *o, QEvent *e
// Purpose   : handles events which come from the viewer.
// Comments  : See docu for details.
bool QCameraKeyPathEditorImpl::eventFilter(QObject *o, QEvent *e)
/*************************************************************/
{
  if( e->type() == QEvent::KeyPress)
    {
      QKeyEvent *k = (QKeyEvent*)e;
      if (k->key() == Key_C)
	{
	  setCamera(m_pMyViewer->getCamera());
	  return true;
	}
    }
  return QWidget::eventFilter( o, e );    // standard event processing
}


// Function  : dragEnterEvent
// Parameters: QDragEnterEvent *pqEvent
// Purpose   : handles QDragEnterEvents.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::dragEnterEvent(QDragEnterEvent *pqEvent)
/*************************************************************/
{
  if ( QCameraDrag::canDecode(pqEvent)) {
    pqEvent->accept();
  }
  if ( QCameraKeyPathDrag::canDecode(pqEvent)) {
    pqEvent->accept();
  }
}



// Function  : dragLeaveEvent
// Parameters: QDragLeaveEvent *pqEvent
// Purpose   : handles QDragLeaveEvents.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::dragLeaveEvent(QDragLeaveEvent *pqEvent)
/*************************************************************/
{
  return;
}



// Function  : dropEvent
// Parameters: QDropEvent *pqEvent
// Purpose   : handles QDropEvents.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::dropEvent(QDropEvent *pqEvent)
/*************************************************************/
{
  CList<CCameraKeyPathPoint> CameraKeyPath;  
  CCamera CCamera;

  if (pqEvent->source() != this) 
    {
      if ( QCameraDrag::decode(pqEvent, CCamera) ) 
	{
	  setCamera(CCamera);
	}
      if ( QCameraKeyPathDrag::decode(pqEvent, CameraKeyPath) )
	{
	  sltSetCameraPath(CameraKeyPath);
	}
    }
}


// Function  : setCamera
// Parameters: const CCamera &CCamera
// Purpose   : Adds given camera to the path and updates all values.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::setCamera(const CCamera &CCamera) 
/*************************************************************/
{
  QString qEndValue;
  m_myCamera = CCamera;
  writeCameraText(&m_myCamera);
  saveInList("new camera");
  setSlider(m_nSliderIndex);
  formatCamBox();
  cameraBox->setEnabled(true);
  cameraBox->setSelected(m_nSliderIndex,true);
  cameraBox->setCurrentItem(m_nSliderIndex);
  pathDropSite->setEnabled(true);
  qEndValue.setNum(m_pCameraKeyPath->getNumObjects()-1);
  sliderEndLabel->setText(qEndValue);
  updateBoundingBox();
  updateHighlights();
}


// Function  : sltSetCameraPath
// Parameters: CList<CCameraKeyPathPoint> &CameraKeyPath
// Purpose   : Sets current path to given path.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::sltSetCameraPath(const CList<CCameraKeyPathPoint> &CameraKeyPath)
/*************************************************************/
{
  int nNumObjects = CameraKeyPath.getNumObjects();
  m_pCameraKeyPath->clear();
  m_highlightedCams.clear();
  for(int i=0;i<nNumObjects; i++)
    {
      m_myCamera = CameraKeyPath[i].getCamera();
      //cout<<"Anzahl: "<<nNumObjects<<endl;
      //cout<<CameraKeyPath[i].getCamera().getEyePos().getZ()<<endl;
      //cout<<CameraKeyPath[i].getName()<<endl;
      saveInList(CameraKeyPath[i].getName());
    }
  QString qNumObjects;
  qNumObjects.setNum(nNumObjects-1);
  sliderEndLabel->setText(qNumObjects);
  CCamera Cam = (*m_pCameraKeyPath)[0].getCamera();
  writeCameraText(&Cam);
  formatCamBox();
  cameraBox->setEnabled(true);
  pathDropSite->setEnabled(true);
  updateBoundingBox();
  updateHighlights();
  setSlider(0);
  cameraBox->setCurrentItem(0);
  cameraBox->setSelected(0,true);
}


// Function  : saveInList
// Parameters: QString qName
// Purpose   : Saves current camera (m_myCamera) with current attributes 
//             (m_attributes) and given qName in the pathlist. 
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::saveInList(QString qName)
/*************************************************************/
{
  QCString qString(qName);
  CCameraKeyPathPoint *pCam = new CCameraKeyPathPoint(m_myCamera,m_attributes,qString);
  Q_CHECK_PTR(pCam);
  
  bool *pBool = new bool(false);
  Q_CHECK_PTR(pBool);

  if (m_pCameraKeyPath->getNumObjects() == 0)                 //empty list
    {
      m_pCameraKeyPath->insertAsLast(pCam);
      m_highlightedCams.insertAsLast(pBool);
      m_nSliderIndex = m_pCameraKeyPath->getNumObjects()-1;
      deleteButton->setEnabled(true);
      replaceButton->setEnabled(true);
      attachButton->setEnabled(true);
    }
  else if (m_nSliderIndex == m_pCameraKeyPath->getNumObjects()-1)      //last element
    {
      m_pCameraKeyPath->insertAsLast(pCam);
      m_highlightedCams.insertAsLast(pBool);
      m_nSliderIndex = m_pCameraKeyPath->getNumObjects()-1;
    }
  else
    {
      m_pCameraKeyPath->insertAfter((*m_pCameraKeyPath)(m_nSliderIndex),pCam);
      m_highlightedCams.insertAfter(m_highlightedCams(m_nSliderIndex),pBool);
      m_nSliderIndex += 1;
    }
}


// Function  : writeAttributes
// Parameters: CCameraKeyPathAttributes Atts
// Purpose   : Writes given attributes into proper fields.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::writeAttributes(CCameraKeyPathAttributes Atts)
/*************************************************************/
{
  QString qValue;
  qValue.setNum(Atts.getFrames());
  frameValue->setText(qValue);
  qValue.setNum(Atts.getTension());
  tensionValue->setText(qValue);
  qValue.setNum(Atts.getContinuity());
  conValue->setText(qValue);
  qValue.setNum(Atts.getBias());
  biasValue->setText(qValue);
}


// Function  : setSlider
// Parameters: int nValue
// Purpose   : Sets the slider to given index.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::setSlider(int nIndex)
/*************************************************************/
{
  if (nIndex == -1)
    {
      cameraSlider->setRange(0,0);
    }
  else
    {
      cameraSlider->setRange(0,m_pCameraKeyPath->getNumObjects()-1);
      cameraSlider->setValue(nIndex);
      m_nSliderIndex = nIndex;
    }
}


// Function  : writeCameraText
// Parameters: CCamera *pMyCamera
// Purpose   : Writes the given camera values into the text fields.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::writeCameraText(CCamera *pMyCamera)
/*************************************************************/
{
  QString qDoubleValue;
  
  if (pMyCamera == NULL)
    {
      xEyeValue->setText("-");
      yEyeValue->setText("-");
      zEyeValue->setText("-");

      xRefValue->setText("-");
      yRefValue->setText("-");
      zRefValue->setText("-");

      xUpValue->setText("-");
      yUpValue->setText("-");
      zUpValue->setText("-");

      verAngleValue->setText("");
    }
  else
    {
      qDoubleValue.setNum(pMyCamera->getEyePos().getX(),'g',4);
      xEyeValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getEyePos().getY(),'g',4);
      yEyeValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getEyePos().getZ(),'g',4);
      zEyeValue->setText(qDoubleValue);

      qDoubleValue.setNum(pMyCamera->getRefPoint().getX(),'g',4);
      xRefValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getRefPoint().getY(),'g',4);
      yRefValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getRefPoint().getZ(),'g',4);
      zRefValue->setText(qDoubleValue);

      qDoubleValue.setNum(pMyCamera->getViewUp().getX(),'g',4);
      xUpValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getViewUp().getY(),'g',4);
      yUpValue->setText(qDoubleValue);
      qDoubleValue.setNum(pMyCamera->getViewUp().getZ(),'g',4);
      zUpValue->setText(qDoubleValue);

      qDoubleValue.setNum(pMyCamera->getFovy());
      verAngleValue->setText(qDoubleValue);
    }
}


// Function  : formatCamBox
// Parameters: 
// Purpose   : Writes the names of the cameras into the camerabox.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::formatCamBox()
/*************************************************************/
{
   cameraBox->clear();
   if (m_pCameraKeyPath->getNumObjects() > 0) {
      for (int i=0; i<m_pCameraKeyPath->getNumObjects(); i++) {
         cameraBox->insertItem((*m_pCameraKeyPath)[i].getName());
      }
   }
}



// Function  : getCurrentEyePos
// Parameters: 
// Purpose   : Returns current EyePos.
// Comments  : See docu for details.
CP3D QCameraKeyPathEditorImpl::getCurrentEyePos()
/*************************************************************/
{
   CP3D EyePos;

   QString qCamString;
    
   qCamString = xEyeValue->text();
   EyePos.setX(qCamString.toFloat());
     
   qCamString = yEyeValue->text();
   EyePos.setY(qCamString.toFloat());
     
   qCamString = zEyeValue->text();
   EyePos.setZ(qCamString.toFloat());
   
   return EyePos;
}


// Function  : getCurrentRefPoint
// Parameters: 
// Purpose   : Returns current RefPoint.
// Comments  : See docu for details.
CP3D QCameraKeyPathEditorImpl::getCurrentRefPoint()
/*************************************************************/
{
   CP3D RefPoint;
 
   QString qCamString;
 
   qCamString = xRefValue->text();
   RefPoint.setX(qCamString.toFloat());
     
   qCamString = yRefValue->text();
   RefPoint.setY(qCamString.toFloat());
     
   qCamString = zRefValue->text();
   RefPoint.setZ(qCamString.toFloat());
   
   return RefPoint;
}


// Function  : getCurrentViewUp
// Parameters: 
// Purpose   : Returns current View Up.
// Comments  : See docu for details.
CV3D QCameraKeyPathEditorImpl::getCurrentViewUp()
/*************************************************************/
{
   CV3D ViewUp;

   QString qCamString;

   qCamString = xUpValue->text();
   ViewUp.setX(qCamString.toFloat());
  
   qCamString = yUpValue->text();
   ViewUp.setY(qCamString.toFloat());
  
   qCamString = zUpValue->text();
   ViewUp.setZ(qCamString.toFloat());
  
   return ViewUp;
}


// Function  : getCurrentFovy
// Parameters: 
// Purpose   : Returns current Fovy
// Comments  : See docu for details.
int QCameraKeyPathEditorImpl::getCurrentFovy()
/*************************************************************/
{
   int nFovy;
 
   QString qCamString;

   qCamString = verAngleValue->text();
   nFovy = qCamString.toInt();

   return nFovy;
}


// Function  : getCurrentFrames
// Parameters: 
// Purpose   : Returns current Frames
// Comments  : See docu for details.
int QCameraKeyPathEditorImpl::getCurrentFrames()
/*************************************************************/
{
   int nFrames;

   QString qCamString;

   qCamString = frameValue->text();
   nFrames = qCamString.toInt();

   return nFrames;
}


// Function  : getCurrentTension
// Parameters: 
// Purpose   : Returns current Tension.
// Comments  : See docu for details.
float QCameraKeyPathEditorImpl::getCurrentTension()
/*************************************************************/
{
   float rfTension;

   QString qCamString;

   qCamString = tensionValue->text();
   rfTension = qCamString.toFloat();

   return rfTension;
}


// Function  : getCurrentContinuity
// Parameters: 
// Purpose   : Returns current Continuity.
// Comments  : See docu for details.
float QCameraKeyPathEditorImpl::getCurrentContinuity()
/*************************************************************/
{
   float rfCon;

   QString qCamString;

   qCamString = conValue->text();
   rfCon = qCamString.toFloat();

   return rfCon;
}


// Function  : getCurrentBias
// Parameters: 
// Purpose   : Returns current Bias.
// Comments  : See docu for details.
float QCameraKeyPathEditorImpl::getCurrentBias()
/*************************************************************/
{
   float rfBias;

   QString qCamString;

   qCamString = biasValue->text();
   rfBias = qCamString.toFloat();

   return rfBias;
}


// Function  : setMenuBar
// Parameters: QLayout *qLayout
// Purpose   : Sets the menu bar for the tool and adds it to the given QLayout.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::setMenuBar(QLayout *qLayout)
/*************************************************************/
{
   QPopupMenu *pqFile = new QPopupMenu();
   Q_CHECK_PTR(pqFile);
   pqFile->insertItem("Open",this,SLOT(sltLoadFile()), CTRL+Key_O);
   pqFile->insertItem("Save as...",this,SLOT(sltSaveInXML()),CTRL+Key_S);

   m_pqOptions = new QIconOptions(this, tr("Icon Options"));
   Q_CHECK_PTR(m_pqOptions);
   connect(m_pqOptions, SIGNAL(sigArrowChanged(double)),
           this, SLOT(sltSetIconSize(double)));
   connect(m_pqOptions, SIGNAL(sigFrustumChanged(double)),
           this, SLOT(sltSetFrustumSize(double)));
   QPopupMenu *pqOptions = new QPopupMenu();
   Q_CHECK_PTR(pqOptions);
   pqOptions->insertItem("Icon Size", this, SLOT(sltShowOptions()), CTRL+Key_I);
    
   QPopupMenu *pqHelp = new QPopupMenu();
   Q_CHECK_PTR(pqHelp);
   pqHelp->insertItem("About...",this,SLOT(sltAbout()));
  
   m_pqMenuBar = new QMenuBar( this, "menu" );
   Q_CHECK_PTR( m_pqMenuBar );
   m_pqMenuBar->insertItem( "&File", pqFile );
   m_pqMenuBar->insertItem( "&Options", pqOptions);
   m_pqMenuBar->insertSeparator();
   m_pqMenuBar->insertItem( "&Help", pqHelp );
   m_pqMenuBar->setSeparator( QMenuBar::InWindowsStyle );

   qLayout->setMenuBar(m_pqMenuBar);  
}


// Function  : makeArrowDispList
// Parameters: 
// Purpose   : Makes the Open-GL display list to draw an arrow.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::makeArrowDispList()
/*************************************************************/
{
   const int nSteps = 20;
   float rfX,rfY,rfZ;
   float rfAngle = 2*M_PI/ nSteps;
  
   if (glIsList(m_glArrowDispList))
      glDeleteLists(m_glArrowDispList, 1);
   m_glArrowDispList = glGenLists(1);

   glNewList(m_glArrowDispList, GL_COMPILE);
   //draw Vector
   ////////////////
   glBegin(GL_LINES);
   glLineWidth(m_rfIconSize);
   glVertex3f(0,0,0);
   glVertex3f(0,0,m_rfIconSize);
   glEnd();

   glBegin(GL_TRIANGLE_FAN);
   glVertex3f(0,0,m_rfIconSize);
   rfZ = m_rfIconSize;
   for (int l = 0; l<nSteps+1; l++) {
      rfX = cos(l*rfAngle);
      rfY = sin(l*rfAngle);
      glVertex3f(0.1*m_rfIconSize*rfX,0.1*m_rfIconSize*rfY,rfZ);
   }
   glEnd();

   glBegin(GL_TRIANGLE_FAN);
   glVertex3f(0,0,1.2*m_rfIconSize);
   rfZ = m_rfIconSize;
   for (int m = 0; m<nSteps+1; m++) {
      rfX = cos(m*rfAngle);
      rfY = sin(m*rfAngle);
      glVertex3f(0.1*m_rfIconSize*rfX,0.1*m_rfIconSize*rfY,rfZ);
   }
   glEnd();
   glEndList();
}



// Function  : makePathDispList
// Parameters: 
// Purpose   : Makes the Open-GL display list to draw the camera-key-path.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::makePathDispList()
/*************************************************************/
{
   if (glIsList(m_glPathDispList))
      glDeleteLists(m_glPathDispList, 1);

   m_glPathDispList = glGenLists(1);
   glNewList(m_glPathDispList, GL_COMPILE);

   // save lighting state
   /////////////////////////
   glPushAttrib(GL_LIGHTING);
   glDisable(GL_LIGHTING);
 
   // draw path
   //////////////
   glColor3f(1.0, 1.0, 1.0);
   glLineWidth(1.5);
   glBegin(GL_LINE_STRIP);
   for (int k=0; k<m_pCameraKeyPath->getNumObjects(); k++) {
      glVertex3f((*m_pCameraKeyPath)[k].getCamera().getEyePos().getX(),
                 (*m_pCameraKeyPath)[k].getCamera().getEyePos().getY(),
                (*m_pCameraKeyPath)[k].getCamera().getEyePos().getZ());
   }
   glEnd();

   for (int i=0; i<m_pCameraKeyPath->getNumObjects(); i++) {
      CP3D Eye((*m_pCameraKeyPath)[i].getCamera().getEyePos().getX(),
               (*m_pCameraKeyPath)[i].getCamera().getEyePos().getY(),
               (*m_pCameraKeyPath)[i].getCamera().getEyePos().getZ());
      CV3D ViewVector((*m_pCameraKeyPath)[i].getCamera().getViewDir().getX(),
                      (*m_pCameraKeyPath)[i].getCamera().getViewDir().getY(),
                      (*m_pCameraKeyPath)[i].getCamera().getViewDir().getZ());
      CV3D UpVector((*m_pCameraKeyPath)[i].getCamera().getViewUp().getX(),
                    (*m_pCameraKeyPath)[i].getCamera().getViewUp().getY(),
                    (*m_pCameraKeyPath)[i].getCamera().getViewUp().getZ());
      CV3D CrossVector = -(UpVector | ViewVector);

      glPushMatrix();
      
      float m[16];
      m[0] = CrossVector.getX(); m[4] = UpVector.getX();  m[8] = ViewVector.getX();  m[12] = Eye.getX(); 
      m[1] = CrossVector.getY(); m[5] = UpVector.getY();  m[9] = ViewVector.getY();  m[13] = Eye.getY(); 
      m[2] = CrossVector.getZ(); m[6] = UpVector.getZ();  m[10] = ViewVector.getZ(); m[14] = Eye.getZ(); 
      m[3] = 0.0;                m[7] = 0.0;              m[11] = 0.0;               m[15] = 1.0; 

      glMultMatrixf(m);
  
      glColor3f(1.0,0.0,0.0);                  //ViewVector
      glCallList(m_glArrowDispList);
      if (m_highlightedCams[i]) 
         glColor3f(0.0,1.0,1.0);
      else 
         glColor3f(0.3,0.3,0.3);

      // draw frustum
      /////////////////
      glBegin(GL_TRIANGLE_FAN);
      glVertex3f(0,0,0);
      
      double rdVAngleDeg, rdHAngleDeg;
      double rdVAngleRad, rdHAngleRad;
      double rdRatio;

      rdVAngleDeg = (((*m_pCameraKeyPath)[i].getCamera().getFovy())/2);
      rdRatio = (*m_pCameraKeyPath)[i].getCamera().getRatio();
      rdHAngleDeg = rdRatio * rdVAngleDeg;
      rdVAngleRad = rdVAngleDeg * M_PI / 180;
      rdHAngleRad = rdHAngleDeg * M_PI / 180;
    
      glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
                 sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
                 sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
                 -sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
                 -sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
                 sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glEnd();
      
      glBegin(GL_LINES);
      glColor3f(0.7,0.7,0.7);
      glVertex3f(0.0,0.0,0.0);
      glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
                 sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glEnd();
      
      glBegin(GL_LINES);
      glVertex3f(0.0,0.0,0.0);
      glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
                 sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glEnd();
      
      glBegin(GL_LINES);
      glVertex3f(0.0,0.0,0.0);
      glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
                 -sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glEnd();
      
      glBegin(GL_LINES);
      glVertex3f(0.0,0.0,0.0);
      glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
                 -sin(rdVAngleRad)*m_rfFrustumSize,
                 cos(rdVAngleRad)*m_rfFrustumSize);
      glEnd();

      glPushMatrix();
      glRotatef(-90,1,0,0);
      glColor3f(0.0,1.0,0.0);                  //UpVector
      glCallList(m_glArrowDispList);
      glPopMatrix();

      glPushMatrix();
      glRotatef(90,0,1,0);
      glColor3f(0.0,0.0,1.0);                  //CrossVector
      glCallList(m_glArrowDispList);
      glPopMatrix();
      
      glPopMatrix();
   }
   // restore lighting state
   ///////////////////////////
   glPopAttrib();

   glEndList();
}



// Function  : showError
// Parameters: QString qError
// Purpose   : Opens a dialog with the error message qError.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::showError(QString qError)
/*************************************************************/
{
   QDialog *pqErrorDialog = new QDialog(this,"qError",true,0);
   Q_CHECK_PTR(pqErrorDialog);

   pqErrorDialog->setCaption("Error");
   pqErrorDialog->resize(120,100);
   pqErrorDialog->setMaximumSize(120,100);
   pqErrorDialog->setSizeGripEnabled(false);
   QVBoxLayout *pqLayout = new QVBoxLayout(pqErrorDialog);
   Q_CHECK_PTR(pqLayout);
   pqLayout->setSpacing( 6 );

   QLabel *pqLabel = new QLabel(pqErrorDialog);
   Q_CHECK_PTR(pqLabel);
   pqLabel->setText( qError );
   pqLabel->setMargin(6);
   pqLayout->addWidget(pqLabel);

   QPushButton *pqButton = new QPushButton(pqErrorDialog);
   Q_CHECK_PTR(pqButton);
   pqButton->setText( tr( "&OK") );
   pqButton->setAutoDefault(true);
   pqLayout->addWidget(pqButton);

   connect(pqButton, SIGNAL(clicked()), pqErrorDialog, SLOT(accept()));

   pqErrorDialog->exec();
}



// Function  : updateHighlights
// Parameters: 
// Purpose   : Marks current highlighted cameras.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::updateHighlights()
/*************************************************************/
{
   for (int i=0; i<m_highlightedCams.getNumObjects(); i++) {
      if (cameraBox->isSelected(i)) 
         m_highlightedCams[i] = true;
      else 
         m_highlightedCams[i] = false;
   }
   redraw();
}



// Function  : redraw
// Parameters: 
// Purpose   : Redraws the scene.
// Comments  : See docu for details.
void QCameraKeyPathEditorImpl::redraw()
/*************************************************************/
{
   emit sigRedrawGL();
}

