/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "fgotocodegen.h"
#include "redfsm.h"
#include "parsetree.h"
#include "bstmap.h"


/* Init base data. */
FGotoCodeGen::FGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
: 
	GotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &FGotoCodeGen::EXEC_ACTIONS()
{
	/* Loop the actions. */
	for ( ActionTableMap::Iter redAct = redFsm->actionMap; redAct.lte(); redAct++ ) {
		/* 	We are at the start of a glob, write the case. */
		out << "f" << redAct->actListId << ":\n";

		/* Loop the items in the list of action items. */
		for ( ActionTable::Iter item = redAct->key; item.lte(); item++ ) {
			/* Get the action data. */
			Action *act = parseData->actionIndex[item->value];

			/* Write the preprocessor line info for going into the 
			 * source file. */
			out << "#line " << act->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";

			/* Wrap the block in brakets. */
			out << "\t{"; INLINE_LIST(act->inlineList, 0, false) << "}\n";
		}

		out << "\tgoto again;\n";
	}
	return out;
}

std::ostream &FGotoCodeGen::FINISH_CASES()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* States that are final and have an out action need a case. */
		if ( st->eofAction != 0 ) {
			/* Write the case label. */
			out << "\t\tcase " << st->id << ": ";

			/* Jump to the func. */
			out << "goto f" << st->eofAction->actListId << ";\n";
		}
	}

	return out;
}


/* Init base data. */
CFGotoCodeGen::CFGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FGotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CFGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CFGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	return out;
}

std::ostream &CFGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void CFGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n"
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";
	
	INIT_CODE() <<
		"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() << 
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = fsm->curs";
	
	if ( anyRegCurStateRef() )
		out << ", _ps";

	out << 
		";\n"
		"	fsm->curs = -1;\n"
		"\n";
	
	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	out <<
		"	switch ( _cs ) {\n";
		STATE_GOTOS() <<
		"	}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() )
		EXEC_ACTIONS() << "\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"	switch( _cs ) {\n";
			FINISH_CASES() <<
			"	}\n"
			"\n";
	}

	out << 
		"out:\n"
		"	fsm->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* Call into execute to invoke out actions. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		/* Write out the vect. */
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
CppFGotoCodeGen::CppFGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FGotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CppFGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{this->_st[this->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CppFGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = this->_st[--this->_top]; goto again;}";
	return out;
}

std::ostream &CppFGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{this->_st[this->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void CppFGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n"
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n"
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = this->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";

	out << 
		";\n"
		"	this->curs = -1;\n"
		"\n";

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	out <<
		"	switch ( _cs ) {\n";
		STATE_GOTOS() <<
		"	}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() )
		EXEC_ACTIONS() << "\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"	switch( _cs ) {\n";
			FINISH_CASES() <<
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute action code. Call into exectue to handle the
		 * finishing code. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
ObjCFGotoCodeGen::ObjCFGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FGotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &ObjCFGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{self->_st[self->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &ObjCFGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = self->_st[--self->_top]; goto again;}";
	return out;
}

std::ostream &ObjCFGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{self->_st[self->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void ObjCFGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n";
		
	out <<
		"@implementation "; FSM_NAME() << "\n\n"

		"- (int)initFsm;\n"
		"{\n"
		"\tself->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "\tself->_top = 0;\n";

	INIT_CODE() <<
		"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	out <<
		"- (int) executeWithData:("; EL_TYPE() << " *)_data len:(int)_len;\n"
		"{\n"
		"\t"; EL_TYPE() << " *_p = ("; EL_TYPE() << " *)_data - 1;\n"
		"\t"; EL_TYPE() << " *_pe = ("; EL_TYPE() << " *)_data + _len;\n"
		"\tint _cs = self->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";

	out << ";\n";
		
	out <<
		"\tself->curs = -1;\n"
		"\n";

	out << 
		"again:\n"
		"\tif ( ++_p == _pe )\n"
		"\t\tgoto out;\n";

	if ( anyEofActions() ) {
		out <<
			"\tif ( _p == 0 )\n"
			"\t\tgoto eofActions;\n"
			"\n";
	}

	out <<
		"\tswitch ( _cs ) {\n";
		STATE_GOTOS() <<
		"\t}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() )
		EXEC_ACTIONS() << "\n";

	if ( anyEofActions() ) {
		out <<
		"eofActions:\n"
			"\t_p = 0; _pe = ( ("; EL_TYPE() << "*)0 ) + 1;\n"
			"\tswitch( _cs ) {\n";
			FINISH_CASES() <<
			"\t}\n"
		"\n";
	}

	out <<
		"out:\n"
			"\tself->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"\tif ( _cs == "; ERROR_STATE() << " ) return -1;\n";
	}

	out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	out <<
		"- (int)finish;\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute action code. Call into exectue to handle the
		 * finishing code. */
		out <<
			"\treturn [self executeWithData:0 len:1];\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
			"\tif ( self->curs == "; ERROR_STATE() << " ) return -1;\n";
		}

		out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		/* Write out the vect. */
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}

	out <<
		"@end\n\n";
}
