/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "ftabcodegen.h"
#include "redfsm.h"
#include "parsetree.h"

/* Init base data. */
FTabCodeGen::FTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	TabCodeGen(fsmName, parseData, redFsm, out)
{ }

/* Determine if we should use indicies or not. */
void FTabCodeGen::calcIndexSize()
{
	int sizeWithInds = 0, sizeWithoutInds = 0;

	/* Calculate cost of using with indicies. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		int totalIndex = st->outSingle.length() + st->outRange.length() + 
				(st->defTrans == 0 ? 0 : 1);
		sizeWithInds += arrayTypeSize(maxIndex) * totalIndex;
	}
	sizeWithInds += arrayTypeSize(maxState) * redFsm->transSet.length();
	if ( anyActions() )
		sizeWithInds += arrayTypeSize(maxActListId) * redFsm->transSet.length();

	/* Calculate the cost of not using indicies. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		int totalIndex = st->outSingle.length() + st->outRange.length() + 
				(st->defTrans == 0 ? 0 : 1);
		sizeWithoutInds += arrayTypeSize(maxState) * totalIndex;
		if ( anyActions() )
			sizeWithoutInds += arrayTypeSize(maxActListId) * totalIndex;
	}

	/* If using indicies reduces the size, use them. */
	useIndicies = sizeWithInds < sizeWithoutInds;
}


/* Write out the out-func for a state. */
std::ostream &FTabCodeGen::STATE_OUT_ACTION( RedStateAp *state )
{
	/* This function is only called if there are any out actions, so need
	 * not guard against there being none. */
	int act = 0;
	if ( state->eofAction != 0 )
		act = state->eofAction->actListId+1;
	out << act;
	return out;
}

/* Write out the function for a transition. */
std::ostream &FTabCodeGen::TRANS_ACTION( RedTransAp *trans )
{
	int action = 0;
	if ( trans->action != 0 )
		action = trans->action->actListId+1;
	out << action;
	return out;
}

/* Write out the function switch. This switch is keyed on the values
 * of the func index. */
std::ostream &FTabCodeGen::ACTION_SWITCH()
{
	/* Loop the actions. */
	for ( ActionTableMap::Iter redAct = redFsm->actionMap; redAct.lte(); redAct++ ) {
		/* Write the entry label. */
		out << "\tcase " << redAct->actListId+1 << ":\n";

		/* Loop the items in the list of action items. */
		for ( ActionTable::Iter item = redAct->key; item.lte(); item++ ) {
			/* Get the action data. */
			Action *act = parseData->actionIndex[item->value];

			/* Write the preprocessor line directive for going into the source
			 * file. */
			out << "#line " << act->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";
			
			/* Wrap the block in brakets. */
			out << "\t{"; INLINE_LIST(act->inlineList, 0, false) << "}\n";
		}

		out << "\tbreak;\n";
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}

std::ostream &FTabCodeGen::STATE_DATA()
{
	out << 
		"static "; ARRAY_TYPE(maxKeyOffset) << " "; FSM_NAME() << "_ko[] = {\n";
		KEY_OFFSETS() <<
		"};\n"
		"\n";

	out <<
		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxSingleLen) << " "; FSM_NAME() << "_sl[] = {\n";
		SINGLE_LENS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxRangeLen) << " "; FSM_NAME() << "_rl[] = {\n";
		RANGE_LENS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxIndexOffset) << " "; FSM_NAME() << "_io[] = {\n";
		INDEX_OFFSETS() <<
		"};\n"
		"\n";

	if ( useIndicies ) {
		out << 
			"static "; ARRAY_TYPE(maxIndex) << " "; FSM_NAME() << "_i[] = {\n";
			INDICIES() <<
			"};\n"
			"\n";

		out <<
			"static "; ARRAY_TYPE(maxState) << " "; FSM_NAME() << "_tt[] = {\n";
			TRANS_TARGS_WI() <<
			"};\n"
			"\n";

		if ( anyActions() ) {
			out <<
				"static "; ARRAY_TYPE(maxActListId) << " "; FSM_NAME() << "_ta[] = {\n";
				TRANS_ACTIONS_WI() <<
				"};\n"
				"\n";
		}
	}
	else {
		out <<
			"static "; ARRAY_TYPE(maxState) << " "; FSM_NAME() << "_tt[] = {\n";
			TRANS_TARGS() <<
			"};\n"
			"\n";

		if ( anyActions() ) {
			out <<
				"static "; ARRAY_TYPE(maxActListId) << " "; FSM_NAME() << "_ta[] = {\n";
				TRANS_ACTIONS() <<
				"};\n"
				"\n";
		}
	}

	if ( anyEofActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActListId) << " "; FSM_NAME() << "_ea[] = {\n";
			EOF_ACTIONS() <<
			"};\n"
			"\n";
	}

	out <<
		"static int "; FSM_NAME() << "_start = " << 
				redFsm->startState->id << ";\n"
		"\n";

	return out;
}


/* Init base class. */
CFTabCodeGen::CFTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FTabCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CFTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CFTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	return out;
}

std::ostream &CFTabCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void CFTabCodeGen::writeOutCode()
{
	/* Write out arrays of states/transitions. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() <<
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _rlen, _cs = fsm->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";

	if ( anyActions() ) 
		out << ", _acts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	int _trans;\n"
		"	fsm->curs = -1;\n"
		"\n";

	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	if ( useIndicies ) {
		out << "	_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	switch ( _acts ) {\n";
			ACTION_SWITCH() <<
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	/* If there are any actions, then jump to the func execution. */
	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out << 
		"out:\n"
		"	fsm->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out << 
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
CppFTabCodeGen::CppFTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FTabCodeGen(fsmName, parseData, redFsm, out) 
{ }

std::ostream &CppFTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{this->_st[this->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CppFTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = this->_st[--this->_top]; goto again;}";
	return out;
}

std::ostream &CppFTabCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{this->_st[this->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}


void CppFTabCodeGen::writeOutCode()
{
	/* Write out arrays of states/transitions. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _rlen, _cs = this->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	
	if ( anyActions() ) 
		out << ", _acts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	int _trans;\n"
		"	this->curs = -1;\n"
		"\n";

	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";
	
	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	if ( useIndicies ) {
		out << "	_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		if ( anyEofActions() )
			out << "execFuncs:\n";

		out << 
			"	switch ( _acts ) {\n";
			ACTION_SWITCH() <<
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	/* If there are any functions, then jump to the func execution. */
	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}
	
	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
ObjCFTabCodeGen::ObjCFTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FTabCodeGen(fsmName, parseData, redFsm, out) 
{ }

std::ostream &ObjCFTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{self->_st[self->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &ObjCFTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = self->_st[--self->_top]; goto again;}";
	return out;
}

std::ostream &ObjCFTabCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{self->_st[self->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}


void ObjCFTabCodeGen::writeOutCode()
{
	/* Write out arrays of states/transitions. */
	STATE_DATA();

	out <<
		"@implementation "; FSM_NAME() << "\n\n"

		"- (int)initFsm;\n"
		"{\n"
		"\tself->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "\tself->_top = 0;\n\n";

	INIT_CODE() <<
		"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	/* The binary search. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out <<
		"- (int) executeWithData:("; EL_TYPE() << " *)_data len:(int)_len;\n"
		"{\n"
		"\t"; EL_TYPE() << " *_p = _data - 1;\n"
		"\t"; EL_TYPE() << " *_pe = _data + _len;\n"
		"\tint _slen, _rlen, _cs = self->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	
	if ( anyActions() ) 
		out << ", _acts";

	out <<
		";\n"
		"\t"; ALPH_TYPE() << " *_keys;\n"
		"\tint _trans;\n"
		"\tself->curs = -1;\n"
		"\n";

	out <<
		"again:\n"
			"\tif ( ++_p == _pe )\n"
				"\t\tgoto out;\n";

	if ( anyEofActions() ) {
		out <<
			"\tif ( _p == 0 )\n"
				"\t\tgoto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";
	
	if ( anyRegCurStateRef() )
		out << "\t_ps = _cs;\n";

	if ( useIndicies ) {
		out << "\t_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
			"\t_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out << 
				"\tif ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
					"\t\tgoto again;\n"
				"\t_acts = "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		if ( anyEofActions() )
			out << "execFuncs:\n";

		out << 
				"\tswitch ( _acts ) {\n";
				ACTION_SWITCH() <<
				"\t}\n"
			"\n";
	}

	out <<
			"\tgoto again;\n"
		"\n";

	/* If there are any functions, then jump to the func execution. */
	if ( anyEofActions() ) {
		out <<
		"eofActions:\n"
			"\tif ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
				"\t\t_acts = "; FSM_NAME() << "_ea[_cs];\n"
				"\t\t_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
				"\t\tgoto execFuncs;\n"
			"\t}\n"
			"\n";
	}

	out <<
		"out:\n"
			"\tself->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"\tif ( _cs == "; ERROR_STATE() << " ) return -1;\n";
	}
	
	out <<
			"\treturn ( _cs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	out <<
		"- (int)finish;\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out <<
			"\treturn [self executeWithData:0 len:1];\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
			"\tif ( self->curs == "; ERROR_STATE() << " ) return -1;\n";
		}

		out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
	
	out <<
		"@end\n\n";
}
