/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "rfbClient.h"

#include <stdlib.h>

//#define DEBUG


#ifdef DEBUG
#include <iostream>
#define debug(msg) (cerr << msg << endl)
#else // DEBUG
#define debug(msg)
#endif // DEBUG




namespace rfb {



ClientStateAuthentication::ClientStateAuthentication( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &serverAuthentication, 4 )
{}

void ClientStateAuthentication::update()
{
  if ( size == 4 ) {
    ClientStateReaderBase::update();
    if ( offset == size ) {
      switch ( serverAuthentication.authentication_scheme ) {
        case 0: client->handleAuthenticationFailed(); break;
        case 1: client->handleAuthenticated(); break;
	case 2: size = 20; break;
	default: client->handleAuthenticationFailed(); return;
      }
    }
  }
  if ( size == 20 ) {
    ClientStateReaderBase::update();
    if ( offset == size ) {
      client->handleVNCAuthentication( serverAuthentication.challenge );
      size = 24;
    }
  }
  if ( size == 24 ) {
    ClientStateReaderBase::update();
    if ( offset == size ) {
      switch ( serverAuthentication.status ) {
        case 0: client->handleAuthenticated(); break;
        case 1: client->handleAuthenticationFailed(); break;
        case 2: client->handleAuthenticationFailed(); break;
        default: client->handleAuthenticationFailed(); return;
      }
    }
  }
}



ClientStateInitialisation::ClientStateInitialisation( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &serverInitialisation, 24 )
{
  serverInitialisation.name_string = NULL;
}

void ClientStateInitialisation::update()
{
  if ( !serverInitialisation.name_string ) {
    ClientStateReaderBase::update();
    if ( size == 24 ) {
      size = serverInitialisation.name_length;
      offset = 0;
      serverInitialisation.name_string = (CARD8 *) malloc( size + 1 );
      serverInitialisation.name_string[size] = 0;
      target = serverInitialisation.name_string;
    }
  }
  if ( serverInitialisation.name_string ) {
    ClientStateReaderBase::update();
    if ( size == offset ) client->handleServerInitialisation( serverInitialisation );
  }
}



ClientStateHandleMessages::ClientStateHandleMessages( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &message_type, 1 )
{}

void ClientStateHandleMessages::update()
{
  ClientStateReaderBase::update();
  if ( size == offset ) {
    offset = 0;
    switch ( message_type ) {
      case 0: client->currentState = &(client->stateFramebufferUpdate); break;
      case 1: break; // Colour Maps not yet supported
      case 2: client->handleBell(); break;
      case 3: client->currentState = &(client->stateCutText);
              client->stateCutText.init();
              break;
#ifdef USE_ZLIB_WARREN
      case 10: client->handleZlibEnabled(); break;
#endif // USE_ZLIB_WARREN
      default: break;
    }
  }
}




ClientStateFramebufferUpdate::ClientStateFramebufferUpdate( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &framebufferUpdate, 3 )
  , remaining_rectangles( 0 )
{}

void ClientStateFramebufferUpdate::update()
{
  if ( remaining_rectangles ) {
    remaining_rectangles--;
    client->currentState = &(client->stateReadingRectangle);
    return;
  }
  if ( offset == size ) {
    client->handleFramebufferUpdateCompleted( framebufferUpdate );
    offset = 0;
    return;
  }
  ClientStateReaderBase::update();
  if ( offset == size ) {
    remaining_rectangles = framebufferUpdate.number_of_rectangles;
    debug( "FramebufferUpdate" );
    client->handleFramebufferUpdate( framebufferUpdate );
  }
}




ClientStateReadingRectangle::ClientStateReadingRectangle( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &rectangle, 12 )
{}

void ClientStateReadingRectangle::update()
{
  ClientStateReaderBase::update();
  if ( offset == size ) {
    offset = 0;
    client->handleDecodeRectangle( rectangle );
  }
}



ClientStateDecodingRaw::ClientStateDecodingRaw( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
{}



void ClientStateDecodingRaw::startDecoder( Rectangle &rectangle )
{
  offset = 0;
  remaining_lines = rectangle.height;
  size = rectangle.width;
  size *= (client->framebuffer->pixelFormat.bits_per_pixel >> 3);
  target = client->framebuffer->data
           + rectangle.y_position
             * client->framebuffer->bytesPerLine
	   + rectangle.x_position
	     * ( client->framebuffer->pixelFormat.bits_per_pixel >> 3);
}



void ClientStateDecodingRaw::update()
{
  ClientStateReaderBase::update();
  if ( offset == size ) {
    remaining_lines--;
    if ( remaining_lines ) {
      offset = 0;
      target += client->framebuffer->bytesPerLine;
      return;
    }
    client->handleRectangleDecoded( client->stateReadingRectangle.rectangle );
  }
}





ClientStateDecodingCopyRect::ClientStateDecodingCopyRect( Client &client )
  : ClientStateReaderBase( client, (unsigned char*) &data, 4 )
{}



void ClientStateDecodingCopyRect::startDecoder( Rectangle &_rectangle )
{
  offset = 0;
  rectangle = _rectangle;
}



void ClientStateDecodingCopyRect::update()
{
  ClientStateReaderBase::update();
  if ( offset == size ) {

    client->framebuffer->copyRect( rectangle.x_position,
                                   rectangle.y_position,
                                   rectangle.width,
                                   rectangle.height,
                                   data.src_x_position,
                                   data.src_y_position
                                 );

    client->handleRectangleDecoded( rectangle );
  }
}





ClientStateDecodingRRESubrectangle::ClientStateDecodingRRESubrectangle( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
{}


void ClientStateDecodingRRESubrectangle::startDecoder()
{
  subrect_pixel_value = 0;
  has_subrect_pixel_value = 0;
  offset = 0;
}

void ClientStateDecodingRRESubrectangle::update()
{
  if ( !has_subrect_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &subrect_pixel_value.value[3]; size = 1; break;
      case 16: target = &subrect_pixel_value.value[2]; size = 2; break;
      case 24: target = &subrect_pixel_value.value[1]; size = 3; break;
      case 32: target = &subrect_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_subrect_pixel_value = 1;
      offset = 0;
    } else return;
  }
  if ( offset != size ) {
    target = (unsigned char*) &subrectangle;
    size = 8;
    ClientStateReaderBase::update();
  }
  if ( offset == size ) {

    client->framebuffer->fillRect( subrectangle.x_position
                                   + client->stateDecodingRRE.rectangle.x_position,
                                   subrectangle.y_position
                                   + client->stateDecodingRRE.rectangle.y_position,
                                   subrectangle.width,
                                   subrectangle.height,
                                   subrect_pixel_value
                                 );

    client->stateDecodingRRE.remaining_subrectangles--;
    if ( client->stateDecodingRRE.remaining_subrectangles )
      startDecoder();
    else {
      client->currentState = &client->stateDecodingRRE;
      client->update();
    }
  }
}





ClientStateDecodingRRE::ClientStateDecodingRRE( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
  , stateDecodingRRESubrectangle( client )
{}

void ClientStateDecodingRRE::startDecoder( Rectangle &_rectangle )
{
  rectangle = _rectangle;
  has_number_of_subrectangles = 0;
  background_pixel_value = 0;
  has_background_pixel_value = 0;
  remaining_subrectangles = 0;
  offset = 0;
}


void ClientStateDecodingRRE::update()
{
  if ( !has_number_of_subrectangles ) {
    target = (unsigned char*) &number_of_subrectangles;
    size = 4;
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_number_of_subrectangles = 1;
      remaining_subrectangles = number_of_subrectangles;
      offset = 0;
    } else return;
  }
  if ( !has_background_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &background_pixel_value.value[3]; size = 1; break;
      case 16: target = &background_pixel_value.value[2]; size = 2; break;
      case 24: target = &background_pixel_value.value[1]; size = 3; break;
      case 32: target = &background_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_background_pixel_value = 1;
      offset = 0;
      client->framebuffer->fillRect( rectangle.x_position,
                                     rectangle.y_position,
                                     rectangle.width,
                                     rectangle.height,
                                     background_pixel_value
                                   );
    } else return;
  }
  if ( remaining_subrectangles ) {
    client->currentState = &stateDecodingRRESubrectangle;
    stateDecodingRRESubrectangle.startDecoder();
  } else client->handleRectangleDecoded( rectangle );
}








ClientStateDecodingCoRRESubrectangle::ClientStateDecodingCoRRESubrectangle( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
{}


void ClientStateDecodingCoRRESubrectangle::startDecoder()
{
  subrect_pixel_value = 0;
  has_subrect_pixel_value = 0;
  offset = 0;
}

void ClientStateDecodingCoRRESubrectangle::update()
{
  if ( !has_subrect_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &subrect_pixel_value.value[3]; size = 1; break;
      case 16: target = &subrect_pixel_value.value[2]; size = 2; break;
      case 24: target = &subrect_pixel_value.value[1]; size = 3; break;
      case 32: target = &subrect_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_subrect_pixel_value = 1;
      offset = 0;
    } else return;
  }
  if ( offset != size ) {
    target = (unsigned char*) &subrectangle;
    size = 4;
    ClientStateReaderBase::update();
  }
  if ( offset == size ) {

    client->framebuffer->fillRect( subrectangle.x_position
                                   + client->stateDecodingCoRRE.rectangle.x_position,
                                   subrectangle.y_position
                                   + client->stateDecodingCoRRE.rectangle.y_position,
                                   subrectangle.width,
                                   subrectangle.height,
                                   subrect_pixel_value
                                 );

    client->stateDecodingCoRRE.remaining_subrectangles--;
    if ( client->stateDecodingCoRRE.remaining_subrectangles )
      startDecoder();
    else {
      client->currentState = &client->stateDecodingCoRRE;
      client->update();
    }
  }
}





ClientStateDecodingCoRRE::ClientStateDecodingCoRRE( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
  , stateDecodingCoRRESubrectangle( client )
{}

void ClientStateDecodingCoRRE::startDecoder( Rectangle &_rectangle )
{
  rectangle = _rectangle;
  has_number_of_subrectangles = 0;
  background_pixel_value = 0;
  has_background_pixel_value = 0;
  remaining_subrectangles = 0;
  offset = 0;
}


void ClientStateDecodingCoRRE::update()
{
  if ( !has_number_of_subrectangles ) {
    target = (unsigned char*) &number_of_subrectangles;
    size = 4;
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_number_of_subrectangles = 1;
      remaining_subrectangles = number_of_subrectangles;
      offset = 0;
    } else return;
  }
  if ( !has_background_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &background_pixel_value.value[3]; size = 1; break;
      case 16: target = &background_pixel_value.value[2]; size = 2; break;
      case 24: target = &background_pixel_value.value[1]; size = 3; break;
      case 32: target = &background_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      has_background_pixel_value = 1;
      offset = 0;
      client->framebuffer->fillRect( rectangle.x_position,
                                     rectangle.y_position,
                                     rectangle.width,
                                     rectangle.height,
                                     background_pixel_value
                                   );
    } else return;
  }
  if ( remaining_subrectangles ) {
    client->currentState = &stateDecodingCoRRESubrectangle;
    stateDecodingCoRRESubrectangle.startDecoder();
  } else client->handleRectangleDecoded( rectangle );
}












ClientStateDecodingHextileTile::ClientStateDecodingHextileTile( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
{}

void ClientStateDecodingHextileTile::startDecoder( unsigned int _x,
                                               unsigned int _y,
                                               unsigned int _width,
                                               unsigned int _height 
                                             )
{
  x = _x;
  y = _y;
  width = _width;
  height = _height;
  has_subencoding_mask = 0;
  raw = 0;
  has_background_pixel_value = 0;
  has_foreground_pixel_value = 0;
  has_number_of_subrectangles = 0;
  has_subrect_pixel_value = 0;
  offset = 0;
}


void ClientStateDecodingHextileTile::update()
{
  if ( raw ) {
    if ( height ) {
      ClientStateReaderBase::update();
      if ( offset == size ) {
        target += client->framebuffer->bytesPerLine;
        offset = 0;
        height--;
      } else return;
    }
    if ( !height ) {
      raw = 0;
      client->currentState = &client->stateDecodingHextile;
      client->update();
    }
    return;
  }
  if ( !has_subencoding_mask ) {
    target = (unsigned char*) &subencoding_mask;
    size = 1;
    ClientStateReaderBase::update();
    if ( offset == size ) {
      offset = 0;
      has_subencoding_mask = 1;
      if ( (subencoding_mask & 1) ) {
        raw = 1;
        target = client->framebuffer->data
                 + y * client->framebuffer->bytesPerLine
	         + x * ( client->framebuffer->pixelFormat.bits_per_pixel >> 3);
        size = width * ( client->framebuffer->pixelFormat.bits_per_pixel >> 3);
        return;
      }
      if ( !(subencoding_mask & 2) ) { 
        has_background_pixel_value = 1;
        client->framebuffer->fillRect( x, y, width, height,
                                       background_pixel_value
                                     );
      }
      if ( !(subencoding_mask & 4) ) has_foreground_pixel_value = 1;
      if ( !(subencoding_mask & 8) ) { 
        has_number_of_subrectangles = 1;
        remaining_subrectangles = 0;
        number_of_subrectangles = 0;
      }
    } else return;
  }
  if ( !has_background_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &background_pixel_value.value[3]; size = 1; break;
      case 16: target = &background_pixel_value.value[2]; size = 2; break;
      case 24: target = &background_pixel_value.value[1]; size = 3; break;
      case 32: target = &background_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      offset = 0;
      has_background_pixel_value = 1;
      client->framebuffer->fillRect( x, y, width, height,
                                     background_pixel_value
                                   );
    } else return;
  }
  if ( !has_foreground_pixel_value ) {
    switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
      case 8:  target = &foreground_pixel_value.value[3]; size = 1; break;
      case 16: target = &foreground_pixel_value.value[2]; size = 2; break;
      case 24: target = &foreground_pixel_value.value[1]; size = 3; break;
      case 32: target = &foreground_pixel_value.value[0]; size = 4; break;
      default: break;
    }
    ClientStateReaderBase::update();
    if ( offset == size ) {
      offset = 0;
      has_foreground_pixel_value = 1;
    } else return;
  }
  if ( !has_number_of_subrectangles ) {
    target = (unsigned char*) &number_of_subrectangles;
    size = 1;
    ClientStateReaderBase::update();
    if ( offset == size ) {
      offset = 0;
      remaining_subrectangles = number_of_subrectangles;
      has_number_of_subrectangles = 1;
    } else return;
  }
  if ( !(subencoding_mask & 16) ) {
    has_subrect_pixel_value = 1;
    subrect_pixel_value = foreground_pixel_value;
  }
  if ( remaining_subrectangles ) {
    if ( !has_subrect_pixel_value ) {
      switch ( client->framebuffer->pixelFormat.bits_per_pixel ) {
        case 8:  target = &subrect_pixel_value.value[3]; size = 1; break;
        case 16: target = &subrect_pixel_value.value[2]; size = 2; break;
        case 24: target = &subrect_pixel_value.value[1]; size = 3; break;
        case 32: target = &subrect_pixel_value.value[0]; size = 4; break;
        default: break;
      }
      ClientStateReaderBase::update();
      if ( offset == size ) {
        offset = 0;
        has_subrect_pixel_value = 1;
      } else return;
    }
    target = (unsigned char*) &subrectangle;
    size = 2;
    ClientStateReaderBase::update();
    if ( size == offset ) {
      offset = 0;
      client->framebuffer->fillRect( (subrectangle.x_and_y_position >> 4) + x,
                                     (subrectangle.x_and_y_position & 15) + y,
                                     (subrectangle.width_and_height >> 4) + 1,
                                     (subrectangle.width_and_height & 15) + 1,
                                     subrect_pixel_value
                                   );
      has_subrect_pixel_value = 0;
//      cout << "remaining: " << remaining_subrectangles << endl;
      remaining_subrectangles--;
      if ( !remaining_subrectangles ) {
//        cout << "tile completed" << endl;
        client->currentState = &client->stateDecodingHextile;
        client->update();
      }
    }
    return;
  }
//  cout << "tile completed" << endl;
  client->currentState = &client->stateDecodingHextile;
  client->update();
}





ClientStateDecodingHextile::ClientStateDecodingHextile( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
  , stateDecodingHextileTile( client )
{}

void ClientStateDecodingHextile::startDecoder( Rectangle &_rectangle )
{
  rectangle = _rectangle;
  current_y = 0;
  current_x = 0;
  offset = 0;
}


void ClientStateDecodingHextile::update()
{
//  cout << "next tile" << endl;
  if ( current_y < rectangle.height ) {
    if ( current_x >= rectangle.width ) {
      int height = rectangle.height - current_y;
      if ( height > 16 ) {
        current_y += 16;
        current_x = 0;
      } else { 
        client->handleRectangleDecoded( rectangle );
      }
    }
  }
//  cout << "X: " << current_x << "  Y: " << current_y << endl;
  if ( current_x < rectangle.width ) {
    int width = rectangle.width - current_x;
    if ( width > 16 ) width = 16;
    int height = rectangle.height - current_y;
    if ( height > 16 ) height = 16;
    stateDecodingHextileTile.startDecoder( rectangle.x_position + current_x,
                                           rectangle.y_position + current_y,
                                           width, height );
    client->currentState = &stateDecodingHextileTile;
    current_x += 16;
    return;
  }
}







ClientStateCutText::ClientStateCutText( Client &client )
  : ClientStateReaderBase( client, NULL, 0 )
{}


void ClientStateCutText::init()
{
  size = 7;
  offset = 0;
  target = (unsigned char*) &data;
}

void ClientStateCutText::update()
{
  ClientStateReaderBase::update();
  if ( size == offset ) {
    if ( target == (unsigned char*) &data ) {
      target = client->handleServerCutText( data.length );
      size = data.length;
      offset = 0;
    } else client->handleServerCutTextCompleted();
  }
}







/**************************************************/
/* ClientStateReaderBase                          */
/**************************************************/

ClientStateReaderBase::ClientStateReaderBase( Client &_client,
                                              unsigned char *_target,
					      unsigned int _size )
  : State()
  , client( &_client )
  , target( _target )
  , size( _size )
  , offset( 0 )
{}

ClientStateReaderBase::~ClientStateReaderBase()
{}

void ClientStateReaderBase::update() {
  unsigned int count;
  if ( target )
    count = client->connection->receive( target + offset, size - offset );
  else {
    unsigned char* tmp = (unsigned char*) malloc( size - offset );
    count = client->connection->receive( tmp, size - offset );
    free( tmp );
  }
  offset += count;
}




ClientStateProtocolVersion::ClientStateProtocolVersion( Client &client )
  : ClientStateReaderBase( client, (unsigned char *) protocolVersion, 12 )
{}


void ClientStateProtocolVersion::update() {
  ClientStateReaderBase::update();
  if ( size == offset ) { client->handleProtocolVersion( protocolVersion ); }
}





Client::Client()
  : stateProtocolVersion( *this )
  , stateAuthentication( *this )
  , stateInitialisation( *this )
  , stateHandleMessages( *this )
  , stateFramebufferUpdate( *this )
  , stateReadingRectangle( *this )
  , stateCutText( *this )
  , stateDecodingRaw( *this )
  , stateDecodingCopyRect( *this )
  , stateDecodingRRE( *this )
  , stateDecodingCoRRE( *this )
  , stateDecodingHextile( *this )
  , currentState( &stateProtocolVersion )
//  , connection( &_connection )
{}


Client::~Client()
{}


void Client::update() {
  if ( currentState ) currentState->update();
}

void Client::handleProtocolVersion( ProtocolVersion &protocolVersion ) {
  ProtocolVersion clientProtocolVersion = "RFB 003.003\n";
  connection->send( clientProtocolVersion, 12 );
  currentState = &stateAuthentication;
}


void Client::handleAuthenticated()
{
/*
  MessageClientInitialisation message;
  message.clientInitialisation.shared_flag = 1;
  sendMessage( message );
  currentState = &stateInitialisation;
*/
}

void Client::handleAuthenticationFailed()
{}

void Client::handleVNCAuthentication( CARD8 challenge[16] )
{}


void Client::handleServerInitialisation( ServerInitialisation &serverInitialisation )
{
  currentState = &stateHandleMessages;
}


void Client::handleFramebufferUpdate( FramebufferUpdate &framebufferUpdate )
{}


void Client::handleFramebufferUpdateCompleted( FramebufferUpdate &framebufferUpdate )
{
  currentState = &stateHandleMessages;
}


void Client::handleDecodeRectangle( Rectangle &rectangle )
{
  switch( rectangle.encoding_type ) {
    case RFB_ENCODING_RAW:
      currentState = &stateDecodingRaw;
      stateDecodingRaw.startDecoder( rectangle );
      break;
    case RFB_ENCODING_COPYRECT:
      currentState = &stateDecodingCopyRect;
      stateDecodingCopyRect.startDecoder( rectangle );
      break;
    case RFB_ENCODING_RRE:
      currentState = &stateDecodingRRE;
      stateDecodingRRE.startDecoder( rectangle );
      break;
    case RFB_ENCODING_CORRE:
      currentState = &stateDecodingCoRRE;
      stateDecodingCoRRE.startDecoder( rectangle );
      break;
    case RFB_ENCODING_HEXTILE:
      currentState = &stateDecodingHextile;
      stateDecodingHextile.startDecoder( rectangle );
      break;
    default: break;
  }
}


void Client::handleRectangleDecoded( Rectangle &rectangle )
{
  currentState = &stateFramebufferUpdate;
  update();
}


void Client::handleBell()
{}


#ifdef USE_ZLIB_WARREN
void Client::handleZlibEnabled()
{}
#endif // USE_ZLIB_WARREN


unsigned char* Client::handleServerCutText( CARD32 &length )
{
  return NULL;
}

void Client::handleServerCutTextCompleted()
{
  currentState = &stateHandleMessages;
}

/*
void Client::sendMessage( Message &message )
{
  unsigned int size = message.getMessageSize();
  message.copyMessageData( 0, 
                           size,
			   connection->senderBuffer->data 
			   + connection->senderBuffer->end );
  connection->senderBuffer->end += size;
}
*/


} // namespace rfb

