#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#include "sgmldocbook_generator.h"
#include "xmldocbook_generator.h"
#include "util.h"
#include "robodoc.h"
#include "globals.h"
#include "links.h"
#include "generator.h"
#include "items.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif

char               *RB_String_Replace( char *str, char *find, char *repl );
char               *RB_Convert_To_DB_Id( char *label );

/****ih* ROBODoc/SGMLDocBook_Generator
 * FUNCTION
 *   Generator for SGML Docbook output
 ******
 */

/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Doc_Start
 * NAME
 *   RB_SGMLDocBook_Generate_Doc_Start --
 *****
 */

void
RB_SGMLDocBook_Generate_Doc_Start( FILE * dest_doc, char *src_name,
                                   char *name, char toc )
{
    if ( !( course_of_action & DO_SINGLEDOC ) )
    {
		fprintf( dest_doc, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        fprintf( dest_doc,
                 "<!DOCTYPE article PUBLIC \"-//OASIS//DTD DocBook V3.1//EN\">\n" );
        fprintf( dest_doc,
                 "<article>\n<!-- Header -->\n<artheader><title>%s</title>\n"
                 "\n<indexterm>\n<primary>%s</primary>\n</indexterm>",
                 RB_String_Replace( name, ".sgml", "     " ),
                 RB_String_Replace( name, ".sgml", "     " ) );

    }

    /* append SGML-comment with document- and copyright-info. This code
     * ensures that every line has an own comment to avoid problems with
     * buggy browsers */
    fprintf( dest_doc, "\n<!-- Source: %s -->\n", src_name );
    {
        static const char   copyright_text[]
            = COMMENT_ROBODOC COMMENT_COPYRIGHT;
        size_t              i = 0;
        char                previous_char = '\n';
        char                current_char = copyright_text[i];

        while ( current_char )
        {
            if ( previous_char == '\n' )
            {
                fprintf( dest_doc, "<!-- " );
            }
            if ( current_char == '\n' )
            {
                fprintf( dest_doc, " -->" );
            }
            else if ( ( current_char == '-' ) && ( previous_char == '-' ) )
            {
                /* avoid "--" inside SGML-comment, and use "-_" instead; this
                 * looks a bit strange, but one should still be able to figure
                 * out what is meant when reading the output */
                current_char = '_';
            }
            fputc( current_char, dest_doc );
            i += 1;
            previous_char = current_char;
            current_char = copyright_text[i];
        }
    }
    /* append heading and start list of links to functions */
    if ( !( course_of_action & DO_SINGLEDOC ) )
    {
        fprintf( dest_doc, "</artheader>\n" );
    }
    fprintf( dest_doc, "<para>\n<anchor id=\"%s\">Generated from %s with ROBODoc v" VERSION " on ", RB_Convert_To_DB_Id( source_file ), src_name );     /* was src_name */
    RB_TimeStamp( dest_doc );
    fprintf( dest_doc, "\n" );
}


/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Doc_End
 * NAME
 *   RB_SGMLDocBook_Generate_Doc_End --
 *****
 */

void
RB_SGMLDocBook_Generate_Doc_End( FILE * dest_doc, char *name )
{
    if ( !( course_of_action & DO_SINGLEDOC ) )
    {                           /* dont' print this if using SINGLEDOC option */
        fprintf( dest_doc, "</article>\n"
                 "<!-- Keep this comment at the end of the file "
                 "Local variables: "
                 "mode: sgml "
                 "sgml-omittag:t "
                 "sgml-shorttag:t "
                 "sgml-namecase-general:t "
                 "sgml-general-insert-case:lower "
                 "sgml-minimize-attributes:nil "
                 "sgml-always-quote-attributes:t "
                 "sgml-indent-step:1 "
                 "sgml-indent-data:nil "
                 "sgml-parent-document:nil "
                 "sgml-exposed-tags:nil "
                 "sgml-local-catalogs:nil "
                 "sgml-local-ecat-files:nil " "End: " "-->\n" );
    }
}


/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Header_Start
 * NAME
 *   RB_SGMLDocBook_Generate_Header_Start --
 *****
 */

void
RB_SGMLDocBook_Generate_Header_Start( FILE * dest_doc,
                                      struct RB_header *cur_header )
{
    if ( cur_header->name && cur_header->function_name )
    {
        /* DCD: Make the full header A Sect 1 */
        fprintf( dest_doc, "\n<!-- Section1: %s-->\n" "<sect1 id=\"%s\"><title>%s</title>\n<indexterm>\n<primary>%s</primary>\n</indexterm>", cur_header->name, RB_Convert_To_DB_Id( cur_header->name ),        /*DB doesn't like some characters in id names */
                 cur_header->name, cur_header->name );
        fprintf( dest_doc, "\n<anchor id=\"%s\">\n\n",
                 RB_Convert_To_DB_Id( cur_header->function_name ) );
    }
}


/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Header_End
 * NAME
 *   RB_SGMLDocBook_Generate_Header_End --
 *****
 */

void
RB_SGMLDocBook_Generate_Header_End( FILE * dest_doc,
                                    struct RB_header *cur_header )
{
    fprintf( dest_doc, "\n" );
}


/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Index
 * NAME
 *   RB_SGMLDocBook_Generate_Index --
 *****
 */

void
RB_SGMLDocBook_Generate_Index( FILE * dest, char *source )
{
#if 0
    if ( document_title )
    {
        RB_Generate_Doc_Start( dest, source, document_title, 0 );
        fprintf( dest, "<sect1><title>%s</title>\n", document_title );
    }
    else
    {
        RB_Generate_Doc_Start( dest, source, "Master Index File", 0 );
        fprintf( dest, "<sect1><title>Master Index</title>\n" );
    }
    if ( RB_Number_Of_Links( MAIN_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, MAIN_HEADER, "Project Modules" );
    }
    RB_Generate_Index_Table( dest, NO_HEADER, "Source Files" );
    if ( RB_Number_Of_Links( CLASS_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, CLASS_HEADER, "Classes" );
    }
    if ( RB_Number_Of_Links( METHOD_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, METHOD_HEADER, "Methods" );
    }
    if ( RB_Number_Of_Links( STRUCT_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, STRUCT_HEADER, "Structures" );
    }
    if ( RB_Number_Of_Links( TYPE_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, TYPE_HEADER, "Types" );
    }
    if ( RB_Number_Of_Links( PROCEDURE_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, PROCEDURE_HEADER, "Procedures" );
    }
    if ( RB_Number_Of_Links( FUNCTION_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, FUNCTION_HEADER, "Functions" );
    }
    if ( RB_Number_Of_Links( VARIABLE_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, VARIABLE_HEADER, "Variables" );
    }
    if ( RB_Number_Of_Links( CONSTANT_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, CONSTANT_HEADER, "Constants" );
    }
    if ( RB_Number_Of_Links( EXCEPTION_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, EXCEPTION_HEADER, "Exceptions" );
    }
    if ( RB_Number_Of_Links( GENERIC_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, GENERIC_HEADER, "Generic" );
    }
    if ( RB_Number_Of_Links( INTERNAL_HEADER, NULL ) )
    {
        RB_Generate_Index_Table( dest, INTERNAL_HEADER, "Internal" );
    }
    RB_Generate_Doc_End( dest, source );
#endif
}

/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Index_Table
 * NAME
 *   RB_SGMLDocBook_Generate_Index_Table --
 *****
 */

void
RB_SGMLDocBook_Generate_Index_Table( FILE * dest, int type, char *title )
{
#if 0
    struct RB_link     *cur_link;
    int                 cur_column;
    int                 number_of_columns;

    {
        fprintf( dest, "<sect2><title>%s</title>\n", title );

        if ( course_of_action & DO_NO_TABLE )
        {
            cur_column = 0;

            for ( cur_link = first_link;
                  cur_link; cur_link = cur_link->next_link )
            {
                if ( cur_link->type == type )
                {
                    if ( cur_column == 0 )
                    {
                        /* fprintf (dest, "<row>\n"); */
                    }
                    if ( type == NO_HEADER )
                    {
                        if ( RB_Number_Of_Links
                             ( NO_HEADER, cur_link->file_name ) > 1 )
                        {
                            fprintf( dest, "<sect3>\n<title><link linkend=\"%s\"><literal>%s</literal></link>\n</title>\n<literallayout></literallayout>\n", RB_Convert_To_DB_Id( cur_link->label_name ), cur_link->label_name );       /* cur_link->file_name, cur_link->label_name, */
                        }
                        else
                        {
                            fprintf( dest,
                                     "<sect3><title>%s</title><literallayout></literallayout>\n",
                                     cur_link->label_name );
                        }
                    }
                    else
                    {
                        fprintf( dest,
                                 "<sect3>\n<title><link linkend=\"%s\"><literal>%s</literal></link>\n</title>\n<literallayout></literallayout>\n",
                                 RB_Convert_To_DB_Id( cur_link->label_name ),
                                 cur_link->label_name );


                    };
                    ++cur_column;
                    if ( cur_column > number_of_columns )
                    {
                        /*  fprintf (dest, "</row>\n"); */
                        cur_column = 0;
                    }
                }
            }
        }
        else
        {
            fprintf( dest,
                     "<informaltable frame=\"none\">\n<tgroup cols=\"%u\"<tbody>",
                     20 );
            cur_column = 0;

            for ( cur_link = first_link;
                  cur_link; cur_link = cur_link->next_link )
            {
                if ( cur_link->type == type )
                {
                    if ( cur_column == 0 )
                    {
                        fprintf( dest, "<row>\n" );
                    }
                    if ( type == NO_HEADER )
                    {
                        if ( RB_Number_Of_Links
                             ( NO_HEADER, cur_link->file_name ) > 1 )
                        {
                            fprintf( dest, "<entry><link linkend=\"%s\"><literal>%s</literal></link></entry>\n", RB_Convert_To_DB_Id( cur_link->label_name ), cur_link->label_name );   /*cur_link->file_name, cur_link->label_name, */

                        }
                        else
                        {
                            fprintf( dest, "<entry>%s</entry>\n",
                                     cur_link->label_name );
                        }
                    }
                    else
                    {
                        fprintf( dest,
                                 "<entry><link linkend=\"%s\"><literal>%s</literal></link></entry>\n",
                                 RB_Convert_To_DB_Id( cur_link->label_name ),
                                 cur_link->label_name );
                    };
                    ++cur_column;
                    if ( cur_column > number_of_columns )
                    {
                        fprintf( dest, "</row>\n" );
                        cur_column = 0;
                    }
                }
            }
            for ( ; cur_column <= number_of_columns; )
            {
                if ( cur_column == 0 )
                {
                    fprintf( dest, "<row>\n" );
                }
                fprintf( dest, "<entry></entry>\n" );
                cur_column++;
            }
            fprintf( dest, "</row>\n" );
            fprintf( dest, "</tbody></tgroup></informaltable>\n" );

        }                       /* end of DO_NO_TABLE else */

    }                           /*end of case dbsgml */
#endif
}

/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Item_Doc
 * NAME
 *   RB_SGMLDocBook_Generate_Item_Doc --
 *****
 */

#if 0
void
RB_SGMLDocBook_Generate_Item_Doc( FILE * dest_doc, char *dest_name,
                                  char *begin_of_item, char *end_of_item,
                                  char *function_name, int item_type )
{
    fprintf( dest_doc, "<literallayout class=\"monospaced\">" );
    RB_Generate_Item_Body( dest_doc, dest_name, begin_of_item, end_of_item,
                           function_name, item_type, 0 );
    fprintf( dest_doc, "</literallayout>" );
}
#endif

/****if* SGMLDocBook_Generator/RB_SGMLDocBook_Generate_Empty_Item
 * NAME
 *   RB_SGMLDocBook_Generate_Empty_Item --
 *****
 */

void
RB_SGMLDocBook_Generate_Empty_Item( FILE * dest_doc )
{
    fprintf( dest_doc,
             "<literallayout class=\"monospaced\">\n</literallayout>" );
}




void
RB_SGMLDocBook_Generate_Link( FILE * dest_doc, char *dest_name,
                              char *filename, char *labelname, char *linkname,
                              int in_fold )
{
    /* Include the file name in the link if we are in fold
     * PetteriK 08.04.2000
     * Modified for DBSGML David Druffner 8/2001
     */
    if ( in_fold )
    {
        /* We are in fold, always use the file name in the link,
         * in file_name == NULL (i.e. the label is in the current file
         * that we are processing), refer to value in dest_name.
         * This also makes sure that we link correctly if function_name
         * is the same as label_name.
         */
        fprintf( dest_doc, "<link linkend=\"%s\">%s</link>", RB_Convert_To_DB_Id( labelname ), labelname );     /* (file_name ? file_name : dest_name), */
    }
    else if ( filename && strcmp( filename, dest_name ) )
    {
        fprintf( dest_doc, "<link linkend=\"%s\">%s</link>",
                 RB_Convert_To_DB_Id( labelname ), labelname );
    }
    else
    {
        if ( strcmp( labelname, linkname ) )
        {
            fprintf( dest_doc, "<link linkend=\"%s\">%s</link>",
                     RB_Convert_To_DB_Id( labelname ), labelname );
        }
        else
        {
            fprintf( dest_doc, "%s", labelname );
        }
    }
}


/****f* Generator/RB_SGMLDocBook_Generate_Char
 * NAME
 *   RB_SGMLDocBook_Generate_Char
 * SYNOPSIS
 *   void RB_SGMLDocBook_Generate_Char( FILE * dest_doc, char c )
 * FUNCTION
 *   Switchboard to RB_XMLDB_Generate_Char
 * SOURCE
 */

void 
RB_SGMLDocBook_Generate_Char( FILE * dest_doc, int c )
{
	RB_XMLDB_Generate_Char( dest_doc, c );
}

/*****/

/****f* Generator/RB_String_Replace [3.0h]
 * NAME
 *   RB_String_Replace
 * SYNOPSIS
 *   char *RB_String_Replace(char *str, char *find, char *repl)
 *
 * FUNCTION
 *   Replaces oldString with newString in dataString
 *
 * INPUTS
 *   str - string being search
 *    find - old substring to find and replace
 *    repl - new string to replace old substring with
 *
 *   cur_header - pointer to a RB_header structure.
 *
 *  AUTHOR
 *   Found on comp.lang.c news group - posted 11/26/1997 by:
 *   Dan Stubbs (dstubbs@carrot.csc.calpoly.edu)
 *
 *   added and modified by DavidCD 8/2001
 *
 *  NOTES:
 *
 * Return : string str in which each instance of find has been       |
 *        : replaced by an instance of repl.                         |
 * Note 1 : If strlen(repl) > strlen(find) then str must be long     |
 *        : enough to contain all the replacements. If not then the  |
 *        : result is undefined.                                     |
 * Note 2 : If find is the empty string then str is unchanged.       |
 * Note 3 : If repl is the empty string then all instances of find   |
 *        : are deleted from string str.                             |
 * Note 4 : Performance of this implementation is O(n). If find and  |
 *        : repl are the same length then no character in str is     |
 *        : moved. Otherwise, characters following the first instance|
 *        : of find are moved once if strlen(repl) < strlen(find)    |
 *        :              and twice if strlen(repl) > strlen(find).   |
 * Note 5 : If repl contains find as a substring only the instances  |
 *        : of find in the original string are replaced  :-)         |
 * Note 6 : This function can be simplified for various special cases
 *
 * SOURCE
 */


char               *
RB_String_Replace( char *str, char *find, char *repl )
{
    char               *temp;
    char               *p0, *p1, *p2;
    size_t              find_len = strlen( find ),
        repl_len = strlen( repl ), dist = 0, diff = repl_len - find_len;

    temp = malloc( ( strlen( str ) + 1 ) * sizeof( char ) );
    if ( temp )
    {
        strcpy( temp, str );    /* modify the temp string only */

        if ( !find_len || ( p0 = strstr( temp, find ) ) == NULL )
        {
            return temp;
        }

        if ( diff > 0 )
        {                       /*  If extra space needed--insert it. */
            for ( p1 = p0 + 1; p1 != NULL; p1 = strstr( ++p1, find ) )
            {
                dist += diff;
            }
            memmove( p0 + dist, p0, strlen( temp ) - ( p0 - temp ) + 1 );
        }
        p1 = ( diff > 0 ) ? p0 + dist + find_len : p0 + find_len;
        p2 = strstr( p1, find );
        while ( p2 )
        {                       /* While there is another string to replace. */
            memcpy( p0, repl, repl_len );       /* Insert replacement. */
            p0 += repl_len;
            memmove( p0, p1, p2 - p1 ); /* Move just the right segment. */
            p0 += ( p2 - p1 );
            p1 = p2 + find_len;
            p2 = strstr( p1, find );
        }
        memcpy( p0, repl, repl_len );   /* Final replacement. */
        if ( diff < 0 )
        {                       /* If there is a gap at the end of str. */
            p0 += repl_len;
            p2 = strchr( p1, '\0' );
            memmove( p0, p1, p2 - p1 + 1 );
        }
    }
    return ( temp );
}

/*****/


/****f* Generator/RB_Convert_To_DB_Id [3.0h]
 * NAME
 *   RB_Convert_To_DB_Id
 * AUTHOR
 *  David C. Druffner
 * SYNOPSIS           *
 *   char *RB_Convert_To_DB_Id(char *label)
 * FUNCTION
 *  Attempts to change label into a unique conforming Docbook ID.
 *  Adds "ANCHOR" to each label (has to start with a letter) and
 *  substitutes digits for each  non-conforming character.
 * PURPOSE
 *   Used generally on function names.
 * INPUTS
 *   label - string to be changed to Docbook ID
 *  AUTHOR
 *   David Druffner
 *  BUGS
 *    I know I don't have all of the non-conforming characters, so
 *    this will still result in a nsgmls failure for special
 *    characters that aren't converted When modifying, keep in mind
 *    that each resulting digit must be uniquely used by a special
 *    character to maintain a unique id
 *
 * SOURCE
 */

char               *
RB_Convert_To_DB_Id( char *label )
{
    char               *temp;
    char               *dest;
    char               *return_string;

    dest = "ANCHOR";

    temp = malloc( ( strlen( label ) + 4 ) * sizeof( char ) );
    strcpy( temp, label );
    temp = RB_String_Replace( temp, "/", "1" );
    temp = RB_String_Replace( temp, "\\", "2" );
    temp = RB_String_Replace( temp, "_", "3" );
    temp = RB_String_Replace( temp, "$", "4" );
    temp = RB_String_Replace( temp, "#", "5" );
    temp = RB_String_Replace( temp, "*", "6" );
    temp = RB_String_Replace( temp, "(", "7" );
    temp = RB_String_Replace( temp, ")", "8" );
    temp = RB_String_Replace( temp, "@", "9" );
    temp = RB_String_Replace( temp, "&", "10" );
    temp = RB_String_Replace( temp, "!", "11" );
    temp = RB_String_Replace( temp, "~", "12" );

    return_string = malloc( strlen( temp ) + strlen( dest ) + 1 );
    strcpy( return_string, dest );
    strcat( return_string, temp );
    free( temp );
    return return_string;
}

/*****/
