// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <float.h>
#include <limits.h>

#include "fitsimage.h"
#include "framebase.h"
#include "util.h"

#include "mmap.h"
#include "smmap.h"
#include "mmapincr.h"
#include "alloc.h"
#include "allocgz.h"
#include "channel.h"
#include "share.h"
#include "sshare.h"
#include "socket.h"
#include "socketgz.h"
#include "var.h"
#include "iis.h"
#include "hist.h"
#include "nan.h"

// wcs coordinate system strings (for use with wcssubs)

// this is kluge to speed up doug minks wcssubs 'ksearch' routine
extern "C" {
  FitsHead* wcshead = NULL;
  char* ksearchh(char*,char*);

  char* findit(char* cards, char* key)
  {
    if (wcshead)
      return wcshead->find(key);
    else
      return ksearchh(cards, key);
  }
};

FitsImage::FitsImage(FrameBase* p)
{
  parent = p;
  valid = 0;
  objectName = NULL;
  rootBaseFileName = NULL;
  fullBaseFileName = NULL;
  rootFileName = NULL;
  fullFileName = NULL;
  iisFileName = NULL;
  fits = NULL;
  image = NULL;
  data = NULL;

  isHist_ = 0;

  width_ = 0;
  height_ = 0;
  depth_ = 0;
  bitpix_ = 0;

  rotation_ = 0;
  zoom_ = Vector(1,1);
  orientation_ = NORMAL;

  binFunction_ = FitsHist::SUM;
  binFactor_ = Vector(1,1);
  binBufferSize_ = 1024;
  binDepth_ = 1;
  binSmooth_ = 0;
  binSmoothFunc_ = FitsHist::GAUSSIAN;
  binSmoothRadius_ = 3;

  keyLTMV = 0;
  keyATMV = 0;
  keyDTMV = 0;
  keyDATASEC = 0;

  imageToData = Translate(-.5, -.5);
  dataToImage = Translate( .5,  .5);

  memset(iparams,0,5*sizeof(int));
  memset(dparams,0,5*sizeof(int));
  memset(imparams,0,5*sizeof(int));
  memset(irparams,0,5*sizeof(int));
  memset(dmparams,0,5*sizeof(int));
  memset(drparams,0,5*sizeof(int));

  wcs = NULL;
  wcstan = NULL;

  next_ = NULL;

  iisMode = 0;
  iiszt = 0;

  mWidgetToData = widgetToData.mm();
  mPannerToData = pannerToData.mm();
  mMagnifierToData = magnifierToData.mm();
  mPSToData = PSToData.mm();
}

FitsImage::~FitsImage()
{
  if (objectName)
    delete [] objectName;

  if (rootBaseFileName)
    delete [] rootBaseFileName;

  if (fullBaseFileName)
    delete [] fullBaseFileName;

  if (rootFileName)
    delete [] rootFileName;

  if (fullFileName)
    delete [] fullFileName;

  if (iisFileName)
    delete [] iisFileName;

  if (data)
    delete data;

  if (isHist_ && image)
    delete image;

  if (wcs) {
    for (int i=0; i<MULTWCS; i++)
      if (wcs[i])
	wcsfree(wcs[i]);
    delete [] wcs;
  }  

  if (wcstan)
    delete [] wcstan;
}

FitsImageMaster::~FitsImageMaster()
{
  if (fits)
    delete fits;
}

FitsImageSlave::FitsImageSlave(FrameBase* p, const char* fn, FitsFile* f) 
  : FitsImage(p)
{
  fits = f;
  process(fn);
}

// Fits

FitsImageFitsAlloc::FitsImageFitsAlloc(FrameBase* p, const char* fn, 
				       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsFitsAlloc(fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsAllocGZ::FitsImageFitsAllocGZ(FrameBase* p, const char* fn, 
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsFitsAllocGZ(fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsChannel::FitsImageFitsChannel(FrameBase* p, Tcl_Interp* interp, 
					   const char* ch, const char* fn, 
					   FitsFile::FlushMode flush) 
  : FitsImageMaster(p)
{
  fits = new FitsFitsChannel(interp, ch, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsMMap::FitsImageFitsMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsFitsMMap(fn, FitsFile::RELAX);
  process(fn);
}

FitsImageFitsSMMap::FitsImageFitsSMMap(FrameBase* p, const char* hdr, 
				       const char* fn) : FitsImageMaster(p)
{
  fits = new FitsFitsSMMap(hdr, fn);
  process(fn);
}

FitsImageFitsMMapIncr::FitsImageFitsMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsFitsMMapIncr(fn, FitsFile::RELAX);
  process(fn);
}

FitsImageFitsShare::FitsImageFitsShare(FrameBase* p, FrameBase::ShmType type, 
				       int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits = new FitsFitsShareID(id, fn, FitsFile::RELAX);
    break;
  case FrameBase::KEY:
    fits = new FitsFitsShareKey(id, fn, FitsFile::RELAX);
    break;
  }
  process(fn);
}

FitsImageFitsSShare::FitsImageFitsSShare(FrameBase* p, FrameBase::ShmType type,
					 int hdr, int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits = new FitsFitsSShareID(hdr, id, fn);
    break;
  case FrameBase::KEY:
    fits = new FitsFitsSShareKey(hdr, id, fn);
    break;
  }
  process(fn);
}

FitsImageFitsSocket::FitsImageFitsSocket(FrameBase* p, int s, const char* fn, 
					 FitsFile::FlushMode flush) 
  : FitsImageMaster(p)
{
  fits = new FitsFitsSocket(s, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsSocketGZ::FitsImageFitsSocketGZ(FrameBase* p, int s, 
					     const char* fn, 
					     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsFitsSocketGZ(s, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsVar::FitsImageFitsVar(FrameBase* p, Tcl_Interp* interp, 
				   const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsFitsVar(interp, var, fn, FitsFile::RELAX);
  process(fn);
}

// Fits Next

FitsImageFitsNextAlloc::FitsImageFitsNextAlloc(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextAlloc(prev);
  process(fn);
}

FitsImageFitsNextAllocGZ::FitsImageFitsNextAllocGZ(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextAllocGZ(prev);
  process(fn);
}

FitsImageFitsNextChannel::FitsImageFitsNextChannel(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextChannel(prev);
  process(fn);
}

FitsImageFitsNextMMap::FitsImageFitsNextMMap(FrameBase* p, 
					     const char* fn,
					     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextMMap(prev);
  process(fn);
}

FitsImageFitsNextSMMap::FitsImageFitsNextSMMap(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextSMMap(prev);
  process(fn);
}

FitsImageFitsNextMMapIncr::FitsImageFitsNextMMapIncr(FrameBase* p, 
						     const char* fn,
						     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextMMapIncr(prev);
  process(fn);
}

FitsImageFitsNextShare::FitsImageFitsNextShare(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextShare(prev);
  process(fn);
}

FitsImageFitsNextSShare::FitsImageFitsNextSShare(FrameBase* p, 
						 const char* fn,
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextSShare(prev);
  process(fn);
}

FitsImageFitsNextSocket::FitsImageFitsNextSocket(FrameBase* p, 
						 const char* fn,
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextSocket(prev);
  process(fn);
}

FitsImageFitsNextSocketGZ::FitsImageFitsNextSocketGZ(FrameBase* p, 
						     const char* fn, 
						     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextSocketGZ(prev);
  process(fn);
}

FitsImageFitsNextVar::FitsImageFitsNextVar(FrameBase* p, 
					   const char* fn,
					   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsFitsNextVar(prev);
  process(fn);
}

FitsImageFitsNextHist::FitsImageFitsNextHist(FrameBase* p, 
					     FitsImage* fi,
					     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsHistNext(prev);
  process(NULL);

  fits->setpFilter(fi->getHistFilter());
  rootBaseFileName = dupstr(fi->getRootBaseFileName());
  fullBaseFileName = dupstr(fi->getFullBaseFileName());
  iisFileName = dupstr(fi->getFullBaseFileName());
}

// Array

FitsImageArrAlloc::FitsImageArrAlloc(FrameBase* p, const char* fn,
				     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsArrAlloc(fn, flush);
  process(fn);
}

FitsImageArrAllocGZ::FitsImageArrAllocGZ(FrameBase* p, const char* fn,
					 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsArrAllocGZ(fn, flush);
  process(fn);
}

FitsImageArrChannel::FitsImageArrChannel(FrameBase* p, Tcl_Interp* interp, 
					 const char* ch, const char* fn,
					 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsArrChannel(interp, ch, fn, flush);
  process(fn);
}

FitsImageArrMMap::FitsImageArrMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsArrMMap(fn);
  process(fn);
}

FitsImageArrMMapIncr::FitsImageArrMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsArrMMapIncr(fn);
  process(fn);
}

FitsImageArrShare::FitsImageArrShare(FrameBase* p, FrameBase::ShmType type, 
				     int id, const char* fn) 
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits = new FitsArrShareID(id, fn);
    break;
  case FrameBase::KEY:
    fits = new FitsArrShareKey(id, fn);
    break;
  }
  process(fn);
}

FitsImageArrSocket::FitsImageArrSocket(FrameBase* p, int s, const char* fn,
				       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsArrSocket(s, fn, flush);
  process(fn);
}

FitsImageArrSocketGZ::FitsImageArrSocketGZ(FrameBase* p, int s, const char* fn,
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsArrSocketGZ(s, fn, flush);
  process(fn);
}

FitsImageArrVar::FitsImageArrVar(FrameBase* p, Tcl_Interp* interp, 
				 const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsArrVar(interp, var, fn);
  process(fn);
}

// Mosaic

FitsImageMosaicAlloc::FitsImageMosaicAlloc(FrameBase* p, const char* fn,
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicAlloc(fn, flush);
  processMosaic(fn);
}

FitsImageMosaicAllocGZ::FitsImageMosaicAllocGZ(FrameBase* p, const char* fn,
					       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicAllocGZ(fn, flush);
  processMosaic(fn);
}

FitsImageMosaicChannel::FitsImageMosaicChannel(FrameBase* p, 
					       Tcl_Interp* interp,
					       const char* ch, 
					       const char* fn,
					       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicChannel(interp, ch, flush);
  processMosaic(fn);
}

FitsImageMosaicMMap::FitsImageMosaicMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicMMap(fn);
  processMosaic(fn);
}

FitsImageMosaicMMapIncr::FitsImageMosaicMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicMMapIncr(fn);
  processMosaic(fn);
}

FitsImageMosaicShare::FitsImageMosaicShare(FrameBase* p, 
					   FrameBase::ShmType type,
					   int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits = new FitsMosaicShareID(id);
    break;
  case FrameBase::KEY:
    fits = new FitsMosaicShareKey(id);
    break;
  }
  processMosaic(fn);
}

FitsImageMosaicSocket::FitsImageMosaicSocket(FrameBase* p, 
					     int s, const char* fn,
					     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicSocket(s, flush);
  processMosaic(fn);
}

FitsImageMosaicSocketGZ::FitsImageMosaicSocketGZ(FrameBase* p, 
						 int s, const char* fn,
						 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicSocketGZ(s, flush);
  processMosaic(fn);
}

FitsImageMosaicVar::FitsImageMosaicVar(FrameBase* p, Tcl_Interp* interp,
				       const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicVar(interp, var, fn);
  processMosaic(fn);
}

// Mosaic Next

FitsImageMosaicNextAlloc::FitsImageMosaicNextAlloc(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev,
						   FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextAlloc(prev, flush);
  processMosaic(fn);
}

FitsImageMosaicNextAllocGZ::FitsImageMosaicNextAllocGZ(FrameBase* p,
						       const char* fn,
						       FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextAllocGZ(prev, flush);
  processMosaic(fn);
}

FitsImageMosaicNextChannel::FitsImageMosaicNextChannel(FrameBase* p, 
						       const char* fn, 
						       FitsFile* prev,
						     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextChannel(prev, flush);
  processMosaic(fn);
}

FitsImageMosaicNextMMap::FitsImageMosaicNextMMap(FrameBase* p, const char* fn, 
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextMMap(prev);
  processMosaic(fn);
}

FitsImageMosaicNextMMapIncr::FitsImageMosaicNextMMapIncr(FrameBase* p,
							 const char* fn, 
							 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextMMapIncr(prev);
  processMosaic(fn);
}

FitsImageMosaicNextShare::FitsImageMosaicNextShare(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextShare(prev);
  processMosaic(fn);
}

FitsImageMosaicNextSocket::FitsImageMosaicNextSocket(FrameBase* p, 
						     const char* fn, 
						     FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextSocket(prev, flush);
  processMosaic(fn);
}

FitsImageMosaicNextSocketGZ::FitsImageMosaicNextSocketGZ(FrameBase* p,
							 const char* fn, 
							 FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextSocketGZ(prev, flush);
  processMosaic(fn);
}

FitsImageMosaicNextVar::FitsImageMosaicNextVar(FrameBase* p, 
					       const char* fn, 
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits = new FitsMosaicNextVar(prev);
  processMosaic(fn);
}

// IIS

FitsImageIIS::FitsImageIIS(FrameBase* p, int w, int h) : FitsImageMaster(p)
{
  fits = new FitsIIS(w, h);
  process("");
  iisMode = 1;
}

void FitsImageIIS::iisErase()
{
  if (fits)
    ((FitsIIS*)fits)->erase();
}

const char* FitsImageIIS::iisGet(int x, int y)
{
  if (fits)
    return ((FitsIIS*)fits)->get(x,y);
}

void FitsImageIIS::iisSet(int x, int y, const char* d, int l)
{
  if (fits)
    ((FitsIIS*)fits)->set(x,y,d,l);
}

void FitsImageIIS::iisWCS(const Matrix& m, const Vector& z, int zt)
{
  dataToImage = Translate(-.5,-.5) * m;
  imageToData = dataToImage.invert();
  iisz = z;
  iiszt = zt;
}

void FitsImage::process(const char* fn)
{
  if (!fits->isValid()) {
    delete fits;
    fits = NULL;
    image = NULL;
    isHist_ = 0;
    valid = 0;

    return;
  }

  if (fn)
    setFileName(fn);

  if (fits->isImage()) {
    image = fits;
    isHist_ = 0;

    FitsHead* head = image->head();
    width_ = head->naxis(1);
    height_ = head->naxis(2);
    depth_ = head->naxis(3);
    bitpix_ = head->bitpix();

    valid = load();
  }
  else if (fits->isTable()) {
    isHist_ = 1;
    initHist();
  }
}

void FitsImage::processMosaic(const char* fn)
{
  if (fits->isValid() && fits->isImage()) {
    setFileName(fn);

    image = fits;
    isHist_ = 0;

    FitsHead* head = image->head();
    width_ = head->naxis(1);
    height_ = head->naxis(2);
    depth_ = 1;
    bitpix_ = head->bitpix();

    valid = load();
    return;
  }

  delete fits;
  fits = NULL;
  image = NULL;
  isHist_ = 0;
  valid = 0;
}

#if __GNUC__ >= 3
void FitsImage::setFileName(const char* fn)
{
  if (fullBaseFileName)
    delete [] fullBaseFileName;
  fullBaseFileName = NULL;

  if (rootBaseFileName)
    delete [] rootBaseFileName;
  rootBaseFileName = NULL;

  if (iisFileName)
    delete [] iisFileName;
  iisFileName = NULL;

  // no filename to set
  if (!fn)
    return;

  // strip any '[]'
  char* ffn = strip(fn);

  if(fits->find("EXTNAME")) {
    char* c = fits->getString("EXTNAME");
    {
      ostringstream str;
      str << ffn << '[' << c << ']' << ends;
      fullBaseFileName = dupstr(str.str().c_str());
    }
    {
      char* m = root(ffn);
      ostringstream str;
      str << m << '[' << c << ']' << ends;
      delete [] m;
      rootBaseFileName = dupstr(str.str().c_str());
    }
    delete [] c;
  }
  else if (fits->ext()) {
    {
      ostringstream str;
      str << ffn << '[' << fits->ext() << ']' << ends;
      fullBaseFileName = dupstr(str.str().c_str());
    }
    {
      char* m = root(ffn);
      ostringstream str;
      str << m << '[' << fits->ext() << ']' << ends;
      delete [] m;
      rootBaseFileName = dupstr(str.str().c_str());
    }
  }
  else {
    fullBaseFileName = dupstr(ffn);
    rootBaseFileName = root(ffn);
  }
  
  // by default, iisFileName is fullBaseFileName
  if (fullBaseFileName)
    iisFileName = dupstr(fullBaseFileName);

  delete [] ffn;
  updateFileName();
}
#else
void FitsImage::setFileName(const char* fn)
{
  if (fullBaseFileName)
    delete [] fullBaseFileName;
  fullBaseFileName = NULL;

  if (rootBaseFileName)
    delete [] rootBaseFileName;
  rootBaseFileName = NULL;

  if (iisFileName)
    delete [] iisFileName;
  iisFileName = NULL;

  // no filename to set
  if (!fn)
    return;

  // strip any '[]'
  char* ffn = strip(fn);

  char lbuf[256];
  if(fits->find("EXTNAME")) {
    char* c = fits->getString("EXTNAME");
    {
      ostrstream str(lbuf,256);
      str << ffn << '[' << c << ']' << ends;
      fullBaseFileName = dupstr(lbuf);
    }
    {
      char* m = root(ffn);
      ostrstream str(lbuf,256);
      str << m << '[' << c << ']' << ends;
      delete [] m;
      rootBaseFileName = dupstr(lbuf);
    }
    delete [] c;
  }
  else if (fits->ext()) {
    {
      ostrstream str(lbuf,256);
      str << ffn << '[' << fits->ext() << ']' << ends;
      fullBaseFileName = dupstr(lbuf);
    }
    {
      char* m = root(ffn);
      ostrstream str(lbuf,256);
      str << m << '[' << fits->ext() << ']' << ends;
      delete [] m;
      rootBaseFileName = dupstr(lbuf);
    }
  }
  else {
    fullBaseFileName = dupstr(ffn);
    rootBaseFileName = root(ffn);
  }
  
  // by default, iisFileName is fullBaseFileName
  if (fullBaseFileName)
    iisFileName = dupstr(fullBaseFileName);

  delete [] ffn;
  updateFileName();
}
#endif

#if __GNUC__ >= 3
void FitsImage::updateFileName()
{
  if (fullFileName)
    delete [] fullFileName;
  fullFileName = NULL;

  if (rootFileName)
    delete [] rootFileName;
  rootFileName = NULL;

  char lbuf[256];
  char* filter = (char*)fits->pFilter();
  char* slice = (char*)fits->pSliceFilter();
  int doFilter = (filter && *filter);
  int doSlice = (slice && *slice);

  if (fullBaseFileName) {
    ostringstream str;
    if (doFilter && doSlice)
      str << fullBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << fullBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << fullBaseFileName << '[' << slice << ']' << ends;
    else
      str << fullBaseFileName << ends;

    fullFileName = dupstr(str.str().c_str());
  }

  if (rootBaseFileName) {
    ostringstream str;
    if (doFilter && doSlice)
      str << rootBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << rootBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << rootBaseFileName << '[' << slice << ']' << ends;
    else
      str << rootBaseFileName << ends;

    rootFileName = dupstr(str.str().c_str());
  }
}
#else
void FitsImage::updateFileName()
{
  if (fullFileName)
    delete [] fullFileName;
  fullFileName = NULL;

  if (rootFileName)
    delete [] rootFileName;
  rootFileName = NULL;

  char lbuf[256];
  char* filter = (char*)fits->pFilter();
  char* slice = (char*)fits->pSliceFilter();
  int doFilter = (filter && *filter);
  int doSlice = (slice && *slice);

  if (fullBaseFileName) {
    ostrstream str(lbuf,256);
    if (doFilter && doSlice)
      str << fullBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << fullBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << fullBaseFileName << '[' << slice << ']' << ends;
    else
      str << fullBaseFileName << ends;

    fullFileName = dupstr(lbuf);
  }

  if (rootBaseFileName) {
    ostrstream str(lbuf,256);
    if (doFilter && doSlice)
      str << rootBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << rootBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << rootBaseFileName << '[' << slice << ']' << ends;
    else
      str << rootBaseFileName << ends;

    rootFileName = dupstr(lbuf);
  }
}
#endif

void FitsImage::iisSetFileName(const char* fn)
{
  if (iisFileName)
    delete [] iisFileName;
  iisFileName = dupstr(fn);
}

char* FitsImage::strip(const char* fn)
{
  if (fn) {
    char* r = dupstr(fn);           // dup the string
    char* ptr = r;                  // init the ptr
    while(*ptr != '[' && *ptr)      // walk it forward til '[' or end
      ptr++;
    *ptr = '\0';                    // zero out rest

    return r;                       // got it!
  }
  else
    return NULL;
}

char* FitsImage::root(const char* fn)
{
  if (fn) {
    const char* ptr = fn;           // init the ptr
    while(*ptr++);                  // walk it forward to end of string
    ptr--;                          // backup one
    while(*ptr != '/' && ptr != fn) // walk it backward til last / or beginning
      ptr--;
    if (*ptr == '/')                // step it over the last '/'
      ptr++;
    return dupstr(ptr);             // got it!
  }
  else
    return NULL;
}

void FitsImage::updateMatrices(FrameBase::MosaicType mt, 
			       Matrix& rgbToRef, Matrix& refToUser,
			       Matrix& userToWidget, Matrix& widgetToCanvas,
			       Matrix& userToPanner)
{
  // basic matrices

  // if in mosaic mode, we need to flip the image, based on orientation
  // which is set via the ATM/V keywords when the file is parsed.

  switch (mt) {
  case FrameBase::NOMOSAIC:
    dataToRef = rgbToRef;
    refToData = dataToRef.invert();
    break;
  case FrameBase::IRAF:
    {
      // this is kluge, put it works ;-)
      // if the segment is flipped, we can have a discontinuity at
      // the edges, due to round off errors, so we 'nudge' it

      Vector oo = origin_;
      Matrix flip;
      switch (orientation_) {
      case XX:
	flip = FlipX();
	oo -= Vector(FLT_EPSILON,0);
	break;
      case YY:
	flip = FlipY();
	oo -= Vector(0,FLT_EPSILON);
	break;
      case XY:
	flip = FlipXY();
	oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	break;
      }

      Vector c = datasec.center() * imageToData;
      dataToRef =
	Translate(-c) *
	Rotate(rotation_) *
	flip *
	Scale(zoom_) *
	Translate(c) *
	Translate(oo) *
	rgbToRef;
      refToData = dataToRef.invert();
    }
    break;
  case FrameBase::WCSMOSAIC:
  case FrameBase::WFPC2:
    {
      // this is kluge, put it works ;-)
      // if the segment is flipped, we can have a discontinuity at
      // the edges, due to round off errors, so we 'nudge' it

      Vector oo = origin_;
      Matrix flip;
      double r = degToRad(180);
      if ((rotation_ > r-FLT_EPSILON && rotation_ < r+FLT_EPSILON) || 
	  (-rotation_ > r-FLT_EPSILON && -rotation_ < r+FLT_EPSILON)) {
	switch (orientation_) {
	case NORMAL:
	  oo -= Vector(0,FLT_EPSILON);
	  break;
	case XX:
	  flip = FlipX();
	  oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	  break;
	case YY:
	  flip = FlipY();
	  break;
	case XY:
	  flip = FlipXY();
	  oo -= Vector(FLT_EPSILON,0);
	  break;
	}
      }
      else {
	switch (orientation_) {
	case NORMAL:
	  break;
	case XX:
	  flip = FlipX();
	  oo -= Vector(FLT_EPSILON,0);
	  break;
	case YY:
	  flip = FlipY();
	  oo -= Vector(0,FLT_EPSILON);
	  break;
	case XY:
	  flip = FlipXY();
	  oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	  break;
	}
      }

      Vector c = getWCScrpix(WCS) * imageToData;
      dataToRef =
	Translate(-c) *
	Rotate(rotation_) *
	flip *
	Scale(zoom_) *
	Translate(c) *
	Translate(oo) *
	rgbToRef;
      refToData = dataToRef.invert();
    }
    break;
  }

  // derived matrices

  refToCanvas = refToUser * userToWidget * widgetToCanvas;
  canvasToRef = refToCanvas.invert();

  physicalToData = physicalToImage * imageToData;
  dataToPhysical = physicalToData.invert();

  physicalToRef = physicalToImage * imageToData * dataToRef;
  refToPhysical = physicalToRef.invert();

  physicalToUser = physicalToImage * imageToData * dataToRef * refToUser;
  userToPhysical = physicalToUser.invert();

  physicalToCanvas = physicalToImage * imageToData * dataToRef * refToUser * 
    userToWidget * widgetToCanvas;
  canvasToPhysical = physicalToCanvas.invert();

  physicalToPanner = physicalToImage * imageToData * dataToRef * refToUser * 
    userToPanner;
  pannerToPhysical = physicalToPanner.invert();

  imageToRef = imageToData * dataToRef;
  refToImage = imageToRef.invert();

  imageToUser = imageToData * dataToRef * refToUser;
  userToImage = imageToUser.invert();

  imageToWidget = imageToData * dataToRef *  refToUser * userToWidget;
  widgetToImage = imageToWidget.invert();

  imageToCanvas = imageToData * dataToRef *  refToUser *
    userToWidget *  widgetToCanvas;
  canvasToImage = imageToCanvas.invert();

  imageToPanner = imageToData * dataToRef * refToUser * userToPanner;
  pannerToImage = imageToPanner.invert();

  dataToUser = dataToRef * refToUser;
  userToData = dataToUser.invert();

  dataToWidget = dataToRef *  refToUser *  userToWidget;
  widgetToData = dataToWidget.invert();

  dataToCanvas = dataToRef *  refToUser *  userToWidget * widgetToCanvas;
  canvasToData = dataToCanvas.invert();

  dataToPanner = dataToRef * refToUser * userToPanner;
  pannerToData = dataToPanner.invert();

  amplifierToRef = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef;
  refToAmplifier = amplifierToRef.invert();

  amplifierToUser = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser;
  userToAmplifier = amplifierToUser.invert();

  amplifierToCanvas = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser *
    userToWidget * 
    widgetToCanvas;
  canvasToAmplifier = amplifierToCanvas.invert();

  amplifierToPanner = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser * 
    userToPanner;
  pannerToAmplifier = amplifierToPanner.invert();

  amplifierToImage =
    amplifierToPhysical *
    physicalToImage;
  imageToAmplifier = amplifierToImage.invert();

  detectorToRef = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef;
  refToDetector = detectorToRef.invert();

  detectorToUser = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser;
  userToDetector = detectorToUser.invert();

  detectorToCanvas = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser *
    userToWidget * 
    widgetToCanvas;
  canvasToDetector = detectorToCanvas.invert();

  detectorToPanner = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser * 
    userToPanner;
  pannerToDetector = detectorToPanner.invert();

  detectorToImage =
    detectorToPhysical *
    physicalToImage;
  imageToDetector = detectorToImage.invert();

}

#if __GNUC__ >= 3
const char* FitsImage::getValue(const Vector& v)
{
  if (!isIIS() || (iiszt != 1))
    return data->getValue(v);
  else {
    double value = data->getValueDouble(v);

    ostringstream str;
    if (value <= 1)
      str << '<' << iisz[0] << ends;
    else if (value >= 200)
      str << '>' << iisz[1] << ends;
    else
      str << ((value-1) * (iisz[1]-iisz[0]))/199 + iisz[0] << ends;

    memcpy(buf,str.str().c_str(), str.str().length());
    return buf;
  }
}
#else
const char* FitsImage::getValue(const Vector& v)
{
  if (!isIIS() || (iiszt != 1))
    return data->getValue(v);
  else {
    double value = data->getValueDouble(v);

    ostrstream str(buf,32);
    if (value <= 1)
      str << '<' << iisz[0] << ends;
    else if (value >= 200)
      str << '>' << iisz[1] << ends;
    else
      str << ((value-1) * (iisz[1]-iisz[0]))/199 + iisz[0] << ends;

    return buf;
  }
}
#endif

void FitsImage::setBinCursor(const Vector& v)
{
  histCursor = v * userToPhysical;
}

void FitsImage::setBinFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] *= bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] *= bb[1] <= 0 ? 1 : bb[1];
}

void FitsImage::setBinToFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] = bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] = bb[1] <= 0 ? 1 : bb[1];
}

Matrix FitsImage::updateHistCenter()
{
  if (isHist_)
    return nextHist(getHistCenter());
  else
    return Matrix();
}

Matrix FitsImage::updateHistCursor()
{
  if (isHist_)
    return nextHist(histCursor);
  else
    return Matrix();
}

Matrix FitsImage::updateHist(const Vector& v)
{
  if (isHist_)
    return nextHist(v * userToPhysical);
  else
    return Matrix();
}

void FitsImage::initHist()
{
  // just in case
  if (!fits->isTable())
    return;

  // make sure we have cols to bin on
  if (!fits->pBinX())
    fits->setpBinX("X");

  if (!fits->pBinY())
    fits->setpBinY("Y");

  if (!fits->pBinZ())
    fits->setpBinZ("TIME");

  // update default values from parent
  binFunction_ = parent->binFunction_;
  binBufferSize_ = parent->binBufferSize_;
  binDepth_ = parent->binDepth_;
  binFactor_ = parent->binFactor_;
  binSmooth_ = parent->binSmooth_;
  binSmoothFunc_ = parent->binSmoothFunc_;
  binSmoothRadius_ = parent->binSmoothRadius_;

  nextHist(getHistCenter());
}

Matrix FitsImage::nextHist(const Vector& c)
{
  // remember where we are pointing
  histCursor = c;

  // cursor is in bin (physical) coords
  calcHistSize();
  Vector center = Vector(width_, height_)/2;

  if (DebugBin) {
    cerr << "width height: " << width_ << ' ' << height_ << endl;
    cerr << "center: " << center << endl;
    cerr << "center.ceil(): " << center.ceil() << endl;
  }

  if (binFactor_[0]<1 || binFactor_[1]<1) {
    actualHistCursor = histCursor;

    if (DebugBin)
      cerr << "histCursor: " << histCursor << endl;
  }
  else {
    // force to a bin boundary, then translate to center of bin cell
    //    actualHistCursor = ((histCursor/binFactor_).floor() * binFactor_) +
    //      Vector(.5,.5);
    actualHistCursor[0] = (floor(histCursor[0]/binFactor_[0]) * binFactor_[0])
      +  .5;
    actualHistCursor[1] = (floor(histCursor[1]/binFactor_[1]) * binFactor_[1])
      +  .5;

    // now, figure out any offset due to mod(lowerleft,binFactor_)
    FitsTableHDU* hdu = (FitsTableHDU*)(fits->head())->hdu();
    Vector xd = hdu->dimension(fits->pBinX());
    Vector yd = hdu->dimension(fits->pBinY());
    Vector ll(xd[0],yd[0]);

    //    Vector a = ((ll/binFactor_).floor() * binFactor_) + Vector(.5,.5);
    Vector a;
    a[0] = (floor(ll[0]/binFactor_[0]) * binFactor_[0]) + .5;
    a[1] = (floor(ll[1]/binFactor_[1]) * binFactor_[1]) + .5;
    Vector offset = a-ll;
    actualHistCursor -= offset;

    if (DebugBin) {
      cerr << "histCursor: " << histCursor << endl;
      cerr << "actualHistCursor: " << actualHistCursor << endl;
      cerr << "ll: " << ll << endl;
      cerr << "offset: " << offset << endl;
    }
  }

  // new physicalToData
  Matrix mm = 
    Translate(-actualHistCursor) * 
    Scale(1./binFactor_[0],1./binFactor_[1]) *
    Translate(center.ceil());

  if (DebugBin)
    cerr << "mm: " << mm << endl << endl;

  if (image)
    delete image;

  image = new FitsHist(fits, width_, height_, depth_, mm,
		       binFunction_, binFactor_, 
		       binSmooth_, binSmoothFunc_, binSmoothRadius_);

  // reset LTMV keywords
  keyLTMV = 0;

  if (image->isValid()) {
    valid = load();
    return refToPhysical * mm * dataToRef;
  }
  else {
    delete fits;
    fits = NULL;
    if (image)
      delete image;
    isHist_ = 0;
    valid = 0;
    return Matrix();
  }
}

void FitsImage::calcHistSize()
{
  //  Vector s = getHistDim()/binFactor_;
  Vector d = getHistDim();
  Vector s;
  s[0] = d[0]/binFactor_[0];
  s[1] = d[1]/binFactor_[1];

  width_ = (int)(s[0]<binBufferSize_? s[0] : binBufferSize_);
  height_= (int)(s[1]<binBufferSize_? s[1] : binBufferSize_);
  depth_ = binDepth_;
}

Vector FitsImage::getHistColMinMax(const char* col)
{
  if (fits->isTable())
    return fits->getColMinMax(col);
  else
    return Vector();
}

Vector FitsImage::getHistDim()
{
  // assumes we aready have our columns

  FitsTableHDU* hdu = (FitsTableHDU*)(fits->head())->hdu();
  Vector xd = hdu->dimension(fits->pBinX());
  Vector yd = hdu->dimension(fits->pBinY());

  // if DBL_MAX, we will get NaN
  Vector r(xd[1]-xd[0],yd[1]-yd[0]);

  if (isNaNd(r[0]) || isNaNd(r[1]))
    return Vector(DBL_MAX,DBL_MAX);
  else
    return r;
}

Vector FitsImage::getHistCenter()
{
  // assumes we aready have our columns

  FitsTableHDU* hdu = (FitsTableHDU*)(fits->head())->hdu();
  Vector xd = hdu->dimension(fits->pBinX());
  Vector yd = hdu->dimension(fits->pBinY());

  // if DBL_MAX, we will get NaN
  Vector r = Vector(xd[1]-xd[0],yd[1]-yd[0])/2 + Vector(xd[0],yd[0]);

  if (isNaNd(r[0]) || isNaNd(r[1]))
    return Vector();
  else
    return r;
}

const char* FitsImage::getHistFilter()
{
  if (fits)
    return fits->pFilter();
  else
    return NULL;
}

const char* FitsImage::getHistX()
{
  return fits ? fits->pBinX() : NULL;
}

const char* FitsImage::getHistY()
{
  return fits ? fits->pBinY() : NULL;
}

const char* FitsImage::getHistZ()
{
  return fits ? fits->pBinZ() : NULL;
}

char* FitsImage::getHistList()
{
  if (fits && fits->isTable()) {
    FitsHead* head = fits->head();
    return ((FitsTableHDU*)head->hdu())->list();
  }
  else
    return NULL;
}

Vector FitsImage::getHistZlim()
{
  return fits ? fits->getBinZlim() : Vector();
}

void FitsImage::setBinX(const char* str)
{
  if (fits)
    fits->setpBinX(str);
}

void FitsImage::setBinY(const char* str)
{
  if (fits)
    fits->setpBinY(str);
}

void FitsImage::setBinZ(const char* str)
{
  if (fits)
    fits->setpBinZ(str);
}

void FitsImage::setBinFilter(const char* filter)
{
  if (fits)
    fits->setpFilter(filter);
}

void FitsImage::setBinSliceFilter(const char* filter)
{
  if (fits)
    fits->setpSliceFilter(filter);
}

void FitsImage::setBinZlim(const Vector& lim)
{
  if (fits)
    fits->setBinZlim(lim);
}

void FitsImage::updatePS(Matrix ps)
{
  dataToPS = dataToRef * ps;
  PSToData = dataToPS.invert();
}

void FitsImage::updateMagnifierMatrices(Matrix& refToMagnifier)
{
  dataToMagnifier = dataToRef * refToMagnifier;
  magnifierToData = dataToMagnifier.invert();
}

BBox FitsImage::getIRAFbb()
{
  // return in IMAGE coords
  Vector ll = detsize.ll * imageToData * Scale(ccdsum) * dataToImage;
  Vector ur = detsize.ur * imageToData * Scale(ccdsum) * dataToImage;
  return BBox(ll,ur);
}

int* FitsImage::getDataParams(FrScale::ScanMode which)
{
  // return in 0 to width_, 0 to height_
  switch (which) {
  case FrScale::NODATASEC:
    return iparams;
  case FrScale::DATASEC:
    return dparams;
  case FrScale::UNODATASEC:
    return irparams;
  case FrScale::UDATASEC:
    return drparams;
  }
}

int* FitsImage::getMinMaxParams(FrScale::ScanMode which)
{
  // return in IMAGE coords
  switch (which) {
  case FrScale::NODATASEC:
    return iparams;
  case FrScale::DATASEC:
    return dparams;
  case FrScale::UNODATASEC:
    return imparams;
  case FrScale::UDATASEC:
    return dmparams;
  }
}

void FitsImage::setDataParams(int x0, int y0, int x1, int y1)
{
  irparams[0] = drparams[0] = x1;
  irparams[1] = drparams[1] = x0;
  irparams[2] = drparams[2] = x1;
  irparams[3] = drparams[3] = y0;
  irparams[4] = drparams[4] = y1;

  // now clip by DATASEC

  if (dparams[0] < drparams[0])
    drparams[0] = dparams[0];
  if (dparams[1] > drparams[1])
    drparams[1] = dparams[1];
  if (dparams[2] < drparams[2])
    drparams[2] = dparams[2];
  if (dparams[3] > drparams[3])
    drparams[3] = dparams[3];
  if (dparams[4] < drparams[4])
    drparams[4] = dparams[4];
}

void FitsImage::setMinMaxParams(int x0, int y0, int x1, int y1)
{
  imparams[0] = dmparams[0] = x1;
  imparams[1] = dmparams[1] = x0;
  imparams[2] = dmparams[2] = x1;
  imparams[3] = dmparams[3] = y0;
  imparams[4] = dmparams[4] = y1;

  // now clip by DATASEC

  if (dparams[0] < dmparams[0])
    dmparams[0] = dparams[0];
  if (dparams[1] > dmparams[1])
    dmparams[1] = dparams[1];
  if (dparams[2] < dmparams[2])
    dmparams[2] = dparams[2];
  if (dparams[3] > dmparams[3])
    dmparams[3] = dparams[3];
  if (dparams[4] < dmparams[4])
    dmparams[4] = dparams[4];

  if (data)
    data->resetMinMax();
}

int FitsImage::load()
{
  resetWCS();

  memset(iparams,0,5*sizeof(int));
  memset(dparams,0,5*sizeof(int));
  memset(imparams,0,5*sizeof(int));
  memset(irparams,0,5*sizeof(int));
  memset(dmparams,0,5*sizeof(int));
  memset(drparams,0,5*sizeof(int));

  if (!processKeywords())
    return 0;

  // create a valid FitsData
  FitsHead* head = image->head();
  if (data)
    delete data;

  switch (head->bitpix()) {
  case 8: 
    data = new FitsDatam<unsigned char>(image, parent);
    break;
  case 16:
    data = new FitsDatam<short>(image, parent);
    break;
  case -16:
    data = new FitsDatam<unsigned short>(image, parent);
    break;
  case 32:
    data = new FitsDatam<int>(image, parent);
    break;
  case -32:
    data = new FitsDatam<float>(image, parent);
    break;
  case -64:
    data = new FitsDatam<double>(image, parent);
    break;
  }

  return data ? 1 : 0;
}

void FitsImage::resetWCS()
{
  // Process OBJECT keyword
  if (objectName)
    delete [] objectName;
  objectName = image->getString("OBJECT");

  // Process WCS keywords
  initWCS(image->head());
}

void FitsImage::replaceWCS(FitsHead* h)
{
  // Process OBJECT keyword
  if (objectName)
    delete [] objectName;
  objectName = h->getString("OBJECT");

  // Process WCS keywords
  initWCS(h);
}

void FitsImage::appendWCS(FitsHead* app)
{
  // process OBJECT keyword

  char* obj = app->getString("OBJECT");
  if (obj) {
    if (objectName)
      delete [] objectName;
    objectName = obj;
  }

  // Process WCS keywords
  FitsHead* head = image->head();

  // append wcs keywords to the end of the header
  int ll = head->headbytes()+app->headbytes();
  char* cards = new char[ll];

  // copy default wcs
  memcpy(cards, head->cards(), head->headbytes());

  // find first END and zero out
  for (int i=0; i<head->headbytes(); i+=80)
    if (!strncmp(cards+i,"END",3)) {
      memcpy(cards+i, "   ",3);
      break;
    }

  // copy appended wcs
  memcpy(cards+head->headbytes(), app->cards(), app->headbytes());


  FitsHead* hhead = new FitsHead(cards,ll,FitsHead::EXTERNAL);
  initWCS(hhead);
  delete hhead;
  delete [] cards;
}

void FitsImage::initWCS(FitsHead* head)
{
  if (wcs) {
    for (int i=0; i<MULTWCS; i++)
      if (wcs[i])
	wcsfree(wcs[i]);
    delete [] wcs;
  }

  if (wcstan)
    delete [] wcstan;

  // wcsinit is sloooowwww! so try to figure it out first
  wcs = new (WorldCoor*)[MULTWCS];
  for (int i=0; i<MULTWCS; i++)
    wcs[i] = NULL;
  
  wcstan = new Vector[MULTWCS];

  // look first for default WCS. Let wcsinit figure it out since there can
  // be many different non-standard wcs's present
  wcshead = head;
  wcs[0] = wcsinit(head->cards());
  wcshead  = NULL;

  // now look for WCSA - WCSZ
  // we can take a short cut here, since only valid FITS wcs's are available
  for (int i=1; i<MULTWCS; i++) {
    char str[] = "CTYPE1 ";
    str[6] = '@'+i;
    if (head->find(str)) {
      wcshead = head;
      wcs[i] = wcsinitc(head->cards(),(char)('@'+i));
      wcshead  = NULL;
    }
  }

  // determine what point to use in determining rotation and orientation
  // if projection is a Zenithal projection (section 5.1 WCS FITS II)
  // then use the tangent point, otherwise, use the center of the image
  for (int i=0; i<MULTWCS; i++) {
    if (wcs[i])
      wcstan[i] = (wcs[i]->prjcode>=WCS_AZP && wcs[i]->prjcode<=WCS_AIR) ?
	Vector(wcs[i]->crpix[0],wcs[i]->crpix[1]) : Vector(width_,height_)/2;
  }

  // now see if we have a 'physical' wcs, if so, set LTMV keywords
  for (int i=1; i<MULTWCS; i++) {
    if (wcs[i] && wcs[i]->wcsname && !strcmp(wcs[i]->wcsname,"PHYSICAL")) {
      keyLTMV = 1;

      double ltm11 = wcs[i]->cd[0] != 0 ? 1/wcs[i]->cd[0] : 0;
      double ltm12 = wcs[i]->cd[1] != 0 ? 1/wcs[i]->cd[1] : 0;
      double ltm21 = wcs[i]->cd[2] != 0 ? 1/wcs[i]->cd[2] : 0;
      double ltm22 = wcs[i]->cd[3] != 0 ? 1/wcs[i]->cd[3] : 0;

      double ltv1 = wcs[i]->crpix[0] -
	wcs[i]->crval[0]*ltm11 - wcs[i]->crval[1]*ltm21;
      double ltv2 = wcs[i]->crpix[1] -
	wcs[i]->crval[0]*ltm12 - wcs[i]->crval[1]*ltm22;

      physicalToImage = Matrix(ltm11, ltm12, ltm21, ltm22, ltv1, ltv2);
      imageToPhysical = physicalToImage.invert();
    }
  }

  if (DebugWCS) {
    for (int i=0; i<MULTWCS; i++) {
      if (wcs[i]) {
	cerr << "wcs " << (char)(!i ? ' ' : '@'+i) << endl;
	cerr << "wcs->wcsname " << (wcs[i]->wcsname ? wcs[i]->wcsname : "")
	     << endl;
	cerr << "wcs->rot=" << wcs[i]->rot << endl;
	cerr << "wcs->equinox=" << wcs[i]->equinox << endl;
	cerr << "wcs->epoch=" << wcs[i]->epoch << endl;
	cerr << "wcs->longpole=" << wcs[i]->longpole << endl;
	cerr << "wcs->latpole=" << wcs[i]->latpole << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->crpix[" << j << "]=" << wcs[i]->crpix[j] << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->crval[" << j << "]=" << wcs[i]->crval[j] << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->cdelt[" << j << "]=" << wcs[i]->cdelt[j] << endl;

	for (int j=0; j<4; j++)
	  cerr << "wcs->cd[" << j << "]=" << wcs[i]->cd[j] << endl;

	for (int j=0; j<4; j++)
	  cerr << "wcs->pc[" << j << "]=" << wcs[i]->pc[j] << endl;

	cerr << "wcs->imrot=" << wcs[i]->imrot << endl;
	cerr << "wcs->pa_north=" << wcs[i]->pa_north << endl;
	cerr << "wcs->pa_east=" << wcs[i]->pa_east << endl;
	cerr << "wcs->imflip=" << wcs[i]->imflip << endl;
	cerr << "wcs->prjcode=" << wcs[i]->prjcode << endl;

	cerr << "wcs->coorflip=" << wcs[i]->coorflip << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->ctype[" << j << "]=" << wcs[i]->ctype[j] << endl;

	cerr << "wcs->c1type=" << wcs[i]->c1type << endl;
	cerr << "wcs->c2type=" << wcs[i]->c2type << endl;
	cerr << "wcs->ptype=" << wcs[i]->ptype << endl;
	cerr << "wcs->radecsys=" << wcs[i]->radecsys << endl;
	cerr << "wcs->radecout=" << wcs[i]->radecout << endl;
	cerr << "wcs->syswcs=" << wcs[i]->syswcs << endl;
	cerr << "wcs->wcsproj=" << wcs[i]->wcsproj << endl;

	cerr << "wcs->wcstan=" << wcstan[i] << endl;
      }
    }
  }
}

void FitsImage::listWCS(ostream& str, CoordSystem sys, char delim)
{
  if (wcs[0]) {
    WorldCoor* wcss = wcs[sys-WCS];

    // required keywords
    str << "# WCSAXES =" << wcss->naxes << delim;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CRVAL" << i+1 << "  =" << wcss->crval[i] << delim;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CRPIX" << i+1 << "  =" << wcss->crpix[i] << delim;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CDELT" << i+1 << "  =" << wcss->cdelt[i] << delim;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CTYPE" << i+1 << "  =" << wcss->ctype[i] << delim;
    for (int i=0; i<wcss->naxes; i++)
      if (wcss->units[i])
	str << "# CUNIT" << i+1 << "  =" << wcss->units[i] << delim;
    str << "# CD1_1   =" << wcss->cd[0] << delim;
    str << "# CD1_2   =" << wcss->cd[1] << delim;
    str << "# CD2_1   =" << wcss->cd[2] << delim;
    str << "# CD2_2   =" << wcss->cd[3] << delim;

    int ilat = 1;
    if (!strncmp(wcss->ctype[1],"LAT",3) || !strncmp(wcss->ctype[1],"DEC",3))
      ilat = 2;
    for (int i=0; i<10; i++)
      if (wcss->prj.p[i])
	str << "# PV" << ilat << "_" << i << "  ="<< wcss->prj.p[i] << delim;

    if (wcss->wcsname)
      str << "# WCSNAME =" << wcss->wcsname << delim;

    // not required, but needed anyways
    if (!strncmp("RA",wcss->ctype[0],2) || !strncmp("RA",wcss->ctype[1],2))
      if (!strncmp("FK4",wcss->radecsys,3) ||
	  !strncmp("FK5",wcss->radecsys,3) ||
	  !strncmp("ICRS",wcss->radecsys,4))
	str << "# RADESYS =" << wcss->radecsys << delim;

    str << "# MJD-OBS = " 
	<< (wcss->epoch-1900)*365.242198781+15019.81352 << delim;

    str << "# EQUINOX =" << wcss->equinox << delim;
    if (wcss->latpole != 999)
      str << "# LATPOLE =" << wcss->latpole << delim;
    if (wcss->longpole != 999)
      str << "# LONPOLE =" << wcss->longpole << delim;
  }
}

FitsHead* FitsImage::getHead()
{
  if (image)
    return image->head();
  else    
    return NULL;
}

char* FitsImage::getCards()
{
  if (image) {
    FitsHead* head = image->head();
    if (head)
      return head->cards();
  }
  return NULL;
}

int FitsImage::getNCards()
{
  if (image) {
    FitsHead* head = image->head();
    if (head)
      return head->ncard();
  }
  return 0;
}

char* FitsImage::getKeyword(const char* key)
{
  return fits->getKeyword(key);
}

int FitsImage::findKeyword(const char* key)
{
  return fits->find(key);
}

char* FitsImage::displayHeader()
{
  // always return the header to the native fits ext, not the image
  if (DebugBin)
    return display(image->head());
  else
    return display(fits->head());
}

char* FitsImage::displayPrimary()
{
  // always return the header to the native fits ext, not the image
  if (DebugBin)
    return display(image->primary());
  else
    return display(fits->primary());
}

char* FitsImage::display(FitsHead* head)
{
  int size = head->ncard() * (FTY_CARDLEN+1);
  char* lbuf = new char[size+1];

  char* lptr = lbuf;
  char* cptr = head->cards();
  for (int i=0; i<head->ncard(); i++,cptr+=FTY_CARDLEN) {
    memcpy(lptr, cptr, FTY_CARDLEN);
    lptr+=FTY_CARDLEN;
    *(lptr++) = '\n';
  }

  lbuf[size] = '\0';
  return lbuf;
}

#if __GNUC__ >= 3
void FitsImage::parseSection(char* lbuf, Vector* v1, Vector* v2)
{
  double x1, y1, x2, y2;
  char d; // dummy char
  string x(lbuf);
  istringstream str(x);
  str >> d >> (*v1)[0] >> d >> (*v2)[0] >> d >> (*v1)[1] >> d >> (*v2)[1] >> d;
}
#else
void FitsImage::parseSection(char* lbuf, Vector* v1, Vector* v2)
{
  double x1, y1, x2, y2;
  char d; // dummy char
  istrstream str(lbuf);
  str >> d >> (*v1)[0] >> d >> (*v2)[0] >> d >> (*v1)[1] >> d >> (*v2)[1] >> d;
}
#endif

int FitsImage::processKeywords()
{
  // Physical to Image (LTM/LTV keywords) (with no wcsname already located)

  if (!keyLTMV) {
    if (image->find("LTM1_1") ||
	image->find("LTM1_2") ||
	image->find("LTM2_1") ||
	image->find("LTM2_2") ||
	image->find("LTV1") ||
	image->find("LTV2"))
      keyLTMV = 1;

    double ltm11 = image->getReal("LTM1_1", 1);
    double ltm12 = image->getReal("LTM1_2", 0);
    double ltm21 = image->getReal("LTM2_1", 0);
    double ltm22 = image->getReal("LTM2_2", 1);

    double ltv1 = image->getReal("LTV1", 0);
    double ltv2 = image->getReal("LTV2", 0);

    physicalToImage = Matrix(ltm11, ltm12, ltm21, ltm22, ltv1, ltv2);
    imageToPhysical = physicalToImage.invert();
  }

  // CDD to Detector (DTM/DTV keywords)

  if (image->find("DTM1_1") ||
      image->find("DTM1_2") ||
      image->find("DTM2_1") ||
      image->find("DTM2_2") ||
      image->find("DTV1") ||
      image->find("DTV2"))
    keyDTMV = 1;

  double dtm11 = image->getReal("DTM1_1", 1);
  double dtm12 = image->getReal("DTM1_2", 0);
  double dtm21 = image->getReal("DTM2_1", 0);
  double dtm22 = image->getReal("DTM2_2", 1);

  double dtv1 = image->getReal("DTV1", 0);
  double dtv2 = image->getReal("DTV2", 0);

  physicalToDetector = Matrix(dtm11, dtm12, dtm21, dtm22, dtv1, dtv2);
  detectorToPhysical = physicalToDetector.invert();

  // Physical to Amplifier (ATM/ATV keywords)

  if (image->find("ATM1_1") ||
      image->find("ATM1_2") ||
      image->find("ATM2_1") ||
      image->find("ATM2_2") ||
      image->find("ATV1") ||
      image->find("ATV2"))
    keyATMV = 1;

  double atm11 = image->getReal("ATM1_1", 1);
  double atm12 = image->getReal("ATM1_2", 0);
  double atm21 = image->getReal("ATM2_1", 0);
  double atm22 = image->getReal("ATM2_2", 1);

  double atv1 = image->getReal("ATV1", 0);
  double atv2 = image->getReal("ATV2", 0);

  physicalToAmplifier = Matrix(atm11, atm12, atm21, atm22, atv1, atv2);
  amplifierToPhysical = physicalToAmplifier.invert();

  if (DebugMosaic) {
    cerr << endl;
    cerr << rootBaseFileName << endl;
    cerr << "ATM/V: " << physicalToAmplifier << endl;
    cerr << "ATM/V-1: " << amplifierToPhysical << endl;
    cerr << "DTM/V: " << physicalToDetector << endl;
    cerr << "DTM/V-1: " << detectorToPhysical << endl;
    cerr << "LTM/V: " << physicalToImage << endl;
    cerr << "LTM/V-1: " << imageToPhysical << endl;
  }

  // DATASEC

  iparams[0] = width_;
  iparams[1] = 0;
  iparams[2] = width_;
  iparams[3] = 0;
  iparams[4] = height_;

  char* datstr = image->getString("DATASEC");
  if (datstr && *datstr) {
    Vector v1,v2;
    parseSection(datstr,&v1,&v2);

    if (v1[0] < 1)
      v1[0] = 1;
    if (v1[1] < 1)
      v1[1] = 1;
    if (v2[0] > width_)
      v2[0] = width_;
    if (v2[1] > height_)
      v2[1] = height_;

    keyDATASEC = 1;
    datasec = BBox(v1,v2);

    // DATASEC is in IMAGE, dparams go from 0 to width
    v1 -= Vector(1,1);
    dparams[0] = width_;
    dparams[1] = v1[0];
    dparams[2] = v2[0];
    dparams[3] = v1[1];
    dparams[4] = v2[1];
  }
  else {
    keyDATASEC = 0;
    datasec = BBox(1,1,width_,height_);

    dparams[0] = width_;
    dparams[1] = 0;
    dparams[2] = width_;
    dparams[3] = 0;
    dparams[4] = height_;
  }
  if (datstr)
    delete [] datstr;

  if (DebugMosaic) {
    cerr << "iparams " 
	 << iparams[0] << ' ' << iparams[1] << ' ' <<  iparams[2] << ' ' 
	 << iparams[3] << ' ' << iparams[4] << endl;

    cerr << "dparams " 
	 << dparams[0] << ' ' << dparams[1] << ' ' <<  dparams[2] << ' ' 
	 << dparams[3] << ' ' << dparams[4] << endl;
  }

  switch (*(parent->currentMosaic)) {
  case FrameBase::NOMOSAIC:
    return 1;
  case FrameBase::IRAF:
    return processKeywordsIRAF();
  case FrameBase::WCSMOSAIC:
  case FrameBase::WFPC2:
    return processKeywordsWCS();
  }
}

int FitsImage::processKeywordsIRAF() 
{
  // reset
  rotation_ = 0;
  zoom_ = Vector(1,1);
  orientation_ = NORMAL;
  origin_ = Vector();

  // DETSIZE
  {
    char* sizestr = image->getString("DETSIZE");
    if (!(sizestr && *sizestr)) {
      if (sizestr)
	delete [] sizestr;
      return 0;
    }
    
    Vector v1,v2;
    parseSection(sizestr,&v1,&v2);
    delete [] sizestr;

    detsize = BBox(v1,v2);
  }

  // DETSEC
  {
    char* detstr =  image->getString("DETSEC");
    if (!(detstr && *detstr)) {
      if (detstr)
	delete [] detstr;
      return 0;
    }

    Vector v1,v2;
    parseSection(detstr,&v1,&v2);
    delete [] detstr;

    int xx = (v1[0] < v2[0]);
    int yy = (v1[1] < v2[1]);

    if (xx && yy)
      orientation_ = NORMAL;
    else if (!xx && yy)
      orientation_ = XX;
    else if (!xx && !yy)
      orientation_ = XY;
    else if (xx && !yy)
      orientation_ = YY;

    detsec = BBox(v1,v2);
  }

  // CCDSUM
  ccdsum = Vector(1,1);
  {
    char* ccdstr = image->getString("CCDSUM");
    if (ccdstr && *ccdstr) {
      double Ns, Np, Ns1, Np1;
#if __GNUC__ >= 3
      string x(ccdstr);
      istringstream str(x);
#else
      istrstream str(ccdstr);
#endif

      str >> Ns >> Np >> Ns1 >> Np1;
      ccdsum = Vector(1/Ns, 1/Np);
    }

    if (ccdstr)
      delete [] ccdstr;
  }

  origin_ = detsec.ll * Scale(ccdsum) * Translate(-datasec.ll);

  if (*(parent->currentCount) == 0) {
    Vector c = datasec.center() * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
      break;
    case YY:
      flip = FlipY();
      break;
    case XY:
      flip = FlipXY();
      break;
    }

    Matrix m =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c) *
      Translate(origin_);
    *(parent->currentWCSmatrix) = m.invert();
  }

  if (DebugMosaic) {
    cerr << "ProcessKeywordsIRAF" << endl
	 << " datasec: " << datasec << endl
	 << " ccdsum : " << ccdsum << endl
	 << " detsize: " << detsize << endl
	 << " detsec : " << detsec << endl
	 << " orientation: " << orientation_ << endl
	 << " origin: " << origin_ << endl
	 << " matrix " << *(parent->currentWCSmatrix) << endl;
  }

  return 1;
}

int FitsImage::processKeywordsWCS()
{
  CoordSystem sys =  *(parent->currentMosaicSystem);

  int* params = getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  if (DebugMosaic)
    cerr << "ProcessKeywordsWCS " << coordSystemStr_[sys-IMAGE] << endl
	 << " crpix " << getWCScrpix(sys) << endl
	 << " cdelt " << getWCScdelt(sys) << endl
	 << " rotation " << radToDeg(getWCSRotation(sys,FK5)) << endl
	 << " orientation " << getWCSOrientation(sys,FK5) << endl
	 << " xmin xmax " << xmin << ' ' << xmax << endl
	 << " ymin ymax " << ymin << ' ' << ymax << endl;

  if (*(parent->currentCount) == 0) {
    // orientation
    Matrix flip;
    orientation_ = NORMAL;

    // rotation
    rotation_ = 0;

    // zoom
    zoom_ = Vector(1,1);

    // origin
    origin_ = Vector();

    // WCS bbox
    Vector c = getWCScrpix(sys) * imageToData;

    Matrix m =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c) *
      Translate(origin_);

    Vector aa = Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage;
    BBox bb(aa,aa);
    bb.bound(Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    bb.bound(Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    bb.bound(Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    // save for next
    *(parent->currentWCScdelt) = getWCScdelt(sys);
    *(parent->currentWCSrot)   = getWCSRotation(sys,FK5);
    *(parent->currentWCSorientation) = getWCSOrientation(sys,FK5);
    *(parent->currentWCSmatrix) = m;
    *(parent->currentWCSbb) = bb;

    if (DebugMosaic)
      cerr << " WCS 1 matrix " << endl
	   << "  center " << c << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
  else {
    // orientation
    Orientation o1 = getWCSOrientation(sys,FK5);
    Orientation o2 = *(parent->currentWCSorientation);
    Matrix flip;
    switch (o1) {
    case NORMAL:
      {
	switch (o2) {
	case NORMAL:
	  orientation_ = NORMAL;
	  break;
	case XX:
	  orientation_ = XX;
	  flip = FlipX();
	  break;
	}
      }
      break;
    case XX:
      {
	switch (o2) {
	case NORMAL:
	  orientation_ = XX;
	  flip = FlipX();
	  break;
	case XX:
	  orientation_ = NORMAL;
	  break;
	}
      }
      break;
    }

    // rotation
    double r1 = getWCSRotation(sys,FK5);
    double r2 = *(parent->currentWCSrot);
    switch (o1) {
    case NORMAL:
      {
	switch (o2) {
	case NORMAL:
	  rotation_ = r2-r1;
	  break;
	case XX:
	  rotation_ = r2+r1;
	  break;
	}
      }
      break;
    case XX:
      {
	switch (o2) {
	case NORMAL:
	  rotation_ = -r2-r1;
	  break;
	case XX:
	  rotation_ = -r2+r1;
	  break;
	}
      }
      break;
    }

    // zoom
    double zz = fabs(getWCScdelt(sys)[0]/(*(parent->currentWCScdelt))[0]);
    zoom_ = Vector(zz,zz);

    // origin
    Vector aa = getWCScrpix(sys);
    Vector bb = pix2wcs(aa,sys,FK5);
    Vector org = (*(parent->channelFits))->wcs2pix(bb,sys,FK5) * imageToData;

    Vector c = getWCScrpix(sys) * imageToData;

    Matrix mm =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c);
    origin_ = org*(*(parent->currentWCSmatrix)) - c*mm;

    // WCS bbox
    Matrix m =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c) *
      Translate(origin_);

    (parent->currentWCSbb)->bound(
       Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << " WCS 1+ matrix " << endl
	   << "  center " << c << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }

  return 1;
}

void FitsImage::updateWCSbb(CoordSystem sys, int first)
{
  int* params = getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  if (first) {
    Vector c = getWCScrpix(sys) * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
    }

    // WCS bbox
    Matrix m =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c) *
      Translate(origin_);

    Vector aa = Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage;
    *(parent->currentWCSbb) = BBox(aa,aa);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << "updateWCSbb " << endl
	   << " WCS 1 matrix " << endl
	   << "  center " << c << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
  else {
    Vector c = getWCScrpix(sys) * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
    }

    // WCS bbox
    Matrix m =
      Translate(-c) *
      Rotate(rotation_) *
      flip *
      Scale(zoom_) *
      Translate(c) *
      Translate(origin_);

    (parent->currentWCSbb)->bound(
      Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << "updateWCSbb " << endl
	   << " WCS 1+ matrix " << endl
	   << "  center " << c << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
}

int FitsImage::hasWCS(CoordSystem sys)
{
  return (sys>=WCS && wcs && wcs[sys-WCS]) ? 1 : 0;
}

int FitsImage::hasWCSEqu(CoordSystem sys)
{
  if (hasWCS(sys)) {
    const char* str = ::getradecsys(wcs[sys-WCS]);

    if (!strncmp("FK4", str, 3))
      return 1;
    else if (!strncmp("FK5", str, 3))
      return 1;
    else if (!strncmp("ICR", str, 3))
      return 1;
    else if (!strncmp("GAL", str, 3))
      return 1;
    else if (!strncmp("ECL", str, 3))
      return 1;
    else if (!strncmp("NPOLE", str, 4)) // special north polar angle
      return 1;
    else
      return 0;
  }
  else
    return 0;
}

int FitsImage::hasWCSLinear(CoordSystem sys)
{
  
  if (hasWCS(sys))
    return hasWCSEqu(sys) ? 0 : 1;
  else
    return 0;
}

char* FitsImage::pix2wcs(Vector in, CoordSystem sys, SkyFrame sky,
			 SkyFormat format, char* lbuf, int len)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsoutinit(wcs[i], skyFrameStr_[sky-FK4]);

      switch (format) {
      case ARCMIN:
      case ARCSEC:
      case DEGREES:
	setwcsdeg(wcs[i],1);
	wcs[i]->ndec = 5;
	break;
      case SEXAGESIMAL:
      case HMS:
	setwcsdeg(wcs[i],0);
	wcs[i]->ndec = 3;
	break;
      }
    }
    else {
      wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      setwcslin(wcs[i],2);
    }

    ::pix2wcst(wcs[i], in[0], in[1], lbuf, len);
  }

  return lbuf;
}

Vector FitsImage::pix2wcs(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
    double x,y;
    ::pix2wcs(wcs[i], in[0], in[1], &x, &y);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::pix2wcs(Vector in, CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsoutinit(wcs[i], skyFrameStr_[sky-FK4]);
    }
    else
      wcsoutinit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    ::pix2wcs(wcs[i], in[0], in[1], &x, &y);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::wcs2pix(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    wcsininit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    int off;
    ::wcs2pix(wcs[i], in[0], in[1], &x, &y, &off);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::wcs2pix(Vector in, CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsininit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsininit(wcs[i], skyFrameStr_[sky-FK4]);
    }
    else
      wcsininit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    int off;
    ::wcs2pix(wcs[i], in[0], in[1], &x, &y, &off);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::pix2wcsDist(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    Vector cd = getWCScdelt(sys);
    return Vector(in[0]*cd[0], in[1]*cd[1]).abs();
  }
  else
    return in;
}

Vector FitsImage::wcs2pixDist(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    Vector cd = getWCScdelt(sys);
    return Vector(in[0]/cd[0], in[1]/cd[1]).abs();
  }
  else
    return in;
}

double FitsImage::wcsdist(Vector a, Vector b, CoordSystem sys)
{
  if (hasWCS(sys))
    return ::wcsdist(a[0],a[1],b[0],b[1]);
  else
    return 0;
}

Vector FitsImage::getWCScrval(CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (!wcs[i]->coorflip)
      return Vector(wcs[i]->crval[0], wcs[i]->crval[1]);
    else
      return Vector(wcs[i]->crval[1], wcs[i]->crval[0]);
  }
  else
    return Vector();
}

Vector FitsImage::getWCScrpix(CoordSystem sys)
{
  if (hasWCS(sys))
      return Vector(wcs[sys-WCS]->crpix[0], wcs[sys-WCS]->crpix[1]);
  else
    return Vector();
}

Vector FitsImage::getWCScdelt(CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (!wcs[i]->coorflip)
      return Vector(wcs[i]->cdelt[0], wcs[i]->cdelt[1]);
    else
      return Vector(wcs[i]->cdelt[1], wcs[i]->cdelt[0]);
  }
  else
    return Vector();
}

Vector FitsImage::getWCStan(CoordSystem sys)
{
  if (wcstan)
    return wcstan[sys-WCS];
  else
    return Vector();
}

double FitsImage::getWCSRotation(CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    Vector orval = pix2wcs(getWCStan(sys), sys, sky);
    Vector orpix = wcs2pix(orval, sys, sky);
    Vector delta = getWCScdelt(sys).abs();
    Vector npix = wcs2pix(Vector(orval[0],orval[1]+delta[1]), sys, sky);
    Vector north = npix - orpix;

    return -(north.angle()-M_PI/2);
  }

  return 0;
}

Orientation FitsImage::getWCSOrientation(CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    Vector orval = pix2wcs(getWCStan(sys), sys, sky);
    Vector orpix = wcs2pix(orval, sys, sky);
    Vector delta = getWCScdelt(sys).abs();
    Vector npix = wcs2pix(Vector(orval[0],orval[1]+delta[1]), sys, sky);
    Vector north = npix - orpix;
    Vector epix = wcs2pix(Vector(orval[0]+delta[0],orval[1]), sys, sky);
    Vector east = epix - orpix;

    // take the cross product and see which way the 3rd axis is pointing
    double w = cross(east, north)[2];

    if (!hasWCSEqu(sys))
      return w>0 ? NORMAL : XX;
    else
      return w<0 ? NORMAL : XX;
  }

  return NORMAL;
}

