// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "projection.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Projection::Projection(const Projection& a) : Marker(a)
{
  p1 = a.p1;
  p2 = a.p2;
  width = a.width;
  method = a.method;

  system = a.system;
  skyframe = a.skyframe;
}

Projection::Projection(FrameBase* p, const Vector& ptr1, const Vector& ptr2, 
		       double wd,
		       const char* mvcb, const char* delcb,
		       const char* clr, int w, const char* f, 
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag) 
  : Marker(p, ptr1, clr, w, f, t, prop, c, tag)
{
  p1 = ptr1;
  p2 = ptr2;
  width = wd;
  method = AVERAGE;

  system = PHYSICAL;
  skyframe = FK5;

  strcpy(type,"projection");
  handle = new Vector[3];
  numHandle = 3;

  if (mvcb && *mvcb) {
    setCallBack(MOVECB, mvcb, parent->options->cmdName);
    setCallBack(EDITCB, mvcb, parent->options->cmdName);
    setCallBack(UPDATECB, mvcb, parent->options->cmdName);
  }
  if (delcb && *delcb)
    setCallBack(DELETECB, delcb, parent->options->cmdName);

  updateBBox();
}

void Projection::update()
{
  center = (p2-p1)/2 + p1;
  angle = (p2-p1).angle();
}

void Projection::updateBBox()
{
  update();

  // bound marker

  Matrix imm = Translate(-center) * Rotate(angle);
  Matrix mm = Rotate(-angle) * Translate(center) * parent->refToCanvas;

  Vector a = p1*imm;
  Vector b = p2*imm;

  Vector ll = a*mm;
  Vector ul = (a+Vector(0,width))*mm;
  Vector lr = b*mm;
  Vector ur = (b+Vector(0,width))*mm;

  // bbox in canvas coords

  bbox = BBox(ll,lr);
  bbox.bound(ul);
  bbox.bound(ur);

  // generate handles in canvas coords

  handle[0] = ll;
  handle[1] = lr;
  handle[2] = ((center*imm)+Vector(0,width))*mm;

  // make room for handles

  bbox.expand(3);

  // calculate overall bbox

  calcAllBBox();
}

void Projection::updateCoords(const Matrix& m)
{
  p1*=m;
  p2*=m;
  update();

  Marker::updateCoords(m);
}

void Projection::move(const Vector& v)
{
  p1+=v;
  p2+=v;
  updateBBox();

  doCallBack(&moveCB);
}

void Projection::moveTo(const Vector& v)
{
  Vector diff = v - p1;
  p1=v;
  p2+=diff;
  updateBBox();

  doCallBack(&moveCB);
}

void Projection::edit(const Vector& v, int h)
{
  switch (h) {
  case 1:
    p1 = v;
    break;
  case 2:
    p2 = v;
    break;
  case 3:
    width = (v*Translate(-center)*Rotate(angle))[1];
    if (width<0)
      width = 0;
    break;
  }
  updateBBox();
  doCallBack(&editCB);
}

#if __GNUC__ >= 3
void Projection::ps(int mode)
{
  Marker::ps(mode);

  {
    Vector a = p1 * parent->refToCanvas;
    Vector b = p2 * parent->refToCanvas;

    ostringstream str;
    str << "newpath " 
	<< a.TkCanvasPs(parent->canvas) << "moveto"
	<< b.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  if (width>0) {
    psLineDash();

    Matrix imm = Translate(-center) * Rotate(angle);
    Matrix mm = Rotate(-angle) * Translate(center) * parent->refToCanvas;

    Vector a = p1*imm;
    Vector b = p2*imm;

    Vector ll = a*mm;
    Vector ul = (a+Vector(0,width))*mm;
    Vector lr = b*mm;
    Vector ur = (b+Vector(0,width))*mm;

    ostringstream str;
    str << "newpath " 
	<< lr.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< ul.TkCanvasPs(parent->canvas) << "lineto"
	<< ll.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void Projection::ps(int mode)
{
  Marker::ps(mode);

  {
    Vector a = p1 * parent->refToCanvas;
    Vector b = p2 * parent->refToCanvas;

    char buf[256];
    ostrstream str(buf,256);
    str << "newpath " 
	<< a.TkCanvasPs(parent->canvas) << "moveto"
	<< b.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  if (width>0) {
    psLineDash();

    Matrix imm = Translate(-center) * Rotate(angle);
    Matrix mm = Rotate(-angle) * Translate(center) * parent->refToCanvas;

    Vector a = p1*imm;
    Vector b = p2*imm;

    Vector ll = a*mm;
    Vector ul = (a+Vector(0,width))*mm;
    Vector lr = b*mm;
    Vector ur = (b+Vector(0,width))*mm;

    char buf[1024];
    ostrstream str(buf,1024);
    str << "newpath " 
	<< lr.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< ul.TkCanvasPs(parent->canvas) << "lineto"
	<< ll.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

int Projection::isIn(const Vector& v)
{
  if (width*parent->getZoom() > 3) {
    double a = (p2-p1).angle();
    Matrix m = Translate(-p1) * Rotate(a);

    Vector vv = v * parent->canvasToRef * m;
    Vector ur = (p2 * m) + Vector(0,width);
    return (vv[0]>0 && vv[0]<ur[0] && vv[1]>0 && vv[1]<ur[1]);
  }
  else {
    Vector l1 = p1 * parent->refToCanvas;
    Vector l2 = p2 * parent->refToCanvas;
    double a = (l2-l1).angle();
    Matrix m = Translate(-l1) * Rotate(a);

    Vector vv = v * m;
    Vector ur = l2 * m;
    return (vv[0]>0 && vv[0]<ur[0] && vv[1]>-3 && vv[1]<3);
  }
}

void Projection::set(const Vector& v1, const Vector& v2, double wd, int m)
{
  p1 = v1;
  p2 = v2;
  width = wd;
  method = (Method)m;

  updateBBox();
  doCallBack(&editCB);
}

void Projection::setWidth(double wd)
{
  width = wd;
  if (width<0)
    width = 0;

  updateBBox();
  doCallBack(&editCB);
}

void Projection::setPoints(const Vector& v1, const Vector& v2)
{
  p1 = v1;
  p2 = v2;

  updateBBox();
  doCallBack(&editCB);
}

// Private

void Projection::render(Drawable drawable, const Matrix& matrix, double zoom,
		  RenderMode mode)
{
  {
    switch (mode) {
    case SRC:
      XSetForeground(display, gc, color);
      XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
      setLineNoDash();
      break;
    case XOR:
      XSetForeground(display, gc, parent->getWhiteColor());
      XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
      setLineDash();
      break;
    }

    Vector a = (p1 * matrix).round();
    Vector b = (p2 * matrix).round();
    XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	      (int)b[0], (int)b[1]);
  }

  if (width>0) {
    switch (mode) {
    case SRC:
      XSetForeground(display, gc, color);
      XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
      setLineDash();
      break;
    case XOR:
      XSetForeground(display, gc, parent->getWhiteColor());
      XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
      setLineDash();
      break;
    }

    Matrix imm = Translate(-center) * Rotate(angle);
    Matrix mm = Rotate(-angle) * Translate(center) * matrix;

    Vector a = p1*imm;
    Vector b = p2*imm;

    Vector ll = (a*mm).round();
    Vector ul = ((a+Vector(0,width))*mm).round();
    Vector lr = (b*mm).round();
    Vector ur = ((b+Vector(0,width))*mm).round();

    XDRAWLINE(display, drawable, gc, (int)lr[0], (int)lr[1], 
	      (int)ur[0], (int)ur[1]);
    XDRAWLINE(display, drawable, gc, (int)ur[0], (int)ur[1], 
	      (int)ul[0], (int)ul[1]);
    XDRAWLINE(display, drawable, gc, (int)ul[0], (int)ul[1], 
	      (int)ll[0], (int)ll[1]);
  }
}

// list

void Projection::list(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v1 = ptr->mapFromRef(p1,sys);
      Vector v2 = ptr->mapFromRef(p2,sys);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ',' 
	  << ptr->mapLenFromRef(width,sys) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v1 = ptr->mapFromRef(p1,sys,sky);
	    Vector v2 = ptr->mapFromRef(p2,sys,sky);
	    str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		<< ',' << v2[0] << ',' << v2[1] << ',' 
		<< ptr->mapLenFromRef(width,sys,ARCSEC) << '"' << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra1[16], ra2[16];
	    char dec1[16], dec2[16];
	    {
	      ptr->mapFromRef(p1,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra1 >> dec1;
	    }
	    {
	      ptr->mapFromRef(p2,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra2 >> dec2;
	    }
	    str << type << '(' << ra1 << ',' << dec1 << ',' 
		<< ra2 << ',' << dec2 << ','
		<< ptr->mapLenFromRef(width,sys,ARCSEC) << '"' << ')';
	  }
	  break;
	}
      }
      else {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	    << ',' << v2[0] << ',' << v2[1] << ',' 
	    << ptr->mapLenFromRef(width,sys) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}
