// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu

//---------------------------------------------------------------------------
#include "IIR_SimpleName.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


IIRScram_SimpleName::~IIRScram_SimpleName() {}

ostream &
IIRScram_SimpleName::_print( ostream &os ){
  return get_prefix()->_print( os );
}


void 
IIRScram_SimpleName::_publish_vhdl(ostream &_vhdl_out) {
  get_prefix()->_publish_vhdl(_vhdl_out);
}


void 
IIRScram_SimpleName::_publish_cc_lvalue( published_file & ){
  cerr << "IIRScram_SimpleName::_publish_cc_lvalue( published_file &_cc_out ) called !! ";
  abort();
}


void
IIRScram_SimpleName::_publish_cc_wait_data( published_file &_cc_out ) {
  _cc_out << "s->";
  _cc_out << *((IIR_Identifier*)get_prefix());
}


void 
IIRScram_SimpleName::_publish_cc_data( published_file &_cc_out ) {
  get_prefix()->_publish_cc_data( _cc_out );
}


void 
IIRScram_SimpleName::_publish_cc_bounds( published_file &_cc_out ) {
  get_prefix()->_publish_cc_bounds( _cc_out );
}


void
IIRScram_SimpleName::_publish_cc_elaborate( published_file &_cc_out ){
  get_prefix()->_publish_cc_elaborate( _cc_out );
}

set<IIR_Declaration> *
IIRScram_SimpleName::_symbol_lookup() {
  set<IIR_Declaration> *retval = NULL;

  IIR *prefix = get_prefix();
  if( prefix->_is_iir_declaration() == true ){
    retval = new set<IIR_Declaration>( (IIR_Declaration *)prefix );
  }
  else if( prefix->_is_text_literal() == TRUE ){
    IIR_TextLiteral *prefix_text = _get_prefix_string();
    retval = new set<IIR_Declaration>( *_get_symbol_table()->find_set( prefix_text ) );
    if( IIR_TextLiteral::_cmp( prefix_text, "work" ) == 0 ){
      retval->add( _get_work_library() );
    }
  }
  else{
    cerr << "Error in IIRScram_SimpleName::_symbol_lookup() - "
	 << " don'tt know how to resolve a " << prefix->get_kind_text()
	 << endl;

    abort();
  }
  return retval;
}


set<IIR_Declaration> *
IIRScram_SimpleName::_symbol_lookup( set<IIR_Declaration> *set_to_look_in ) {
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;

  IIR_Declaration *current_decl = set_to_look_in->get_element();
  while( current_decl != NULL ){
    set<IIR_Declaration> *found = current_decl->_find_declarations( this );
    if( found != NULL ){
      retval->add( found );
      delete found;
    }
    current_decl = set_to_look_in->get_next_element();
  }

  if ( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }
  
  return retval;
}


set<IIR_TypeDefinition> *
IIRScram_SimpleName::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ) {
  set<IIR_TypeDefinition> *retval = NULL;

  // Get the set of declarations that correspond to our name, and then
  // get all of their RVALS.
  set<IIR_Declaration> *decl_set = _symbol_lookup();
  if( decl_set != NULL ){
    retval = decl_set_to_typedef_set( decl_set, constraint_function );
  }
  return retval;
}


void 
IIRScram_SimpleName::_type_check( set<IIR_TypeDefinition> * ){ }


IIR *
IIRScram_SimpleName::_rval_to_decl(IIR_TypeDefinition *my_rval) {
  IIR *retval = NULL;

  set<IIR_Declaration> *my_decls = _symbol_lookup();
  if( my_decls == NULL ){
    ostringstream err;
    err << "Internal error in IIRScram_SimpleName::_rval_to_decl - no declaration for |"
	<< *this << "| found.";
    report_error( this, err.str() );
    return this;
  }

  set<IIR_Declaration> possibilities;
  IIR_Declaration *current_decl = my_decls->get_element();
  while( current_decl != NULL ){
    if ( (current_decl->get_subtype() != NULL  
	  && current_decl->get_subtype()->_is_compatible( my_rval ) != NULL )
 	 && 
	  (current_decl->_is_subprogram() == FALSE || 
	   ((IIR_SubprogramDeclaration *)current_decl)->_num_required_args() == 0 )) {
      possibilities.add( current_decl );
    }
    current_decl = my_decls->get_next_element();
  }

  // In the case of:
  // constant V1: I1 := 1;
  //    constant V2: I2 := 20;
  //    type I5 is range V1 to V2;
  // We have a situation where V1 isn't compatible with I5.  The following snippet
  // basically says, "if you didn't find a compatible match, go with an incompatible
  // match.  There's got to be a better way to do this though.
  if( possibilities.num_elements() == 0 ){
    current_decl = my_decls->get_element();
    while( current_decl != NULL ){
      if ( current_decl->_is_subprogram() == FALSE || 
	   ((IIR_SubprogramDeclaration *)current_decl)->_num_required_args() == 0 ) {
	possibilities.add( current_decl );
      }
      current_decl = my_decls->get_next_element();
    }
  }

  delete my_decls;  

  switch( possibilities.num_elements() ){
  case 0:{
    ostringstream err;
    err << "Internal error in IIRScram_SimpleName::_rval_to_decl - zero possibilities"
	<< " found for |" << *this << "|.";
    report_error( this, err.str() );
    abort();
    break;
  }
  case 1:{
    retval = possibilities.get_element();
    retval = _convert_to_function_call( (IIR_Declaration *)retval );
    break;
  }
  default:{
    ostringstream err;
    err << "Internal error in IIRScram_SimpleName::_rval_to_decl - multiple possibilities"
	<< " found for |" << *this << "|.";
    report_error( this, err.str() );
    report_ambiguous_error( this, &possibilities );
    abort();
  }
  }

  return retval;
}


IIR *
IIRScram_SimpleName::_decl_to_decl( IIR_Declaration *my_decl ){
  IIR *retval = _convert_to_function_call( my_decl );
  delete this;
  return retval;
}


IIR_TextLiteral *
IIRScram_SimpleName::_get_string(){
  return (IIR_TextLiteral *)get_prefix();
}


const string
IIRScram_SimpleName::_convert_to_library_name() {
  ASSERT( get_prefix() != NULL );
  return get_prefix()->_convert_to_library_name(); 
}


IIR *
IIRScram_SimpleName::_rval_to_decl( IIR_TypeDefinition *prefix, IIR_TypeDefinition *suffix ){
  IIR *retval = NULL;

  set<IIR_Declaration> *decls = prefix->_find_declarations( this );
  ASSERT( decls != NULL );
  ASSERT( decls->num_elements() == 1 );
  
  IIR_Declaration *decl = decls->get_element();
  delete decls;

  ASSERT( decl->get_subtype()->_is_compatible( suffix ) != NULL );
  retval = _convert_to_function_call( decl );

  return retval;
}


IIR *
IIRScram_SimpleName::_rval_to_decl( IIR_Declaration *prefix, IIR_TypeDefinition *suffix ){
  IIR *retval = NULL;

  set<IIR_Declaration> *decls = prefix->_find_declarations( this );
  ASSERT( decls != NULL );
  ASSERT( decls->num_elements() == 1 );
  
  IIR_Declaration *decl = decls->get_element();
  delete decls;
  
  ASSERT( decl->get_subtype()->_is_compatible( suffix ) != NULL );
  retval = _convert_to_function_call( decl );

  return retval;
}


IIR_TypeDefinition *
IIRScram_SimpleName::_determine_rval_in_set( set<IIR_TypeDefinition> *search_in,
					     IIR_TypeDefinition *looking_for ){

  IIR_TypeDefinition *current_outer = search_in->get_element();
  while( current_outer != NULL ){
    set<IIR_Declaration> *decls = current_outer->_find_declarations( this );
    if( decls != NULL ){
      IIR_Declaration *current_inner = decls->get_element();
      while( current_inner != NULL ){
	if( current_inner->get_subtype()->_is_compatible( looking_for ) != NULL ){
	  delete decls;
	  return current_outer;
	}
	current_inner = decls->get_next_element();
      }
    }
    current_outer = search_in->get_next_element();
  }

  // If we made it here, something bad happened...
  ostringstream err;
  err << "Internal error in IIR_SimpleName::_determine_rval_in_set - no possiblilites found.";
  report_error( this, err.str() );
  return 0;
}


IIR_Declaration *
IIRScram_SimpleName::_determine_decl_in_set( set<IIR_Declaration> *search_in,
					     IIR_TypeDefinition *looking_for ){
  IIR_Declaration *current_outer = search_in->get_element();
  while( current_outer != NULL ){
    set<IIR_Declaration> *decls = current_outer->_find_declarations( this );
    if( decls != NULL ){
      IIR_Declaration *current_inner = decls->get_element();
      while( current_inner != NULL ){
	if( current_inner->get_subtype()->_is_compatible( looking_for ) != NULL ){
	  delete decls;
	  return current_outer;
	}
	current_inner = decls->get_next_element();
      }
    }
    current_outer = search_in->get_next_element();
  }

  // If we made it here, something bad happened...
  ostringstream err;
  err << "Internal error in IIR_SimpleName::_determine_rval_in_set - no possiblilites found.";
  report_error( this, err.str() );
  return 0;
}


IIR *
IIRScram_SimpleName::_clone(){
  IIR_SimpleName *retval = new IIR_SimpleName();
  IIR_Name::_clone( retval );
  return retval;
}

IIR *
IIRScram_SimpleName::_convert_to_function_call( IIR_Declaration *my_decl ){
  ASSERT( my_decl != NULL );
  IIR *retval = my_decl;

  if( retval->get_kind() == IIR_FUNCTION_DECLARATION ){
    // This assertion isn't true in the case of an attribute of a function.
    // i.e. function foo( x : integer )
    // foo'bar
    //    ASSERT( ((IIR_Declaration *)retval)->_num_required_args() == 0 );
    // In this very special case, after the function declaration get's converted into
    // a function call, the caller will have to convert it _back_!
    IIR_FunctionCall *new_retval = new IIR_FunctionCall();
    copy_location( this, new_retval );
    new_retval->set_implementation( (IIR_SubprogramDeclaration *)retval );
    IIR_TypeDefinition *my_rval = retval->get_subtype();
    
    retval = new_retval->_semantic_transform( my_rval );
    retval->_type_check( my_rval );
    retval = retval->_rval_to_decl( my_rval );
  }

  return retval;
}

visitor_return_type *
IIRScram_SimpleName::_accept_visitor( node_visitor *visitor, 
				      visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_SimpleName(this, arg);
}
