/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_dataset.h"
#include "python/python_command.h"

#define NAME_LEN 100


static void sg_dataset_class_init			(SGdatasetClass *klass);
static void sg_dataset_init				(SGdataset *dataset);
static void sg_dataset_destroy				(GtkObject *object);
static void sg_dataset_refresh_name			(SGdataset *dataset);
static void sg_dataset_refresh_python                   (SGdataset *dataset);
static gdouble function 				(GtkPlot *plot, 
							 GtkPlotData *data, 
							 gdouble x, 
							 gboolean *error);
static void main_iterator				(GtkPlot *plot,
							 GtkPlotData *data,
							 gint iter,
							 gdouble *x,
							 gdouble *y,
							 gdouble *z,
							 gdouble *a,
							 gdouble *dx,
							 gdouble *dy,
							 gdouble *dz,
							 gdouble *da,
                					 gchar **label,
							 gboolean *error);

static void matrix_iterator				(GtkPlot *plot,
							 GtkPlotData *data,
							 gint iter,
							 gdouble *x,
							 gdouble *y,
							 gdouble *z,
							 gdouble *a,
							 gdouble *dx,
							 gdouble *dy,
							 gdouble *dz,
							 gdouble *da,
                					 gchar **label,
							 gboolean *error);

static GtkObjectClass *parent_class = NULL;

GtkType
sg_dataset_get_type (void)
{
  static GtkType dataset_type = 0;

  if (!dataset_type)
    {
      GtkTypeInfo dataset_info =
      {
        "SGdataset",
        sizeof (SGdataset),
        sizeof (SGdatasetClass),
        (GtkClassInitFunc) sg_dataset_class_init,
        (GtkObjectInitFunc) sg_dataset_init,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      dataset_type = gtk_type_unique (GTK_TYPE_OBJECT, &dataset_info);
    }
  return dataset_type;
}

static void
sg_dataset_class_init (SGdatasetClass *klass)
{
  GtkObjectClass *object_class;

  parent_class = (GtkObjectClass *) gtk_type_class (gtk_object_get_type ());

  object_class = (GtkObjectClass *) klass;

  object_class->destroy = sg_dataset_destroy;
}

SGdataset *
sg_dataset_new (SGdataStyle style)
{
  SGdataset *dataset;

  dataset = SG_DATASET(gtk_type_new(sg_dataset_get_type()));

  switch(style){
    case SG_STYLE_VBARS:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_bar_new(GTK_ORIENTATION_VERTICAL)); 
        break;
    case SG_STYLE_HBARS:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_bar_new(GTK_ORIENTATION_HORIZONTAL)); 
        break;
    case SG_STYLE_VBOXES:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_box_new(GTK_ORIENTATION_VERTICAL)); 
        break;
    case SG_STYLE_HBOXES:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_box_new(GTK_ORIENTATION_HORIZONTAL)); 
        break;
    case SG_STYLE_FLUX:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_flux_new()); 
        break;
    case SG_STYLE_CONTOUR:
    case SG_STYLE_CSURFACE:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_csurface_new()); 
        break;
    case SG_STYLE_DMAP:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_surface_new()); 
        GTK_PLOT_SURFACE(dataset->real_data)->height_gradient = TRUE;
        break;
    case SG_STYLE_SURFACE:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_surface_new()); 
        break;
    case SG_STYLE_COLORS:
    case SG_STYLE_BUBBLES:
    case SG_STYLE_LPOINTS:
    default:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_data_new()); 
        break;
  }

  gtk_plot_data_set_link(dataset->real_data, dataset);

  dataset->style = style;

  return dataset;
}

static void
sg_dataset_init(SGdataset *dataset)
{
  gint i;

  dataset->id = -1;

  dataset->type = SG_DATA_COLUMNS;

  dataset->worksheet = NULL;

  dataset->x = -1;
  dataset->y = -1;
  dataset->z = -1;
  dataset->a = -1;
  dataset->dx = -1;
  dataset->dy = -1;
  dataset->dz = -1;
  dataset->da = -1;
  dataset->l = -1;

  dataset->exp = NULL;

  dataset->ref_count = 0;

  for (i=0; i<9; i++)
    dataset->p_exp[i] = NULL;
}

static SGdataset *
sg_dataset_new_with_worksheet  (SGdataStyle style,
                                SGworksheet *worksheet,
				gint col_x,
				gint col_y,
				gint col_z,
				gint col_a,
				gint col_ex,
				gint col_ey,
				gint col_ez,
				gint col_ea,
				gint col_l)
{
  SGdataset *dataset;

  dataset = sg_dataset_new(style);
  sg_dataset_set_worksheet(dataset, worksheet, 
                           col_x, col_y, col_z, col_a,
                           col_ex, col_ey, col_ez, col_ea, col_l);

  return dataset;
}

static SGdataset *
sg_dataset_new_with_matrix     (SGdataStyle style,
                                SGworksheet *matrix)
{
  SGdataset *dataset;

  dataset = sg_dataset_new(style);
  sg_dataset_set_matrix(dataset, matrix); 

  return dataset;
}

SGdataset *
sg_dataset_new_function  (gchar *exp)
{
  SGdataset *dataset;

  dataset = sg_dataset_new(SG_STYLE_LPOINTS);
  dataset->type = SG_DATA_FUNCTION;
  sg_dataset_set_exp(dataset, exp);

  return dataset;
}


SGdataset *
sg_dataset_new_python  (SGdataStyle style, gchar *p_exp[9])
{ gint i;
  SGdataset *dataset;

  dataset = sg_dataset_new(style);
  dataset->type = SG_DATA_PYTHON;

  sg_dataset_set_python(dataset,p_exp);

  return dataset;
}

SGdataset *
sg_dataset_clone(SGdataset *dataset)
{
  SGdataset *copy;
  GtkPlotData *copy_data, *real_data;

  real_data = GTK_PLOT_DATA(dataset->real_data);

  switch(dataset->type){
    case SG_DATA_FUNCTION:
      copy = sg_dataset_new_function(dataset->exp);
      break;
    case SG_DATA_PYTHON:
      copy = sg_dataset_new_python(dataset->style, dataset->p_exp);
      break;
    case SG_DATA_MATRIX:
    case SG_DATA_COLUMNS:
    default:
      copy = sg_dataset_new(dataset->style);
      break;
  }

  copy->id = dataset->id;

  copy->worksheet = dataset->worksheet;
  copy->x = dataset->x;
  copy->y = dataset->y;
  copy->z = dataset->z;
  copy->a = dataset->a;
  copy->dx = dataset->dx;
  copy->dy = dataset->dy;
  copy->dz = dataset->dz;
  copy->da = dataset->da;
  copy->l = dataset->l;

  copy_data = copy->real_data;

  copy_data->x = real_data->x;
  copy_data->y = real_data->y;
  copy_data->z = real_data->z;
  copy_data->a = real_data->a;
  copy_data->dx = real_data->dx;
  copy_data->dy = real_data->dy;
  copy_data->dz = real_data->dz;
  copy_data->da = real_data->da;
  copy_data->labels = real_data->labels;

  copy_data->is_iterator = real_data->is_iterator;
  copy_data->is_function = real_data->is_function;
  copy_data->iterator = real_data->iterator;
  copy_data->function = real_data->function;
  copy_data->function3d = real_data->function3d;
  copy_data->num_points = real_data->num_points;
  copy_data->iterator_mask = real_data->iterator_mask;

  copy_data->symbol = real_data->symbol;
  copy_data->line = real_data->line;
  copy_data->line_connector = real_data->line_connector;

  copy_data->show_legend = real_data->show_legend;
  copy_data->show_labels = real_data->show_labels;
  copy_data->show_gradient = real_data->show_gradient;
  copy_data->fill_area = real_data->fill_area;
  copy_data->labels_offset = real_data->labels_offset;
  copy_data->legends_precision = real_data->legends_precision;

  copy_data->x_step = real_data->x_step;
  copy_data->y_step = real_data->y_step;
  copy_data->z_step = real_data->z_step;

  copy_data->x_line = real_data->x_line;
  copy_data->y_line = real_data->y_line;
  copy_data->z_line = real_data->z_line;

  copy_data->show_xerrbars = real_data->show_xerrbars;
  copy_data->show_yerrbars = real_data->show_yerrbars;
  copy_data->show_zerrbars = real_data->show_zerrbars;

  copy_data->gradient_mask = real_data->gradient_mask;
  copy_data->color_min = real_data->color_min;
  copy_data->color_max = real_data->color_max;

  if(GTK_IS_PLOT_SURFACE(copy_data)){
    GTK_PLOT_SURFACE(copy_data)->show_grid = GTK_PLOT_SURFACE(real_data)->show_grid;
    GTK_PLOT_SURFACE(copy_data)->show_mesh = GTK_PLOT_SURFACE(real_data)->show_mesh;

    GTK_PLOT_SURFACE(copy_data)->color = GTK_PLOT_SURFACE(real_data)->color;
    GTK_PLOT_SURFACE(copy_data)->shadow = GTK_PLOT_SURFACE(real_data)->shadow;
    GTK_PLOT_SURFACE(copy_data)->grid_foreground = GTK_PLOT_SURFACE(real_data)->grid_foreground;
    GTK_PLOT_SURFACE(copy_data)->grid_background = GTK_PLOT_SURFACE(real_data)->grid_background;

    GTK_PLOT_SURFACE(copy_data)->light = GTK_PLOT_SURFACE(real_data)->light;
    GTK_PLOT_SURFACE(copy_data)->ambient = GTK_PLOT_SURFACE(real_data)->ambient;
    GTK_PLOT_SURFACE(copy_data)->height_gradient = GTK_PLOT_SURFACE(real_data)->height_gradient;

    GTK_PLOT_SURFACE(copy_data)->nx = GTK_PLOT_SURFACE(real_data)->nx;
    GTK_PLOT_SURFACE(copy_data)->ny = GTK_PLOT_SURFACE(real_data)->ny;

    GTK_PLOT_SURFACE(copy_data)->xstep = GTK_PLOT_SURFACE(real_data)->xstep;
    GTK_PLOT_SURFACE(copy_data)->ystep = GTK_PLOT_SURFACE(real_data)->ystep;
  }

  if(GTK_IS_PLOT_CSURFACE(copy_data)){
    GTK_PLOT_CSURFACE(copy_data)->lines_visible = GTK_PLOT_CSURFACE(real_data)->lines_visible;
    GTK_PLOT_CSURFACE(copy_data)->lines_only= GTK_PLOT_CSURFACE(real_data)->lines_only;
    GTK_PLOT_CSURFACE(copy_data)->project_xy= GTK_PLOT_CSURFACE(real_data)->project_xy;
    GTK_PLOT_CSURFACE(copy_data)->labels_visible= GTK_PLOT_CSURFACE(real_data)->labels_visible;
  }

  gtk_plot_data_set_link(copy_data, dataset);
  dataset->ref_count++;
  return copy;
}

void
sg_dataset_set_worksheet  (SGdataset *dataset,
			   SGworksheet *worksheet,
		   	   gint col_x,
			   gint col_y,
			   gint col_z,
			   gint col_a,
			   gint col_ex,
			   gint col_ey,
			   gint col_ez,
			   gint col_ea,
			   gint col_l)
{
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  real_data->worksheet = worksheet;

  dataset->x = real_data->x = col_x;
  dataset->y = real_data->y = col_y;
  dataset->z = real_data->z = col_z;
  dataset->a = real_data->a = col_a;
  dataset->dx = real_data->dx = col_ex;
  dataset->dy = real_data->dy = col_ey;
  dataset->dz = real_data->dz = col_ez;
  dataset->da = real_data->da = col_ea;
  dataset->l = real_data->l = col_l;

  real_data->real_data->is_iterator = TRUE;
  real_data->real_data->iterator = main_iterator;

  sg_dataset_refresh(real_data);
}

void
sg_dataset_set_matrix     (SGdataset *dataset,
                           SGworksheet *matrix)
{
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  real_data->worksheet = matrix;

  real_data->real_data->is_iterator = TRUE;
  real_data->real_data->iterator = matrix_iterator;

  sg_dataset_refresh(real_data);
}

void
sg_dataset_add_point(SGdataset *dataset,
                     gdouble *x, gdouble *y, gdouble *z, gdouble *a,
                     gdouble *ex, gdouble *ey, gdouble *ez, gdouble *ea,
                     gchar *label)
{
  gint num_points;
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  num_points = real_data->real_data->num_points + 1;

  if(real_data->x != -1){
    real_data->real_data->x = g_renew(gdouble,real_data->real_data->x,num_points);
    real_data->real_data->x[num_points-1] = *x;
  }

  if(real_data->y != -1){
    real_data->real_data->y = g_renew(gdouble,real_data->real_data->y,num_points);
    real_data->real_data->y[num_points-1] = *y;
  }

  if(real_data->z != -1){
    real_data->real_data->z = g_renew(gdouble,real_data->real_data->z,num_points);
    real_data->real_data->z[num_points-1] = *z;
  }

  if(real_data->a != -1){
    real_data->real_data->a = g_renew(gdouble,real_data->real_data->a,num_points);
    real_data->real_data->a[num_points-1] = *a;
  }

  if(real_data->dx != -1){
    real_data->real_data->dx = g_renew(gdouble,real_data->real_data->dx,num_points);
    real_data->real_data->dx[num_points-1] = *ex;
  }

  if(real_data->dy != -1){
    real_data->real_data->dy = g_renew(gdouble,real_data->real_data->dy,num_points);
    real_data->real_data->dy[num_points-1] = *ey;
  }

  if(real_data->dz != -1){
    real_data->real_data->dz = g_renew(gdouble,real_data->real_data->dz,num_points);
    real_data->real_data->dz[num_points-1] = *ez;
  }

  if(real_data->da != -1){
    real_data->real_data->da = g_renew(gdouble,real_data->real_data->da,num_points);
    real_data->real_data->da[num_points-1] = *ea;
  }

  if(real_data->l != -1){
    real_data->real_data->labels = g_renew(gchar *,real_data->real_data->labels, num_points);
    real_data->real_data->labels[num_points-1] = NULL;
    if(label)
      real_data->real_data->labels[num_points-1] = g_strdup(label);
  }

  real_data->real_data->num_points++;
}


void
sg_dataset_refresh(SGdataset *dataset)
{
  SGworksheet *worksheet;
  GtkSheet *sheet;
  gchar **labels = NULL;
  gint row, col_x, col_y, col_z, col_a, col_ex, col_ey, col_ez, col_ea, col_l;
  gchar *text[9];
  gdouble val[9];
  gint numrows = 0, numcols = 0;
  gint begin;
  gint end;
  gint n;
  gboolean error;
  SGdataset *real_data;
  guint16 mask = 0;

  if(dataset->type == SG_DATA_FUNCTION) return;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  if (real_data->type == SG_DATA_PYTHON)
  {
    sg_dataset_refresh_python(dataset);
    return;
  }
  if(!real_data->worksheet) return;

  worksheet = real_data->worksheet;
  sheet = GTK_SHEET(worksheet->sheet);

  if(worksheet->mode == SG_MODE_WORKSHEET){
    col_x = real_data->x;
    col_y = real_data->y;
    col_z = real_data->z;
    col_a = real_data->a;
    col_ex = real_data->dx;
    col_ey = real_data->dy;
    col_ez = real_data->dz;
    col_ea = real_data->da;
    col_l = real_data->l;
  
    if(col_x >= 0) mask |= GTK_PLOT_DATA_X; 
    if(col_y >= 0) mask |= GTK_PLOT_DATA_Y; 
    if(col_z >= 0) mask |= GTK_PLOT_DATA_Z; 
    if(col_a >= 0) mask |= GTK_PLOT_DATA_A; 
    if(col_ex >= 0) mask |= GTK_PLOT_DATA_DX; 
    if(col_ey >= 0) mask |= GTK_PLOT_DATA_DY; 
    if(col_ez >= 0) mask |= GTK_PLOT_DATA_DZ; 
    if(col_ea >= 0) mask |= GTK_PLOT_DATA_DA; 
    if(col_l >= 0) mask |= GTK_PLOT_DATA_LABELS; 
  
    real_data->real_data->iterator_mask = mask;
    dataset->real_data->iterator_mask = mask;
  
    begin = MAX(0, worksheet->begin);
    end = sheet->maxallocrow;
   
    if(worksheet->end >= 0)
      end = MIN(end, worksheet->end);
  
    for(row = begin; row <= end; row++){
/*
      gdouble xval, yval;
      xval = sg_worksheet_cell_get_double(worksheet, row, col_x, &error);      
      yval = sg_worksheet_cell_get_double(worksheet, row, col_y, &error);      
      if(error) break;
      numrows++;
*/
     
      text[0] = sg_worksheet_cell_get_text(worksheet, row, col_x);      
      text[1] = sg_worksheet_cell_get_text(worksheet, row, col_y);      
 
      if(text[0] && strlen(text[0]) > 0 &&
         text[1] && strlen(text[1]) > 0) numrows++;
      else break;
    }

    real_data->real_data->num_points = numrows;
    dataset->real_data->num_points = numrows;

    if(GTK_IS_PLOT_SURFACE(real_data->real_data)){ 
      GTK_PLOT_SURFACE(dataset->real_data)->recalc_dt = TRUE;
    }

  } else if(worksheet->mode == SG_MODE_MATRIX){
    if(real_data->style == SG_STYLE_BUBBLES)
      mask = GTK_PLOT_DATA_X | GTK_PLOT_DATA_Y | GTK_PLOT_DATA_A;
    else if(real_data->style == SG_STYLE_COLORS)
      mask = GTK_PLOT_DATA_X | GTK_PLOT_DATA_Y | GTK_PLOT_DATA_DA;
    else
      mask = GTK_PLOT_DATA_X | GTK_PLOT_DATA_Y | GTK_PLOT_DATA_Z;

    real_data->real_data->iterator_mask = mask;
    dataset->real_data->iterator_mask = mask;
 
    numrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet->sheet)); 
    numcols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet->sheet)); 

    real_data->real_data->num_points = numrows * numcols;
    dataset->real_data->num_points = numrows * numcols;
   
    if(GTK_IS_PLOT_SURFACE(real_data->real_data)){ 
      GTK_PLOT_SURFACE(real_data->real_data)->nx = numcols;
      GTK_PLOT_SURFACE(real_data->real_data)->ny = numrows;
      GTK_PLOT_SURFACE(dataset->real_data)->nx = numcols;
      GTK_PLOT_SURFACE(dataset->real_data)->ny = numrows;
      GTK_PLOT_SURFACE(dataset->real_data)->recalc_dt = TRUE;
    }
  }

  gtk_plot_data_update(dataset->real_data); 
  sg_dataset_refresh_name(real_data);
}

static void
sg_dataset_refresh_python(SGdataset *dataset)
{
  gdouble *x[8];
  gchar **labels = NULL;
  gchar *p_exp[9];
  gchar name[NAME_LEN];
  gint i,num,len[4],j;
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(GTK_PLOT_DATA(dataset->real_data));

  sg_dataset_free_points(real_data);

  len[0]=-1;
  for (i=0;i<8;i++)
    { 
       if (real_data->p_exp[i] && strlen(real_data->p_exp[i])){
         x[i]=sg_eval_expr_double(real_data->p_exp[i], &len[i], len[0]);
       }else{
         x[i]=NULL;
       }
       if (!len[0]) return;
    }

  if (real_data->p_exp[i] && strlen(real_data->p_exp[i]))
    labels=sg_eval_expr_string(real_data->p_exp[i], &num);


  gtk_plot_data_set_points(real_data->real_data, x[0], x[1], x[4], x[5], len[0]);
  gtk_plot_data_set_labels(real_data->real_data, labels);
  gtk_plot_data_set_z(real_data->real_data, x[2]);
  gtk_plot_data_set_dz(real_data->real_data, x[6]);
  gtk_plot_data_set_a(real_data->real_data, x[3]);
  gtk_plot_data_set_da(real_data->real_data, x[7]);

  gtk_plot_data_set_points(dataset->real_data, x[0], x[1], x[4], x[5], len[0]);
  gtk_plot_data_set_labels(dataset->real_data, labels);
  gtk_plot_data_set_z(dataset->real_data, x[2]);
  gtk_plot_data_set_dz(dataset->real_data, x[6]);
  gtk_plot_data_set_a(dataset->real_data, x[3]);
  gtk_plot_data_set_da(dataset->real_data, x[7]);

  dataset->x = real_data->x = -1;
  dataset->y = real_data->y = -1;
  dataset->z = real_data->z = -1;
  dataset->a = real_data->a = -1;
  dataset->dx = real_data->dx = -1;
  dataset->dy = real_data->dy = -1;
  dataset->dz = real_data->dz = -1;
  dataset->da = real_data->da = -1;
  dataset->l = real_data->l = -1;

  dataset->worksheet = real_data->worksheet = NULL;
  
  gtk_plot_data_update(dataset->real_data); 
}

static void
sg_dataset_destroy(GtkObject *object)
{
  SGdataset *dataset;
  gint i;

  dataset = SG_DATASET(object);

  for(i = 0; i < 9; i++)
   if(dataset->p_exp[i]) g_free(dataset->p_exp[i]);
 
  if(dataset->exp) g_free(dataset->exp);

  gtk_widget_destroy(GTK_WIDGET(dataset->real_data));

  GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

void
sg_dataset_free_points(SGdataset* dataset)
{
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  if(real_data->real_data->x) g_free(real_data->real_data->x);
  if(real_data->real_data->y) g_free(real_data->real_data->y);
  if(real_data->real_data->z) g_free(real_data->real_data->z);
  if(real_data->real_data->dx) g_free(real_data->real_data->dx);
  if(real_data->real_data->dy) g_free(real_data->real_data->dy);
  if(real_data->real_data->dz) g_free(real_data->real_data->dz);
  if(real_data->real_data->labels) {
      gint i;
      for (i = 0; i < real_data->real_data->num_points; i++){
            g_free(real_data->real_data->labels[i]);
      }
      g_free(real_data->real_data->labels);
  }

  real_data->real_data->x = dataset->real_data->x = NULL;
  real_data->real_data->y = dataset->real_data->y = NULL;
  real_data->real_data->z = dataset->real_data->z = NULL;
  real_data->real_data->dx = dataset->real_data->dx = NULL;
  real_data->real_data->dy = dataset->real_data->dy = NULL;
  real_data->real_data->dz = dataset->real_data->dz = NULL;
  real_data->real_data->labels = dataset->real_data->labels = NULL;
}


void
sg_dataset_set_name(SGdataset *dataset, gchar *name)
{
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  gtk_plot_data_set_name(real_data->real_data, name);
}
                     
void
sg_dataset_set_exp(SGdataset *dataset, gchar *exp)
{
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  if(real_data->exp){
    g_free(real_data->exp);
    real_data->exp = NULL;
  }

  real_data->real_data->is_function = TRUE;
  if(exp){
     real_data->exp = g_strdup(exp);
     real_data->real_data->function = (GtkPlotFunc)function;
  } else {
     real_data->real_data->function = NULL;
  }
}

void
sg_dataset_set_python(SGdataset *dataset, gchar *p_exp[9])
{ 
  gint i;
  SGdataset *real_data;

  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);
  dataset->type = real_data->type = SG_DATA_PYTHON;

  dataset->real_data->iterator = real_data->real_data->iterator = NULL;
  dataset->real_data->is_iterator = real_data->real_data->is_iterator = FALSE;

  for (i=0;i<9;i++)
  {  
    if(real_data->p_exp[i])
      { 
        g_free(real_data->p_exp[i]);
        real_data->p_exp[i] = NULL;
      }

     if(p_exp[i])
        real_data->p_exp[i] = g_strdup(p_exp[i]);
  }

}


static gdouble
function (GtkPlot *plot, GtkPlotData *data, gdouble x, gboolean *error)
{
  SGdataset *dataset;
  gdouble val;

  dataset = (SGdataset *)gtk_plot_data_get_link(data);

  if (sg_eval_func(dataset->exp, x, &val)) *error=FALSE;
  else *error = TRUE;

  return val;
}

static void
sg_dataset_refresh_name(SGdataset *dataset)
{
  gint i;
  SGdataType data_type;
  SGdataStyle data_style;
  gint dataset_columns[9];
  gchar name[100];
  gchar column_text[9][100];
  SGdataset *real_data;
  gchar *column_labels[][10] = {
  {"X", "Y", "Z", "", "Xerr", "Yerr", "Zerr", "", "Lbls"},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "DX", "DY", "DZ", "Amp", ""},
  {"X", "Y", "Z", "Size", "", "", "", "Amp", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "", "", "", "", ""},};
  gchar *names[] = {
                     "scatter",
                     "vbars",
                     "hbars",
                     "vboxes",
                     "hboxes",
                     "bubbles",
                     "vectors",
                     "color",
                     "density",
                     "contour",
                     "contour",
                     "surface" };


  real_data = (SGdataset *)gtk_plot_data_get_link(dataset->real_data);

  data_type = real_data->type;
  data_style = real_data->style;

  if(!real_data->worksheet) return;

  if(data_type == SG_DATA_FUNCTION || data_type == SG_DATA_PYTHON) 
     return;

  if(real_data->worksheet->mode == SG_MODE_MATRIX){
    g_snprintf(name, 100, "%s:%s", names[data_style], 
                                   real_data->worksheet->name);
    sg_dataset_set_name(real_data, name);
    return;
  }

  dataset_columns[0] = real_data->x;
  dataset_columns[1] = real_data->y;
  dataset_columns[2] = real_data->z;
  dataset_columns[3] = real_data->a;
  dataset_columns[4] = real_data->dx;
  dataset_columns[5] = real_data->dy;
  dataset_columns[6] = real_data->dz;
  dataset_columns[7] = real_data->da;
  dataset_columns[8] = real_data->l;

  for(i = 0; i < 9; i++){
    gchar *column_name;
    if(dataset_columns[i] != -1){
      column_name = GTK_SHEET(real_data->worksheet->sheet)->column[dataset_columns[i]].name;
      if(i == 0)
        g_snprintf(column_text[i], 100, "%s(%s)", column_name,
                                         column_labels[data_style][i]);
      else
        g_snprintf(column_text[i], 100, ",%s(%s)", column_name,
                                         column_labels[data_style][i]);
    }
    else
      sprintf(column_text[i],"");
  }
  g_snprintf(name, 100, "%s:%s->%s%s%s%s%s%s%s%s%s",
                                       names[data_style],
                                       real_data->worksheet->name,
                                       column_text[0],
                                       column_text[1],
                                       column_text[2],
                                       column_text[3],
                                       column_text[4],
                                       column_text[5],
                                       column_text[6],
                                       column_text[7],
                                       column_text[8]);

  sg_dataset_set_name(real_data, name);
}

static void
main_iterator	(GtkPlot *plot,
		 GtkPlotData *data,
		 gint iter,
		 gdouble *x,
		 gdouble *y,
		 gdouble *z,
		 gdouble *a,
		 gdouble *dx,
		 gdouble *dy,
		 gdouble *dz,
		 gdouble *da,
                 gchar **label,
		 gboolean *error)
{
  SGdataset *real_data;
  SGworksheet *worksheet;
  gchar *text[2];
  gint begin;
  gint row;
  static gchar real_label[1000];

  real_data = (SGdataset *)gtk_plot_data_get_link(data);
  worksheet = real_data->worksheet;

  *error = FALSE;
  begin = MAX(0, worksheet->begin);

  row = begin + iter;

  text[0] = sg_worksheet_cell_get_text(worksheet, row, real_data->x);      
  text[1] = sg_worksheet_cell_get_text(worksheet, row, real_data->y);      

  if(!text[0] || strlen(text[0]) == 0 || !text[1] || strlen(text[1]) == 0){
      *error = TRUE;
      return;
  } 

  *x = sg_worksheet_cell_get_double(worksheet, row, real_data->x, error);
  *y = sg_worksheet_cell_get_double(worksheet, row, real_data->y, error);
  if(*error) return;

  if(data->iterator_mask & GTK_PLOT_DATA_Z && real_data->z >= 0)
     *z = sg_worksheet_cell_get_double(worksheet, row, real_data->z, error);  
  if(data->iterator_mask & GTK_PLOT_DATA_A && real_data->a >= 0)
     *a = sg_worksheet_cell_get_double(worksheet, row, real_data->a, error);  
  if(data->iterator_mask & GTK_PLOT_DATA_DX && real_data->dx >= 0)
     *dx = sg_worksheet_cell_get_double(worksheet, row, real_data->dx, error); 
  if(data->iterator_mask & GTK_PLOT_DATA_DY && real_data->dy >= 0)
     *dy = sg_worksheet_cell_get_double(worksheet, row, real_data->dy, error); 
  if(data->iterator_mask & GTK_PLOT_DATA_DZ && real_data->dz >= 0)
     *dz = sg_worksheet_cell_get_double(worksheet, row, real_data->dz, error); 
  if(data->iterator_mask & GTK_PLOT_DATA_DA && real_data->da >= 0)
     *da = sg_worksheet_cell_get_double(worksheet, row, real_data->da, error); 

  if(data->iterator_mask & GTK_PLOT_DATA_LABELS && real_data->l >= 0)
  {   
     snprintf(real_label, 1000, sg_worksheet_cell_get_text(worksheet, row, real_data->l));
     *label = real_label;
  }
}

static void
matrix_iterator	(GtkPlot *plot,
		 GtkPlotData *data,
		 gint iter,
		 gdouble *x,
		 gdouble *y,
		 gdouble *z,
		 gdouble *a,
		 gdouble *dx,
		 gdouble *dy,
		 gdouble *dz,
		 gdouble *da,
                 gchar **label,
		 gboolean *error)
{
  SGdataset *real_data;
  SGworksheet *worksheet;
  gchar *text;
  gint row, col;
  gint nrows, ncols;
  gdouble xstep, ystep;

  real_data = (SGdataset *)gtk_plot_data_get_link(data);
  worksheet = real_data->worksheet;

  *error = FALSE;

  nrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet->sheet));
  ncols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet->sheet));

  row = iter/ncols;
  col = iter - row*ncols;

  xstep = (worksheet->xmax - worksheet->xmin)/ncols;
  ystep = (worksheet->ymax - worksheet->ymin)/nrows;
  
  text = sg_worksheet_cell_get_text(worksheet, row, col);      

  *x = col * xstep;
  *y = row * ystep;

  if(!text || strlen(text) == 0){
      *z = 0.;
      *a = 0.;
      return;
  } 

  if(data->iterator_mask & GTK_PLOT_DATA_Z)
     *z = sg_worksheet_cell_get_double(worksheet, row, col, error);  
  if(data->iterator_mask & GTK_PLOT_DATA_A)
     *a = sg_worksheet_cell_get_double(worksheet, row, col, error);  
  if(data->iterator_mask & GTK_PLOT_DATA_DA)
     *da = sg_worksheet_cell_get_double(worksheet, row, col, error);  

  if(*error){
      *error = FALSE;
      *z = 0.;
      *a = 0.;
  }
}
