/*
 * dvutils.c -- DV Utils for ffmpeg2raw
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdint.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <sys/wait.h>

#include "dvutils.h"
#include <libdv/dv.h>

/** Bi-directional pipe structure.
*/

typedef struct rwpipe
{
	int pid;
	int reader;
	int writer;
}
rwpipe;

/** Create a bidirectional pipe for the given command.
*/

rwpipe *rwpipe_open( char *command )
{
	rwpipe *this = calloc( 1, sizeof( rwpipe ) );

	if ( this != NULL )
	{
        int input[ 2 ];
        int output[ 2 ];

        pipe( input );
        pipe( output );

        this->pid = fork();

        if ( this->pid == 0 )
        {
            dup2( output[ 0 ], STDIN_FILENO );
            dup2( input[ 1 ], STDOUT_FILENO );

            close( input[ 0 ] );
            close( input[ 1 ] );
            close( output[ 0 ] );
            close( output[ 1 ] );

            execl("/bin/sh", "sh", "-c", command, (char *) 0);
			exit( 255 );
        }
        else
        {
            close( input[ 1 ] );
            close( output[ 0 ] );

            this->reader = input[ 0 ];
            this->writer = output[ 1 ];
        }

	}

	return this;
}

/** Read data from the pipe.
*/

int rwpipe_read( rwpipe *this, void *data, int size )
{
	if ( this != NULL )
		return read( this->reader, data, size );
	else
		return -1;
}

/** Write data to the pipe.
*/

int rwpipe_write( rwpipe *this, void *data, int size )
{
	if ( this != NULL )
		return write( this->writer, data, size );
	else
		return -1;
}

/** Close the pipe and process.
*/

void rwpipe_close( rwpipe *this )
{
	if ( this != NULL )
	{
		close( this->reader );
		close( this->writer );
		waitpid( this->pid, NULL, 0 );
		free( this );
	}
}

typedef struct
{
	uint8_t *data;
	int width;
	int height;
	int dupe;
}
ppm_frame;

static ppm_frame last_frame;
static char *filter = NULL;
static int every = 1;
static int encode_count = 0;

/** Convert a ppm frame to a raw dv frame.
*/

int ppm_to_rawdv( rwpipe *this, ppm_frame *frame, uint8_t *dv, int len )
{
	int bytes = 0;
	int len1;
	char temp[ 132 ];
	uint8_t *p = frame->data;
	int i;
	int linesize = frame->width * 3;

	if ( frame->dupe && filter == NULL )
		return len;
	else if ( frame->dupe && filter )
		frame = &last_frame;
	else if ( encode_count ++ % every != 0 )
		return len;

	sprintf( temp, "P6\n%d %d\n255\n", frame->width, frame->height );

	rwpipe_write( this, temp, strlen( temp ) );
   	for(i=0;i< frame->height;i++) 
	{
		rwpipe_write( this, p, frame->width * 3 );
		p += linesize;
   	}

	p = dv;
	while ( len > 0 )
	{
		len1 = rwpipe_read( this, p, len );
		if ( len1 < 0 )
			break;
		p += len1;
		len -= len1;
		bytes += len1;
	}

	return bytes;
}

// PAL flag indicator
static int pal;
// Wide screen flag
static int wide;
// Audio frequency
static int frequency;
// Frame multiplier
static double multiplier;

static double delta;
static double current;

// DV encoder for audio
static dv_encoder_t *encoder;
// Number of frames calculated
static int frame_count = 0;

// Maximum number of frames to keep
#define VIDEO_FRAME_COUNT 	100
// Frames
static ppm_frame *frames[ VIDEO_FRAME_COUNT ];
// The currently used frame
static int used = 0;

// Maximum number of audio frames to hold
#define AUDIO_FRAME_COUNT	500
// The number of samples used in the current frame
static int audio_count = 0;
// The number of audio frames in use
static int audio_used = 0;
// The number of samples per frame
static int samples_per_frame[ AUDIO_FRAME_COUNT ];
// Audio frames 
static int16_t *sample_frames[ AUDIO_FRAME_COUNT ];

// Output meta info
static int output_count = 0;
static time_t datetime;

static rwpipe *pip = NULL;
static uint8_t dvframe[ 144000 ];
static double start_time;
static double end_time;
static double current_time;
static int audio_output = 0;

/** Initialise the dvframe handling writer.
*/

void dvframes_init( dvframes_info info, double use_multiplier )
{
	char command[ 512 ];
	int i;
	for ( i = 0; i < VIDEO_FRAME_COUNT; i ++ )
		frames[ i ] = calloc( 1, sizeof( ppm_frame ) );
	for ( i = 0; i < AUDIO_FRAME_COUNT; i ++ )
		sample_frames[ i ] = calloc( 4, sizeof( int16_t ) * DV_AUDIO_MAX_SAMPLES );
	pal = info.pal;
	wide = info.wide;
	frequency = info.frequency;
	audio_output = info.audio_output;
	every = info.every;

	multiplier = use_multiplier;
	delta = 1.0 / multiplier;
	current = delta < 1 ? 1 - delta : 0;
    encoder = dv_encoder_new( 0, !pal, !pal );
    encoder->isPAL = pal;
	encoder->is16x9 = wide;
	frame_count = 0;
	datetime = time( NULL );

	start_time = info.start * ( pal ? 25 : 29.97 );
	if ( info.end > 0 )
		end_time = info.end * ( pal ? 25 : 29.97 );
	else
		end_time = INTMAX_MAX;
	current_time = 0;

	memset( &last_frame, 0, sizeof( ppm_frame ) );
	last_frame.data = malloc( info.width * info.height * 3 );
	memset( last_frame.data, 0, info.width * info.height * 3 );
	last_frame.width = info.width;
	last_frame.height = info.height;

	if ( info.ppm_output == 0 )
	{
		strcpy( command, "" );
		if ( info.filter )
		{
			filter = info.filter;
			sprintf( command, "%s |", info.filter );
		}
		sprintf( command + strlen( command ), "ppm2raw -scaler %d", info.scaler );
		if ( info.scaler < 2 )
			strcat( command, " -p 1 -q 1" );
		if ( info.scale == 1 )
			strcat( command, " -a" );
		else if ( info.scale == 2 )
			strcat( command, " -s" );
		if ( !info.pal )
			strcat( command, " -n" );
		if ( info.twopass )
			strcat( command, " -2" );
		if ( info.wide )
			strcat( command, " -w" );
		if ( info.preview )
			strcat( command, " | rawplay -q 3 -p -p -u 50" );

		pip = rwpipe_open( command );
	}
}

void dvframes_refresh_last_frame( ppm_frame *frame )
{
	if ( !frame->dupe )
	{
		if ( last_frame.data == NULL || frame->width != last_frame.width || frame->height != last_frame.height )
		{
			last_frame.data = realloc( last_frame.data, frame->width * frame->height * 3 );
			last_frame.width = frame->width;
			last_frame.height = frame->height;
		}
		memcpy( last_frame.data, frame->data, frame->width * frame->height * 3 );
	}
}

/** Add a video frame
*/

int dvframes_add_video( uint8_t *data, int width, int height, int linesize )
{
	if ( used < VIDEO_FRAME_COUNT )
	{
		int i;
		ppm_frame *frame = frames[ used ++ ];
		uint8_t *p = data;
		uint8_t *q;

		if ( frame == NULL )
			return -1;

		if ( data != NULL )
		{
			if ( frame->data == NULL || frame->width != width || frame->height != height )
			{
				frame->data = realloc( frame->data, width * height * 3 );
				frame->width = width;
				frame->height = height;
			}
			if ( frame->data == NULL )
				return -1;

			q = frame->data;

	   		for( i = 0; i< height; i++ ) 
			{
				memcpy( q, p, width * 3 );
				p += linesize;
				q += width * 3;
			}

			frame->dupe = 0;
		}
		else
		{
			frame->dupe = 1;
		}

		return 0;
	}
	else
	{
		return -1;
	}
}

/** Remove oldest video frame
*/

void dvframes_finished_with_oldest( )
{
	if ( used )
	{
		int i = 1;
		ppm_frame *frame = frames[ 0 ];
		for ( i = 1; i < used; i ++ )
			frames[ i - 1 ] = frames[ i ];
		used --;
		frames[ used ] = frame;
	}
}

/** Get the oldest video frame
*/

ppm_frame *dvframes_get_oldest( )
{
	if ( used > 0 )
		return frames[ 0 ];
	else
		return NULL;
}

/** Return number of video frames used.
*/

int dvframes_used( )
{
	return used;
}

/** Dub and output the frame.
*/

void dvframes_dub_output( uint8_t *dv, int in_range )
{
	int16_t *audio_buffers[ 4 ];
	int16_t *frame = sample_frames[ 0 ];
	int i = 1;
	int s;

	if ( dv != NULL && in_range )
	{
		audio_buffers[ 0 ] = &frame[ 0 ];
		audio_buffers[ 1 ] = &frame[ DV_AUDIO_MAX_SAMPLES ];
		audio_buffers[ 2 ] = &frame[ 2 * DV_AUDIO_MAX_SAMPLES ];
		audio_buffers[ 3 ] = &frame[ 3 * DV_AUDIO_MAX_SAMPLES ];

		s = dv_calculate_samples( encoder, frequency, output_count );

		if ( samples_per_frame[ 0 ] != s && samples_per_frame[ 0 ] != 0 )
			fprintf( stderr, "Mismatch %d != %d\n", s, samples_per_frame[ 0 ] );

		if ( output_count % 25 == 0 )
			fprintf( stderr, "Writing frame : %6d\r", output_count );

		dv_encode_full_audio( encoder, audio_buffers, 2, frequency, dv );
		dv_encode_metadata( dv, pal, wide, &datetime, output_count );
		dv_encode_timecode( dv, pal, output_count ++ );

		fwrite( dv, pal ? 144000 : 120000, 1, stdout );
		fflush( stdout );
	}
	else if ( dv == NULL && in_range )
	{
		if ( output_count % 25 == 0 )
			fprintf( stderr, "Writing frame : %6d\r", output_count );
		if ( output_count % every == 0 )
		{
			fprintf( stdout, "P6\n%d %d\n255\n", last_frame.width, last_frame.height );
			fwrite( last_frame.data, last_frame.width * last_frame.height, 3, stdout );
			fflush( stdout );
		}
		if ( audio_output )
		{
			fprintf( stdout, "A6 %d %d %d\n", frequency, 2, samples_per_frame[ 0 ] );
			fwrite( sample_frames[ 0 ], samples_per_frame[ 0 ] * sizeof( int16_t ), 1, stdout );
			fwrite( &sample_frames[ 0 ][ DV_AUDIO_MAX_SAMPLES ], samples_per_frame[ 0 ] * sizeof( int16_t ), 1, stdout );
			fflush( stdout );
		}
		output_count ++;
	}
	else
	{
		if ( output_count % 25 == 0 )
			fprintf( stderr, "Skipping frame: %6d\r", output_count );
		output_count ++;
	}

	if ( audio_used > 0 )
	{
		for ( i = 1; i <= audio_used; i ++ )
		{
			samples_per_frame[ i - 1 ] = samples_per_frame[ i ];
			sample_frames[ i - 1 ] = sample_frames[ i ];
		}
		sample_frames[ audio_used -- ] = frame;
	}
}

/** Flush the audio and video buffers.
*/

void dvframes_flush( int ignore_audio )
{
	int audio_frames_needed;
	uint8_t *dvframe_ptr = pip == NULL ? NULL : dvframe;

	while( dvframes_used() > 0 )
	{
		int actual = 0;
		ppm_frame *frame = dvframes_get_oldest( );

		if ( multiplier <= 1 )
		{
			audio_frames_needed = ( current + multiplier ) >= 1 ? 1 : 0;
		}
		else
		{
			double target = ceil( current );
			if ( target == current )
				target ++;
			audio_frames_needed = ( int )( ( target - current ) / delta ) + 1;
		}

		if ( audio_used < audio_frames_needed && !ignore_audio )
			break;

		dvframes_refresh_last_frame( frame );

		if ( current_time >= start_time && current_time <= end_time )
		{
			/* Convert frame to dv, but only if it will get output. */
			if ( audio_frames_needed > 0 && dvframe_ptr != NULL )
				ppm_to_rawdv( pip, frame, dvframe_ptr, pal ? 144000 : 120000 );

			if ( multiplier <= 1 )
			{
				current += multiplier;
				if ( current >= 1 )
				{
					dvframes_dub_output( dvframe_ptr, 1 );
					current = current - 1;
					actual ++;
					current_time ++;
				}
			}
			else if ( multiplier > 1 )
			{
				double target = ceil( current );
				if ( target == current )
					target ++;
				for ( ; current <= target; current += delta ) 
				{
					dvframes_dub_output( dvframe_ptr, 1 );
					actual ++;
					current_time ++;
				}
			}

			if ( actual != audio_frames_needed )
				fprintf( stderr, "Got %d instead of %d (%f)\n", actual, audio_frames_needed, multiplier );
		}
		else
		{
			int i;
			for ( i = 0; i < audio_frames_needed; i ++ )
			{
				dvframes_dub_output( NULL, 0 );
				current_time ++;
				if ( multiplier <= 1 )
					current += multiplier;
				else
					current += delta;
			}
		}

		dvframes_finished_with_oldest( );
	}
}

/** Add audio samples to the audio buffer
*/

int dvframes_add_audio( int16_t *samples, int count )
{
	int ret = 0;
	int16_t *p = samples;
	int16_t *c[ 2 ];
	int i;
	int offset;

	while( count && ret == 0 )
	{
		if ( audio_count == 0 )
		{
			if ( audio_used >= AUDIO_FRAME_COUNT )
			{
				ret = -1;
				break;
			}
			audio_count = dv_calculate_samples( encoder, frequency, frame_count ++ );
			memset( sample_frames[ audio_used ], 0, 4 * DV_AUDIO_MAX_SAMPLES * sizeof( int16_t ) );
			samples_per_frame[ audio_used ] = audio_count;
		}

		offset = samples_per_frame[ audio_used ] - audio_count;
		c[ 0 ] = &sample_frames[ audio_used ][ offset ];
		c[ 1 ] = &sample_frames[ audio_used ][ offset + DV_AUDIO_MAX_SAMPLES ];

		if ( count < audio_count )
		{
			for( i = 0; i < count * 2; i ++ )
				*c[ i % 2 ] ++ = *p ++;
			audio_count -= count;
			count = 0;
		}
		else
		{
			for( i = 0; i < audio_count * 2; i ++ )
				*c[ i % 2 ] ++ = *p ++;
			count -= audio_count;
			audio_count = 0;
			audio_used ++;
		}

		dvframes_flush( 0 );

		if ( current_time > end_time )
			ret = 1;
	}

	return ret;
}

/** Close the dvframes structure
*/

void dvframes_close( )
{
	int i;
	for ( i = 0; i < VIDEO_FRAME_COUNT; i ++ )
	{
		free( frames[ i ]->data );
		free( frames[ i ] );
	}
	for ( i = 0; i < AUDIO_FRAME_COUNT; i ++ )
		free( sample_frames[ i ] );

	if ( pip != NULL )
		rwpipe_close( pip );
}

