/*
 * xwd2raw.cc -- Obtains xwd images and converts to a raw dv stream
 * Copyright (C) 2002 Raphael Amaury Jacquot <sxpert@esitcom.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#ifdef __linux__
#include <stdint.h>
#else
#include <inttypes.h>
#endif

#include <string.h>
#include <sys/time.h>
#include <ctype.h>

#include <libdv/dv.h>
#include <pthread.h>

#define FRAME_RING		10

typedef struct
{
	int interactive;
	int is_pal;
	int running;
	int current;
	uint8_t *frame[ FRAME_RING ];
	uint8_t *image;
	int width;
	int height;
	dv_encoder_t *encoder;
	char *window;
	char *id;
	int reported_error;
	int last_width;
	int last_height;
	int image_magick;
}
xwd2raw_t, *xwd2raw;

static xwd2raw xwd2raw_init( )
{
	int i;
	xwd2raw frames = (xwd2raw)malloc( sizeof( xwd2raw_t ) );
	memset( frames, 0, sizeof( xwd2raw_t ) );
	for ( i = 0; i < FRAME_RING; i ++ )
		frames->frame[ i ] = (uint8_t *)malloc( 144000 );
	frames->current = -1;
	frames->running = 1;
	frames->is_pal = 1;
	return frames;
}

static void xwd2raw_set_pal( xwd2raw frames, int is_pal )
{
	frames->is_pal = is_pal;
}

static void xwd2raw_set_window( xwd2raw frames, char *window )
{
	frames->window = window;
}

static void xwd2raw_set_id( xwd2raw frames, char *id )
{
	frames->id = id;
}

static void xwd2raw_set_interactive( xwd2raw frames, int interactive )
{
	frames->interactive = interactive;
}

static void xwd2raw_set_image_magick( xwd2raw frames, int image_magick )
{
	frames->image_magick = image_magick;
}

static void xwd2raw_obtain_id( xwd2raw frames )
{
	if ( frames->id == NULL && frames->window == NULL && frames->interactive )
	{
		FILE *pipe = popen( "xwininfo | grep \"xwininfo: Window id\" | sed -e 's/.*0x/0x/ ; s/ .*$//'", "r" );
		char id[ 132 ];
		if ( pipe != NULL )
		{
			fgets( id, 132, pipe );
			id[ strlen( id ) - 1 ] = '\0';
			frames->id = strdup( id );
			pclose( pipe );
		}
	}
}

static int xwd2raw_get_dv_size( xwd2raw frames )
{
	return frames->is_pal ? 144000 : 120000;
}

static void xwd2raw_set_active( xwd2raw frames, int number )
{
	frames->current = number % FRAME_RING;
}

static int xwd2raw_frame_available( xwd2raw frames )
{
	return frames->current != -1;
}

static uint8_t *xwd2raw_get_inactive( xwd2raw frames )
{
	return frames->frame[ ( frames->current + 1 ) % FRAME_RING ];
}

static uint8_t *xwd2raw_get_active( xwd2raw frames )
{
	if ( xwd2raw_frame_available( frames ) )
		return frames->frame[ frames->current ];
	else
		return NULL;
}

static int xwd2raw_is_running( xwd2raw frames )
{
	return frames->running;
}

static void xwd2raw_stop( xwd2raw frames )
{
	frames->running = 0;
}

int read_ppm_number( FILE *pipe, int *value )
{
	int c = '\0';
	*value = 0;

	do 
	{
		while( !feof( pipe ) && !isdigit( c ) && c != '#' )
			c = fgetc( pipe );

		if ( c == '#' )
			while( !feof( pipe ) && c != '\n' )
				c = fgetc( pipe );
	}
	while ( !isdigit( c ) && !feof( pipe ) );

	while( isdigit( c ) && !feof( pipe ) )
	{
		*value = *value * 10 + ( c - '0' );
		c = fgetc( pipe );
	}

	return *value;
}

int read_ppm( FILE *pipe, int *width, int *height )
{
	char type[ 100 ];
	int maxval;

	if ( fscanf( pipe, "%s", type ) == 1 && !strcmp( type, "P6" ) )
	{
		read_ppm_number( pipe, width );
		read_ppm_number( pipe, height );
		read_ppm_number( pipe, &maxval );
	}

	return !strcmp( type, "P6" ) && *width != 0 && *height != 0;
}

static int xwd2raw_capture( xwd2raw frames )
{
	FILE *pipe = NULL;
	char command[ 512 ];
	int screen_width;
	int screen_height;

	if ( frames->image == NULL )
	{
		frames->width = 720;
		frames->height = frames->is_pal ? 576 : 480;
		frames->image = (uint8_t *)malloc( frames->width * frames->height * 3 );
		frames->encoder = dv_encoder_new( !frames->is_pal, !frames->is_pal, !frames->is_pal  );
		frames->encoder->isPAL = frames->is_pal;
		frames->encoder->is16x9 = FALSE;
		frames->encoder->vlc_encode_passes = 3;
		frames->encoder->static_qno = 0;
		frames->encoder->force_dct = DV_DCT_AUTO;
	}

	strcpy( command, "xwd -silent " );

	if ( frames->id != NULL )
	{
		strcat( command, "-id " );
		strcat( command, frames->id );
		strcat( command, " " );
	}
	else if ( frames->window != NULL )
	{
		strcat( command, "-name \"" );
		strcat( command, frames->window );
		strcat( command, "\" " );
	}
	else
	{
		strcat( command, "-root " );
	}

	if ( frames->image_magick )
		strcat( command, "| convert xwd:- ppm:- 2> /dev/null" );
	else
		strcat( command, "| xwdtopnm 2> /dev/null" );

	pipe = popen( command, "r" );

	if ( pipe != NULL )
	{
		int line = 0;
		uint8_t *scan_line = NULL;
		int start_line;
		int end_line;
		int start_offset;
		int line_width;
		int output_offset;

		if ( read_ppm( pipe, &screen_width, &screen_height ) )
		{
			scan_line = (uint8_t *)malloc( screen_width * 3 );

			if ( screen_width != frames->last_width || screen_height != frames->last_height )
			{
				memset( frames->image, 0, frames->width * frames->height * 3 );
				frames->last_width = screen_width;
				frames->last_height = screen_height;
			}

			start_line = screen_height / 2 - frames->height / 2;
			end_line = screen_height / 2 + frames->height / 2;

			if ( screen_width >= frames->width )
			{
				start_offset = screen_width / 2 - frames->width / 2;
				line_width = frames->width;
				output_offset = 0;
			}
			else
			{
				start_offset = 0;
				line_width = screen_width;
				output_offset = frames->width / 2 - screen_width / 2;
			}

			for ( line = 0; line < screen_height; line ++ )
			{
				fread( scan_line, 1, screen_width * 3, pipe );
				if ( line >= start_line && line < end_line )
					memcpy( frames->image + ( line - start_line ) * frames->width * 3 + output_offset * 3, scan_line + start_offset * 3, line_width * 3 );
			}
			free( scan_line );

			dv_encode_full_frame( frames->encoder, &frames->image, e_dv_color_rgb, xwd2raw_get_inactive( frames ) );
		}
		pclose( pipe );
		frames->reported_error = 0;
	}
	else
	{
		if ( frames->reported_error == 0 )
			fprintf( stderr, "Failed: %s\n", command );
		frames->reported_error = 1;
		return 1;
	}

	return 0;
}

static void xwd2raw_close( xwd2raw frames )
{
	free( frames->frame[ 0 ] );
	free( frames->frame[ 1 ] );
	free( frames->image );
	free( frames );
}

static void *xwd2raw_reader( void *ptr )
{
	xwd2raw frames = (xwd2raw)ptr;
	struct timespec sleeper = { 0, 0 };
	int frame_number = 0;

	while( xwd2raw_is_running( frames ) )
	{
		if ( xwd2raw_capture( frames ) == 0 )
			xwd2raw_set_active( frames, frame_number ++ );
		nanosleep( &sleeper, NULL );
	}

	return NULL;
}

static void *xwd2raw_writer( void *ptr )
{
	xwd2raw frames = (xwd2raw)ptr;
	struct timespec sleeper = { 0, 0 };
	FILE *pipe = stdout;
	
	while( xwd2raw_is_running( frames ) )
	{
		if ( xwd2raw_frame_available( frames ) )
		{
			uint8_t *frame = xwd2raw_get_active( frames );
			int size = xwd2raw_get_dv_size( frames );
			fwrite( frame, size, 1, pipe );
			fflush( pipe );
		}
		nanosleep( &sleeper, NULL );
	}

	return NULL;
}

static void wait_for_end( )
{
	char string[ 132 ];
	fprintf( stderr, "Press return to stop..." );
	fgets( string, 132, stdin );
}

static void usage( )
{
	fprintf( stderr, "xwd2raw [ -n ] [ -w window ] [ -i id ]\n" );
	fprintf( stderr, "Where: -n        - for NTSC (default is PAL)\n" );
	fprintf( stderr, "       -w window - for window to capture (default is whole screen)\n" );
	fprintf( stderr, "       -i id     - for id to capture (default is whole screen)\n" );
	fprintf( stderr, "       -s        - interactively select the window\n" );
	fprintf( stderr, "       -m        - use imagemagick for xwd to ppm conversion\n" );
	fprintf( stderr, "NB: Use xwininfo to obtain the wm_name or id - only one is needed\n" );
	fprintf( stderr, "    or use the interactive selector (when no -i or -w is specified)\n" );
	exit( 0 );
}

int main( int argc, char **argv )
{
	xwd2raw dv_frames = xwd2raw_init( );

	pthread_t reader;
	pthread_t writer;

	int index = 1;

	for ( index = 1; index < argc; index ++ )
	{
		if ( !strcmp( argv[ index ], "--help" ) )
			usage( );
		else if ( !strcmp( argv[ index ], "-n" ) )
			xwd2raw_set_pal( dv_frames, 0 );
		else if ( !strcmp( argv[ index ], "-w" ) )
			xwd2raw_set_window( dv_frames, argv[ ++ index ] );
		else if ( !strcmp( argv[ index ], "-i" ) )
			xwd2raw_set_id( dv_frames, argv[ ++ index ] );
		else if ( !strcmp( argv[ index ], "-s" ) )
			xwd2raw_set_interactive( dv_frames, 1 );
		else if ( !strcmp( argv[ index ], "-m" ) )
			xwd2raw_set_image_magick( dv_frames, 1 );
		else
			usage( );
	}

	xwd2raw_obtain_id( dv_frames );

	pthread_create( &reader, NULL, xwd2raw_reader, dv_frames );
	pthread_create( &writer, NULL, xwd2raw_writer, dv_frames );

	wait_for_end( );

	xwd2raw_stop( dv_frames );

	pthread_join( writer, NULL );
	pthread_join( reader, NULL );

	xwd2raw_close( dv_frames );

	return 0;
}

