/*
 * Copyright (c) 1997-2003 Motonori Nakamura <motonori@media.kyoto-u.ac.jp>
 * Copyright (c) 1997-2003 WIDE Project
 * Copyright (c) 1997-2003 Kyoto University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 4. Neither the name of the Project, the University nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char *_id_ = "$Id: dns.c,v 1.86 2003/08/20 05:12:58 motonori Exp $";
#endif

# include "common.h"
# include "extern.h"

#if PACKETSZ > 1024
#define MAXPACKET	PACKETSZ
#else
#define MAXPACKET	1024
#endif

#ifdef ultrix
#define NSADDR_LIST(x)	_res.ns_list[x].addr	/* ultrix implementation is strange */
#endif

#ifndef NSADDR_LIST
# if defined(INET6) && !defined(RESOLVER_V4)
#  ifdef RESOLVER_KAME
#   define NSADDR_LIST(x)	_res_ext.nsaddr_list[x]	/* for standerd BIND */
#  else
#   ifdef RESOLVER_OLDKAME
#    define NSADDR_LIST(x)	_res.nsaddr_list_un[x]	/* for standerd BIND */
#   else
#    error unknown resolver type
#   endif
#  endif
# else
#  define NSADDR_LIST(x)	_res.nsaddr_list[x]		/* for standerd BIND */
# endif
#endif

#define CNAMELOOPMAX	10

#define FAKE_AR_CHECK	0

#if FAKE_AR_CHECK
# define ARCHECKSZ	32
#endif

#ifndef RES_INSECURE1
# define RES_INSECURE1	0x00000400	/* type 1 security disabled */
#endif

int
pregetmx(dl)
struct domain *dl;
{
	struct domain *dp;
	char **map_arg;
	long stime, etime;
	int n;
	static int query_domain(), process_query();
#ifdef INET6
	SockAddr dsin;	/* dummy */
#endif

	for (n = 0; n < _res.nscount; n++)
	{
		struct sockaddr_in *srv;

#if INET6
		srv = (struct sockaddr_in *)&NSADDR_LIST(n);
#else
		srv = &NSADDR_LIST(n);
#endif
		if (srv->sin_family == AF_INET
		 && srv->sin_addr.s_addr == INADDR_ANY)
		{
			/* use 127.0.0.1 instead */
			srv->sin_addr.s_addr = htonl(0x7f000001);
		}
	}

	log(LOG_DEBUG, "start getting MXRRs with caching");
	n = 0;
	stime = time(NULL);
	if (env.queueid != NULL)
		setproctitle("[%s] pre-getting MX", env.queueid);
	else
		setproctitle("pre-getting MX", NULL);

	if (cnf.pgateway != NULL)
	{
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "pregetting A of protocol gateway: %s",
			cnf.pgateway);
#ifdef INET6
		if (cnf.inetdom & SMTP_V4)
		{
			if (query_domain(cnf.pgateway, T_A) < 0)
				return -1;
			n++;
		}
		if (cnf.inetdom & SMTP_V6)
		{
			if (query_domain(cnf.pgateway, T_AAAA) < 0)
				return -1;
			n++;
		}
#else
		if (query_domain(cnf.pgateway, T_A) < 0)
			return -1;
		n++;
#endif
	}
	if (cnf.fallbackmx != NULL)
	{
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "pregetting A of fallbackmx: %s",
			cnf.fallbackmx);
#ifdef INET6
		if (cnf.inetdom & SMTP_V4)
		{
			if (query_domain(cnf.fallbackmx, T_A) < 0)
				return -1;
			n++;
		}
		if (cnf.inetdom & SMTP_V6)
		{
			if (query_domain(cnf.fallbackmx, T_AAAA) < 0)
				return -1;
			n++;
		}
#else
		if (query_domain(cnf.fallbackmx, T_A) < 0)
			return -1;
		n++;
#endif
	}

	if (cnf.sourceIP != NULL)
	{
		char *p, *q;

		p = cnf.sourceIP;
		while (p != NULL)
		{
			q = strchr(p, ',');
			if (q != NULL)
				*q = '\0';

#ifdef INET6
			if (inet_addr(p) == -1
			 && inet_pton(INET6, p, &dsin.in6.sin6_addr) <= 0)
#else
			if (inet_addr(p) == -1)
#endif
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG,
					"pregetting A of sourceIP: %s", p);
#ifdef INET6
				if (cnf.inetdom & SMTP_V4)
				{
					if (query_domain(p, T_A) < 0)
						return -1;
					n++;
				}
				if (cnf.inetdom & SMTP_V6)
				{
					if (query_domain(p, T_AAAA) < 0)
						return -1;
					n++;
				}
#else
				if (query_domain(p, T_A) < 0)
					return -1;
				n++;
#endif
			}
			if (q != NULL)
			{
				*q = ',';
				p = q + 1;
			}
			else
				p = NULL;
		}
	}

	for (dp = dl; dp != NULL; dp = dp->next)
	{
		char *map_opt;

		/* known by previous transaction */
		if (dp->firstmx != NULL || dp->stat != EX_OK)
			continue;

#if 0
		/* skip numeric address spec */
		if (*(dp->name) == '[')
			continue;
#endif

		map_arg = dp->map_arg;
		map_opt = strchr(*map_arg, '/');
		while (*map_arg != NULL)
		{
			int len;

			if (map_opt != NULL)
				*map_opt = '\0'; /* ignore /opt portion */

			if ((strcasecmp(*map_arg, "MX") == 0)
			 || (strcasecmp(*map_arg, "MX?") == 0))
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "pregetting MX: %s", dp->name);
				if (query_domain(dp->name, T_MX) < 0)
					return -1;
				n++;
			}
			else if (strcasecmp(*map_arg, "A") == 0)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "pregetting A: %s", dp->name);
#ifdef INET6
				if (cnf.inetdom & SMTP_V4)
				{
#endif
					if (query_domain(dp->name, T_A) < 0)
						return -1;
					n++;
#ifdef INET6
				}
				if (cnf.inetdom & SMTP_V6)
				{
					if (query_domain(dp->name, T_AAAA) < 0)
						return -1;
					n++;
				}
#endif
			}
			else if (**map_arg != '[')
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "pregetting MX: %s", *map_arg);
				if (query_domain(*map_arg, T_MX) < 0)
					return -1;
				n++;
			}
			else if ((len = strlen(*map_arg)) > 2
			      && (*map_arg)[len - 1] == ']')
			{	/* [...] spec */
				(*map_arg)[len - 1] = '\0';
#ifdef INET6
				if (inet_addr((*map_arg) + 1) != -1
				 || inet_pton(INET6, (*map_arg) + 1,
					      &dsin.in6.sin6_addr) > 0)
#else
				if (inet_addr((*map_arg) + 1) != -1)
#endif
					goto skip_preget_a;

				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG,
					"pregetting A of sourceIP: %s",
					(*map_arg) + 1);
#ifdef INET6
				if (cnf.inetdom & SMTP_V4)
				{
#endif
					if (query_domain((*map_arg) + 1,
							 T_A) < 0)
					{
						(*map_arg)[len - 1] = ']';
						return -1;
					}
					n++;
#ifdef INET6
				}
				if (cnf.inetdom & SMTP_V6)
				{
					if (query_domain((*map_arg) + 1,
							 T_AAAA) < 0)
					{
						(*map_arg)[len - 1] = ']';
						return -1;
					}
					n++;
				}
#endif
  skip_preget_a:
				(*map_arg)[len - 1] = ']';
			}
			if (map_opt != NULL)
				*map_opt = '/';	/* restore /opt portion */
			map_arg++;
		}
	}
	if (n > 0)
	{
		if (env.queueid != NULL)
			setproctitle("[%s] syncing MX: %d", env.queueid, n);
		else
			setproctitle("syncing MX: %d", n);
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "syncing responses of MXRRs");
		if (process_query(1) < 0)	/* sync */
			return -1;
		etime = time(NULL);
		log(LOG_DEBUG, "got MXRRs: %d MXs in %d seconds",
			n, etime - stime);
	}
	else
	{
		log(LOG_DEBUG, "no MXRRs to sync");
	}
	return 0;
}

static int
query_domain(name, type)
char *name;
int type;
{
	struct dns_stat *qrp, **hashp;
	static int make_query();

	qrp = hash_query_lookup(name, &hashp);
	if (qrp == NULL) {
		qrp = (struct dns_stat *)MALLOC(sizeof(struct dns_stat));
		bzero(qrp, sizeof(struct dns_stat));
		if (qrp == NULL)
		{
			log(LOG_NOTICE, "out of memory (dns_query)");
			return -1;
		}
		qrp->name = newstr(name);
		qrp->next = *hashp;
		*hashp = qrp;
	}

	switch (type) {
	  case T_MX:
	  case T_A:
	  case T_AAAA:
		break;
	  default:
		log(LOG_NOTICE, "query_domain: unknown query type: %d", type);
		return 0;
	}

	make_query(qrp, type);
	return 0;
}

static struct query_chain *QueryChain = NULL;
static struct query_chain *RevQueryChain = NULL;
static query_chain_count = 0;

static int
make_query(qrp, type)
struct dns_stat *qrp;
int type;
{
	int qflag;
	u_char buf[MAXPACKET];
	int n;
	struct query_chain *qchain;
	static int process_query();

	switch (type) {
	  case T_MX:
		qflag = RR_MX;
		break;
	  case T_A:
		qflag = RR_A;
		break;
	  case T_AAAA:
		qflag = RR_AAAA;
		break;
	  default:
		/* error */
		return -1;
	}

	if ((qrp->rr_got & qflag)
	 || (qrp->rr_timeout & qflag)
	 || (qrp->rr_noex & qflag))
	{
		/* already got */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "ignore request n=%s t=%d",
			qrp->name, type);
		return 0;
	}

	qrp->rr_need |= qflag;

	if (cnf.tryANYfirst && type == T_MX && (qrp->rr_sent & RR_ANY) == 0)
	{
		type = T_ANY;
		qrp->rr_sent |= RR_ANY;
	}
	else if ((qrp->rr_sent & qflag) == 0)
	{
		qrp->rr_sent |= qflag;
	}
	else
	{
		/* already in processing */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "skip request n=%s t=%d", qrp->name, type);
		return 0;
	}

	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "res_mkquery n=%s t=%d", qrp->name, type);
	n = res_mkquery(QUERY, qrp->name, C_IN, type, NULL, 0, NULL,
			buf, sizeof(buf));
	if (n <= 0)
	{
		/* error */
		log(LOG_NOTICE, "res_mkquery failed for %s type %d",
			qrp->name, type);
		return -1;
	}

	qchain = (struct query_chain *)MALLOC(sizeof(struct query_chain));
	if (qchain == NULL)
	{
		/* error */
		log(LOG_NOTICE, "out of memory (make_query)");
		return -1;
	}
	bzero(qchain, sizeof(struct query_chain));
	qchain->chain = qrp->requests;
	qrp->requests = qchain;
	qchain->request = (char *)MALLOC(n);
	if (qchain->request == NULL)
	{
		/* error */
		log(LOG_NOTICE, "out of memory (make_query)");
		return -1;
	}
	bcopy(buf, qchain->request, n);
	qchain->reqlen = n;
	qchain->stat = qrp;
	qchain->type = type;
	qchain->need = qflag;

	if (QueryChain == NULL)
	{
		QueryChain = RevQueryChain = qchain;
		qchain->next = NULL;
	}
	else
	{
		qchain->next = QueryChain;
		qchain->next->prev = qchain;
		QueryChain = qchain;
	}
	qchain->prev = NULL;
	query_chain_count++;
	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "DNS: query concurrency = %d", query_chain_count);
	if (sti.maxcquery < query_chain_count)
		sti.maxcquery = query_chain_count;

#if 0
	if (cnf.debug & DEBUG_DNS)
	{
		struct query_chain *qcp;

		log(LOG_DEBUG, "Chain after linked:");
		for (qcp = QueryChain; qcp != NULL; qcp = qcp->next)
		{
			log(LOG_DEBUG, " %s", qcp->stat->name);
		}
	}
#endif

	if (process_query(0) < 0)
		return -1;
	return 0;
}

int
whichserver(inp)
#if INET6
const SockAddr *inp;
#else
const struct sockaddr_in *inp;
#endif
{
	register int ns;

	for (ns = 0; ns < _res.nscount; ns++)
	{
#if INET6
		const SockAddr *srv = (SockAddr *)&NSADDR_LIST(ns);

		if (srv->in.sin_family == inp->in.sin_family
		 && srv->in.sin_port == inp->in.sin_port
		 && srv->in.sin_addr.s_addr == inp->in.sin_addr.s_addr)
		{
			return ns;
		}
		else if (srv->in6.sin6_family == inp->in6.sin6_family
		 && srv->in6.sin6_port == inp->in6.sin6_port
		 && bcmp(&srv->in6.sin6_addr, &inp->in6.sin6_addr, 16) == 0)
		{
			return ns;
		}
#else
		register const struct sockaddr_in *srv = &NSADDR_LIST(ns);

		if (srv->sin_family == inp->sin_family
		 && srv->sin_port == inp->sin_port
		 && srv->sin_addr.s_addr == inp->sin_addr.s_addr)
		{
			return ns;
		}
#endif
	}
	return -1;
}

#define SLEEPFACTOR	10
#define SLEEPTIME	10000	/* micro second */
#ifndef CHECK_SRVR_ADDR
# define CHECK_SRVR_ADDR	1
#endif

static int
process_query(sync)
int sync;
{
	static int s = -1;
#if INET6
	static int s6 = -1;
#endif
	struct query_chain *qcp, *nextqcp;
	time_t now, latest;
	int itvl, class, type;
#if INET6
	SockAddr from, *nsap;
#else
	struct sockaddr_in from, *nsap;
#endif
	struct timeval timeout;
	static int nsent = 0;
	static fd_set dsmask;
        int max, n, fromlen, resplen, ns;
	union {
		HEADER  qb1;
		char    qb2[MAXPACKET];
	} answer;
	int anssiz;
        HEADER *qp, *hp;
	u_char *eom, *ap;
	char nbuf[MAXDNAME+1];
	struct dns_stat *qrp;
	int pending;
	static void got_answer();
	static void remove_query();

	if (sync == 0 && cnf.cquery_max != 0
	 && query_chain_count >= cnf.cquery_max)
	{
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "reached to max concurrency (%d)....syncing",
			query_chain_count);
		sync = -1;
	}
#if INET6
	for (n = 0; n < _res.nscount; n++)
	{
		const SockAddr *srv = (SockAddr *)&NSADDR_LIST(n);

		if (s == -1 && srv->in.sin_family == AF_INET)
		{
			s = socket(AF_INET, SOCK_DGRAM, PF_UNSPEC);
			if (s < 0) {
				/* error */
				log(LOG_NOTICE,
					"socket failed (process_query)");
				return -1;
			}
		}
		else if (s6 == -1 && srv->in6.sin6_family == AF_INET6)
		{
			s6 = socket(AF_INET6, SOCK_DGRAM, PF_UNSPEC);
			if (s6 < 0) {
				/* error */
				log(LOG_NOTICE,
					"socket(INET6) failed (process_query)");
				return -1;
			}
		}
	}
#else
	if (s == -1) {
		s = socket(AF_INET, SOCK_DGRAM, PF_UNSPEC);
		if (s < 0) {
			/* error */
			log(LOG_NOTICE, "socket failed (process_query)");
			return -1;
		}
	}
#endif

again:
	now = time(NULL);

	/* send request packets */
	for (qcp = QueryChain; qcp != NULL;)
	{
		if (qcp->time <= now && qcp->time != 0)
		{
			/* time to work (not the first time) */
			do {
				++qcp->ns;
			} while (qcp->badns & (1<<qcp->ns));
			if (qcp->ns >= (u_char)_res.nscount)
			{
				qcp->ns = 0;
				while (qcp->badns & (1<<qcp->ns))
					qcp->ns++;
				if (qcp->ns >= (u_char)_res.nscount)
				{
					/* no valid ns, set status of timeout */
					qcp->stat->rr_timeout |= qcp->need;
					nextqcp = qcp->next;
					remove_query(qcp);
					if (cnf.debug & DEBUG_DNS)
					log(LOG_DEBUG, "DNS: no valid ns, removing %s type %d",
						qcp->stat->name, qcp->type);
					qcp = nextqcp;
					continue;
				}
				qcp->try++;
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: retry count=%d", qcp->try);
			}
		}
		if (qcp->try >= (u_char)_res.retry)
		{
			/* set status of timeout */
			qcp->stat->rr_timeout |= qcp->need;
			nextqcp = qcp->next;
			remove_query(qcp);
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: no valid ns, removing %s type %d",
				qcp->stat->name, qcp->type);
			qcp = nextqcp;
			continue;
		}
		if (qcp->time <= now)	/* the time to work */
		{
			itvl = _res.retrans << qcp->try;
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: waiting %d sec.", itvl);
			if (qcp->try > 0)
				itvl /= _res.nscount;
			if (itvl <= 0)
				itvl = 1;
			qcp->time = now + itvl;
#if INET6
			nsap = (SockAddr *)&NSADDR_LIST(qcp->ns);
			if (nsap->in.sin_family == AF_INET)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "send query n=%s t=%d (s=%s)",
					qcp->stat->name, qcp->type,
					inet_ntoa(nsap->in.sin_addr));

				if (sendto(s, qcp->request, qcp->reqlen, 0,
					   (struct sockaddr *)&nsap->in,
					   sizeof(struct sockaddr_in))
				    != qcp->reqlen)
				{
					qcp->badns |= 1<<qcp->ns;
					if (cnf.debug & DEBUG_DNS)
					log(LOG_DEBUG, "sendto failed: invalid ns (%s)", strerror(errno));
				}
			}
			if (nsap->in6.sin6_family == AF_INET6)
			{
				char buf[64];

				if (cnf.debug & DEBUG_DNS)
				{
				inet_ntop(AF_INET6, &nsap->in6.sin6_addr,
					buf, sizeof(buf));
				log(LOG_DEBUG, "send query n=%s t=%d (s=%s)",
					qcp->stat->name, qcp->type, buf);
				}

				if (sendto(s6, qcp->request, qcp->reqlen, 0,
					   (struct sockaddr *)&nsap->in6,
					   sizeof(struct sockaddr_in6))
				    != qcp->reqlen)
				{
					qcp->badns |= 1<<qcp->ns;
					if (cnf.debug & DEBUG_DNS)
					log(LOG_DEBUG, "sendto failed: invalid ns (%s)", strerror(errno));
				}
			}
#else
			nsap = &NSADDR_LIST(qcp->ns);
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "send query n=%s t=%d (s=%s)",
				qcp->stat->name, qcp->type,
				inet_ntoa(nsap->sin_addr));

			if (sendto(s, qcp->request, qcp->reqlen, 0,
				   (struct sockaddr *)nsap,
				   sizeof(struct sockaddr)) != qcp->reqlen)
			{
				qcp->badns |= 1<<qcp->ns;
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "sendto failed: invalid ns");
			}
#endif
			sti.nqueries++;
			sti.dns_outb += qcp->reqlen;
			if (++nsent == SLEEPFACTOR) {
#if 0 /* HAVE_USLEEP */
				usleep(SLEEPTIME);
#else
				timeout.tv_sec = 0;
				timeout.tv_usec = SLEEPTIME;
				(void)select(0, (fd_set *)NULL, (fd_set *)NULL,
					     (fd_set *)NULL, &timeout);
#endif
				nsent = 0;
			}
		}
		qcp = qcp->next;
	}

	do {
		latest = 0;

		if (sync != 0) {
			/* calculate next action time */
			for (qcp = QueryChain; qcp != NULL; qcp = qcp->next)
			{
				if (latest == 0 || qcp->time < latest)
					latest = qcp->time;
			}

			if (latest == 0)	/* no more pending queries */
				break;
			if (latest > now)
				timeout.tv_sec = latest - now;
			else
				timeout.tv_sec = 1;	/* XXX */
			timeout.tv_usec = 0;
		} else {     
			/* no wait */
			timeout.tv_sec = 0;
			timeout.tv_usec = 0;
		}
		if ((cnf.debug & DEBUG_DNS) && (timeout.tv_sec > 0))
		log(LOG_DEBUG, "DNS: waiting %d sec in %s mode",
			timeout.tv_sec, sync?"sync":"nosync");
		FD_ZERO(&dsmask);
#if INET6
		max = 0;
		if (s >= 0)
		{
			FD_SET(s, &dsmask);
			max = s + 1;
		}
		if (s6 >= 0)
		{
			FD_SET(s6, &dsmask);
			if (s6 + 1 > max)
				max = s6 + 1;
		}
#else
		FD_SET(s, &dsmask);
		max = s + 1;
#endif
		if ((n = select(max, &dsmask, (fd_set *)NULL, (fd_set *)NULL,
				&timeout)) > 0)
		{
			/* packet arrived */
			fromlen = sizeof(struct sockaddr_in);
			anssiz = sizeof(answer);
#if INET6
			if (s >= 0 && FD_ISSET(s, &dsmask))
			{
				resplen = recvfrom(s, &answer, anssiz, 0,
					   (struct sockaddr *)&from, &fromlen);
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got answer from %s",
					inet_ntoa(from.in.sin_addr));
			}
			else if (s6 >= 0 && FD_ISSET(s6, &dsmask))
			{
				char buf[64];

				resplen = recvfrom(s6, &answer, anssiz, 0,
					   (struct sockaddr *)&from, &fromlen);
				if (cnf.debug & DEBUG_DNS)
				{
					inet_ntop(AF_INET6, &from.in6.sin6_addr,
						buf, sizeof(buf));
					log(LOG_DEBUG,
						"DNS: got answer from %s", buf);
				}
			}
#else
			resplen = recvfrom(s, &answer, anssiz, 0,
					   (struct sockaddr *)&from, &fromlen);
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: got answer from %s",
				inet_ntoa(from.sin_addr));
#endif

			if (0) {
				show_query("DNS: answer received",
					&answer, &answer+resplen);
			}
			now = time(NULL);

			if (resplen <= 0) {
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: bad anssiz %d",
					resplen);
				continue;	/* error */
			}

			ns = whichserver(&from);
			if (ns < 0)
			{
#if CHECK_SRVR_ADDR
				if (!(_res.options & RES_INSECURE1))
				{
					if (cnf.debug & DEBUG_DNS)
					log(LOG_DEBUG,
						"DNS: not from our server");
					continue;
				}
#endif
				ns = 0;
			}

			hp  = (HEADER *) &answer;
			eom = (u_char *) &answer + resplen;
			ap = (u_char *) hp + sizeof(HEADER);

			n = dn_expand((u_char *)hp, eom, ap, nbuf, sizeof nbuf);
			ap += n;
			GETSHORT(type, ap);
			GETSHORT(class, ap);
 
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: got answer for %s type %d",
				nbuf, type);

			qrp = hash_query_lookup(nbuf, NULL);
			if (qrp == NULL)
			{
				show_query("DNS: no associated query sent",
					&answer, &answer+resplen);
				continue;	/* no associated request */
			}
			qcp = qrp->requests;
			while (qcp != NULL && qcp->type != type)
				qcp = qcp->chain;
			if (qcp == NULL)
			{
				show_query("DNS: no associated query",
					&answer, &answer+resplen);
				continue;	/* no associated request */
			}

			if (hp->rcode == SERVFAIL ||
			    hp->rcode == FORMERR ||
			    hp->rcode == NOTIMP ||
			    hp->rcode == REFUSED)
			{
				qcp->badns |= (1<<ns);
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: badnsF rcode=%d %x/%x",
					hp->rcode,
					qcp->badns, (1<<_res.nscount)-1);
				if (qcp->badns != (1<<_res.nscount)-1)
					continue; /* others may be good */

				qrp->rr_timeout |= qcp->need;
				remove_query(qcp);
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: TCP requested, removing %s type %d",
					qcp->stat->name, qcp->type);
				continue;
			}
			if (hp->tc) {
				/* TCP requested */
				qrp->rr_timeout |= qcp->need;
				remove_query(qcp);
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: TCP requested, removing %s type %d",
					qcp->stat->name, qcp->type);
				continue;
			}

			sti.nanswers++;
			sti.dns_inb += resplen;
			got_answer(qcp, (u_char *)&answer, resplen);

		}
		if (n < 0) {
			/* error in select */
			break;
		}
	} while(n > 0);

	if (sync != 0) {
		pending = 0;
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "DNS: sync: get nearest timeout from %d", now);
		for (qcp = QueryChain; qcp != NULL;)
		{
			if (qcp->try >= (u_char)_res.retry)
			{
				/* set status of timeout */
				qcp->stat->rr_timeout |= qcp->need;
				nextqcp = qcp->next;
				remove_query(qcp);
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: sync timeout (n=%s, t=%d)",
					qcp->stat->name, qcp->type);
				qcp = nextqcp;
				continue;
			}

			pending++;
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: event timeout=%d", qcp->time);
			qcp = qcp->next;
		}
		if (sync < 0)
			return (0); /* concurrency control */
		if (pending)
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: retry in sync mode");
			/* sleep(1); * XXX */
			goto again;
		}

		for (qcp = QueryChain; qcp != NULL;)
		{
			qcp->stat->rr_timeout |= qcp->need;
			nextqcp = qcp->next;
			remove_query(qcp);
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: final timeout (n=%s, t=%d)",
				qcp->stat->name, qcp->type);
			qcp = nextqcp;
		}
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "DNS: closing socket");
#if INET6
		if (s >= 0)
		{
			close(s);
			s = -1;
		}
		if (s6 >= 0)
		{
			close(s6);
			s6 = -1;
		}
#else
		close(s);
		s = -1;
#endif
	}
	return (0);
}

static void
remove_query(qcp)
struct query_chain *qcp;
{
	if (qcp->request == NULL)
		return;
	else
		free(qcp->request);
	qcp->request = NULL;
	if (qcp->prev != NULL)
		qcp->prev->next = qcp->next;
	if (qcp->next != NULL)
		qcp->next->prev = qcp->prev;
	if (QueryChain == qcp)
		QueryChain = qcp->next;
	if (RevQueryChain == qcp)
		RevQueryChain = qcp->prev;
	qcp->prev = NULL;
	qcp->next = NULL;
	query_chain_count--;

#if 0
	if (cnf.debug & DEBUG_DNS)
	{
		log(LOG_DEBUG, "Chain after removed:");
		for (qcp = QueryChain; qcp != NULL; qcp = qcp->next)
		{
			log(LOG_DEBUG, " %s", qcp->stat->name);
		}
	}
#endif
}

static void
got_answer(qcp, answer, anssiz)
struct query_chain *qcp;
u_char *answer;
int anssiz;
{
	u_char *b, *eom, *ap;
	HEADER *hp;
	int ancount, qdcount, nscount, arcount;
	register int n;
	int type, class, pref, domain;
	char nbuf[MAXDNAME+1], nbuf2[MAXDNAME+1];
	struct mx *mxp, mxtmp;
	struct dns_stat *qrp, *qrp2, **hashp;
	int qflag;
	int mx_found, addr_found, cname_found;
#if FAKE_AR_CHECK
	static char archeck[ARCHECKSZ][MAXDNAME+1];
	int narcheck, found;
#endif

	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "DNS: got answer, removing %s type %d",
		qcp->stat->name, qcp->type);

	b  = (u_char *) answer;
	hp = (HEADER *) b;
	ap = (u_char *) hp + sizeof(HEADER);
	eom = (u_char *) answer + anssiz;

	if (hp->aa == 0 && hp->ra == 0) {
		/* this nameserver does not support recursion */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "non recursive ns=%d", qcp->ns);
		qcp->badns |= 1<<qcp->ns;
		return;	/* keep waiting for a reply from other NS */
	}

	remove_query(qcp);

	qdcount = ntohs(hp->qdcount);
	if (qdcount <= 0)
		return;

	ancount = ntohs(hp->ancount);

	if (hp->rcode == NXDOMAIN
	 || qcp->type == T_ANY && hp->rcode == NOERROR && ancount == 0)
	{
		/* host not found */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "DNS: host (any other data) not found");

		qcp->stat->rr_noex = RR_ALL;
		return;
	}

	if (hp->rcode == NOERROR && ancount == 0
	 || hp->rcode == FORMERR || hp->rcode == NOTIMP || hp->rcode == REFUSED)
	{
		/* no data, other data exist */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "DNS: no data for %d", qcp->type);

		if (qcp->type == T_ANY)
		{
			/* query with exact record type */
			if (qcp->need == RR_MX)
			{
				make_query(qcp->stat, T_MX);
			}
			else if (qcp->need == RR_AAAA)
			{
				make_query(qcp->stat, T_AAAA);
			}
			else if (qcp->need == RR_A)
			{
				make_query(qcp->stat, T_A);
			}
			return;
		}
		qcp->stat->rr_noex |= qcp->need;
		if (qcp->type == T_MX)
		{
			/* no MX, try A/AAAA */
#ifdef INET6
			if (cnf.inetdom & SMTP_V6 || cnf.v4v6fallback)
			{
				make_query(qcp->stat, T_AAAA);
			}
			if (cnf.inetdom & SMTP_V4 || cnf.v4v6fallback)
			{
				make_query(qcp->stat, T_A);
			}
#else
			make_query(qcp->stat, T_A);
			if (cnf.v4v6fallback)
			{
				make_query(qcp->stat, T_AAAA);
			}
#endif
		}
		return;
	}

	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "DNS: valid data");

	while (qdcount--)
	{
		n = dn_expand((u_char *) b, eom, ap, nbuf, sizeof nbuf);
		ap += n;
		if (ap + INT16SZ*2 > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(type, ap);
		GETSHORT(class, ap);
	}

#if FAKE_AR_CHECK
	narcheck = 0;
#endif
	addr_found = mx_found = cname_found = 0;
	ancount = ntohs(hp->ancount);
	while (--ancount >= 0 && ap < eom)
	{
		n = dn_expand((u_char *)b, eom, ap, nbuf, sizeof nbuf);
		if (n < 0)
			return;
		ap += n;
		if (ap + INT16SZ*3 + INT32SZ > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(type, ap);
		GETSHORT(class, ap);
		ap += INT32SZ; /* TTL */
		GETSHORT(n, ap);
		if (ap + n > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		if (class != C_IN)
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: unexpected class %d in an", class);
			ap += n;
			continue;
		}
		if (type == T_MX)
		{
			if (strcasecmp(qcp->stat->name, nbuf) != 0)
			{
				/* may be a MX pointed by CNAME */
				mx_found = 1;
#if 1
				cname_found = 0;
#endif
				qrp = hash_query_lookup(nbuf, NULL);
				if (qrp == NULL) {
					/* chain broken */
					return;
				}
#if 0
				ap += n;
				continue;
#endif
			}
			else
				qrp = qcp->stat;
			if (qcp->need == RR_MX)
			{
				mx_found = 1;
				GETSHORT(pref, ap);
				if ((n = dn_expand((u_char *)b, eom, ap,
						   nbuf2, sizeof nbuf2)) < 0)
					return;

				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got MX for %s, %s p=%d",
					nbuf, nbuf2, pref);

				qrp->rr_got |= RR_MX;
				mxp = (struct mx*)MALLOC(sizeof(struct mx));
				mxp->next = qrp->mx;
				qrp->mx = mxp;
				mxp->name = newstr(nbuf2);
				mxp->pref = pref;
#if FAKE_AR_CHECK
				if (narcheck < ARCHECKSZ)
					strncpy(archeck[narcheck++], nbuf2,
						MAXDNAME+1);
#endif
			}
#if FAKE_AR_CHECK
			else
			{
				GETSHORT(pref, ap);
				if ((n = dn_expand((u_char *)b, eom, ap,
						   nbuf2, sizeof nbuf2)) < 0)
					return;
				if (narcheck < ARCHECKSZ)
					strncpy(archeck[narcheck++], nbuf2,
						MAXDNAME+1);
			}
#endif
		}
#ifdef INET6
		else if (type == T_AAAA
		      && ((cnf.inetdom & SMTP_V6) || cnf.v4v6fallback))
#else
		else if (type == T_AAAA && cnf.v4v6fallback)
#endif
		{
			if (strcasecmp(qcp->stat->name, nbuf) == 0)
			{
				addr_found = 1;
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got AAAA for %s",
					qcp->stat->name);

				qcp->stat->rr_got |= RR_AAAA;
			}
			else
			{
				/* may be address for an alias */
				cname_found = 0;
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got aliased (%s) AAAA for %s",
					nbuf, qcp->stat->name);
				qrp = hash_query_lookup(nbuf, NULL);
				if (qrp == NULL)
				{
					/* chain broken */
					return;
				}
				qrp->rr_got |= RR_AAAA;
			}
			bzero(&mxtmp, sizeof(mxtmp));
			mxtmp.name = nbuf;
#ifdef INET6
			if (addinetaddress(&mxtmp, AF_INET6, n, ap) < 0)
				return;
#else
			if (addinetaddress(&mxtmp, AF_UNSPEC, 0, "") < 0)
				return;
#endif
		}
#ifdef INET6
		else if (type == T_A
		      && ((cnf.inetdom & SMTP_V4) || cnf.v4v6fallback))
#else
		else if (type == T_A)
#endif
		{
			if (strcasecmp(qcp->stat->name, nbuf) == 0)
			{
				addr_found = 1;
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got A for %s",
					qcp->stat->name);

				qcp->stat->rr_got |= RR_A;
			}
			else
			{
				cname_found = 0;
				/* may be address for an alias */
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: got aliased (%s) A for %s",
					nbuf, qcp->stat->name);
				qrp = hash_query_lookup(nbuf, NULL);
				if (qrp == NULL)
				{
					/* chain broken */
					return;
				}
				qrp->rr_got |= RR_A;
			}
			bzero(&mxtmp, sizeof(mxtmp));
			mxtmp.name = nbuf;
			if (addinetaddress(&mxtmp, AF_INET, n, ap) < 0)
				return;
		}
		else if (type == T_CNAME)
		{
			cname_found = 1;
			if (dn_expand((u_char *) b, eom, ap,
					   nbuf2, sizeof nbuf2) < 0)
				return;

			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: got CNAME %s for %s",
				nbuf2, nbuf);

			if (strcasecmp(nbuf, nbuf2) == 0)
			{
				/* CNAME points back itself, ignore it */
				log(LOG_INFO,
					"CNAME %s points back itself (q=%d)",
					nbuf, qcp->type);
				return;
			}

			qcp->stat->rr_got |= qcp->need; /* XXX */
			if (strcasecmp(qcp->stat->name, nbuf) != 0)
			{
				/* CNAME chain found in the answer */
				qrp = hash_query_lookup(nbuf, NULL);
				if (qrp == NULL) {
					/* chain broken */
					return;
				}
			}
			else
			{
				qrp = qcp->stat;
			}

			qrp2 = hash_query_lookup(nbuf2, &hashp);
			if (qrp2 == NULL) {
				qrp2 = (struct dns_stat *)
					MALLOC(sizeof(struct dns_stat));
				bzero(qrp2, sizeof(struct dns_stat));
				if (qrp2 == NULL)
				{
					log(LOG_NOTICE,
						"out of memory (dns_query)");
					return;
				}
				qrp2->name = newstr(nbuf2);
				qrp2->next = *hashp;
				*hashp = qrp2;
			}
			else
			{
				/* XXX break loop */
			}
			qrp->cname = qrp2;
		}
#if FAKE_AR_CHECK
		else if (type == T_NS)
		{
			if (dn_expand((u_char *) b, eom, ap,
					   nbuf2, sizeof nbuf2) < 0)
				return;
			if (narcheck < ARCHECKSZ)
				strncpy(archeck[narcheck++], nbuf2, MAXDNAME+1);
		}
#endif
		ap += n;
	}
	if (cname_found)
	{
		int cnamechain;

		qrp = qcp->stat->cname;
		cnamechain = 0;
		while (qrp->cname != NULL && ++cnamechain < CNAMELOOPMAX)
			qrp = qrp->cname;

		if (cnamechain < CNAMELOOPMAX)
		{
			switch (qcp->need)
			{
			  case RR_MX:
				make_query(qrp, T_MX);
				break;
			  case RR_A:
				make_query(qrp, T_A);
				break;
			  case RR_AAAA:
				make_query(qrp, T_AAAA);
				break;
			}
		}
	}

	nscount = ntohs(hp->nscount);
	arcount = ntohs(hp->arcount);
	if (arcount == 0)
		goto noadditional;

	while (--ancount >= 0 && ap < eom)
	{
		if ((n = dn_skipname(ap, eom)) < 0)
			goto noadditional;
		ap += n;
		ap += INT16SZ + INT16SZ + INT32SZ;
		if (ap + INT16SZ > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(n, ap);
		if (ap + n > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		ap += n;
	}
	while (--nscount >= 0 && ap < eom)
	{
		if ((n = dn_skipname(ap, eom)) < 0)
			goto noadditional;
		ap += n;
 		ap += INT16SZ + INT16SZ + INT32SZ;
		if (ap + INT16SZ > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(n, ap);
		if (ap + n > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		ap += n;
	}
	while (--arcount >= 0 && ap < eom)
	{
		int addrsize;

		if ((n = dn_expand((u_char *) b, eom, ap,
				nbuf, sizeof nbuf)) < 0)
			return;
		ap += n;
		if (ap + INT16SZ*3 + INT32SZ > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(type, ap);
		GETSHORT(class, ap);
 		ap += INT32SZ; /* TTL */
		GETSHORT(addrsize, ap);
		if (ap + addrsize > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		if (class != C_IN)
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: unexpected class %d in ar", class);
			ap += addrsize;
			continue;
		}
		switch (type)
		{
#ifdef INET6
		  case T_A:
			qflag = RR_A;
			domain = AF_INET;
			if (cnf.inetdom & SMTP_V4)
				break;
			ap += addrsize;
			continue;
		  case T_AAAA:
			qflag = RR_AAAA;
			domain = AF_INET6;
			if (cnf.inetdom & SMTP_V6)
				break;
			ap += addrsize;
			continue;
#else
		  case T_A:
			qflag = RR_A;
			domain = AF_INET;
			break;
#endif
		  default:
			ap += addrsize;
			continue;
		}

		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "DNS: got additional Address RR for %s type=%d",
			nbuf, type);

#if FAKE_AR_CHECK
		found = 0;
		for (n = 0; n < narcheck; n++)
		{
			if (strcasecmp(nbuf, archeck[n]) == 0)
			{
				found = 1;
				break;
			}
		}
		if (!found)
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: ignoring additional Address RR");
			ap += addrsize;
			continue;
		}
#endif

		qrp = hash_query_lookup(nbuf, &hashp);
		if (qrp == NULL) {
			qrp = (struct dns_stat *)
				MALLOC(sizeof(struct dns_stat));
			bzero(qrp, sizeof(struct dns_stat));
			if (qrp == NULL)
			{
				log(LOG_NOTICE, "out of memory (dns_query)");
				return;
			}
			qrp->name = newstr(nbuf);
			qrp->next = *hashp;
			*hashp = qrp;
		}
		qrp->rr_sent |= qflag;
		qrp->rr_got |= qflag;

		bzero(&mxtmp, sizeof(mxtmp));
		mxtmp.name = nbuf;
		if (addinetaddress(&mxtmp, domain, addrsize, ap) < 0)
			return;

		ap += addrsize;
	}
  noadditional:

	if (mx_found)
	{
		qrp = qcp->stat;
		while (qrp->cname != NULL)
			qrp = qrp->cname;
		mxp = qrp->mx;
		while (mxp != NULL)
		{
			qrp = hash_query_lookup(mxp->name, &hashp);
			if (qrp == NULL) {
				qrp = (struct dns_stat *)
					MALLOC(sizeof(struct dns_stat));
				bzero(qrp, sizeof(struct dns_stat));
				if (qrp == NULL)
				{
					log(LOG_NOTICE,
						"out of memory (dns_query)");
					return;
				}
				qrp->name = newstr(mxp->name);
				qrp->next = *hashp;
				*hashp = qrp;
			}

#ifdef INET6
			if (cnf.inetdom & SMTP_V6 || cnf.v4v6fallback)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: getting AAAA RR for %s",
					mxp->name);
				make_query(qrp, T_AAAA);
			}
			if (cnf.inetdom & SMTP_V4 || cnf.v4v6fallback)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "DNS: getting A RR for %s",
					mxp->name);
				make_query(qrp, T_A);
			}
#else
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: getting A RR for %s",
				mxp->name);
			make_query(qrp, T_A);
			if (cnf.v4v6fallback)
			{
				make_query(qrp, T_AAAA);
			}
#endif
			mxp = mxp->next;
		}
	}
	if (qcp->type == T_ANY && !mx_found)
	{
		/* try with exact query */
		make_query(qcp->stat, T_MX);
	}
}

int
getmxlist(dl)
struct domain *dl;
{
	struct domain *dp;
	char **map_arg, *map_opt, *p;
	int offset, needfindaddr, rc;
	long opt;
	struct mx *mxp;

	for (dp = dl; dp != NULL; dp = dp->next)
	{
		/* known by previous transaction */
		if (dp->firstmx != NULL || dp->stat != EX_OK)
			continue;

		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "getting MX for %s", dp->name);

		offset = 0;
		needfindaddr = 0;
		map_arg = dp->map_arg;
		while (*map_arg != NULL)
		{
			map_opt = strchr(*map_arg, '/');
			if (map_opt != NULL)
				*map_opt++ = '\0'; /* split /opt portion */
			opt = 0;
			if ((p = map_opt) != NULL)
			{
				while (*p != '\0')
				{
					switch (*p)
					{
					  case 'a':
						opt |= MAPOPT_TRYALLADDR;
						break;
					}
					p++;
				}
			}

			if (strcasecmp(*map_arg, "MX") == 0)
				rc = getmxrr(dp->name, dp, 1, offset, 1, opt);
			else if (strcasecmp(*map_arg, "MX?") == 0)
				rc = getmxrr(dp->name, dp, 1, offset, 0, opt);
			else if (strcasecmp(*map_arg, "A") == 0)
				rc = getmxrr(dp->name, dp, 0, offset, 0, opt);
			else
				rc = getmxrr(*map_arg, dp, 1, offset, 0, opt);
			if (rc < 0 || dp->stat != EX_OK)
				goto next_domain;
			if (rc > 0)
				needfindaddr = 1;
			offset += 300;

			if (map_opt != NULL)
				*--map_opt = '/'; /* restore */
			map_arg++;
		}

		if (cnf.debug & DEBUG_DNS)
		{
			struct mx *mxp;

			log(LOG_DEBUG, "final MX list for %s", dp->name);
			mxp = dp->firstmx;
			if (mxp == NULL)
			{
				log(LOG_DEBUG, " NO MX!");
			}
			while (mxp != NULL)
			{
				log(LOG_DEBUG, " %s(%d, %d)", mxp->name,
					mxp->pref, mxp->weight);
				mxp = mxp->next;
			}
		}

		if (needfindaddr)
		{
			for (mxp = dp->firstmx; mxp != NULL; mxp = mxp->next)
			{
				if (mxp->host == NULL)
				{
					/* set address information */
					addinetaddress(mxp, AF_UNSPEC, 0, NULL);
				}
			}
		}

		if (cnf.pgateway != NULL && cnf.v4v6fallback)
		{
			struct inetaddr *addrp;
			int found = 0;

			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "protocol gateway checking for %s",
				dp->name);
			for (mxp = dp->firstmx; mxp != NULL; mxp = mxp->next)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "checking MX: %s", mxp->name);
				if (mxp->host == NULL)
				{
					/* no hosts, try next MX */
					mxp = mxp->next;
					continue;
				}
				for(addrp = mxp->host->addr; addrp != NULL;
				    addrp = addrp->next)
				{
/*
					if (cnf.debug & DEBUG_DNS)
					log(LOG_DEBUG, "proto: %d",
						addrp->domain);
*/
#ifdef INET6
					if(((cnf.inetdom & SMTP_V4) == 0
					 && addrp->domain == AF_INET)
					|| ((cnf.inetdom & SMTP_V6) == 0
					 && addrp->domain == AF_INET6))
#else
					if(addrp->domain != AF_INET)
#endif
					{
						struct mx *gwmxp;
						static struct mx* newmx();

						if (cnf.debug & DEBUG_DNS)
						log(LOG_DEBUG, "setting protocol gateway as first MX");
						if ((gwmxp = newmx(cnf.pgateway,
								  -2, 0, dp, 0))
						    == NULL)
						{
							/* out of memory */
							found = 1;
							break;
						}
						/* set address information */
						addinetaddress(gwmxp, AF_UNSPEC,
							       0, NULL);
						found = 1;
						break;
					}
				}
				if (found)
					break;
			}
		}
  next_domain:;
	}
	return 0;
}

static void
linkmx(domp, newmxp)
struct domain *domp;
struct mx *newmxp;
{
	struct mx *mxp, *mxpp;

	mxpp = NULL;
	for (mxp = domp->firstmx; mxp != NULL; mxpp = mxp, mxp = mxp->next)
	{
		if (mxp->pref > newmxp->pref
		 || (mxp->pref == newmxp->pref
		  && mxp->weight > newmxp->weight)
		 || (mxp->pref == newmxp->pref
		  && mxp->weight == newmxp->weight
		  && strcasecmp(mxp->name, newmxp->name) > 0))
		{
			/* insertion point */
			break;
		}
	}

	newmxp->next = mxp;
	if (mxpp == NULL)
	{
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "linking MX %s(%d) at the top",
			newmxp->name, newmxp->pref);

		domp->firstmx = newmxp;
	}
	else
	{
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "linking MX %s(%d) after MX %s(%d)",
			newmxp->name, newmxp->pref,
			mxpp->name, mxpp->pref);

		mxpp->next = newmxp;
	}
}

static struct mx*
newmx(name, pref, weight, dp, opt)
char *name;
int pref, weight;
struct domain *dp;
long opt;
{
	struct mx *mxp;
	struct mx *pmxp = NULL;	/* just for warning suppression */
	int len;

	/* duplication check */
	for (mxp = dp->firstmx; mxp != NULL; mxp = mxp->next)
	{
		if (strcasecmp((char *)name, mxp->name) == 0)
			break;
		pmxp = mxp;
	}

	if (mxp != NULL)
	{
		/* already registered, drop lower one */
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "duplicated MX %s(%d) found",
			mxp->name, mxp->pref);

		if (mxp->pref > pref
		 || (mxp->pref == pref && mxp->weight > weight))
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "remove it and reuse");

			if (mxp == dp->firstmx)
				dp->firstmx = mxp->next;
			else
				pmxp->next = mxp->next;
		}
		else
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "leave it and ignore newer");

			return mxp;	/* do nothing */
		}
	}

	if (mxp == NULL)
	{
		/* new MX */
		mxp = (struct mx*)MALLOC(sizeof(struct mx));
		if (mxp == NULL)
		{
			log(LOG_NOTICE, "out of memory (newmx)");
			return NULL;
		}
		bzero(mxp, sizeof(struct mx));
		len = strlen(name);
		while (len > 0 && name[len-1] == '.')
			name[--len] = '\0';
		mxp->name = newstr(name);
		if (mxp->name == NULL)
			return NULL;
		mxp->pref = pref;
		mxp->weight = weight;
		mxp->domain = dp;
		mxp->opt = opt;
	}

	linkmx(dp, mxp);

	return mxp;
}

int
getmxrr(name, dp, query_mx, offset, wait, opt)
char *name;
struct domain *dp;
int query_mx;
int offset;
int wait;	/* wait till next trial if MX lookup failure */
long opt;	/* option described in map file */
{
	int seenlocal, localpref, seenfallback;
	struct mx *mxp, *pmxp, *cmxp;
	struct dns_stat *qrp, *qrp2;
	int cnamechain;
	char msgbuf[MAXLINE], *mxname;

	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "getmxrr(%s, %s) offset=%d opt=%x",
		name, dp->name, offset, opt);

	if (!query_mx)
	{
		/* A RR lookup */
		mxp = newmx(name, offset - 1, -1, dp, opt);
		if (mxp == NULL)
			return -1;
		return 1;	/* find adderess */
	}

	if (*name == '[')
	{
		/* numeric address or A RR lookup */
		int len;
		SockAddr nsin;

		len = strlen(name);
		if (len <= 2 || name[len-1] != ']')
		{
			dp->stat = EX_NOHOST;
			snprintf(msgbuf, sizeof(msgbuf),
				"Bad host name: %s", name);
			dp->response = newstr(msgbuf);
			if (dp->response == NULL)
				return -1;
			return 0;
		}
		dp->stat = EX_OK;
		name[len-1] = '\0';
		if ((nsin.in.sin_addr.s_addr = inet_addr(name + 1)) != -1)
		{
			/* numeric IPv4 address */
			name[len-1] = ']';
			mxp = newmx(name, offset - 1, -1, dp, opt);
			if (mxp == NULL)
				return -1;
			addinetaddress(mxp, AF_INET, sizeof(nsin.in.sin_addr),
				&nsin.in.sin_addr);
		}
#ifdef INET6
		else if (inet_pton(INET6, name + 1, &nsin.in6.sin6_addr) > 0)
		{
			/* numeric IPv6 address */
			name[len-1] = ']';
			mxp = newmx(name, offset - 1, -1, dp, opt);
			if (mxp == NULL)
				return -1;
			addinetaddress(mxp, AF_INET6,
				sizeof(nsin.in6.sin6_addr),
				&nsin.in6.sin6_addr);
		}
#endif
		else
		{
			/* A RR lookup */
			mxp = newmx(name + 1, offset - 1, -1, dp, opt);
			name[len-1] = ']';
			if (mxp == NULL)
				return -1;
		}
		return 1;	/* find adderess */
	}

  retry:
	seenlocal = 0;
	localpref = 256;
	seenfallback = 0;

	qrp = hash_query_lookup(name, NULL);

	cnamechain = 0;
	while (qrp->cname != NULL && ++cnamechain < CNAMELOOPMAX)
		qrp = qrp->cname;

	if (cnf.debug & DEBUG_DNS && cnamechain > 0)
	log(LOG_DEBUG, "CNAME traced: %s", qrp->name);

	if (cnamechain >= CNAMELOOPMAX)
	{
		dp->stat = EX_TEMPFAIL;
		dp->response = "too deep CNAME chain";
		return 0;
	}

	if (cnf.debug & DEBUG_DNS)
	log(LOG_DEBUG, "%s: timeout=%x noex=%x got=%x",
		qrp->name, qrp->rr_timeout, qrp->rr_noex, qrp->rr_got);

	if (qrp == NULL || qrp->rr_timeout & RR_MX) {
		/* couldn't connect to the name server */
		if (cnf.fallbackmx != NULL)
		{
			dp->stat = EX_OK;
			mxp = newmx(cnf.fallbackmx, offset - 1, 0, dp, opt);
			if (mxp == NULL)
				return -1;
			return 1;	/* adderess found */
		}
		if (!wait)
		{
			dp->stat = EX_OK;
			return 1;
		}
		dp->stat = EX_TEMPFAIL;
		dp->response = "DNS temporary lookup failure";
		return 0;
	}
	if (qrp->rr_noex == RR_ALL)
	{
		/* just for route map configuration error */
		if (dp->firstmx == NULL)
		{
			dp->stat = EX_NOHOST;
			snprintf(msgbuf, sizeof(msgbuf),
				"Host unknown (No MX/A found for %s)", name);
			dp->response = newstr(msgbuf);
			if (dp->response == NULL)
				return -1;
		}
		else
			log(LOG_INFO, "No MX/A on %s for %s; ignored",
				name, dp->name);
		return 0;
	}
	if (qrp->rr_noex & RR_MX)
	{
		/* no MX on this host, may be A exists */
		dp->stat = EX_OK;
		/* give error if A for the host is not found */
		mxp = newmx(qrp->name, offset - 1, -1, dp, opt);
		if (mxp == NULL)
			return -1;
		if (cnf.fallbackmx != NULL)
		{
			mxp = newmx(cnf.fallbackmx, offset - 1, 0, dp, opt);
			if (mxp == NULL)
				return -1;
		}
		return 1;	/* find adderess */
	}

	if ((qrp->rr_got & RR_MX) == 0)
	{
		/* may be CNAME tail, no MX on this host, may be A exists */
		dp->stat = EX_OK;
		/* give error if A for the host is not found */
		mxp = newmx(qrp->name, offset - 1, -1, dp, opt);
		if (mxp == NULL)
			return -1;
		if (cnf.fallbackmx != NULL)
		{
			mxp = newmx(cnf.fallbackmx, offset - 1, 0, dp, opt);
			if (mxp == NULL)
				return -1;
		}
		return 1;	/* find adderess */
	}

	dp->stat = EX_OK;

	for (cmxp = qrp->mx; cmxp != NULL; cmxp = cmxp->next)
	{
		mxname = cmxp->name;
#if 0
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "mx=%s pref=%d", mxname, cmxp->pref);
#endif
		if (isamyalias(mxname))
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "found localhost (%s) in MX list",
				mxname);
			if (!seenlocal || cmxp->pref < localpref)
				localpref = cmxp->pref;
			seenlocal = 1;
			continue;
		}
		if (cnf.fallbackmx != NULL
		 && strcasecmp(mxname, cnf.fallbackmx) == 0)
			seenfallback = 1;

		/* truncate lower prefered MX (imperfect) */
		if (seenlocal && localpref <= cmxp->pref)
			continue;

		/* resolv cnames */
		qrp2 = hash_query_lookup(mxname, NULL);
		if (qrp2 != NULL && qrp2->cname != NULL)
		{
			cnamechain = 0;
			while (qrp2->cname != NULL
			    && ++cnamechain < CNAMELOOPMAX)
				qrp2 = qrp2->cname;

			if (cnamechain >= CNAMELOOPMAX)
				return -1;
			mxname = qrp2->name;
		}

		/* duplication check */
		pmxp = NULL;
		for (mxp = dp->firstmx; mxp != NULL; mxp = mxp->next)
		{
			if (strcasecmp(mxname, mxp->name) == 0)
				break;
			pmxp = mxp;
		}
		if (mxp != NULL)
		{
			/* already registered, drop lower one */

			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "duplicated MX %s(%d) found",
				mxp->name, mxp->pref);

			if (mxp->pref > offset + cmxp->pref)
			{
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "remove it for re-link");
				mxp->pref = offset + cmxp->pref;
				if (mxp == dp->firstmx)
					dp->firstmx = mxp->next;
				else
				{
					pmxp->next = mxp->next;
				}
				/* re-link it */
				linkmx(dp, mxp);
			}
			continue;
		}

		mxp = newmx(mxname, offset + cmxp->pref,
			mxrand(mxname, qrp->name), dp, opt);
		if (mxp == NULL)
			return -1;
	}

	if (seenlocal)
	{
		/* truncate MX records */

		pmxp = NULL;
		for (mxp = dp->firstmx; mxp != NULL; mxp = mxp->next)
		{
			if (mxp->pref >= localpref)
			{
				if (pmxp == NULL)
				{
					/* free! */
					dp->firstmx = NULL;
				}
				else
				{
					/* free! */
					pmxp = NULL;
				}
				break;
			}
			pmxp = mxp;
		}
	}

	if (seenlocal && dp->firstmx == NULL)
	{
		if (cnf.useAfirstMX)
		{
			mxp = newmx(qrp->name, offset - 1, -1, dp, opt);
			if (mxp == NULL)
				return -1;
		}
		else
		{
			dp->stat = EX_CONFIG;
			dp->response = "MX list points back myself";
			return 0;
		}
	}

	if (cnf.fallbackmx != NULL && !seenfallback)
	{
		mxp = newmx(cnf.fallbackmx, offset + 256, 0, dp, opt);
		if (mxp == NULL)
			return -1;
	}

	/* no sorting; MX RRs are linked in order by linkmx() */

	return 1;

  formaterror:
	dp->stat = EX_TEMPFAIL;
	dp->response = "DNS data format error";
	return 0;
}

show_query(t, b, e)
u_char *t, *b, *e;
{
	char nbuf[MAXDNAME+1];
	register int n;
	int type, class, pref;
	HEADER *hp;
	u_char *eom, *ap;
	/* struct hostent *hep; */
	int ancount, qdcount;

	hp = (HEADER *) b;
	ap = (u_char *) b + sizeof(HEADER);
	eom = (u_char *) e;

	qdcount = ntohs(hp->qdcount);
	if (qdcount <= 0)
		return;

	/* skip question part of response -- we know what we asked */
	while (qdcount--)
	{
		n = dn_expand((u_char *) b, eom, ap, nbuf, sizeof nbuf);
		ap += n;
		if (ap + INT16SZ*2 > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(class, ap);
		GETSHORT(type, ap);
		if (cnf.debug & DEBUG_DNS)
		log(LOG_DEBUG, "%s: %s(%d,%d), rcode=%d",
			t, nbuf, class, type, ntohs(hp->rcode));
	}

	if (ntohs(hp->ancount) == 0)
		return;

	for (ancount = ntohs(hp->ancount); --ancount >= 0 && ap < eom;)
	{
		n = dn_expand((u_char *) b, eom, ap, nbuf, sizeof nbuf);
		if (n < 0)
			break;
		ap += n;
		if (ap + INT16SZ*3 + INT32SZ > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		GETSHORT(class, ap);
		GETSHORT(type, ap);
		ap += INT32SZ; /* TTL */
		GETSHORT(n, ap);
		if (ap + n > eom)
		{
			/* log(LOG_NOTICE, "corrupt DNS answer"); */
			return;
		}
		if (class != C_IN)
		{
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "DNS: unexpected class %d in an", class);
			ap += n;
			continue;
		}
		if (type == T_MX) {
			GETSHORT(pref, ap);
			if ((n = dn_expand((u_char *) b, eom, ap,
					   nbuf, sizeof nbuf)) < 0)
				break;
			ap += n;
			if (cnf.debug & DEBUG_DNS)
			log(LOG_DEBUG, "got MX pref=%d,host=(%s)", pref, nbuf);
#if 0 /* for debug */
			hep = gethostbyname(nbuf);
			if (hep == NULL) {
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "get A --  h_errno=%d", h_errno);
			} else {
				if (cnf.debug & DEBUG_DNS)
				log(LOG_DEBUG, "get A -- OK");
			}
#endif
		} else {
			ap += n;
			/* log(LOG_DEBUG, "unexpected answer type: %d", type); */
		}
	}
}
