#ifndef _MATHDEFS_H_
#define _MATHDEFS_H_

#include "../../lib/compat.h"
#include <cmath>
#include <cstdlib>
#include <climits>
#include <string>
#include <sstream>

// useful constants:
#ifndef M_PI
# define M_PI		3.14159265358979323846
#endif
#ifndef M_PI_2
# define M_PI_2		1.57079632679489661923
#endif
#ifndef M_SQRT2 
# define M_SQRT2	1.41421356237309504880
#endif

#define RAD_PER_HOUR	0.26179938779914943654	/* == (pi / 12)  */
#define RAD_PER_DEGREE	0.017453292519943295769	/* == (pi / 180) */
#define HOURS		* RAD_PER_HOUR
#define DEGREES		* RAD_PER_DEGREE

// Not all of the below are actually used, but maybe I will eventually
// support other units in StarPlot

#define C_KM_PER_SEC	299792.458    // exact (by definition of the meter)
#define SEC_PER_YEAR	31556930.0    // tropical year, source C&O page A-1
#define KM_PER_AU	149597870.0   // source: CRC 75th ed., page 14-6
#define AU_PER_PARSEC	206264.806247 // 360*60*60 / 2*pi to be exact
#define KM_PER_MILE	1.609344      // exact (since 1 inch defined as 25.4 mm)

#define KM_PER_LY	(C_KM_PER_SEC * SEC_PER_YEAR)
#define AU_PER_LY	(KM_PER_LY / KM_PER_AU)
#define MILE_PER_LY	(KM_PER_LY / KM_PER_MILE)
#define KM_PER_PARSEC	(AU_PER_PARSEC * KM_PER_AU)
#define MILE_PER_PARSEC (KM_PER_PARSEC / KM_PER_MILE)
#define LY_PER_PARSEC	(KM_PER_PARSEC / KM_PER_LY)

/* source for the following two constants: _Intro to Modern Astrophysics_,
 * Carroll and Ostlie, 1996, page A-1 */
#define SUN_DIAMETER	(1391980.0 / KM_PER_LY)
#define SUN_TEMP	5770.0 /* Kelvins */
#define SUN_BOLO_MAG	+4.70

/* XXX Euler angles for Celestial -> Galactic transform go here */
// for now, the transform is hardcoded in with values in vector3.h
// (not sure how they correspond to Euler angles)

#define ERROR_VAL	-1.0e9

// useful macros

#define ROUND(x) (static_cast<int>(std::floor((x) + 0.5)))
#define FLOOR(x) (static_cast<int>(std::floor(x)))
#define FRAC(x)  ((x) - FLOOR(x))

// useful functions:

namespace starmath {
  // for rounding to a given number of significant figures
  // examples:	sigdigits(375.25, 1) == 400.0
  //		sigdigits(375.25, 4) == 375.3
  double sigdigits(double value, int numdigits);

  // for rounding to a given number of decimal places
  // examples:	roundoff(375.25, 1) == 375.3
  //		roundoff(375.25, -2) == 400.0
  inline double roundoff(double value, int numdigits)
  { 
    return std::pow(10.0, -numdigits) *
      std::floor(std::pow(10.0, numdigits) * value + 0.5);
  }

  // safe versions of atoi, atol, atof
  inline double atof(const std::string &numstring)
    { return std::strtod(numstring.c_str(), 0); }

  inline long int atol(const std::string &numstring)
    { return std::strtol(numstring.c_str(), 0, 10); }

  inline int atoi(const std::string &numstring)
    {
      long int temp = starmath::atol(numstring);
      return (temp < INT_MIN) ? INT_MIN :
	      (temp > INT_MAX) ? INT_MAX : static_cast<int>(temp);
    }

  // conversion of apparent and absolute magnitudes and distance
  inline double get_absmag(double apparentmag, double distance /* in LY */)
    { return apparentmag + 5 - 5 * std::log10(distance / LY_PER_PARSEC); }

  inline double get_appmag(double absolutemag, double distance /* in LY */)
    { return absolutemag - 5 + 5 * std::log10(distance / LY_PER_PARSEC); }

  inline double /* LY */ get_distance(double apparentmag, double absolutemag)
    { return 10*LY_PER_PARSEC*std::pow(10.0, 0.2*(apparentmag-absolutemag)); }
};

#endif
