// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/*
  convert.cc
  Contains the main() function for the starconvert utility and all the
  functions allowing it to convert a star data file to StarPlot format.
*/

#include "convert.h"
#include <iostream>

using std::istream;
using std::ostream;
using std::string;
using std::cin;
using std::cout;
using std::cerr;
using std::endl;

// defined in names.cc:
extern void get_names(const string &, namedata, StringList *, StringList *);

Star parse_star_record(const string &, parsedata);
void get_coordinates(const string &, coordinates, StringList *);
void get_mag_distance(const string &, const string &, characteristics,
		      double *, double *);
void check_for_double_stars(const string &, systems, comments, Star *, Star *);


void parse_star_file(istream & infile, ostream & outfile, parsedata format,
		     bool using_stdout)
{
  int lineno = 0;
  bool sun_found = false, moredata = true;
  char record[1000];
  Star tempstar, currentstar = BADSTAR, previousstar;

  // boring legal disclaimer:
  outfile << starstrings::ssprintf(_(
"This StarPlot data file was automatically generated by starconvert %s\n\
If this file was derived from a copyrighted source, you may not redistribute\n\
it without the permission of the original author.\n\n"), STARPLOT_VERSION)
	  << _(
"If you obtained this data file by installing one of the data packages\n\
available on the StarPlot web page, you should see the file copyright notice\n\
in that package for more information.") << endl << endl;

  do {
    moredata = infile.getline(record, 999, '\n');
    record[999] = 0;

    // $ ; and , have special meanings to StarPlot, so purge them:
    for (unsigned int i = 0; i < strlen(record); i++)
      if (record[i] == '$' || record[i] == ';' || record[i] == ',')
        record[i] = ' ';
  
    tempstar = parse_star_record(record, format);
    if (tempstar.GetStarNames()[0] != BADSTARNAME || !moredata) {
      previousstar = currentstar;
      currentstar = tempstar;
    }
    else continue;

    if (previousstar.GetStarNames()[0] != BADSTARNAME) {
      Rules r;
      r.CelestialCoords = true;
      check_for_double_stars(record, format.Systems, format.Comments,
			     &previousstar, &currentstar);
      StringList starinfo = previousstar.GetInfo(r, false, ',');
      
      // write star information to output
      outfile << "$ " << starinfo[1];
      if (starinfo.size() >= 11) {
	for (unsigned int i = 10; i < starinfo.size(); i++)
	  outfile << ", " << starinfo[i];
      }
      outfile << ";" << endl << "  ";
      
      outfile << starinfo[2] << "; " << starinfo[3] << "; " << starinfo[4] 
	      << "; 0;" << endl << "  " << starinfo[5] << "; ";
      outfile << (starstrings::uppercase(starinfo[1]) == "SUN" ?
		  (sun_found = true, "4.85") : starinfo[6]);
      outfile << "; " << starinfo[9] << "; " << starinfo[7] << ";" << endl;
      
      if (starinfo[8].size())
	outfile << "  " << starinfo[8] << endl;
      outfile << endl;
    }
    record[0] = 0;

    lineno++;
    if (!using_stdout && !(lineno % 1000))
      cout << starstrings::ssprintf(_("%d records converted."), lineno) << endl;

  } while (moredata) ;

  // if the Sun is not in the data file, append it at the end!
  if (!sun_found)
    outfile << "$ Sun, Sol;" << endl << "  ; ; 0; 0;" << endl 
    	    << "  G2 V; 4.85; 0; ;" << endl << endl;
  return;
}


// parse_star_record(): converts the current record to a Star

Star parse_star_record(const string &record, parsedata format)
{
  if (starstrings::isempty(record)) return BADSTAR;

  double starmag, stardist;
  string spectrum;
  StringList starnames, starcoord, starcomments, starmembership;
  Star result;

  // find the first non-empty spectral class datum out of those specified
  //  in the specification file
  unsigned int i = 0;
  do {
    if (format.Charact.s[i].start < record.size()) {
      spectrum = record.substr(format.Charact.s[i].start,
			       format.Charact.s[i].len);
      starstrings::stripspace(spectrum);
    }
    i++;
  } while (i < format.Charact.s.size() &&
	   starstrings::isempty(spectrum)) ;

  // convert comments (if any) to a StringList
  if (format.Comments.s.len > 0 && format.Comments.s.start < record.size()) {
    string commentstring = record.substr(format.Comments.s.start,
		    			 format.Comments.s.len);
    starcomments = StringList(commentstring, ' ');
    starcomments.eraseempty();
  }

  get_coordinates(record, format.Coord, &starcoord);
  get_mag_distance(record, spectrum, format.Charact, &stardist, &starmag);
  get_names(record, format.Names, &starnames, &starcomments);

  if (starnames.size() && (starstrings::uppercase(starnames[0]) == "SUN"))
    stardist = 0.0;

  if (stardist >= 0.0) {
    std::ostringstream starstring;

    citerate_until (StringList, starnames, strptr, -1)
      starstring << *strptr << ", ";
    if (starnames.size())
      starstring << starnames[starnames.size() - 1];
    else
      starstring << _("<no name>");
    starstring << "; ";

    for (unsigned int i = 0; i < 6; i++)
      starstring << starcoord[i] << (((i + 1) % 3) ? ", " : "; ");
    starstring << stardist << "; 0; " << spectrum << "; "
	       << starmag << "; " << "0; ; ";

    citerate (StringList, starcomments, strptr)
      starstring << *strptr << " ";

    result = Star(starstring.str(),
		  false /* no fast conversion */,
		  false /* no name translation */);
  }

  else {
    cerr << starstrings::ssprintf(_(
	"*** Cannot obtain distance to star %s; leaving it out."),
	starnames[0].c_str()) << endl;
    result = BADSTAR;
  }

  return result;
}


void get_coordinates(const string & record, coordinates c, 
		     StringList * coordstrings)
{
  coordstrings->clear();
  
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++) {
    if (c.s[i].len > 0 && c.s[i].start < record.size())
      coordstrings->push_back(record.substr(c.s[i].start, c.s[i].len));
    else
      coordstrings->push_back(string(""));
  }
  coordstrings->stripspace();

  // fourth element is the sign, so concat it with fifth element and erase
  (*coordstrings)[3] += (*coordstrings)[4];
  coordstrings->erase(coordstrings->begin() + 4);
  
  if (!c.isCelestial) {
    // in this case we convert from Galactic coordinates so that contents
    // of output file are in celestial coordinates
    double theta, phi;
    SolidAngle omega;

    phi = starstrings::strs_to_ra((*coordstrings)[0], (*coordstrings)[1],
		  		  (*coordstrings)[2], GALACTIC);
    theta = starstrings::strs_to_dec((*coordstrings)[3], (*coordstrings)[4],
				     (*coordstrings)[5]);
    omega = SolidAngle(phi, theta).toCelestial();
    coordstrings->clear();
    coordstrings->push_back(starstrings::ra_to_strs(omega.getPhi()));
    coordstrings->push_back(starstrings::dec_to_strs(omega.getTheta()));
  }
  return;
}


// get_mag_distance(): This function obtains the distance and magnitude of
//  the star.  It goes through the distance/magnitude formatting pairs 
//  specified in the config file, and uses the first pair which gives an 
//  acceptable result.

void get_mag_distance(const string &record, const string &spectrum,
		      characteristics c, double *dist, double *mag)
{
  double tempmag, tempdist;

  for (unsigned int i = 0; i < c.distarray.size(); i++) {
    if (c.magarray[i].s.len <= 0 || c.magarray[i].s.start >= record.size())
      continue;
    char *endptr;
    string magstring = record.substr(c.magarray[i].s.start,
				     c.magarray[i].s.len);
    starstrings::stripspace(magstring);
    tempmag = std::strtod(magstring.c_str(), &endptr);
    if (starstrings::isempty(magstring) || (*endptr != 0 && *endptr != '.'))
      continue;

    if (c.distarray[i].type == SPECCLASS) {
      if (starstrings::isempty(spectrum) || c.magarray[i].type == ABSOLUTE)
	continue;
      SpecClass sclass = SpecClass(spectrum);
      sclass.initialize();
      double absmag = sclass.absmag();
      if (absmag < -25 || absmag > 25) continue;
      
      // Technically this should also account for the effect of 
      //  interstellar dust on the visual magnitude...
      tempdist = starmath::get_distance(tempmag, absmag);

      // should NOT be using spectral class distance estimate for nearby stars:
      if (tempdist < 20 /* LY */) continue;

      *dist = starmath::sigdigits(tempdist, 3);
      *mag = absmag;
      return;
    }

    else if (c.distarray[i].s.len > 0 &&
	     c.distarray[i].s.start < record.size()) {
      string diststring = record.substr(c.distarray[i].s.start, 
		      			c.distarray[i].s.len);
      starstrings::stripspace(diststring);
      tempdist = std::strtod(diststring.c_str(), &endptr);
      if (starstrings::isempty(diststring) || (*endptr != 0 && *endptr != '.'))
	continue;

      double parallaxerr = 0.0;
      if ((c.distarray[i].type == ARCSEC || c.distarray[i].type == MILLIARCSEC)
	  && c.distarray[i].err.len > 0
	  && c.distarray[i].err.start < record.size()) {
	string errstring = record.substr(c.distarray[i].err.start,
					 c.distarray[i].err.len);
	starstrings::stripspace(errstring);
	parallaxerr = std::strtod(errstring.c_str(), &endptr);
	if (starstrings::isempty(errstring) || parallaxerr < 0 
	    || (*endptr != 0 && *endptr != '.'))
	  continue;
      }
	
      switch (c.distarray[i].type) {
      case MILLIARCSEC: tempdist /= 1000.0; parallaxerr /= 1000.0;
      case ARCSEC:
	{
	  // if parallax is < min. acceptable parallax, go to next rule
	  if (tempdist < c.distarray[i].minparallax) break;
	  // if relative error is > max. acceptable error, go to next rule
	  else if (c.distarray[i].err.len > 0 
		   && parallaxerr > c.distarray[i].maxerror * tempdist) break;
	  else tempdist = 1.0 / tempdist;
	}
      case PC:          tempdist *= LY_PER_PC;
      case LY:
	{
	  *dist = starmath::sigdigits(tempdist, 3);
	  if (c.magarray[i].type == ABSOLUTE)
	    *mag = tempmag;
	  else {
	    tempmag = starmath::get_absmag(tempmag, tempdist);
	    *mag = starmath::roundoff(tempmag, 1);
	  }
	  return;
	}
      case SPECCLASS: break; // do nothing
      } // end switch statement
    } 
  } // end for loop
  
  // if we haven't returned yet, something is wrong; flag the values.
  *mag = *dist = -9999;
  return;
}


// Check for double systems.  This could certainly be improved but it
// may not be worth the effort.

void check_for_double_stars(const string &record, systems s, comments c,
			    Star *previous, Star *current)
{
  // Check for component of multiple star system and save in the "sPlace"
  // variable of the current Star.
  if (s.comp.len > 0 && s.comp.start < record.size()) {
    string component = record.substr(s.comp.start, s.comp.len);
    starstrings::stripspace(component);
    current->SetPlace(component[0]);
  }

  // Check for separation of multiple star system components.
  if (current->GetPlace() != 'A') {
    double separation = 0.0;
    string sepstr;
    if (s.isSeparationCommented && c.s.len > 0 && c.s.start < record.size()) {
      string comments = record.substr(c.s.start, c.s.len);
      citerate (StringList, s.sep_prefixes, prefix_ptr) {
	size_t posn = comments.find(*prefix_ptr);
	if (posn < comments.size()) {
	  sepstr = comments.substr(posn+(*prefix_ptr).size(), comments.size());
	  starstrings::stripspace(sepstr);
	  break;
	}
      }
    }
    else if (s.sep.len > 0 && s.sep.start < record.size())
      sepstr = record.substr(s.sep.start, s.sep.len);
    starstrings::stripspace(sepstr);

    separation = starmath::atof(sepstr);
    if (sepstr.size() && sepstr[sepstr.size() - 1] == '\'')
      separation *= 60;

    current->SetStarPrimaryDistance(separation * RAD_PER_DEGREE / 3600.0
				    * current->GetStarDistance());
  }
  
  // Assume that two stars that are adjacent in raw data file and separated
  // by less than 1/5 LY are paired or part of a multiple star system.
  double distdiff = (previous->GetStarXYZ()-current->GetStarXYZ()).magnitude();
  if (distdiff < 0.2 /* light-years */) {
    Star *brighter, *dimmer;

    if (previous->GetPlace() && current->GetPlace())
      brighter = (previous->GetPlace() > current->GetPlace()) ?
		 current : previous;
    else
      brighter = (previous->GetStarMagnitude() > current->GetStarMagnitude()) ?
		 current : previous;
    dimmer = (brighter == previous) ? current : previous;
    
    if (previous->GetStarMembership().size() == 0) {
      previous->SetStarMembership(brighter->GetStarNames()[0]);
      current->SetStarMembership(brighter->GetStarNames()[0]);
    }
    else { // at least 3 stars; we are stuck with whatever was decided upon
           // as the system name when looking at the first pair
      current->SetStarMembership(previous->GetStarMembership()[0]);
    }
    if (dimmer->GetStarPrimaryDistance() == 0.0)
      dimmer->SetStarPrimaryDistance(distdiff);
    if (dimmer->GetStarPrimaryDistance() == 0.0)
      // flag primary distance as small but unknown:
      dimmer->SetStarPrimaryDistance(-1.0);
  }
}


void printusage()
{
  cout << PROGNAME << " " << _("usage:") << endl
       << PROGNAME << " " << _("specfile infile [outfile]") << endl << _(
"Uses specifications in `specfile' to convert `infile' to StarPlot\n\
data format and writes the results to `outfile' if specified, or\n\
standard output if not.  The first two arguments are required.\n\
Exactly ONE may be replaced with a dash for reading from stdin.\n") << endl;
  cout << PROGNAME << ": " << _("version") << " " << STARPLOT_VERSION
       /* TRANSLATORS: this is a proper name */
       << _(" (C) 2000-2004 Kevin B. McCarty") << endl
       << starstrings::ssprintf(_(
"%s comes with ABSOLUTELY NO WARRANTY; for details, see\n\
the file %s."), PROGNAME, DOCDIR "/COPYING") << endl;
  return;
}


// main(): This doesn't do much besides error checking, and calling the two
//  top-level parsing functions (one for the specification file, and one for
//  the data file).

int main(int argc, char **argv)
{
  if (argc != 3 && argc != 4) {
    printusage();
    return 0;
  }

  parsedata specdata;
  std::ifstream infile, specfile;
  std::ofstream outfile;

  // Set up i18n
  setlocale (LC_ALL, "");
  setlocale (LC_NUMERIC, "C"); // avoid problems with comma as decimal point
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
  
  if (strcmp(argv[1], "-") == 0) {
    if (strcmp(argv[2], "-") == 0) {
      printusage();
      return 0;
    }
    else {
      if (! cin.good()) {
	cerr << _("Can't read specifications from stdin - not a tty?") << endl;
	return EXIT_FAILURE;
      }
      parse_config_file(cin, &specdata);
    }
  }
  else {
    specfile.open(argv[1]);
    if (! specfile.good()) {
      cerr << starstrings::ssprintf(_(
	"Can't open specification file `%s' for input; exiting."), argv[1])
	   << endl;
      return EXIT_FAILURE;
    }
    parse_config_file(specfile, &specdata);
    specfile.close();
  }

  if (argc == 4) {
    outfile.open(argv[3]);
    if (! outfile.good()) {
      cerr << starstrings::ssprintf(_(
	"Can't open output file `%s' for output; exiting."), argv[3])
	   << endl;
      return EXIT_FAILURE;
    }
  }
  
  if (strcmp(argv[2], "-") == 0) {
    if (! cin.good()) {
      cerr << _("Can't read star data from stdin - not a tty?") << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(cin, outfile, specdata, false);
    else           parse_star_file(cin, cout, specdata, true);
  }

  else {
    infile.open(argv[2]);
    if (! infile.good()) {
      cerr << starstrings::ssprintf(_(
	"Can't open star data file `%s' for input; exiting."), argv[2])
	   << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(infile, outfile, specdata, false);
    else           parse_star_file(infile, cout, specdata, true);
    infile.close();
  }

  if (argc == 4) {
    outfile.close();
  }

  return 0;
}

