
/* $Id */

#include "settings.h"
#include "filesettings.h"
#include "stdebug.h"

#include <stdlib.h>
#include <string.h>

static char *boolnames[] = {
    "ring-mode",
    "vibrate-mode",
    "message-reminder-tone",
    "dtmf-is-long",
    "dtmf-is-on",
    "service-tone",
    "silent-mode",
    "roam-ring",
    "call-reminder-tone",
    "keypad-tone",
    "instant-redial",
    "multi-key-answer",
    "call-guard",
    "answer-on-open",
    "auto-answer",
    "quick-menu",
    "auto-hyphen",
    "24-hour-clock",
    "service-light",
    "confirm-minibrowse",
    "analog-only",
    "individual-timer",
    "one-minute-timer",
    "use-repeating-timer",
    "use-single-timer",
    "auto-lock",
    "priority-call",
};

static char *ucharnames[] = {
    "ringer-style",
    "service-level",
    "backlight-mode",
};

static char *stringnames[] = {
    "greeting",
};

static char *uintnames[] = {
    "repeating-timer-duration",
    "single-timer-duration",
    "lock-code",
};

static char *langnames[] = {
    "language",
};

static char *suintnames[] = {
  "security-code",
};

static char **settingnames[STARTALK_SET_TYPE_LAST+1] = {
  boolnames,
  ucharnames,
  stringnames,
  uintnames,
  langnames,
  suintnames,
};

static int lastsetting[STARTALK_SET_TYPE_LAST+1] = {
  STARTALK_SET_BOOL_LAST,
  STARTALK_SET_UCHAR_LAST,
  STARTALK_SET_STRING_LAST,
  STARTALK_SET_UINT_LAST,
  STARTALK_SET_LANG_LAST,
  STARTALK_SET_SUINT_LAST,
};

struct startalk_settings *
startalk_new_settings(void)
{
  struct startalk_settings *n;
  int i;
  
  n = malloc(sizeof(struct startalk_settings));
  if (!n)
    return (void *) reterr(0,"malloc error:");

  for (i=0;i<=STARTALK_SET_BOOL_LAST;i++)
    n->ss_bool[i] = -1;
  for (i=0;i<=STARTALK_SET_UCHAR_LAST;i++)
    n->ss_uchar[i] = -1;
  for (i=0;i<=STARTALK_SET_STRING_LAST;i++)
    n->ss_string[i] = NULL;
  for (i=0;i<=STARTALK_SET_UINT_LAST;i++)
    n->ss_uint[i] = -1;
  for (i=0;i<=STARTALK_SET_SUINT_LAST;i++)
    n->ss_suint[i] = -1;
  for (i=0;i<=STARTALK_SET_LANG_LAST;i++)
    n->ss_lang[i] = -1;

  n->gotbuf = 0;
  n->databuf[0] = n->buf1;
  n->databuf[1] = n->buf2;
  n->databuf[2] = n->buf3;
  
  return n;
}

void
startalk_delete_settings(struct startalk_settings *n)
{
  if (n)
    free(n);
}

signed char
startalk_get_setting_bool(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num > STARTALK_SET_BOOL_LAST)
    die("Unknown boolean setting requested: number %d\n", setting_num);

  return (signed char) n->ss_bool[setting_num];
}

signed int
startalk_get_setting_uchar(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num > STARTALK_SET_UCHAR_LAST)
    die("Unknown uchar setting requested: number %d\n", setting_num);

  return (signed int) n->ss_uchar[setting_num];
}

char *
startalk_get_setting_string(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num != 0)
    die("Unknown string setting requested: number %d\n", setting_num);

  return n->ss_string[0];
}

signed long
startalk_get_setting_uint(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num > STARTALK_SET_UINT_LAST)
    die("Unknown uint setting requested: number %d\n", setting_num);

  return (signed long) n->ss_uint[setting_num];
}

signed long
startalk_get_setting_suint(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num > STARTALK_SET_SUINT_LAST)
    die("Unknown suint setting requested: number %d\n", setting_num);

  return (signed long) n->ss_suint[setting_num];
}

signed int
startalk_get_setting_lang(struct startalk_settings *n, unsigned int setting_num)
{
  if (setting_num > STARTALK_SET_LANG_LAST)
    die("Unknown language setting requested: number %d\n", setting_num);

  return (signed int) n->ss_lang[setting_num];
}

void
startalk_set_setting_bool(struct startalk_settings *n, unsigned int setting_num, signed char val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num > STARTALK_SET_BOOL_LAST)
    die("Unknown boolean setting %d\n",setting_num);

  n->ss_bool[setting_num] = val;
}

void
startalk_set_setting_uchar(struct startalk_settings *n, unsigned int setting_num, unsigned int val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num > STARTALK_SET_UCHAR_LAST)
    die("Unknown UCHAR setting %d\n",setting_num);

  n->ss_uchar[setting_num] = val;
}

void
startalk_set_setting_string(struct startalk_settings *n, unsigned int setting_num, char *val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num != 0 )
    die("Unknown UCHAR setting %d\n",setting_num);

  memset(n->ss_greeting_buf, 0, 13);
  strncpy(n->ss_greeting_buf, val, 12);
  n->ss_string[0] = n->ss_greeting_buf;
}


void
startalk_set_setting_uint(struct startalk_settings *n, unsigned int setting_num, unsigned long val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num > STARTALK_SET_UINT_LAST)
    die("Unknown UINT setting %d\n",setting_num);

  n->ss_uint[setting_num] = val;
}

void
startalk_set_setting_suint(struct startalk_settings *n, unsigned int setting_num, unsigned long val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num > STARTALK_SET_SUINT_LAST)
    die("Unknown SUINT setting %d\n",setting_num);

  n->ss_suint[setting_num] = val;
}


void
startalk_set_setting_lang(struct startalk_settings *n, unsigned int setting_num, unsigned int val)
{
  if (!n)
    die("Requested setting from NULL settings structure!\n");

  if (setting_num > STARTALK_SET_LANG_LAST)
    die("Unknown LANG setting %d\n",setting_num);

  n->ss_lang[setting_num] = val;
}


void
startalk_print_setting(struct startalk_settings *n,unsigned int setting_type, unsigned int setting_num)
{
  startalk_fprint_setting(stdout, n, setting_type, setting_num);
}


void
startalk_print_settings(struct startalk_settings *n)
{
  startalk_fprint_settings(stdout, n);
}

int
startalk_setting_readonly(unsigned int setting_type, unsigned int setting_num)
{
  if (setting_type == STARTALK_SET_TYPE_BOOL)
  {
    return ( (setting_num == STARTALK_SET_ANALOGONLY) ||
             (setting_num == STARTALK_SET_DTMF_IS_LONG) ||
             (setting_num == STARTALK_SET_AUTOLOCK)
      );
  }
  else if (setting_type == STARTALK_SET_TYPE_UINT)
  {
    return ( (setting_num == STARTALK_SET_LOCKCODE)
      );
  }
  else if (setting_type == STARTALK_SET_TYPE_SUINT)
  {
    return ( (setting_num == STARTALK_SET_SECURECODE)
      );
  }
  else
    return 0;
}

int
startalk_validate_setting(struct startalk_settings *n)
{
  char *r;
  int failures = 0;
  int i;
  
  for (i=0;i<=STARTALK_SET_BOOL_LAST;i++)
  {
    r=startalk_validate_setting_bool(n, i, n->ss_bool[i]);
    if (!r)
    {
      warn("Boolean setting %s failed validation: %s",settingnames[STARTALK_SET_TYPE_BOOL][i],r);
      failures++;
    }
  }
  return !!failures;
}

char *
startalk_validate_setting_bool(struct startalk_settings *n, unsigned int setting_num, signed char value)
{
  switch(setting_num)
  {
    /* If DTMF_ON is off, DTMF_IS_LONG must be, too. */
    case STARTALK_SET_DTMF_IS_LONG:
      if ( (n->ss_bool[STARTALK_SET_DTMF_ON] == 0) && value )
        return "If dtmf-is-long is on, dtmf-is-on must be on";
      break;
    case STARTALK_SET_DTMF_ON:
      if ( (n->ss_bool[STARTALK_SET_DTMF_IS_LONG] == 1) && !value )
        return "If dtmf-is-long is on, dtmf-is-on must be on";
      break;

    /* Either RING or VIBRATE or BOTH must be set. */
    case STARTALK_SET_RING:
      if ( (n->ss_bool[STARTALK_SET_VIBRATE] == 0) && !value)
        return "Either ring-mode or vibrate-mode (or both) must be on";
      break;
      
    case STARTALK_SET_VIBRATE:
      if ( (n->ss_bool[STARTALK_SET_RING] == 0) && !value)
        return "Either ring-mode or vibrate-mode (or both) must be on";
      break;
  }

  return NULL;
}

char *
startalk_validate_setting_uchar(struct startalk_settings *n, unsigned int setting_num, signed int value)
{
  if ( (value < 0) || (value > 255) )
    return "uchar value must be between 0 and 255";
  
  switch(setting_num)
  {
    case STARTALK_SET_RINGER_STYLE:
      if ( (value == 0) || (value > 9) )
        return "ringer-style must be between 1 and 10";
      break;

    case STARTALK_SET_SVCLEVEL:
      if ( (value != 1) && (value != 4) )
        return "service-level must be either 1 or 4";
      break;
  }

  return NULL;
}

char *
startalk_validate_setting_string(struct startalk_settings *n, unsigned int setting_num, char *val)
{
  switch (setting_num)
  {
    case STARTALK_SET_GREETING:
      if (strlen(val) > 12)
        return "Greeting cannot be longer than 12 characters";
      break;
  }
  return NULL;
}

char *
startalk_validate_setting_uint(struct startalk_settings *n, unsigned int setting_num, unsigned long val)
{
  if ( (val < 0L) || (val > (256L*256L-1L)) )
    return "uint must be between 0 and 65535";

  switch(setting_num)
  {
    case STARTALK_SET_REPTIMER_DUR:
    case STARTALK_SET_SINGTIMER_DUR:
      if ( (val < 11) || (val > 999) )
        return "timers must be between 11 and 999 seconds";
      break;
  }
  return NULL;
}

char *
startalk_validate_setting_lang(struct startalk_settings *n, unsigned int setting_num, unsigned int val)
{
  switch(setting_num)
  {
    case STARTALK_SET_LANG:
      if ( (setting_num != 0) && (setting_num != 7) && (setting_num != 6) )
      {
        return "Only languages English (0), Spanish (7), and French (6) are currently recognized";
      }
      break;
  }
  return NULL;
}

char *
startalk_validate_setting_suint(struct startalk_settings *n, unsigned int setting_num, unsigned long val)
{
  if ( (val < 0L) || (val > (256L*256L*256L-1L)) )
    return "suint must be between 0 and 16777215";

  return NULL;
}

char *
startalk_get_setting_name(unsigned int setting_type, unsigned int setting_num)
{

  if (setting_type <= STARTALK_SET_TYPE_LAST)
    if (setting_num <= lastsetting[setting_type])
      return settingnames[setting_type][setting_num];

  warn("Request for unknown setting type %d, number %d\n",setting_type, setting_num);
  return NULL;
}

int
startalk_setting_find(char *name, int *setting_type, int *setting_num)
{
  int curtype, curnum;

  for(curtype=0;curtype<=STARTALK_SET_TYPE_LAST;curtype++)
  {
    for(curnum=lastsetting[curtype];curnum>=0;curnum--)
      if (strcasecmp(name, settingnames[curtype][curnum]) == 0)
      {
        *setting_type = curtype;
        *setting_num = curnum;
        return 1;
      }
  }

  return 0;
}
