/*
 * m i s c . c					-- Misc. functions
 * 
 * Copyright  2000-2003 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date:  9-Jan-2000 12:50 (eg)
 * Last file update: 10-Feb-2003 10:40 (eg)
 */

#include "stklos.h"
#include "gnu-getopt.h"

#ifdef STK_DEBUG
int STk_interactive_debug = 0;
#endif


static void error_bad_string(SCM str)
{
  STk_error("bad string ~S", str);
}


char *STk_strdup(const char *s)
{
  /* Like standard strdup but with our allocator */
  char *res; 
  
  res = STk_must_malloc_atomic(strlen(s) + 1);
  strcpy(res, s);
  return res;
}


void STk_add_primitive(struct primitive_obj *o)
{
  SCM symbol;

  symbol = STk_intern(o->name);
  STk_define_variable(symbol, (SCM) o, STk_current_module);
}


/*===========================================================================*\
 * 
 * Primitives that feet anywhere else 
 * 
\*===========================================================================*/
/*
<doc ext version
 * (version)
 *
 * Returns a string identifying the current version of the system. A version is 
 * constituted of three numbers separated by a point: the version, the release
 * and sub-release numbers.
doc>
 */
DEFINE_PRIMITIVE("version", version, subr0, (void))
{
  return STk_Cstring2string(VERSION);
}


/*
<doc ext void
 * (void)
 * (void arg1 ...)
 *
 * Returns the special @emph{void} object. If arguments are passed to |void|,
 * they are evalued and simply ignored.
doc>
 */
DEFINE_PRIMITIVE("void", scheme_void, vsubr, (int argc, SCM *argv))
{
  return STk_void;
}


/*
<doc address-of
 * (address-of obj)
 *
 * Returns the address of the object |obj| as an integer.
doc>
*/
DEFINE_PRIMITIVE("address-of", address_of, subr1, (SCM object))
{
  char buffer[50];     /* should be sufficient for a while */

  sprintf(buffer, "%lx", (unsigned long) object); /* not very efficient ... */
  return STk_Cstr2number(buffer, 16L);
}


/*
<doc gc
 * (gc)
 *
 * Returns the address of the object |obj| as an integer.
doc>
*/
DEFINE_PRIMITIVE("gc", scheme_gc, subr0, (void))
{
  STk_gc();
  return STk_void;
}


DEFINE_PRIMITIVE("%enter-procedure", enter_proc, subr1, (SCM symbol))
{
  if (!SYMBOLP(symbol)) {
    ENTER_PRIMITIVE(enter_proc);
    STk_error("bad symbol ~S", symbol);
  }
  ENTER_PROCEDURE(SYMBOL_PNAME(symbol));
  return STk_void;
}

/*===========================================================================*\
 * 
 * 			Argument parsing
 * 
\*===========================================================================*/
static int Argc;
static char * optstring;
static char **Argv;
static struct option *long_options;


DEFINE_PRIMITIVE("%initialize-getopt", init_getopt, subr3, (SCM argv, SCM s, SCM v))
{
  int i, len;

  ENTER_PRIMITIVE(init_getopt);

  STk_start_getopt_from_scheme();
  optind = 1;    /* Initialize optind, since it has already be used 
		  * by ouserlves  before initializing the VM. 
		  */

  /* 
   * Argv processing 
   */
  len = STk_int_length(argv);
  if (len < 0) STk_error("bad argument list ~S", argv);
  Argv = STk_must_malloc_atomic((len+1) * sizeof(char *));
  for (i = 0; i < len; i++) {
    if (!STRINGP(CAR(argv))) error_bad_string(CAR(argv));
    Argv[i] = STRING_CHARS(CAR(argv));
    argv    = CDR(argv);
  }
  Argv[len] = NULL;
  Argc      = len;
  
  /*
   * Optstring 
   */
  if (!STRINGP(s)) error_bad_string(s);
  optstring = STRING_CHARS(s);

  /* 
   * Option vector processing
   */
  if (!VECTORP(v)) STk_error("bad vector ~S", v);
  len = VECTOR_SIZE(v);
  /* If there is an else clause, last item of the vector is #f */
  if (VECTOR_DATA(v)[len-1] == STk_false) len -= 1;

  long_options = STk_must_malloc_atomic((len+1) * sizeof(struct option));
  
  /* Copy the values in v in the long_options array */
  for (i=0; i < len; i ++) {
    if (!STRINGP(CAR(VECTOR_DATA(v)[i]))) error_bad_string(CAR(VECTOR_DATA(v)[i]));

    long_options[i].name    = STRING_CHARS(CAR(VECTOR_DATA(v)[i]));
    long_options[i].has_arg = (CDR(VECTOR_DATA(v)[i]) == STk_false) ? no_argument 
      							        : required_argument;
    long_options[i].flag    = 0;
    long_options[i].val     = 0;
  }
  
  long_options[len].name = NULL; long_options[len].has_arg = 0;
  long_options[len].flag = NULL; long_options[len].val     = 0;

  return STk_void;
}

DEFINE_PRIMITIVE("%getopt", getopt, subr0, (void))
{
  int  n, longindex; 

  ENTER_PRIMITIVE(getopt);

  n = getopt_long(Argc, Argv, optstring, long_options, &longindex);

  switch (n) {
    case -1:
      {
	/* We are at the end. Collect all the remaining parameters in a list */
	SCM l = STk_nil;
	while (optind < Argc)
	  l = STk_cons(STk_Cstring2string(Argv[optind++]), l);
	
	return STk_cons(MAKE_INT((long) -1), STk_dreverse(l));
      }
    case '?': /* Error or argument missing */
    case ':': return STk_false;
    case 0  : /* Long option */
      {
	SCM str = (optarg)? STk_Cstring2string(optarg): STk_void;
	return STk_cons(MAKE_INT(longindex),str);
      }
    default: /* short option */
      {
	SCM str = (optarg)? STk_Cstring2string(optarg): STk_void;
	return STk_cons(MAKE_CHARACTER(n),  str);
      }
  }
}


/*===========================================================================*\
 * 
 * 			Debugging Code
 * 
\*===========================================================================*/
#ifdef STK_DEBUG
DEFINE_PRIMITIVE("%debug", set_debug, subr0, (void))
{
  STk_interactive_debug = !STk_interactive_debug;
  STk_debug("Debug mode %d", STk_interactive_debug);
  return STk_void;
}

DEFINE_PRIMITIVE("%test", test, subr1, (SCM s))
{
  /* A special place for doing tests */
  STk_debug("On a ~S |%s|", s, STRING_CHARS(s));
  return STk_void;
}
#endif 


/*===========================================================================*\
 * 
 * 				Initialization
 * 
\*===========================================================================*/
int STk_init_misc(void)
{
  ADD_PRIMITIVE(version);
  ADD_PRIMITIVE(scheme_void);
  ADD_PRIMITIVE(address_of);
  ADD_PRIMITIVE(scheme_gc);
  ADD_PRIMITIVE(enter_proc);

  ADD_PRIMITIVE(init_getopt);
  ADD_PRIMITIVE(getopt);

#ifdef STK_DEBUG
  ADD_PRIMITIVE(set_debug);
  ADD_PRIMITIVE(test);
#endif
  return TRUE;
}

