/* escpg.c
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1997-1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "dvi-2_6.h"
#include "defs.h"
#include "escpg.h"


static int           pcol;
static int           offset_x;
static int           offset_y;
static int           lbp_compress_mode;
static unsigned char lbp_buffer[1024*16];
static int           lbp_buffer_ptr = 0;
static int           param_dvi2escpg_init_mode = ESCPG_INIT_MODE_DEFAULT;
extern int           param_verbose;


#define ESC      (char)0x1b
#define GS       (char)0x1d
#define COMPRESS_YES    0
#define COMPRESS_NO     1

struct s_escpg_paper {
  char  *name;
  int   id;
};
static struct s_escpg_paper  escpg_paper_list[] = {
  {"A3", 13},   {"A4", 14},   {"A5", 15},   {"A6", 16},   
  {"a3", 13},   {"a4", 14},   {"a5", 15},   {"a6", 16},   
  {"B4", 24},   {"B5", 25},   {"LETTER", 30},   {"LEGAL", 32},
  {"b4", 24},   {"b5", 25},   {"letter", 30},   {"legal", 32},
  {NULL, -1}
};

static void  escpg_prologue(void);
static void  escpg_epilogue(void);

static int   send_bitmap(DVI_DEVICE,DVI_BITMAP,int,int);
static void  lbp_set_param(DVI_DEVICE);
static void  lbp_bitmap_compress_mode(DVI_DEVICE,int);
static void  lbp_reset_param(DVI_DEVICE);
static void  lbp_form_feed(DVI_DEVICE);
static void  lbp_mode_escp2escpage(DVI_DEVICE);
static void  lbp_mode_escpage2escp(DVI_DEVICE);
static void  lbp_send(DVI_DEVICE,char*,char*);
static void  lbp_send_nbytes(DVI_DEVICE,char*,int,char*);
static void  lbp_flush(DVI_DEVICE);



void 
dvi2escpg(char *dvi_file, long w, long h) 
{
  int page, ppp;
 
  escpg_prologue();

  pcol = 0;
  page = (param_page_rev == 0) ? param_page_from : param_page_to;
  do {
    if (param_verbose > 0){
      if (pcol > 70){
	fprintf(stderr, "\n");
	pcol = 0;
      }
      fprintf(stderr, "[%d] ", page);
      fflush(stderr);
      pcol += 4;
      ppp = page;
      while (ppp >= 10){
	pcol++;
	ppp /= 10;
      }
    }
    if (DVI_DRAW_PAGE(dvi, dev, page, 1.0) < 0)
      break;
    lbp_form_feed(dev);
    page = page + ((param_page_rev == 0) ? 1 : -1);
  } while ((param_page_from <= page) && (page <= param_page_to));

  escpg_epilogue();

  if (param_verbose > 0)
    fprintf(stderr, "\n");
}

static void
escpg_prologue(void)
{
  offset_x = param_offset_x * param_dpi;
  offset_y = param_offset_y * param_dpi;

  if (param_dvi2escpg_init_mode == ESCPG_INIT_MODE_ESCP)
    lbp_mode_escp2escpage(dev);
  lbp_reset_param(dev);
  lbp_set_param(dev);

}

static void
escpg_epilogue(void)
{
  lbp_flush(dev);
  lbp_reset_param(dev);
  if (param_dvi2escpg_init_mode == ESCPG_INIT_MODE_ESCP)
    lbp_mode_escpage2escp(dev);
}


void  
DEV_put_bitmap_escpg(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
		     int font_id, long key2, long code_point, long x, long y)
{
  char  escseq[256];
  int   bm_length_compress_yes, bm_length_compress_no;

  if (param_debug == 1)
    fprintf(stderr, "Put bitmap\n");

  sprintf(escseq, "%c%ldX", GS, x+offset_x);         /* Set CP x */
  lbp_send(dev, escseq, "CP x");
  sprintf(escseq, "%c%ldY", GS, y+offset_y);         /* Set CP y */
  lbp_send(dev, escseq, "CP y");

  bm_length_compress_yes = send_bitmap(NULL, bm, COMPRESS_YES, -1);
  bm_length_compress_no  = send_bitmap(NULL, bm, COMPRESS_NO,  -1);
  if (lbp_compress_mode == COMPRESS_YES){
    if ((bm_length_compress_no + 5) < bm_length_compress_yes)
      send_bitmap(dev, bm, COMPRESS_NO,  bm_length_compress_no);
    else 
      send_bitmap(dev, bm, COMPRESS_YES, bm_length_compress_yes);
  } else {
    if (bm_length_compress_no > (bm_length_compress_yes + 5))
      send_bitmap(dev, bm, COMPRESS_YES, bm_length_compress_yes);
    else 
      send_bitmap(dev, bm, COMPRESS_NO,  bm_length_compress_no);
  }
}

static int
send_bitmap(DVI_DEVICE dev, DVI_BITMAP bm, int compress_mode, int length)
{
  char  escseq[256], msg[256];
  int   yy, wb, bm_len;
  char  cmp[3];
  int   xx, r, cnt, t;

  if ((length >= 0) && (compress_mode != lbp_compress_mode))
    lbp_bitmap_compress_mode(dev, compress_mode);

  if (compress_mode == COMPRESS_YES){
    wb = (bm->width+7)/8;
    sprintf(escseq, "%c%d;%ld;%ld;%dbi{I",           /* Bitmap */
	    GS, length, bm->width, bm->height, 0);
    sprintf(msg, "Bitmap %ldbytes", wb*bm->height);
    if (length >= 0)
      lbp_send(dev, escseq, msg);
    bm_len = 0;
    for (yy = 0, r = 0; yy < bm->height; yy++, r += bm->raster){
      sprintf(msg, "  bitmap body %d/%ld", yy, bm->height);
      xx = r;
      while (xx < (r+wb)){
	if (xx == (r+wb-1)){
	  cmp[0] = cmp[1] = bm->bitmap[xx];
	  cmp[2] = 1;
	  if (length >= 0)
	    lbp_send_nbytes(dev, cmp, 3, NULL);
	  bm_len += 3;
	  xx++;
	} else if (bm->bitmap[xx] != bm->bitmap[xx+1]){
	  if (length >= 0)
	    lbp_send_nbytes(dev, (char *)&bm->bitmap[xx], 1, NULL);
	  bm_len += 1;
	  xx++;
	} else { /* (xx != (r+wb-1)) && (bm->bitmap[xx]==bm->bitmap[xx+1]) */
	  cmp[0] = cmp[1] = bm->bitmap[xx];
	  for (cnt = 2; (xx+cnt) < (r+wb); cnt++){
	    if (bm->bitmap[xx] != bm->bitmap[xx+cnt])
	      break;
	  }
	  xx = xx + cnt;
	  while (cnt > 0){
	    if ((t = cnt) > 255)
	      t = 255;
	    cnt = cnt - t;
	    cmp[2] = t;
	    if (length >= 0)
	      lbp_send_nbytes(dev, cmp, 3, NULL);
	    bm_len += 3;
	  }
	}
      }
    }
  } else {   /* compress_mode == COMPRESS_NO */
    wb = (bm->width+7)/8;
    if (length >= 0){
      sprintf(escseq, "%c%ld;%ld;%ld;%dbi{I",       /* Bitmap */
	      GS, bm->height * wb, bm->width, bm->height, 0);
      sprintf(msg, "Bitmap %ldbytes", wb*bm->height);
      lbp_send(dev, escseq, msg);
      for (yy = 0, r = 0; yy < bm->height; yy++, r += bm->raster){
	sprintf(msg, "  bitmap body %d/%ld", yy, bm->height);
	lbp_send_nbytes(dev, (char *)&bm->bitmap[r], wb, msg);
      }
    }
    bm_len = wb * bm->height;
  }
  return bm_len;
}

void  
DEV_put_rectangle_escpg(DVI_DEVICE dev, DVI dvi, 
			long x, long y, long w, long h)
{
  char  mode[32], rect[256];
  int   x0, y0;
 
  if (param_debug == 1)
    fprintf(stderr, "Put rectangle\n");
  x0 = x + offset_x;
  y0 = y + offset_y;
  sprintf(mode, "%c1dmG", GS);
  sprintf(rect, "%c%d;%d;%ld;%ld;%drG", GS, x0, y0, x0+w, y0+h, 0);
  lbp_send(dev, mode, "inside fill mode");           /* Inside fill mode */
  lbp_send(dev, rect, "Rectangle");                  /* Rectangle Fill */
  sprintf(mode, "%c0dmG", GS);
  lbp_send(dev, mode, "frame mode");                 /* Frame mode */
  lbp_send(dev, rect, "Rectangle");                  /* Rectangle Frame */
}

int   
DEV_poll_escpg(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  if (interrupted == 1)
    return -1;
  return 0;
}


static void  
lbp_form_feed(DVI_DEVICE dev)
{
  char  ff[2] = { 0x0c, '\0'};                       /* Flush page */

  if (param_debug == 1)
    fprintf(stderr, "Next page\n");
  lbp_send(dev, ff, "Form Feed");
  lbp_flush(dev);
}


static void  
lbp_set_param(DVI_DEVICE dev)
{
  int   id, i;
  char  escseq[256];

  id = 14;  /* A4, by default */
  for (i = 0; escpg_paper_list[i].name != NULL; i++){
    if (strcmp(escpg_paper_list[i].name, param_paper_name) == 0){
      id = escpg_paper_list[i].id;
      break;
    }
  }

  if (param_debug == 1)
    fprintf(stderr, "Set param\n");
  sprintf(escseq, "%c%dpsE", GS, id);    /* Paper Size */
  lbp_send(dev, escseq, "Paper Size");
  sprintf(escseq, "%c0;%smuE", GS,   
	  (param_dpi==600)?"0.12":"0.24");           /* Dimen & Resolution */
  lbp_send(dev, escseq, "Dimension");
  sprintf(escseq, "%c0;%d;%ddrE", GS, param_dpi, param_dpi);  /* Resolution */
  lbp_send(dev, escseq, "Resolution");
  sprintf(escseq, "%c0;%d;%d;%dlwG", GS, 1, 0, 0);   /* Line style */
  lbp_send(dev, escseq, "Line Style");
  lbp_send(dev, escseq, "Bitmap Compression");
  lbp_bitmap_compress_mode(dev, COMPRESS_YES);
}

static void 
lbp_bitmap_compress_mode(DVI_DEVICE dev, int compress_mode)
{
  char  escseq[256];

  if (compress_mode == COMPRESS_NO){
    sprintf(escseq, "%c%dbcI", GS, 0);
    lbp_compress_mode = COMPRESS_NO;
  } else {
    sprintf(escseq, "%c%dbcI", GS, 1);               /* Bitmap compression */
    lbp_compress_mode = COMPRESS_YES;
  }
  lbp_send(dev, escseq, "Compression Mode");
}

static void 
lbp_reset_param(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "Reset param\n");
  sprintf(escseq, "%crpE", GS);                      /* Parameter Reset */
  lbp_send(dev, escseq, "Param Reset");
}

static void  
lbp_mode_escp2escpage(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "ESC/P->ESC/Page");
  sprintf(escseq, "%c%c%c%c", ESC, 0x7a, 0, 0);      /* ESC/P -> ESC/Page */
  lbp_send_nbytes(dev, escseq, 4, "ESC/P->ESC/Page");
}

static void  
lbp_mode_escpage2escp(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "ESC/Page->ESC/P");
  sprintf(escseq, "%c1pmE", GS);                     /* ESC/Page -> ESC/P */
  lbp_send(dev, escseq, "ESC/Page->ESC/P");
  lbp_flush(dev);
}

static void
lbp_send(DVI_DEVICE dev, char *s, char *msg)
{
  lbp_send_nbytes(dev, s, strlen(s), msg);
}

static void
lbp_send_nbytes(DVI_DEVICE dev, char *s, int len, char *msg)
{
  int   m, m0;

  while (len > 0){
    if ((len + lbp_buffer_ptr) < sizeof(lbp_buffer))
      m0 = m = len;
    else
      m0 = m = sizeof(lbp_buffer) - lbp_buffer_ptr;
    while (m > 0){
      lbp_buffer[lbp_buffer_ptr++] = *(s++);
      m--;
    }
    len -= m0;
    if (lbp_buffer_ptr == sizeof(lbp_buffer))
      lbp_flush(dev);
  }
}

void 
lbp_flush(DVI_DEVICE dev)
{ 
  if (lbp_buffer_ptr == 0)
    return;
  if (param_debug == 1)
    fprintf(stderr, "Flush buffer\n");
  fwrite(lbp_buffer, sizeof(char), lbp_buffer_ptr, output_fp);
  lbp_buffer_ptr = 0;
  fflush(output_fp);
}

/*EOF*/
