/*
 * xldvi.c - a core module for xldvi
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/StringDefs.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>

#include "dvi-2_6.h"
#include "defs.h"
#include "cf-xldvi.h"
#include "compiled.h"
#include "resource.h"
#include "window.h"


Public int      current_page;
Public int      start_page;
Public int      min_page, max_page;  
Public double   shrink; 

#define CMD_NONE                0
#define CMD_EXPOSE              1
#define CMD_QUIT                2
#define CMD_NEXT_PAGE           3
#define CMD_PREV_PAGE           4
#define CMD_NEXT10_PAGE         5
#define CMD_PREV10_PAGE         6
#define CMD_ENLARGE             7
#define CMD_SHRINK              8
#define CMD_UP                  9
#define CMD_DOWN               10
#define CMD_LEFT               11
#define CMD_RIGHT              12
#define CMD_LEFTMOST           13
#define CMD_RIGHTMOST          14
#define CMD_BEGIN              15
#define CMD_END                16
#define CMD_WINDOW_ENLARGE     17
#define CMD_WINDOW_SHRINK      18

#define Ctl(ch)        ((ch)-0x40)    /* ASCII CODE ONLY */

Private int      draw_interrupted;
Private int      x_parse_event(XEvent*);
Private void     x_run_cmd(int);

Private XEvent    xevent;


Public void
main_loop(void)
{
  int  cmd;

  min_page     = 1;
  max_page     = dviobj->pages;
  current_page = 1;
  draw_interrupted = 0;
  for (;;){
    if (draw_interrupted == 0)
      XNextEvent(x_display, &xevent);
    if ((cmd = x_parse_event(&xevent)) != CMD_NONE){
      x_run_cmd(cmd);
      draw_interrupted = 0;
      goto_page(current_page);
    }
  }
}

Public int
DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;
  Bool        e;

  if (poll_type != DVI_POLL_PAGE)
    return 0;

  if ((--t) > 0)
    return 0;
  t = Resource.poll;
  e = XCheckWindowEvent(x_display, x_window, EVENT_MASK, &xevent);
  if (e == False) 
    return 0;
  if (x_parse_event(&xevent) == CMD_NONE)
    return 0;
  return 1;
}


Private int
x_parse_event(XEvent* xev)
{
  int    val;
  char   keyin[16];
  KeySym ks;

  val = CMD_NONE;
  switch (xev->type){
  case Expose:
    val = CMD_EXPOSE;    break;
  case ButtonPress:
    switch (xev->xbutton.button){
    case Button1:  
      val = CMD_PREV_PAGE;   break;
    case Button3:
      val = CMD_NEXT_PAGE;   break;
    }
    break;
  case ConfigureNotify:
    x_resized_window(xev->xconfigure.width, xev->xconfigure.height);
    break;
  case KeyPress:
    if (XLookupString(&(xev->xkey), keyin, sizeof(keyin), &ks, NULL) != 1){
      switch (ks){
      case XK_Up:
	val = CMD_UP;     break;
      case XK_Down:
	val = CMD_DOWN;   break;
      case XK_Right:
	val = CMD_RIGHT;  break;
      case XK_Left:
	val = CMD_LEFT;   break;
      case XK_Page_Up:
	val = CMD_PREV_PAGE;   break;
      case XK_Page_Down:
	val = CMD_NEXT_PAGE;   break;
      case XK_Begin: case XK_Home:
	val = CMD_BEGIN;  break;
      case XK_End:
	val = CMD_END;    break;
      }
      break;
    }
    switch (keyin[0]){
    case 'q': case 'Q': 
      val = CMD_QUIT;           break;
    case ' ': case Ctl('V'): case Ctl('M'): case Ctl('J'):
      val = CMD_NEXT_PAGE;      break;
    case 'b': case 'B': case 'v': case 'V': case Ctl('H'): case 0x7f:
      val = CMD_PREV_PAGE;      break;
    case '+': case ']':
      val = CMD_ENLARGE;        break;
    case '-':case '[':
      val = CMD_SHRINK;         break;
    case '}':
      val = CMD_WINDOW_ENLARGE; break;
    case '{':
      val = CMD_WINDOW_SHRINK;  break;
    case 'k': case Ctl('P'):
      val = CMD_UP;             break;
    case 'j': case Ctl('N'):
      val = CMD_DOWN;           break;
    case 'h': case Ctl('B'):
      val = CMD_LEFT;           break;
    case 'l': case Ctl('F'):
      val = CMD_RIGHT;          break;
    case Ctl('A'):
      val = CMD_LEFTMOST;       break;
    case Ctl('E'):
      val = CMD_RIGHTMOST;      break;
    case '<':
      val = CMD_BEGIN;          break;
    case '>':
      val = CMD_END;            break;
    }
  }

  switch (val){
  case CMD_NEXT_PAGE:
    if (max_page <= current_page){
      XBell(x_display, 50);
      val = CMD_NONE;
    }
    break;
  case CMD_PREV_PAGE:
    if (current_page <= min_page){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_NEXT10_PAGE:
    if (max_page < (current_page + 10)){
      XBell(x_display, 50);
      val = CMD_NONE;
    }
    break;
  case CMD_PREV10_PAGE:
    if ((current_page - 10) < min_page){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_ENLARGE:
    if ((shrink/WINDOW_ENLARGE_RATIO) < SHRINK_MIN){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_SHRINK:
    if (SHRINK_MAX < (shrink*WINDOW_ENLARGE_RATIO)){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_UP:
    if (x_shift_view_y(-WIN_MOVE_Y, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_DOWN:
    if (x_shift_view_y(WIN_MOVE_Y, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_LEFT:
    if (x_shift_view_x(-WIN_MOVE_X, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_RIGHT:
    if (x_shift_view_x(WIN_MOVE_X, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_LEFTMOST:
    if (x_shift_view_x(-WIN_MOVE_X_MAX, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  case CMD_RIGHTMOST:
    if (x_shift_view_x(WIN_MOVE_X_MAX, 1) < 0){
      XBell(x_display, 50);    
      val = CMD_NONE;
    }
    break;
  }

  return val;
}

Private void
x_run_cmd(int cmd)
{
  switch (cmd){
  case CMD_EXPOSE:
    break;
  case CMD_QUIT:
    exit(0);
  case CMD_NEXT_PAGE:
    if ((current_page+1) <= max_page)
      current_page += 1;
    break;
  case CMD_PREV_PAGE:
    if (1 <= (current_page-1))
      current_page -= 1;
    break;
  case CMD_NEXT10_PAGE:
    if ((current_page+10) <= max_page)
      current_page += 10;
    break;
  case CMD_PREV10_PAGE:
    if (1 <= (current_page-10))
      current_page -= 10;
    break;
  case CMD_ENLARGE:
    if (SHRINK_MIN <= (shrink/WINDOW_ENLARGE_RATIO))
      change_shrink(shrink/WINDOW_ENLARGE_RATIO);
    break;
  case CMD_SHRINK:
    if ((shrink*WINDOW_ENLARGE_RATIO) <= SHRINK_MAX)
      change_shrink(shrink*WINDOW_ENLARGE_RATIO);
    break;
  case CMD_UP:
    x_shift_view_y(-WIN_MOVE_Y, 0);
    break;
  case CMD_DOWN:
    x_shift_view_y(WIN_MOVE_X, 0);
    break;
  case CMD_LEFT:
    x_shift_view_x(-WIN_MOVE_X, 0);
    break;
  case CMD_RIGHT:
    x_shift_view_x(WIN_MOVE_X, 0);
    break;
  case CMD_LEFTMOST:
    x_shift_view_x(-WIN_MOVE_X_MAX, 0);
    break;
  case CMD_RIGHTMOST:
    x_shift_view_x(WIN_MOVE_X_MAX, 0);
    break;
  case CMD_BEGIN:
    current_page = min_page;
    break;
  case CMD_END:
    current_page = max_page;
    break;
  case CMD_WINDOW_ENLARGE:
    x_window_enlarge();
    break;
  case CMD_WINDOW_SHRINK:
    x_window_shrink();
    break;
  case CMD_NONE:
  default:
    break;
  }
}

Public void
goto_page(int page)
{
  WIN_CACHE  wc;
  int        fin;

  if (max_page <= 0)
    return;
  if ((page < 1) || (max_page < page)){
    XBell(x_display, 50);
    return;
  }

  current_page = page;

  wc = x_cache_get();
  x_correct_view();

  switch (wc->flag){
  default:
  case WIN_CACHE_EMPTY:
    x_change_cursor(CURSOR_DRAWING);
    x_clear_window();
    fin = DVI_DRAW_PAGE(dviobj, dev, current_page, shrink);
    x_change_cursor(CURSOR_READY);
    break;
  case WIN_CACHE_INTERRUPTED:
    x_change_cursor(CURSOR_DRAWING);
    x_restore_current_page();
    fin = DVI_CONTINUE_TO_CONTEXT(dviobj, dev, wc->context, shrink);
    x_change_cursor(CURSOR_READY);
    break;
  case WIN_CACHE_COMPLETED:
    x_restore_current_page();
    fin = DVI_DRAW_OK;
    break;
  }

  switch (fin){
  default:
  case DVI_DRAW_ERROR:
  case DVI_DRAW_OK:
  case DVI_DRAW_LAST_PAGE:
    wc->flag = WIN_CACHE_COMPLETED;
    if (wc->context != NULL)
      DVI_FREE_CONTEXT(dviobj, dev, wc->context);
    wc->context = NULL;
    break;
  case DVI_DRAW_INTERRUPTED:
    wc->flag = WIN_CACHE_INTERRUPTED;
    if (wc->context != NULL)
      DVI_FREE_CONTEXT(dviobj, dev, wc->context);
    wc->context = DVI_GET_CURRENT_CONTEXT(dviobj, dev);
    draw_interrupted = 1;
    break;
  }

  x_put_border();
  XFlush(x_display);
}


Public void
set_shrink(double s)
{
  shrink = s;
  x_offset_x = Resource.dpi * Resource.offset_x / shrink;
  x_offset_y = Resource.dpi * Resource.offset_y / shrink;
}

Public void
change_shrink(double s)
{
  set_shrink(s);
  x_clear_window();
  x_cache_reset();
}

/*EOF*/
